/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2006, 2007, 2008, 2009, 2010  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-02-10 06:50:30 -0700 (Wed, 10 Feb 2010) $
--   $Revision: 4562 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "chunk_info.h"
#include "config.h"
#include "header.h"
#include "log.h"
#include "objectify.h"     /* define nwos_create_root */
#include "user_config.h"


extern void nwos_set_private_objects_path(const char* path);


/**********************************************************/
/* Determine if path is a regular file or a block device. */
/**********************************************************/

bool is_device(const char* path)
{
    assert(path[0] == '/');   /* must be a full path specified */

    return path[1] == 'd' && path[2] == 'e' && path[3] == 'v' && path[4] == '/';
}


/***************************************************************/
/* figure out how many blocks are available on a block device. */
/***************************************************************/

unsigned max_blocks_on_device(const char* path)
{
    int fd;
    off_t lower = 0LL;
    off_t upper = 1099511627776LL;
    off_t middle;
    off_t result;
    unsigned num_sectors;


    fd = open(path, O_RDONLY);

    if (fd < 0)
    {
	perror(path);
	exit(1);
    }

    while ((upper - lower) > 512)
    {
	middle = lower + (upper - lower) / 2;

	result = lseek(fd, middle, SEEK_SET);

	/* printf("%zd: %zd\n", middle, result); */

	if (result < 0)
	{
	    upper = middle;
	}
	else
	{
	    lower = middle;
	}
    }

    close(fd);

    num_sectors = (unsigned) (lower / 512);

    if (((off_t)num_sectors * 512) != lower)
    {
	printf("something is wrong, not a multiple of 512 bytes: %lld\n", (int64)lower);
    }

    return num_sectors * 2;
}


/****************************************************/
/* Print the usage information, is can be different */
/* depending upon if writing to a file or device.   */
/****************************************************/

void print_usage(char* name)
{
    fprintf(stderr, "usage: %s [--no-private] [archive]\n\n", name);

    fprintf(stderr, "  if no archive is specified it uses the default archive.\n");
    fprintf(stderr, "  --no-private: don't create the initial private blocks (if importing old data)\n\n");
}


static void get_response(char* buffer, size_t bufsize)
{
    if (fgets(buffer, bufsize, stdin) == NULL)
    {
	fputc('\n', stderr);

	if (feof(stdin))
	{
	    fprintf(stderr, "ERROR: received end of input from standard input when reading response\n");
	    exit(1);
	}
	else
	{
	    perror("reading responce");
	    exit(1);
	}
    }
}


void goodbye()
{
    fprintf(stderr, "\n");
    fprintf(stderr, "I'm sorry you don't wish to use this software.  If you have any comments,\n");
    fprintf(stderr, "questions, or concerns please e-mail me at `j.scott.edwards.nwos@gmail.com'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-------->   Program terminated without preparing disk!   <--------\n");
    fprintf(stderr, "\n");
    exit(1);
}


void agree_no_warranty()
{
    char buffer[16];

                  /*          1         2         3         4         5         6         7         8 */
                  /* 12345678901234567890123456789012345678901234567890123456789012345678901234567890 */
    fprintf(stderr, "\n");
    fprintf(stderr, "Objectify version %s\n", PACKAGE_VERSION);
    fprintf(stderr, "\n");
    fprintf(stderr, "Copyright (C) 2004-2010  J. Scott Edwards - QRW Software\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY; THE ENTIRE RISK AS TO THE\n");
    fprintf(stderr, "QUALITY AND PERFORMANCE OF THIS PROGRAM IS WITH YOU.  For details read the\n");
    fprintf(stderr, "`GNU General Public License' described in the file `LICENSE', which should\n");
    fprintf(stderr, "have been distributed with this software.  If it is not included, it can be\n");
    fprintf(stderr, "read online at `http://www.gnu.org/copyleft/gpl.html'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This is free software, and you are welcome to redistribute it under certain\n");
    fprintf(stderr, "conditions; also see the `LICENSE' file for details.  If you have any\n");
    fprintf(stderr, "comments or questions you can e-mail me at `qrwsoftware@gmail.com'\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you wish to use this program, enter \"I agree\" at the prompt.  By doing\n");
    fprintf(stderr, "so you agree to the terms specified in the GNU General Public License.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you agree with the license terms of this software? ");
    fflush(stderr);

    get_response(buffer, sizeof(buffer));

    if (strcmp(buffer, "I agree\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


void understand_security()
{
    char buffer[16];

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Please be aware that if you forget or lose the pass phrase, it is unlikely you\n");
    fprintf(stderr, "will be able to recover your data.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you still wish to use this program, enter \"I understand\" at the prompt.\n");
    fprintf(stderr, "By doing so you acknowledge that if you forget or otherwise lose the pass\n");
    fprintf(stderr, "phrase, YOUR DATA WILL BE LOST FOREVER.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you understand this? ");
    fflush(stderr);

    get_response(buffer, sizeof(buffer));

    if (strcmp(buffer, "I understand\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


/***************************************************************************/
/* Program to prepare a disk or file for NWOS storage.                     */
/* Now there is a bit map of used blocks as the first block in every chunk */
/* 8192 * 8 = 65536 blocks * 256 bytes per block = 16777216 bytes in chunk */
/***************************************************************************/

int main(int argc, char* argv[])
{
    int obj_fd;
    const char* private_path;
    struct flock lock;
    uint8 *chunk;
    char answer[8];
    bool is_file = false;
    bool do_private_blocks = true;

    uint32 total_blocks;
    uint32 total_chunks;
    uint32 blocks = 0;
    Disk_Header disk_header;
    char msg[128];
    char* p;
#if 0
    ObjRef root_object_reference;
    bool upgrade = false;
//    struct stat stat_struct;
#endif

    printf("\n");

    nwos_log_arguments(argc, argv);


    /**************************/
    /* sort out the arguments */
    /**************************/


    if (argc == 1)   /* no option or archive specified */
    {
    }
    else if (argc == 2)
    {
	if (*argv[1] == '-')
	{
	    if (strcmp(argv[1], "--no-private") != 0)
	    {
		fprintf(stderr, "\nunknown option: %s\n\n", argv[1]);
		print_usage(argv[0]);
		exit(1);
	    }

	    do_private_blocks = false;
	}
	else
	{
	    nwos_set_private_objects_path(argv[1]);
	}
    }
    else if (argc == 3)
    {
	if (strcmp(argv[1], "--no-private") != 0 || *argv[2] == '-')
	{
	    print_usage(argv[0]);
	    exit(1);
	}

	do_private_blocks = false;
	nwos_set_private_objects_path(argv[2]);
    }
    else
    {
	print_usage(argv[0]);
	exit(1);
    }

    private_path = nwos_get_private_objects_path();
       
    if (!is_device(private_path))
    {
	is_file = true;

	/* NOTE: this is a bad thing to do, it temporarily modifies */
	/* the string returned by nwos_get_private_objects_path.    */
        /* We cheat the system by calling strchr.                   */

	p = strchr(private_path, '/');
	assert(p != NULL);
	p = strchr(p + 1, '/');
	while (p != NULL)
	{
	    *p = '\0';
	    if (mkdir(private_path, 0755) != 0 && errno != EEXIST)
	    {
		perror(p);
		exit(1);
	    }
	    *p = '/';

	    p = strchr(p + 1, '/');
	}
    }


    /* convert arguments */


    if (!is_file)
    {
	total_blocks = max_blocks_on_device(private_path);   /* zero means unlimited */

	total_chunks = total_blocks / BLOCKS_IN_CHUNK - 1;   /* one chunk is for the system (header and chunk info table) */
    }
    else
    {
	total_chunks = 1;	     /* start with enough space for one private chunk */
    }

    if (getenv(TEST_ENVIRONMENT_VARIABLE) == NULL)
    {
	agree_no_warranty();   /* make sure he or she agrees with the licensing */
    }

    snprintf(msg, sizeof(msg), "total chunks: %u  private chunks: %u", total_chunks + 1, total_chunks);
    printf("\n%s\n", msg);


    chunk = malloc(CHUNK_SIZE);

    if (chunk == NULL)
    {
	perror("allocating chunk");
	exit(1);
    }

    printf("\n");
    printf("WARNING: all of the contents of %s will be ERASED!!!\n", private_path);
    printf("\n");
    printf("Do you want to continue? (enter `yes' to erase %s) ", private_path);

    fflush(stdout);

    get_response(answer, sizeof(answer));

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }


    /***************************/
    /* Open the storage device */
    /***************************/

    if (is_file)
    {
	obj_fd = open(private_path, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    }
    else
    {
	obj_fd = open(private_path, O_WRONLY);    /* don't need large file if not seeking? */
    }

    if (obj_fd < 0)
    {
	perror(private_path);
	exit(1);
    }


    lock.l_type = F_WRLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_fd, F_SETLK, &lock) != 0)
    {
	perror(private_path);
	exit(1);
    }



    /*********************************/
    /* finally write the disk header */
    /*********************************/

    memset(&disk_header, 0, sizeof(disk_header));
    
    memcpy(disk_header.magic_number, MAGIC_NUMBER, 4);
    memcpy(disk_header.version_string, VERSION_STRING, 4);

    if (is_file)
    {
	memcpy(disk_header.type_code, TYPE_CODE_FILE, 4);
    }
    else
    {
	memcpy(disk_header.type_code, TYPE_CODE_DISK, 4);
    }

    nwos_get_time_stamp(disk_header.last_prep_disk);

    memcpy(disk_header.last_change, disk_header.last_prep_disk, sizeof(disk_header.last_change));

    nwos_uint32_to_4_uint8(&total_chunks, disk_header.total_chunks);


    /* set the number of used blocks to 0 */
    blocks = 0;
    nwos_uint32_to_4_uint8(&blocks, disk_header.used_blocks);

    /* set the number of used chunks to 0 */
    blocks = 0;
    nwos_uint32_to_4_uint8(&blocks, disk_header.used_chunks);

    /* for now set the chunk offset to 1 chunk (for the header) */
    blocks = BLOCKS_IN_CHUNK;
    nwos_uint32_to_4_uint8(&blocks, disk_header.block_offset_to_chunks);

    memset(chunk, 0, CHUNK_SIZE);

    memcpy(chunk, &disk_header, sizeof(disk_header));

    /* if (write(obj_fd, chunk, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE) */
    if (write(obj_fd, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	perror(private_path);
	exit(1);
    }

    /*************************************/
    /* write private portion of the disk */
    /*************************************/

    if (is_file)
    {
	memset(chunk, 0, CHUNK_SIZE);

	if (write(obj_fd, chunk, CHUNK_SIZE) != CHUNK_SIZE)
	{
	    perror(private_path);
	    exit(1);
	}
    }

    if (close(obj_fd) != 0)
    {
	perror(private_path);
	exit(1);
    }

    free(chunk);

    snprintf(msg, sizeof(msg), "   private blocks: %9u  used: %9u", total_chunks * USABLE_BLOCKS_PER_CHUNK, blocks);
    nwos_log(msg);

    if (do_private_blocks)
    {
	if (getenv(TEST_ENVIRONMENT_VARIABLE) == NULL)
	{
	    understand_security();   /* make sure he or she understands the risks */
	}

	nwos_create_root();
    }

    return 0;
}

