/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-21 05:54:56 -0700 (Thu, 21 Jan 2010) $
--   $Revision: 4470 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include "crc32.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "strlcxx.h"


#define MAX_NAME_CHANGES 1024


typedef struct {
  char* old_name;
  char* new_name;
} NameChange;


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void free_name_list(NameChange* name_list)
{
    int i;

    for (i = 0; i < MAX_NAME_CHANGES; i++)
    {
	if (name_list[i].old_name != NULL) nwos_free(name_list[i].old_name);
	if (name_list[i].new_name != NULL) nwos_free(name_list[i].new_name);
    }

    nwos_free(name_list);
}


NameChange* read_list_file(char* list_file_name)
{
    FILE* fp;
    NameChange* result = NULL;
    int line_num = 0;
    char line[256 * 2];           /* for now max length is 255 each - need to fix! */
    char* newline;
    char* tab;
    size_t old_length;
    size_t new_length;
    char* old_name;
    char* new_name;
    int i;
    int j;
    bool we_have_a_problem_houston = false;


    fp = fopen(list_file_name, "r");

    if (fp == NULL)
    {
	perror(list_file_name);
    }
    else
    {
	result = nwos_malloc(MAX_NAME_CHANGES * sizeof(NameChange));

	memset(result, 0, MAX_NAME_CHANGES * sizeof(NameChange));

	while (fgets(line, sizeof(line), fp) != NULL)
	{
	    line_num++;

	    if (line_num > MAX_NAME_CHANGES)
	    {
		fprintf(stderr, "File contains too many lines, maximum allowed is: %d\n", MAX_NAME_CHANGES);
		we_have_a_problem_houston = true;
	    }

	    newline = strchr(line, '\n');

	    if (newline == NULL)  
	    {
		fprintf(stderr, "line %d is too long, ignored\n", line_num);
		we_have_a_problem_houston = true;

		while (fgets(line, sizeof(line), fp) != NULL)
		{
		    newline = strchr(line, '\n');
		    if (newline != NULL) break;
		}    

		if (newline == NULL) break;    /* loop terminated because of error or end of file */
	    }
	    else
	    {
		tab = strchr(line, '\t');

		if (tab == NULL)
		{
		    fprintf(stderr, "line %d is missing tab\n", line_num);
		    we_have_a_problem_houston = true;
		}
		else if (result != NULL)
		{
		    *tab = '\0';   /* change the tab to a end of line */
		    *newline = '\0';   /* change new line to end of line */

		    if (strchr(tab + 1, '\t') != NULL)
		    {
			fprintf(stderr, "line %d has more than one tab\n", line_num);
			we_have_a_problem_houston = true;
		    }
		    else
		    {
			old_length = strlen(line);
			new_length = strlen(tab + 1);

			if (0 < old_length && old_length <= 255 && 0 < new_length && new_length <= 255)
			{
			    old_name = nwos_malloc(old_length + 1);
			    new_name = nwos_malloc(new_length + 1);

			    strlcpy(old_name, line, old_length + 1);
			    strlcpy(new_name, tab + 1, new_length + 1);

			    result[line_num].old_name = old_name;
			    result[line_num].new_name = new_name;
			}
			else
			{
			    if (old_length > 255) fprintf(stderr, "Old name %s is too long (255 max in this version), line %d\n", line, line_num);
			    if (new_length > 255) fprintf(stderr, "New name %s is too long (255 max in this version), line %d\n", line, line_num);
			    if (old_length > 255) fprintf(stderr, "Old name %s is empty, line %d\n", line, line_num);
			    if (new_length > 255) fprintf(stderr, "New name %s is empty, line %d\n", line, line_num);
			    we_have_a_problem_houston = true;
			}
		    }
		}
	    }
	}

	if (ferror(fp))
	{
	    perror(list_file_name);
	    we_have_a_problem_houston = true;
	}

	fclose(fp);
    }

    /************************/
    /* Check for duplicates */
    /************************/

    if (result != NULL)
    {
	for (i = 0; i < MAX_NAME_CHANGES; i++)
	{
	    if (result[i].old_name != NULL || result[i].new_name != NULL)
	    {
		assert(result[i].old_name != NULL && result[i].new_name != NULL);

		if (strcmp(result[i].old_name, result[i].new_name) == 0)
		{
		    fprintf(stderr, "Names are the same in line: %d\n", i);
		    we_have_a_problem_houston = true;
		}

		for (j = i + 1; j < MAX_NAME_CHANGES; j++)
		{
		    if (result[j].old_name != NULL || result[j].new_name != NULL)
		    {
			assert(result[j].old_name != NULL && result[j].new_name != NULL);

			if (strcmp(result[i].old_name, result[j].old_name) == 0 ||
			    strcmp(result[i].old_name, result[j].new_name) == 0 ||
			    strcmp(result[i].new_name, result[j].old_name) == 0 ||
			    strcmp(result[i].new_name, result[j].new_name) == 0)
			{
			    fprintf(stderr, "Duplicate names in line: %d and line: %d\n", i, j);
			    we_have_a_problem_houston = true;
			}
		    }
		}
	    }
	}
    }

    if (we_have_a_problem_houston)
    {
	free_name_list(result);
	result = NULL;
    }

    return result;
}


static char* file_is_used_in_disc_list(ObjRef* disc_list_class_ref, C_struct_Path_And_File_Association* assoc_obj)
{
    ObjRef object_class;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    static char id[sizeof(ptr_list_obj->id) + 1];
    char* result = NULL;


    ref_list = nwos_malloc_reference_list(&assoc_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, disc_list_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

	    memcpy(id, ptr_list_obj->id, sizeof(id));
	    id[sizeof(ptr_list_obj->id)] = '\0';

	    result = id;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


/**************************/
/* Check the names are ok */
/**************************/

bool names_are_okay(char* old_name, char* new_name)
{
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef old_path_ref;
    ObjRef list_class_ref;
    ObjRef dummy_ref;
    ObjRef old_assoc_ref;
    char *id;

    if (strlen(new_name) > 255)
    {
	fprintf(stderr, "Error: maximum length of file name is 255\n");
	return false;
    }

    if (!nwos_find_file_path(old_name, &old_path_ref))
    {
	fprintf(stderr, "Could not find: %s in system.\n", old_name);
	return false;
    }

    if (nwos_number_of_files_for_path(&old_path_ref) > 1)
    {
	fprintf(stderr, "This version cannot change %s because there is more than one file referencing it.\n", old_name);
	return false;
    }

    if (nwos_find_file_path(new_name, &dummy_ref))
    {
	fprintf(stderr, "Error: %s already exists in the system\n", new_name);
	return false;
    }


    assert(nwos_read_variable_sized_object_from_disk(&old_path_ref, kludge, sizeof(kludge), &get_path_object_size));

    assert(nwos_file_path_to_path_and_file_association(&old_path_ref, 0, &old_assoc_ref));

    assert(nwos_read_object_from_disk(&old_assoc_ref, &assoc_obj, sizeof(assoc_obj)));

    if (!is_void_reference(&assoc_obj.header.object.next_version))
    {
	fprintf(stderr, "The file that had the name %s has already been renamed.\n", old_name);
	return false;
    }


    /******************************************************/
    /* before changing verify it is NOT in any disc lists */
    /******************************************************/

    if (nwos_find_private_class_definition("DISC LIST", &list_class_ref))
    {
	id = file_is_used_in_disc_list(&list_class_ref, &assoc_obj);

	if (id != NULL)   /* the file was found in disc_list with this id */
	{
	    fprintf(stderr, "\nError: found file %s in disc list: %s\n", old_name, id);
	    return false;
	}
    }

    assert(is_same_object(&ptr_path_obj->header.common.id, &old_path_ref));
    assert(is_same_object(&assoc_obj.header.common.id, &old_assoc_ref));

    return true;
}


/*************************************************************************************/
/* Create new path and association objects for this file, but keep the old ones too. */
/*************************************************************************************/

void create_new_path(char* old_path, char* new_path)
{
#ifdef VERIFY_WRITE
    C_struct_Path_And_File_Association temp;
#endif
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef old_path_ref;
    ObjRef new_path_ref;
    ObjRef old_assoc_ref;
    ObjRef new_assoc_ref;


    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    assert(nwos_find_file_path(old_path, &old_path_ref));
    assert(!nwos_find_file_path(new_path, &new_path_ref));

    assert(nwos_number_of_files_for_path(&old_path_ref) == 1);

    assert(nwos_file_path_to_path_and_file_association(&old_path_ref, 0, &old_assoc_ref));

    assert(nwos_read_object_from_disk(&old_assoc_ref, &assoc_obj, sizeof(assoc_obj)));


    /***********************/
    /* Create the new path */
    /***********************/

    assert(nwos_create_file_path(new_path, &new_path_ref) == CREATED_NEW);

    nwos_generate_new_id(&new_assoc_ref);


    /***********************************************************************/
    /* Point to the new association object from the old association object */
    /***********************************************************************/

    assert(is_void_reference(&assoc_obj.header.object.next_version));   /* above code checks this */

    copy_reference(&assoc_obj.header.object.next_version, &new_assoc_ref);

    nwos_crc32_calculate((uint8*) &assoc_obj.header.object, sizeof(ObjectHeader), assoc_obj.header.common.header_chksum);

    nwos_overwrite_object_to_disk(&old_assoc_ref, &assoc_obj, sizeof(assoc_obj));

#ifdef VERIFY_WRITE
    /* read back to verify */
    assert(nwos_read_object_from_disk(&old_assoc_ref, &temp, sizeof(temp)));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);
#endif

    void_reference(&assoc_obj.header.object.next_version);   /* remove it before creating the new */


    /*****************************************/
    /* Now create the new association object */
    /*****************************************/

    nwos_fill_in_common_header(&assoc_obj.header.common, &new_assoc_ref, &assoc_class_ref);

    copy_reference(&assoc_obj.header.object.prev_version, &old_assoc_ref);

    copy_reference(&assoc_obj.path, &new_path_ref);

    nwos_create_reference_list(&new_assoc_ref, &assoc_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &assoc_obj.header.object, sizeof(ObjectHeader), assoc_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&new_assoc_ref, &assoc_obj, sizeof(assoc_obj));

    nwos_add_to_references(&new_assoc_ref, &assoc_class_ref);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj.path);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj.file);

#ifdef VERIFY_WRITE
    /* read back to verify */
    assert(nwos_read_object_from_disk(&new_assoc_ref, &temp, sizeof(temp)));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);
#endif
}


/**************************************/
/* Rename without saving the old name */
/**************************************/

void change_old_path(char* old_path, char* new_path)
{
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    size_t length;
    ObjRef old_path_ref;
    ObjRef new_path_ref;

    assert(nwos_find_file_path(old_path, &old_path_ref));
    assert(!nwos_find_file_path(new_path, &new_path_ref));

    assert(nwos_number_of_files_for_path(&old_path_ref) == 1);

    assert(nwos_read_variable_sized_object_from_disk(&old_path_ref, kludge, sizeof(kludge), &get_path_object_size));

    length = strlen(new_path);

    ptr_path_obj->count = length;

    memcpy(ptr_path_obj->storage, new_path, length);

    nwos_crc32_calculate((uint8*) &ptr_path_obj->count, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

    nwos_overwrite_object_to_disk(&old_path_ref, ptr_path_obj, get_path_object_size(ptr_path_obj));
}


int main(int argc, char* argv[])
{
    char* old_name = "-";
    char* new_name = "-";
    char* list_file_name = NULL;
    bool keep_history = false;
    bool names_okay;
    NameChange* name_change_list = NULL;
    int i;
    uint32 blocks_needed;


    if (argc == 4 && strcmp(argv[1], "--keep-history") == 0)
    {
	keep_history = true;
	old_name = argv[2];
	new_name = argv[3];
    }
    else if (argc == 4 && strcmp(argv[1], "--no-history") == 0)
    {
	old_name = argv[2];
	new_name = argv[3];
    }

    if (strcmp(old_name, "-f") == 0)
    {
	list_file_name = new_name;
	old_name = "";
	new_name = "";
    }

    if (*old_name == '-' || *new_name == '-')
    {
	fprintf(stderr, "\n");
	fprintf(stderr, "usage: %s --no-history | --keep-history old-file-name new-file-name\n", argv[0]);
	fprintf(stderr, "       %s --no-history | --keep-history -f list_of_files_to_change.txt\n", argv[0]);
	fprintf(stderr, "\n");
	fprintf(stderr, "The list_of_files_to_change.txt file has two file names per line, separated\n");
	fprintf(stderr, "by a tab.  In this version of the software, the maximum number of files that\n");
	fprintf(stderr, "can be changed at one time is: %u\n", MAX_NAME_CHANGES);
	fprintf(stderr, "\n");
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    printf("\n");

    if (list_file_name != NULL)
    {
	name_change_list = read_list_file(list_file_name);

	if (name_change_list == NULL)
	{
	    fprintf(stderr, "\n");
	    exit(1);
	}
    }

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);


    if (keep_history)
    {
	blocks_needed = 4;

	if (name_change_list != NULL)
	{
	    for (i = 0; i < MAX_NAME_CHANGES; i++)
	    {
		if (name_change_list[i].old_name != NULL || name_change_list[i].new_name != NULL)
		{
		    assert(name_change_list[i].old_name != NULL && name_change_list[i].new_name != NULL);
		    blocks_needed += 4;
		}
	    }
	}

	if (!nwos_check_blocks_available(blocks_needed))
	{
	    fprintf(stderr, "Cannot rename with history (could rename with --no-history option).\n");
	    nwos_terminate_objectify();
	    exit(1);
	}
    }


    if (name_change_list == NULL)
    {
	names_okay = names_are_okay(old_name, new_name);

	if (names_okay)
	{
	    if (keep_history)
	    {
		create_new_path(old_name, new_name);
	    }
	    else
	    {
		change_old_path(old_name, new_name);
	    }
	}
    }
    else
    {
	for (i = 0; i < MAX_NAME_CHANGES; i++)
	{
	    if (name_change_list[i].old_name != NULL || name_change_list[i].new_name != NULL)
	    {
		assert(name_change_list[i].old_name != NULL && name_change_list[i].new_name != NULL);

		printf("%s -> %s\n", name_change_list[i].old_name, name_change_list[i].new_name);

		if (!names_are_okay(name_change_list[i].old_name, name_change_list[i].new_name)) break;
	    }
	}

	names_okay = (i == MAX_NAME_CHANGES);         /* all names were okay */

	if (names_okay)
	{
	    for (i = 0; i < MAX_NAME_CHANGES; i++)
	    {
		if (name_change_list[i].old_name != NULL || name_change_list[i].new_name != NULL)
		{
		    assert(name_change_list[i].old_name != NULL && name_change_list[i].new_name != NULL);

		    printf("%s ->> %s\n", name_change_list[i].old_name, name_change_list[i].new_name);

		    if (keep_history)
		    {
			create_new_path(name_change_list[i].old_name, name_change_list[i].new_name);
		    }
		    else
		    {
			change_old_path(name_change_list[i].old_name, name_change_list[i].new_name);
		    }
		}
	    }
	}

	free_name_list(name_change_list);
    }

    nwos_terminate_objectify();

    printf("\n");

    return names_okay == false;
}


