/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-29 08:44:14 -0600 (Wed, 29 Jul 2009) $
--   $Revision: 4202 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "../gnu/md5.h"

#include "../chunk_info.h"
#include "../header.h"


static void print_bit_map(uint8* bit_map)
{
    int j;

    printf("bit map:\n");

    for (j = 0; j < BIT_MAP_BYTES; j++)
    {
	if (j % 32 == 0)
	{
	    printf("%d: %02x", j, bit_map[j]);
	}
	else if (j % 32 == 31)
	{
	    printf("\n");
	}
	else
	{
	    printf("%02x", bit_map[j]);
	}
    }

    printf("\n");
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    struct flock lock;
    const char * error_msg;
    char ref_str[128];
    uint8 *chunk;
    int i;
    int j;
    size_t bytes_read;
    int bit_num;
    int byte_num;
    uint64 private_blocks_free = 0;
    uint32 chunk_num;
    uint32 used_blocks;         /* blocks used in a chunk */
    uint64 ref;
    Chunk_Info* chunk_info;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    bool bad_chunk = false;

    if (argc != 2)
    {
	fprintf(stderr, "%s: archive\n", argv[0]);
	exit(1);
    }


    obj_file_desc = open(argv[1], O_RDWR);

    if (obj_file_desc < 0)
    {
	perror(argv[1]);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(argv[1]);
	exit(1);
    }


    /* allocate buffer for reading chunks */

    chunk = malloc(CHUNK_SIZE);

    if (chunk == NULL)
    {
	perror("allocating memory for chunk");
	exit(1);
    }

    if (read(obj_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	snprintf(ref_str, sizeof(ref_str), "reading first chunk from: %s", argv[1]);
	perror(ref_str);
	close(obj_file_desc);
	exit(1);
    }

    error_msg = nwos_load_private_data(chunk, CHUNK_SIZE, false);   /* don't allow compressed files */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[1]);
	exit(1);
    }


    printf("Last change: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c\n",
	   nwos_extract_year_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_month_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_day_of_month_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_hour_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_minute_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_second_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_private_disk_header.version_string[0], nwos_private_disk_header.version_string[1],
	   nwos_private_disk_header.version_string[2], nwos_private_disk_header.version_string[3]);

    printf("total_blocks: %10llu  total_chunks: %5u\n", (uint64)nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK, nwos_total_private_chunks);
    printf("used_blocks:  %10llu  used_chunks:  %5u\n", nwos_used_private_blocks, nwos_used_private_chunks);


    chunk_info = malloc(nwos_total_private_chunks * sizeof(Chunk_Info));

    if (chunk_info == NULL)
    {
	perror("allocating memory for chunk_info table");
	exit(1);
    }

    nwos_used_private_blocks = 0;


    /*****************************/
    /* now do the private blocks */
    /*****************************/

    for (chunk_num = 0; chunk_num < nwos_total_private_chunks; chunk_num++)
    {
	bytes_read = read(obj_file_desc, chunk, CHUNK_SIZE);
	used_blocks = 0;

	memset(&chunk_info[chunk_num], 0, sizeof(Chunk_Info));

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */

	for (i = BIT_MAP_BYTES; i < bytes_read; i += FILE_BLOCK_SIZE)
	{
	    for (j = 0; j < FILE_BLOCK_SIZE; j++)
	    {
		if (chunk[i+j] != 0) break;
	    }

	    byte_num = ((i - BIT_MAP_BYTES) / FILE_BLOCK_SIZE) / 8;
	    bit_num = ((i - BIT_MAP_BYTES) / FILE_BLOCK_SIZE) % 8;

	    if (j < FILE_BLOCK_SIZE)   /* block not empty */
	    {
		nwos_used_private_blocks++;
		used_blocks++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) == 0)
		{
		    printf("bit clear that shouldn't be, chunk: %d  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);

		    print_bit_map(chunk);

		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");

		    bad_chunk = true;
		    break;
		}

		nwos_8_uint8_to_uint64(&chunk[i], &ref);

		if (chunk_info[chunk_num].ref == 0LL)
		{
		    chunk_info[chunk_num].ref = ref & 0xffffffffffff0000ULL;
		    chunk_info[chunk_num].index = chunk_num;
		}
		else
		{
		    if (chunk_info[chunk_num].ref != (ref & 0xffffffffffff0000ULL))
		    {
			printf("block reference %llx doesn't match chunk %llx\n", ref, chunk_info[chunk_num].ref);
			bad_chunk = true;
			break;
		    }
		}

		md5_process_bytes(&chunk[i], FILE_BLOCK_SIZE, &md5_context);    /* include this data in the md5 checksum */
	    }
	    else
	    {
		private_blocks_free++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) != 0)
		{
		    printf("bit set that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);

		    print_bit_map(chunk);

		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");

		    bad_chunk = true;
		    break;
		}
	    }
	}

	if (bad_chunk || used_blocks == 0) break;    /* if bit map didn't match blocks in chunk stop here */

	if (bytes_read != CHUNK_SIZE)
	{
	    printf("last chunk read: %08zx\n", chunk_num + (bytes_read / FILE_BLOCK_SIZE));
	    break;
	}

	chunk_info[chunk_num].flags_used = used_blocks;

	md5_finish_ctx(&md5_context, chunk_info[chunk_num].md5_digest);   /* finish computing the md5 sum */

	printf("%llx %u %u ", chunk_info[chunk_num].ref, chunk_info[chunk_num].flags_used, chunk_info[chunk_num].index);
	for (i = 0; i < MD5_DIGEST_SIZE; i++) printf("%02x", chunk_info[chunk_num].md5_digest[i]);
	printf("\n");
	fflush(stdout);
    }

    nwos_used_private_chunks = chunk_num;

    memset(chunk, 0, FILE_BLOCK_SIZE);
    nwos_store_private_data(chunk, FILE_BLOCK_SIZE, false);

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
#ifndef WORDS_BIGENDIAN
	chunk_info[i].ref = byteswap_uint64(chunk_info[i].ref);
	chunk_info[i].flags_used = byteswap_uint32(chunk_info[i].flags_used);
	chunk_info[i].index = byteswap_uint32(chunk_info[i].index);
#endif
    }

    if (lseek(obj_file_desc, 0LL, SEEK_SET) < 0)
    {
	perror(argv[1]);
    }

    if (write(obj_file_desc, chunk, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	perror(argv[1]);
	close(obj_file_desc);
	exit(1);
    }

    if (write(obj_file_desc, chunk_info, sizeof(Chunk_Info) * nwos_used_private_chunks) != sizeof(Chunk_Info) * nwos_used_private_chunks)
    {
	perror(argv[1]);
	close(obj_file_desc);
	exit(1);
    }

    close(obj_file_desc);

    printf("used_blocks:  %10llu  used_chunks:  %5u\n", nwos_used_private_blocks, nwos_used_private_chunks);

    return 0;
}

