/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-22 18:47:06 -0700 (Fri, 22 Jan 2010) $
--   $Revision: 4479 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file, the disk_io.c file, and the
--   objectify.c file, since this file was created with code taken from the
--   disk_io.c and the objectify.c files.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "bit_map.h"        /* define nwos_calculate_chunk_md5sum */
#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "gen_id.h"
#include "header.h"
#include "log.h"
#include "mem_alloc.h"


/* Warning: if the total_number_of_chunks changes, all of these have to be reallocated. */

#ifndef PUBLIC_MODE

Chunk_Info*    nwos_chunk_info;

static int*    chunk_info_reverse_index;   /* index from chunk index to info table */
static uint16* chunk_used_index;  /* order of chunks sorted by blocks used, fewest to most used */

static uint32 capacity;
static bool chunk_info_modified;


/*************************************************************************************/
/* This updates the chunk_info_reverse_index table to point to the chunk_info table. */
/*                                                                                   */
/*  chunk_info   reverse                                                             */
/*    0 [5]       0 [4]                                                              */
/*    1 [3]       1 [5]                                                              */
/*    2 [4]       2 [3]                                                              */
/*    3 [2]       3 [1]                                                              */
/*    4 [0]       4 [2]                                                              */
/*    5 [1]       5 [0]                                                              */
/*                                                                                   */
/*************************************************************************************/

static void update_chunk_info_reverse_index()
{
    int i;

    assert(nwos_chunk_info != NULL);
    assert(capacity > 0);

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(nwos_chunk_info[i].index < nwos_used_private_chunks);
	chunk_info_reverse_index[nwos_chunk_info[i].index] = i;
    }

    /* verify it is correct */
    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(nwos_chunk_info[chunk_info_reverse_index[i]].index == i);
    }
}


/****************************************/
/* Read the chunk_info table from disk. */
/****************************************/

void nwos_initialize_chunk_info(void)
{
    size_t chunk_info_size_in_bytes;
    uint64 used_blocks;
    int i;
    int j;
    int k;
    uint16 left;
    uint16 right;
    uint16 save;
    char log_msg[256];

    assert(capacity == 0);
    assert(nwos_chunk_info == NULL);
    assert(chunk_info_reverse_index == NULL);
    assert(chunk_used_index == NULL);

    capacity = nwos_total_private_chunks;   /* save the size of the chunk_info table */

    chunk_info_size_in_bytes = capacity * sizeof(Chunk_Info);
    nwos_chunk_info = nwos_malloc(chunk_info_size_in_bytes);

    nwos_read_chunk_info(nwos_chunk_info, chunk_info_size_in_bytes);

    used_blocks = 0;
    for (i = 0; i < (int)capacity; i++)
    {
#ifndef WORDS_BIGENDIAN
	nwos_chunk_info[i].ref = byteswap_uint64(nwos_chunk_info[i].ref);
	nwos_chunk_info[i].flags_used = byteswap_uint32(nwos_chunk_info[i].flags_used);
	nwos_chunk_info[i].index = byteswap_uint32(nwos_chunk_info[i].index);
#endif
	used_blocks += (nwos_chunk_info[i].flags_used & CHUNK_INFO_USED_MASK);

	if (nwos_chunk_info[i].ref > MAXIMUM_32_BIT_PRIVATE_REFERENCE)
	{
	    snprintf(log_msg, sizeof(log_msg), 
		     "WARNING: you have a chunk of data located at %08llx that is above %08x",
		     nwos_chunk_info[i].ref, MAXIMUM_PRIVATE_REFERENCE);
	    nwos_log(log_msg);
	    fprintf(stderr, "%s,\n", log_msg);
	    fprintf(stderr, "which is used for temporary storage in versions after Alpha_29.4.  Please\n"
		            "e-mail me at qrwsoftware@gmail.com for help fixing this.\n\n");
	    sleep(4);
	}
    }

    if (used_blocks != nwos_used_private_blocks)
    {
	snprintf(log_msg, sizeof(log_msg), 
		 "Warning: calculated sum of used blocks (%llu) doesn't match stored (%llu)",
		 used_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
	fprintf(stderr, "%s\n", log_msg);
    }

    chunk_info_modified = false;

    chunk_info_reverse_index = nwos_malloc(capacity * sizeof(int));

    update_chunk_info_reverse_index();

    /* now create the chunk_used_index */

    chunk_used_index = nwos_malloc(capacity * sizeof(uint16));

    /* insert them into a heap to do a heap sort */
    for (i = 0; i < (int)nwos_used_private_chunks; i++)
    {
	j = i + 1;
	while (j / 2 > 0 && (nwos_chunk_info[chunk_used_index[(j / 2) - 1]].flags_used & CHUNK_INFO_USED_MASK) < (nwos_chunk_info[i].flags_used & CHUNK_INFO_USED_MASK))
	{
	    /* parent is smaller, move it lower in the tree */
	    chunk_used_index[j - 1] = chunk_used_index[(j / 2) - 1];
	    j = j / 2;
	}

	chunk_used_index[j - 1] = i;
    }

    for (i = (int)nwos_used_private_chunks; i > 1; i--)
    {
	/* save the index at the top of the tree (smallest) */
	save = chunk_used_index[i - 1];

	/* move the root of the the (largest) to the top */
	chunk_used_index[i - 1] = chunk_used_index[0];

	j = 1;
	while (j < (i / 2) - 1)
	{
	    k = j * 2;

	    /* if there is a right child */
	    if (k < i - 1)
	    {
		left = nwos_chunk_info[chunk_used_index[k - 1]].flags_used & CHUNK_INFO_USED_MASK;
		right = nwos_chunk_info[chunk_used_index[k + 1 - 1]].flags_used & CHUNK_INFO_USED_MASK;

		if (left < right)
		{
		    k++;
		}
	    }

	    if ((nwos_chunk_info[save].flags_used & CHUNK_INFO_USED_MASK) >= (nwos_chunk_info[chunk_used_index[k-1]].flags_used & CHUNK_INFO_USED_MASK)) break;

	    chunk_used_index[j - 1] = chunk_used_index[k - 1];

	    j = k;
	}

	chunk_used_index[j - 1] = save;
    }
}


/*********************************************************************/
/* If the chunk_info table has been modified, write it back to disk. */
/*********************************************************************/

void nwos_terminate_chunk_info(void)
{
    int i;
    int j;
    uint64 used_blocks;
#ifndef PUBLIC_MODE
    size_t chunk_info_size_in_bytes;
#endif
    char log_msg[128];

    assert(nwos_chunk_info != NULL);

    /* first see if any of the chunks have been modified, but the MD5 checksum was not calculated. */

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	for (j = 0; j < MD5_DIGEST_SIZE; j++)
	{
	    if (nwos_chunk_info[i].md5_digest[j] != 0) break;
	}

	if (j == MD5_DIGEST_SIZE)
	{
	    nwos_calculate_chunk_md5sum(nwos_chunk_info[i].index, nwos_chunk_info[i].md5_digest);

	    nwos_clear_chunk_md5_verified(i);

	    chunk_info_modified = true;
	}
    }

    if (chunk_info_modified)
    {
	snprintf(log_msg, sizeof(log_msg), "  chunk index modified - new size: %u", nwos_used_private_chunks);
	nwos_log(log_msg);

	chunk_info_size_in_bytes = capacity * sizeof(Chunk_Info);

	used_blocks = 0;
	for (i = 0; i < (int)capacity; i++)
	{
	    used_blocks += (nwos_chunk_info[i].flags_used & CHUNK_INFO_USED_MASK);
#ifndef WORDS_BIGENDIAN
	    nwos_chunk_info[i].ref = byteswap_uint64(nwos_chunk_info[i].ref);
	    nwos_chunk_info[i].flags_used = byteswap_uint32(nwos_chunk_info[i].flags_used);
	    nwos_chunk_info[i].index = byteswap_uint32(nwos_chunk_info[i].index);
#endif
	}

	if (used_blocks != nwos_used_private_blocks)
	{
	    snprintf(log_msg, sizeof(log_msg), 
		     "Warning: calculated sum of used blocks (%llu) doesn't match stored (%llu)",
		     used_blocks, nwos_used_private_blocks);
	    nwos_log(log_msg);
	    fprintf(stderr, "%s\n", log_msg);
	}

	nwos_write_chunk_info(nwos_chunk_info, chunk_info_size_in_bytes);

	chunk_info_modified = false;
    }

    nwos_free(nwos_chunk_info);
    nwos_chunk_info = NULL;

    nwos_free(chunk_info_reverse_index);
    chunk_info_reverse_index = NULL;

    nwos_free(chunk_used_index);
    chunk_used_index = NULL;

    capacity = 0;
}


/******************************************/
/* Functions to read the block used count */
/******************************************/

uint32 nwos_blocks_used_in_chunk(int info_index)
{
    return nwos_chunk_info[info_index].flags_used & CHUNK_INFO_USED_MASK;
}


bool nwos_chunk_is_full(int info_index)
{
    return (nwos_chunk_info[info_index].flags_used & CHUNK_INFO_FULL_MASK) != 0;
}


/************************************************/
/* Functions to deal with the MD5 verified flag */
/************************************************/

bool nwos_chunk_md5_has_been_verified(int info_index)
{
    return (nwos_chunk_info[info_index].flags_used & CHUNK_INFO_MD5_VERIFIED_MASK) != 0;
}


void nwos_set_chunk_md5_verified(int info_index)
{
    nwos_chunk_info[info_index].flags_used |= CHUNK_INFO_MD5_VERIFIED_MASK;
}


void nwos_clear_chunk_md5_verified(int info_index)
{
    nwos_chunk_info[info_index].flags_used &= ~CHUNK_INFO_MD5_VERIFIED_MASK;
}


/********************************************/
/* Functions to update the block used count */
/********************************************/

void nwos_increment_chunk_info_block_used(int info_index)
{
    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert((nwos_chunk_info[info_index].flags_used & CHUNK_INFO_FULL_MASK) == 0);
    assert(nwos_chunk_info != NULL);

    nwos_chunk_info[info_index].flags_used++;

    chunk_info_modified = true;
}


void nwos_decrement_chunk_info_block_used(int info_index)
{
    assert(nwos_chunk_info != NULL);
    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert((nwos_chunk_info[info_index].flags_used & CHUNK_INFO_USED_MASK) != 0);

    nwos_chunk_info[info_index].flags_used--;

    chunk_info_modified = true;
}


void nwos_reset_chunk_info_blocks_used(int info_index, uint32 blocks_used)
{
    char log_msg[128];

    assert(nwos_chunk_info != NULL);
    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(0 <= blocks_used && blocks_used <= USABLE_BLOCKS_PER_CHUNK);

    snprintf(log_msg, sizeof(log_msg),
	     "Warning: blocks used in chunk mismatch - map: %u  stored: %u, updated stored to match.",
	     blocks_used, (nwos_chunk_info[info_index].flags_used & CHUNK_INFO_USED_MASK));
    nwos_log(log_msg);
    fprintf(stderr, "%s\n", log_msg);

    nwos_chunk_info[info_index].flags_used = (nwos_chunk_info[info_index].flags_used & ~CHUNK_INFO_USED_MASK) | (blocks_used & CHUNK_INFO_USED_MASK);

    chunk_info_modified = true;
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the chunk index passed in or */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

int nwos_chunk_index_to_info_index(uint16 chunk_index)
{
    assert(chunk_index < nwos_used_private_chunks);
    assert(nwos_chunk_info != NULL);

    return chunk_info_reverse_index[chunk_index];
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the reference passed in or   */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

int nwos_uint64_ref_to_info_index(uint64 ref)
{
    int lwr;
    int upr;
    int mid = 0;

    assert(nwos_chunk_info != NULL);
    assert(nwos_used_private_chunks > 0);
    assert(MINIMUM_32_BIT_PRIVATE_REFERENCE <= ref && ref <= MAXIMUM_32_BIT_PRIVATE_REFERENCE);

    lwr = 1;

    upr = nwos_used_private_chunks;

    while (lwr <= upr)
    {
	mid = (upr + lwr) / 2;

	if (ref < nwos_chunk_info[mid - 1].ref)
	{
	    upr = mid - 1;
	}
	else if (ref >= nwos_chunk_info[mid - 1].ref + USABLE_BLOCKS_PER_CHUNK)
	{
	    lwr = mid + 1;
	}
	else
	{
	    mid = mid - 1;  /* adjust to 0 based index */
	    break;
	}
    }

    if (lwr <= upr)
    {
	assert(0 <= mid && mid < nwos_used_private_chunks);
	assert(nwos_chunk_info[mid].ref <= ref && ref < nwos_chunk_info[mid].ref + USABLE_BLOCKS_PER_CHUNK);

	return mid;
    }

    return -1;   /* not found */
}


int nwos_uint32_ref_to_info_index(uint32 ref)
{
    assert(MINIMUM_PRIVATE_REFERENCE <= ref && ref <= MAXIMUM_PRIVATE_REFERENCE);

    return nwos_uint64_ref_to_info_index(BASE_32_BIT_PRIVATE + ref);
}

#endif


/*********************************************************************************************/
/* this function maps a uint32 reference to a block number                                   */
/*                                                                                           */
/* The new formula is to skip over the bit maps instead of leaving holes for them, so every  */
/* 32 bit number is available for use.                                                       */
/*********************************************************************************************/

uint32 nwos_hash_uint32_ref(uint32 ref)
{
#ifdef PUBLIC_MODE
    assert(ref < nwos_used_public_blocks);
    return ref;
#else
    int index;
    uint32 result = 0;
    uint32 chunk;
    uint32 offset;

    if (ref <= MAXIMUM_PUBLIC_REFERENCE)
    {
	result = ref;
    }
    else if (ref <= MAXIMUM_PRIVATE_REFERENCE && nwos_used_private_chunks > 0)
    {
	index = nwos_uint32_ref_to_info_index(ref);

	if (index >= 0)    /* found it */
	{
	    chunk = (uint32) nwos_chunk_info[index].index;

	    offset = (ref - MINIMUM_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK;

	    result = nwos_block_offset_to_chunks + (chunk * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS + offset;
	}

//	printf("total blocks: %u chunks: %u  chunk: %u  offset: %u  result: %u (%08x)\n",
//	       nwos_total_private_blocks, nwos_total_private_chunks, chunk, offset, result, result);
    }

    return result;
#endif
}


uint32 nwos_hash_ref(ObjRef* ref)
{
    uint32 uint32_ref;

    uint32_ref = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
                                  ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    return nwos_hash_uint32_ref(uint32_ref);
}


off_t nwos_ref_to_offset(ObjRef* ref)
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = nwos_hash_ref(ref);

    return hash == 0 ? (off_t) -1 : (off_t) hash << 8;
}


/************************************************************/
/* This function clears the MD5 sum in the chunk_info table */
/************************************************************/

#ifndef PUBLIC_MODE
void nwos_clear_chunk_digest(ObjRef* ref)
{
    uint32 uint32_ref;
    int info_index;

    uint32_ref = nwos_ref_to_word(ref);

    assert(MINIMUM_PRIVATE_REFERENCE <= uint32_ref && uint32_ref <= MAXIMUM_PRIVATE_REFERENCE);

    info_index = nwos_uint32_ref_to_info_index(uint32_ref);

    assert(0 <= info_index && info_index <= nwos_used_private_chunks);

    memset(nwos_chunk_info[info_index].md5_digest, 0, sizeof(nwos_chunk_info[info_index].md5_digest));
}


/*************************************/
/* This function creates a new chunk */
/*************************************/

void nwos_allocate_new_chunk(uint32 uint32_ref)
{
    uint64 ref;
    uint64 offset;
    int i;
    int j;

    assert(nwos_chunk_info != NULL);

    assert(nwos_used_private_chunks < nwos_total_private_chunks);

    assert(MINIMUM_PRIVATE_REFERENCE <= uint32_ref && uint32_ref <= MAXIMUM_PRIVATE_REFERENCE);

    assert(nwos_hash_uint32_ref(uint32_ref) == 0);

    ref = BASE_32_BIT_PRIVATE + uint32_ref;
 
    assert(MINIMUM_32_BIT_PRIVATE_REFERENCE <= ref && ref <= MAXIMUM_32_BIT_PRIVATE_REFERENCE);

    /* first make sure the total_private_chunks hasn't been increased (if it is a file). */

    if (capacity != nwos_total_private_chunks)
    {
	nwos_chunk_info = nwos_realloc(nwos_chunk_info, nwos_total_private_chunks * sizeof(Chunk_Info));

	chunk_info_reverse_index = nwos_realloc(chunk_info_reverse_index, nwos_total_private_chunks * sizeof(int));

	chunk_used_index = nwos_realloc(chunk_used_index, nwos_total_private_chunks * sizeof(uint16));

	for (i = capacity; i < nwos_total_private_chunks; i++)
	{
	    memset(&nwos_chunk_info[i], 0, sizeof(Chunk_Info));
	    chunk_info_reverse_index[i] = 0;
	    chunk_used_index[i] = 0;
	}

	capacity = nwos_total_private_chunks;
    }

    /* figure out where this ref goes in the chunk_info table */
    for (i = nwos_used_private_chunks - 1; i >= 0; i--)
    {
	if (nwos_chunk_info[i].ref < ref) break;

	nwos_chunk_info[i+1].ref = nwos_chunk_info[i].ref;
	nwos_chunk_info[i+1].flags_used = nwos_chunk_info[i].flags_used;
	nwos_chunk_info[i+1].index = nwos_chunk_info[i].index;
	for (j = 0; j < MD5_DIGEST_SIZE; j++) nwos_chunk_info[i+1].md5_digest[j] = nwos_chunk_info[i].md5_digest[j];
    }
    i++;

    offset = (uint32_ref - MINIMUM_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK;

    nwos_chunk_info[i].ref = ref - offset;

    assert(i == 0 || nwos_chunk_info[i-1].ref < nwos_chunk_info[i].ref);
    assert(i == nwos_used_private_chunks || nwos_chunk_info[i+1].ref > nwos_chunk_info[i].ref);

    /* make sure chunk_info.ref is an even multiple of USABLE_BLOCKS_PER_CHUNK */

    offset = nwos_chunk_info[i].ref - MINIMUM_32_BIT_PRIVATE_REFERENCE;

    nwos_chunk_info[i].flags_used = 0;
    nwos_chunk_info[i].index = nwos_used_private_chunks;

    memset(nwos_chunk_info[i].md5_digest, 0, MD5_DIGEST_SIZE);

    assert(nwos_chunk_info[i].index < nwos_total_private_chunks);

    assert((offset % USABLE_BLOCKS_PER_CHUNK) == 0);

    if (nwos_archive_is_block_device())
    {
	nwos_write_empty_chunk(nwos_chunk_info[i].index);
    }

    assert(MINIMUM_32_BIT_PRIVATE_REFERENCE <= nwos_chunk_info[i].ref && nwos_chunk_info[i].ref <= MAXIMUM_32_BIT_PRIVATE_REFERENCE);

    nwos_used_private_chunks += 1;

    update_chunk_info_reverse_index();

    chunk_info_modified = true;
}

#endif

