/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-10-06 08:06:23 -0600 (Tue, 06 Oct 2009) $
--   $Revision: 4384 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../class_definition.h"
#include "../crc32.h"
#include "../disk_io.h"    /* define nwos_next_public_ref */
#include "../objectify.h"


#if 0
static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}
#endif

static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


ObjRef* byte_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("BYTE", &result))
	{
	    fprintf(stderr, "Could not locate BYTE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* character_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("CHARACTER", &result))
	{
	    fprintf(stderr, "Could not locate CHARACTER class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* objref_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("OBJECT REFERENCE", &result))
	{
	    fprintf(stderr, "Could not locate OBJECT REFERENCE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("object_reference_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


void add_feature(C_struct_Class_Definition* class_obj_ptr, ObjRef* type, char* name, int num_elements)
{
    int count;
    ObjRef feature_ref;

    assert(get_class_object_size(class_obj_ptr) < FILE_BLOCK_SIZE);

    count = class_obj_ptr->count;

    nwos_find_or_create_public_feature_definition(type, name, num_elements, &feature_ref);

    copy_reference(&class_obj_ptr->feature[count], &feature_ref);

    nwos_add_to_references(&class_obj_ptr->header.common.id, &feature_ref);

    class_obj_ptr->count = count + 1;

    printf("feature: %d - %s  new size: %d\n", count, name, (int)get_class_object_size(class_obj_ptr));
}


int main(int argc, char* argv[])
{
    ObjRef class_ref;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_obj_ptr = (C_struct_Class_Definition*)kludge;


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(PUBLIC, NULL);


#define NEW_CLASS_NAME "SHA512SUM"

    if (nwos_find_public_class_definition(NEW_CLASS_NAME, &class_ref))
    {
	fprintf(stderr, "\nERROR: class %s already exists: %08x\n\n", NEW_CLASS_NAME, nwos_ref_to_word(&class_ref));
	exit(1);
    }

    printf("Adding URL class on: %s\n", nwos_get_public_objects_path());

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0],
	   nwos_next_public_ref.id[1],
	   nwos_next_public_ref.id[2],
	   nwos_next_public_ref.id[3]);

    memset(kludge, 0, sizeof(kludge));

    nwos_generate_new_id(&class_ref);

    nwos_fill_in_common_header(&class_obj_ptr->header.common, &class_ref, &nwos_public_class_definition_class_ref);

    nwos_create_name(NEW_CLASS_NAME, &class_obj_ptr->name);

    nwos_create_reference_list(&class_ref, &class_obj_ptr->header.object.references);

    printf("class_ref: %02x%02x%02x%02x\n",
	   class_ref.id[0],
	   class_ref.id[1],
	   class_ref.id[2],
	   class_ref.id[3]);

    add_feature(class_obj_ptr, byte_class_ref(),   "sha512sum", 64);

    nwos_crc32_calculate((uint8*) &class_obj_ptr->header.object,
			 sizeof(ObjectHeader),
			 class_obj_ptr->header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &class_obj_ptr->name,
			 get_class_object_size(class_obj_ptr) - sizeof(EveryObject),
			 class_obj_ptr->header.common.data_chksum);

    nwos_write_object_to_disk(&class_ref, class_obj_ptr, get_class_object_size(class_obj_ptr));

    nwos_add_to_references(&class_ref, &nwos_public_class_definition_class_ref);
    nwos_add_to_references(&class_ref, &class_obj_ptr->name);

    printf("  number of features: %d\n", class_obj_ptr->count);

    nwos_terminate_objectify();

    return 0;
}

