/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-11-30 20:06:27 -0700 (Mon, 30 Nov 2009) $
--   $Revision: 4416 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revisions of this file.  Since this file was originally created
--   from a file called "create_person.c".  To make it easier to find I have
--   left the part of the log from when it was called create_person.c below.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.15  2005/12/30 14:13:39  jsedwards
-- Removed Social Security Number stuff, not ready yet.
--
-- Revision 1.14  2005/12/30 05:12:57  jsedwards
-- Tweaked the birth date stuff and added social security number.
--
-- Revision 1.13  2005/12/29 18:40:38  jsedwards
-- Added code for the new long key and encryption stuff.
--
-- Revision 1.12  2005/12/27 19:52:21  jsedwards
-- Added code to generate the root object from a password.  Changed to find
-- class definition instead of hardcoded file name and to generate a random
-- person object id instead of generating it based on contents.
--
-- Revision 1.11  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.10  2005/12/21 23:28:15  jsedwards
-- Fixed so that it adds the references to the name and date objects.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:06:31  jsedwards
-- Moved calculation of header checksum down just before writing object to
-- disk to ensure no changes were made after the checksum was calculated.
-- Also added the "nickname" field.
--
-- Revision 1.7  2005/12/04 14:14:46  jsedwards
-- Changed to actually get input and create a person (finally)!
--
-- Revision 1.6  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.5  2005/12/04 00:35:49  jsedwards
-- Changed to just create a date.
--
-- Revision 1.4  2005/12/04 00:23:46  jsedwards
-- Removed create year, date, and name stuff and put it in the date.c and
-- the name.c files.
--
-- Revision 1.3  2005/12/02 20:33:11  jsedwards
-- Changed so that name objects have "NAME" in the filename and added new
-- parameter to the create_reference_list calls.
--
-- Revision 1.2  2005/12/02 19:30:32  jsedwards
-- Changed to create Year object filename from the year itself.
--
-- Revision 1.1  2005/12/02 13:05:17  jsedwards
-- Initial version that only creates a name and a year.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */

#define MAX_NAME_SIZE 64
#define MAX_DATE_SIZE 16


static bool is_what_is_number(char* command)
{
    static char* what_is = "what is ";
    char* ptr;
    bool result = false;

    if (strncasecmp(command, what_is, strlen(what_is)) == 0)
    {
	ptr = strchr(command, '\'');
	if (ptr != NULL)
	{
	    ptr++;
	    if (*ptr == 's') ptr++;
	    result = (strcasecmp(ptr, " number") == 0 || strcasecmp(ptr, " phone number") == 0
		      || strcasecmp(ptr, " cell number") == 0 || strcasecmp(ptr, " mobile number") == 0);
	}
    }

    return result;
}


static bool space_available(uint32 estimated_blocks)
{
    if (!nwos_check_blocks_available(estimated_blocks))
    {
	printf("Command not completed\n");

	return false;
    }

    return true;
}


void execute_command(char *command)
{
    char* ptr1;
    char* ptr2;
    char* name;
    ObjRef ref;

    if (strcasecmp(command, "add person") == 0)
    {
	if (space_available(36))
	{
	    nwos_add_person();
	}
    }
    else if (strcasecmp(command, "list persons") == 0)
    {
	nwos_list_persons();
    }
    else if (strcasecmp(command, "list all credit unions") == 0)
    {
	nwos_list_public_credit_unions();
    }
    else if (strcasecmp(command, "list all manufacturers") == 0)
    {
	nwos_list_public_manufacturers();
    }
    else if (strcasecmp(command, "add account") == 0)
    {
	if (space_available(12))
	{
	    nwos_add_account();
	}
    }
    else if (strcasecmp(command, "list accounts") == 0)
    {
	nwos_list_accounts();
    }
    else if (strncasecmp(command, "describe ", strlen("describe ")) == 0 && strlen(command) > strlen("describe "))
    {
	/* copy just the name part into name buffer */
	name = malloc(strlen(command));   /* allocate enough room for entire command even though we won't use it all */
	assert(strlcpy(name, command + strlen("describe "), strlen(command)) < strlen(command));

	if (!nwos_find_private_name(name, &ref) && !nwos_find_public_name(name, &ref))
	{
	    printf("I don't know of anyone or anything by the name of: %s\n", name);
	}
	else
	{
	    if (nwos_any_persons_named(name))
	    {
		nwos_describe_person(name);
	    }

	    if (nwos_any_states_named(name))
	    {
		nwos_describe_state(name);
	    }
	}
    }
    else if (strncasecmp(command, "list cities in ", strlen("list cities in ")) == 0 && strlen(command) > strlen("list cities in "))
    {
	nwos_list_cities_in_state(command + strlen("list cities in "));
    }
    else if (strcasecmp(command, "add city") == 0)
    {
	if (space_available(8))
	{
	    ObjRef city_ref;

	    nwos_add_city(&city_ref);
	}
    }
    else if (strstr(command, " birthday is ") != NULL)
    {
	ptr1 = strstr(command, " birthday is ");
	ptr2 = ptr1 + strlen(" birthday is ");      /* point to the start of the phone number */
	if (ptr1[-2] == '\'' && ptr1[-1] == 's')
	{
	    ptr1 -= 2;
	}
	else if (ptr1[-1] == '\'')
	{
	    ptr1 -= 1;
	}
	*ptr1 = '\0';    /* terminate the name here */

	if (space_available(4))
	{
	    nwos_add_birthday(command, ptr2);
	}
    }
    else if (strstr(command, " mobile phone is ") != NULL)
    {
	ptr1 = strstr(command, " mobile phone is ");
	ptr2 = ptr1 + strlen(" mobile phone is ");      /* point to the start of the phone number */
	if (ptr1[-2] == '\'' && ptr1[-1] == 's')
	{
	    ptr1 -= 2;
	}
	else if (ptr1[-1] == '\'')
	{
	    ptr1 -= 1;
	}
	*ptr1 = '\0';    /* terminate the name here */

	if (space_available(4))
	{
	    nwos_add_mobile_phone(command, ptr2);
	}
    }
    else if (is_what_is_number(command))
    {
	*strchr(command, '\'') = '\0';
	nwos_find_phone_for_person(command + strlen("what is "));
    }
    else
    {
	printf("I'm sorry I don't understand\n");
    }
}


int main(int argc, char* argv[])
{
    char command[80];
    int i;

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);


    if (argc == 1)   /* nothing on the command line */
    {
	while (1)
	{
	    printf("\n");

	    nwos_ask_user("How may I help you?", command, sizeof(command));

	    printf("\n");

	    if (nwos_is_quit_command(command)) break;

	    if (strcasecmp(command, "help") == 0 || strcasecmp(command, "commands") == 0 || strcasecmp(command, "?") == 0)
	    {
		printf("The only commands that are available in this version are (without quotes):\n\n");

		printf("   \"add person\" - adds a new person to the database.\n");
		printf("   \"list persons\" - list persons to the database.\n");
		printf("   \"add account\" - adds a new checking account to the database.\n");
		printf("   \"list accounts\" - list checking accounts in the database.\n");
		printf("   \"add city\" - adds a new city to the database (not of much use in this version).\n");
		printf("   \"describe <name>\" - describes any person(s) with the name <name> in the database.\n");
		printf("   \"list cities in <state>\" - lists the cities in the database for state <state>.\n");
		printf("   \"<name>'s birthday is <date>\" - adds the birthday <date> to the person <name>.\n");
		printf("   \"<name>'s mobile phone is <number>\" - adds the phone number <number> to the person <name>.\n");
		printf("   \"what is <name>'s number\" - prints out the phone number for the person <name>.\n");
		printf("   \"list all credit unions\" - list all credit unions in the database.\n");
		printf("   \"list all manufacturers\" - list all manufacturers in the database.\n");

		printf("\n   \"commands\" or \"help\" - print this information.\n");
		printf("   \"done\", \"exit\", \"finish\", \"quit\", \"stop\", or \"terminate\" - exit the program.\n");

		printf("\nPlease note that this version is picky about the spacing, you need to have exactly one space\n");
		printf("as shown in the commands above.\n");
	    }
	    else
	    {
		execute_command(command);
	    }
	}

	printf("Goodbye!\n\n");
    }
    else
    {
	command[0] = '\0';
	for (i = 1; i < argc; i++)
	{
	    if (strlen(command) + strlen(argv[i]) + 1 > sizeof(command))
	    {
		fprintf(stderr, "The word %s exceeds size of input buffer\n", argv[i]);
		exit(1);
	    }
	    if (command[0] != '\0')     /* there are already words in command so add a space in front of this word */
	    {
		strlcat(command, " ", sizeof(command));
	    }
	    strlcat(command, argv[i], sizeof(command));
	}

	execute_command(command);
    }

    nwos_terminate_objectify();

    return 0;
}

