--
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-10-06 08:06:23 -0600 (Tue, 06 Oct 2009) $
--   $Revision: 4384 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  I have kept the portion of the
--   old CVS log before this file was moved from the main directory.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.6  2006/11/11 12:29:19  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.5  2006/11/01 13:30:17  jsedwards
-- Minor modifications to use for convert_c_structs_to_csv program.
--
-- Revision 1.4  2006/02/11 18:49:05  jsedwards
-- Took out the debugging print statement in "read_from_stream".
--
-- Revision 1.3  2006/02/11 15:10:23  jsedwards
-- Fixed off by one error in writting to stream and reading from stream.
-- Added ensure clauses to verify year, month, day, etc. are in correct range.
-- Added "append_in" feature and debugging print statements.
--
-- Revision 1.2  2006/02/08 14:25:01  jsedwards
-- Fixed calls to input stream.
--
-- Revision 1.1  2006/01/25 13:44:14  jsedwards
-- Incomplete version, saving in CVS as back up.
--


class NWOS_TIME_STAMP

creation make, now

feature

   year: INTEGER is
      do
         Result := ((storage.item(0).code.to_bit @<< 4) or (storage.item(1).code.to_bit @>> 4)).to_integer
      ensure
--         2006 <= Result
      end

   month: INTEGER is
      do
         Result := (storage.item(1).code.to_bit and (0xf).to_bit).to_integer
      ensure
         1 <= Result and Result <= 12
      end

   day_of_month: INTEGER is
      do
         Result := (storage.item(2).code.to_bit @>> 3).to_integer
      ensure
         (<< 1, 3, 5, 7, 8, 10, 12 >>).has(month) implies 1 <= Result and Result <= 31
         (<< 4, 6, 9, 11 >>).has(month) implies 1 <= Result and Result <= 30
         month = 2 and not is_leap_year implies 1 <= Result and Result <= 28
         month = 2 and is_leap_year implies 1 <= Result and Result <= 29
      end

   day_of_week: INTEGER is
      do
         Result := (storage.item(2).code.to_bit and (0x7).to_bit).to_integer
      ensure
         1 <= Result and Result <= 7
      end
 
   hour: INTEGER is
      do
         Result := (storage.item(3).code.to_bit and (0x1f).to_bit).to_integer
      ensure
         0 <= Result and Result <= 23
      end

   minute: INTEGER is
      do
         Result := (storage.item(4).code.to_bit @>> 2).to_integer
      ensure
         0 <= Result and Result <= 59
      end

   second: INTEGER is
      do
         Result := (((storage.item(4).code.to_bit @<< 4) or (storage.item(5).code.to_bit @>> 4)) and (0x3f).to_bit).to_integer
      ensure
         0 <= Result and Result <= 59
      end

   microsecond: INTEGER is
      do
         Result := (((storage.item(5).code.to_bit @<< 16)
                     or (storage.item(6).code.to_bit @<< 8)
                        or (storage.item(7).code.to_bit)) and (0xfffff).to_bit).to_integer
      ensure
         0 <= Result and Result <= 999999
      end

   is_leap_year: BOOLEAN is
      do
         Result := (year \\ 4) = 0 and ((year \\ 100) /= 0 or (year \\ 400) = 0)
      end

feature  -- I/O

   read_from_stream(stream: BINARY_INPUT_STREAM) is
      local
         i: INTEGER
      do
         from
            i := 0
         until
            i = count
         loop
           stream.read_byte
           storage.put(stream.last_byte, i)
           i := i + 1
         end
      end

   write_to_stream(stream: BINARY_OUTPUT_STREAM) is
      local
         i: INTEGER
      do
         from
            i := 0
         until
            i = count
         loop
           stream.put_character(storage.item(i))
           i := i + 1
         end
      end


feature

   append_in(str: STRING) is
      local
         usec_format: INTEGER_FORMAT
         two_digit_format: INTEGER_FORMAT
      do
         two_digit_format.set_width(2)
         two_digit_format.set_zero_fill

         usec_format.set_width(6)
         usec_format.set_zero_fill

         year.append_in(str)
         str.extend('-')
         month.append_in_format(str, two_digit_format.to_integer)
         str.extend('-')
         day_of_month.append_in_format(str, two_digit_format.to_integer)

         str.append(" (")
         day_of_week.append_in(str)
         str.append(") ")

         hour.append_in_format(str, two_digit_format.to_integer)
         str.extend(':')
         minute.append_in_format(str, two_digit_format.to_integer)
         str.extend(':')
         second.append_in_format(str, two_digit_format.to_integer)
         str.extend('.')
         microsecond.append_in_format(str, usec_format.to_integer)
      end

   now is
      do
         make
         nwos_get_time_stamp(storage.to_external)
      end

feature {NONE}

   count: INTEGER is 8

   storage: NATIVE_ARRAY[CHARACTER]

   initialized: BOOLEAN

   make is
      do
         if not initialized then
            storage := storage.calloc(count)
         end
      end

   nwos_get_time_stamp(ptr: POINTER) is
      external "C"
      end

end

