/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, and bug tracking
--   go to:
--      http://savannah.nongnu.org/projects/objectify
--
--   For releases from Alpha_30 and up, bug and feature request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2010-01-20 08:10:57 -0700 (Wed, 20 Jan 2010) $
--   $Revision: 4465 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "md5.h"

#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "header.h"
#include "user_config.h"


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s [--verbose] [archive]\n", program);
}


bool block_all_zeros(uint8* block)
{
    int i;

    for (i = 0; i < FILE_BLOCK_SIZE; i++)
    {
	if (block[i] != 0) break;
    }

    return i == FILE_BLOCK_SIZE;
}


bool reference_matches(uint32 ref, uint8 *block)
{
    return (uint8)(ref >> 24) == block[4] && (uint8)(ref >> 16) == block[5] && (uint8)(ref >> 8) == block[6] && (uint8)ref == block[7];
}


int main(int argc, char* argv[])
{
    bool verbose = false;
    int obj_file_desc;
    struct flock lock;
    const char * obj_file_path;
    const char * error_msg;
    char ref_str[128];
    uint8 *chunk;
    int i;
    int j;
    size_t bytes_read;
    int bit_num;
    int byte_num;
    uint64 private_blocks_used = 0;
    uint64 private_blocks_free = 0;
    uint64 private_blocks_total;
    uint32 chunk_num;
    uint32 info_index;
    uint32 last_total = 0;
    uint32 used_blocks;         /* blocks used in a chunk */
    uint64 ref;
    Chunk_Info* chunk_info;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    uint8 md5_digest[MD5_DIGEST_SIZE];


    if (argc == 1)   /* no option or archive specified */
    {
	obj_file_path = nwos_get_private_objects_path();
    }
    else if (argc == 2)
    {
	if (*argv[1] == '-')
	{
	    if (strcmp(argv[1], "--verbose") != 0)
	    {
		fprintf(stderr, "\nunknown option: %s\n\n", argv[1]);
		print_usage(argv[0]);
		exit(1);
	    }

	    obj_file_path = nwos_get_private_objects_path();
	    verbose = true;
	}
	else
	{
	    obj_file_path = argv[1];
	}
    }
    else if (argc == 3)
    {
	if (((*argv[1] == '-') ^ (*argv[2] == '-')) == false)     /* only one or the other can be an option */
	{
	    print_usage(argv[0]);
	    exit(1);
	}

	if ((*argv[1] == '-' && strcmp(argv[1], "--verbose") != 0) || (*argv[2] == '-' && strcmp(argv[2], "--verbose") != 0))
	{
	    fprintf(stderr, "\nunknown option: %s\n\n", argv[1]);
	    print_usage(argv[0]);
	    exit(1);
	}

	if (*argv[1] == '-')
	{
	    obj_file_path = argv[2];
	}
	else
	{
	    obj_file_path = argv[1];
	}

	verbose = true;
    }
    else
    {
	print_usage(argv[0]);
	exit(1);
    }


    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(obj_file_path);
	exit(1);
    }


    /* allocate buffer for reading chunks */

    chunk = malloc(CHUNK_SIZE);

    assert(chunk != NULL);

    if (read(obj_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	snprintf(ref_str, sizeof(ref_str), "reading first chunk from: %s", obj_file_path);
	perror(ref_str);
	close(obj_file_desc);
	exit(1);
    }

    error_msg = nwos_load_private_data(chunk, CHUNK_SIZE, false);   /* don't allow compressed files */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, obj_file_path);
	exit(1);
    }

    printf("Last change: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c\n",
	   nwos_extract_year_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_month_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_day_of_month_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_hour_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_minute_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_extract_second_from_time_stamp(nwos_private_disk_header.last_change),
	   nwos_private_disk_header.version_string[0], nwos_private_disk_header.version_string[1],
	   nwos_private_disk_header.version_string[2], nwos_private_disk_header.version_string[3]);

    printf("total_blocks: %10llu  total_chunks: %5u\n", (uint64)nwos_total_private_chunks * USABLE_BLOCKS_PER_CHUNK, nwos_total_private_chunks);
    printf("used_blocks:  %10llu  used_chunks:  %5u\n", nwos_used_private_blocks, nwos_used_private_chunks);

    if (nwos_used_private_chunks == 0)
    {
	printf("\nArchive is empty, nothing to check!\n\n");
    }

    chunk_info = malloc(nwos_used_private_chunks * sizeof(Chunk_Info));
    assert(chunk_info != NULL);

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	nwos_8_uint8_to_uint64(&chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info)], &chunk_info[i].ref);
	nwos_4_uint8_to_uint32(&chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info) + 8], &chunk_info[i].flags_used);
	nwos_4_uint8_to_uint32(&chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info) + 12], &chunk_info[i].index);
	memcpy(&chunk_info[i].md5_digest, &chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info) + 16], MD5_DIGEST_SIZE);
    }

    /*****************************/
    /* now do the private blocks */
    /*****************************/

    for (chunk_num = 0; chunk_num < nwos_used_private_chunks; chunk_num++)
    {
	bytes_read = read(obj_file_desc, chunk, CHUNK_SIZE);
	used_blocks = 0;

	for (info_index = 0; info_index < nwos_used_private_chunks; info_index++) if (chunk_info[info_index].index == chunk_num) break;

	if (info_index == nwos_used_private_chunks)
	{
	    printf("ERROR: unabled to find chunk number index: %u in chunk_info table, skipping\n", chunk_num);
	    continue;
	}

	if (verbose)
	{
	    printf("%llx %u %u ", chunk_info[info_index].ref, chunk_info[info_index].flags_used, chunk_info[info_index].index);
	    for (i = 0; i < MD5_DIGEST_SIZE; i++) printf("%02x", chunk_info[info_index].md5_digest[i]);
	    printf("\n");
	    fflush(stdout);
	}

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */

	for (i = BIT_MAP_BYTES; i < bytes_read; i += FILE_BLOCK_SIZE)
	{
	    for (j = 0; j < FILE_BLOCK_SIZE; j++)
	    {
		if (chunk[i+j] != 0) break;
	    }

	    byte_num = ((i - BIT_MAP_BYTES) / FILE_BLOCK_SIZE) / 8;
	    bit_num = ((i - BIT_MAP_BYTES) / FILE_BLOCK_SIZE) % 8;

	    if (j < FILE_BLOCK_SIZE)   /* block not empty */
	    {
		private_blocks_used++;
		used_blocks++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) == 0)
		{
		    printf("bit clear that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);
		    for (j = 0; j < BIT_MAP_BYTES; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}

		nwos_8_uint8_to_uint64(&chunk[i], &ref);

		if (ref < chunk_info[info_index].ref || ref > chunk_info[info_index].ref + USABLE_BLOCKS_PER_CHUNK)
		{
		    printf("block_reference: %llx  outside of index: %llx to %llx\n",
			   ref, chunk_info[info_index].ref, chunk_info[info_index].ref + USABLE_BLOCKS_PER_CHUNK);
		}

		md5_process_bytes(&chunk[i], FILE_BLOCK_SIZE, &md5_context);    /* include this data in the md5 checksum */
	    }
	    else
	    {
		private_blocks_free++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) != 0)
		{
		    printf("bit set that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);
		    for (j = 0; j < BIT_MAP_BYTES; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	}

	if (bytes_read != CHUNK_SIZE)
	{
	    printf("last chunk read: %08zx\n", chunk_num + (bytes_read / FILE_BLOCK_SIZE));
	    break;
	}

	if (used_blocks != (chunk_info[info_index].flags_used & CHUNK_INFO_USED_MASK))
	{
	    printf("chunk: %d  used blocks (%u) doesn't match map count (%u)\n", 
		   chunk_num, (chunk_info[info_index].flags_used & CHUNK_INFO_USED_MASK), used_blocks);
	    break;
	}


	if (!verbose && chunk_num % 64 == 0)
	{
	    printf("%u - %u  used: %lld\n", chunk_num / 64, chunk_num * BLOCKS_IN_CHUNK, private_blocks_used - last_total);
	    fflush(stdout);
	    last_total = private_blocks_used;
	}

	md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */

	if (memcmp(chunk_info[info_index].md5_digest, md5_digest, sizeof(md5_digest)) != 0)
	{
	    printf("chunk: %d md5sum mismatch:\n", chunk_num);

	    printf("chunk_info: ");
	    for (i = 0; i < MD5_DIGEST_SIZE; i++) printf("%02x", chunk_info[info_index].md5_digest[i]);

	    printf("calculated: ");
	    for (i = 0; i < MD5_DIGEST_SIZE; i++) printf("%02x", md5_digest[i]);
	}
    }

    if (!verbose)
    {
	printf("%u - %u  used: %lld\n", chunk_num / 64, chunk_num * BLOCKS_IN_CHUNK, private_blocks_used - last_total);
	fflush(stdout);
    }

    close(obj_file_desc);

    private_blocks_total = private_blocks_used + private_blocks_free;

    printf("private blocks used: %llu  stored: %llu - %s\n", private_blocks_used, nwos_used_private_blocks,
	   private_blocks_used == nwos_used_private_blocks ? "CORRECT" : "MISMATCH");

    printf("private blocks free: %llu  total: %llu\n", private_blocks_free, private_blocks_total);

    printf("usage: %.1f%%\n", ((float)private_blocks_used / (float)private_blocks_total) * 100.0f );

    return 0;
}

