/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 05:36:52 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4253 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"

#define MAX_NAME_SIZE 128
#define MAX_NUMBER_SIZE 18
#define MAX_ID_SIZE 4



bool nwos_find_public_manufacturer(char* name, ObjRef* ref)
{
    C_struct_Manufacturer mfr_obj;
    EveryObject name_header;
    ObjRef mfr_class_ref;
    ObjRef nora_ref;       /* name or acronym reference */
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;


    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    if (nwos_find_public_name(name, &nora_ref) || nwos_find_public_acronym(name, &nora_ref))
    {
	nwos_read_object_headers_from_disk(&nora_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("number of public manufacturers: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &mfr_class_ref))   /* it is a word object */
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &mfr_obj, sizeof(mfr_obj)));

		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


#ifdef PUBLIC_MODE
/* url_list needs to have a null at the end */
void nwos_create_public_manufacturer(char* brand_name, char* acronym, char* url_list[], ObjRef* ref)
{
    C_struct_Manufacturer mfr_obj;
    ObjRef mfr_class_ref;
    ObjRef url_ref[64];
    int i;

    assert(brand_name != NULL || acronym != NULL);   /* at least one has to be non-null */

    if (brand_name != NULL)
    {
	assert(!nwos_find_public_manufacturer(brand_name, ref));
    }

    if (acronym != NULL)
    {
	assert(!nwos_find_public_manufacturer(acronym, ref));
    }

    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    memset(&mfr_obj, 0, sizeof(mfr_obj));

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&mfr_obj.header.common, ref, &mfr_class_ref);

    if (brand_name != NULL)
    {
	nwos_create_name(brand_name, &mfr_obj.brand_name);
    }

    if (acronym != NULL)
    {
	nwos_create_public_acronym(acronym, &mfr_obj.acronym);
    }

    if (url_list != NULL)
    {
	nwos_create_reference_list(ref, &mfr_obj.url_list);
	for (i = 0; url_list[i] != NULL; i++)
	{
	    nwos_create_url(url_list[i], &url_ref[i]);
	    nwos_add_to_reference_list(&url_ref[i], &mfr_obj.url_list);
	}
    }

    nwos_create_reference_list(ref, &mfr_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &mfr_obj.header.object, sizeof(ObjectHeader), mfr_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &mfr_obj.corporation, sizeof(mfr_obj) - sizeof(EveryObject), mfr_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &mfr_obj, sizeof(mfr_obj));

    nwos_add_to_references(ref, &mfr_class_ref);

    if (!is_void_reference(&mfr_obj.brand_name))
    {
	nwos_add_to_references(ref, &mfr_obj.brand_name);
    }

    if (!is_void_reference(&mfr_obj.acronym))
    {
	nwos_add_to_references(ref, &mfr_obj.acronym);
    }

    if (!is_void_reference(&mfr_obj.url_list))
    {
	for (i = 0; url_list[i] != NULL; i++)
	{
	    nwos_add_to_references(ref, &url_ref[i]);
	}
    }

    printf("Object created: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
}
#endif


void nwos_list_public_manufacturers()
{
    C_struct_Manufacturer mfr_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef mfr_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    char name[32];
    ReferenceList* url_list;
    int num_urls;

    assert(nwos_find_public_class_definition("MANUFACTURER", &mfr_class_ref));

    nwos_read_object_headers_from_disk(&mfr_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("mfr class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &mfr_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &mfr_obj, sizeof(mfr_obj)));

	    if (!is_void_reference(&mfr_obj.brand_name))
	    {
		nwos_name_to_string(&mfr_obj.brand_name, name, sizeof(name));
		printf("%s", name);
	    }

	    if (!is_void_reference(&mfr_obj.acronym))
	    {
		assert(nwos_acronym_to_string(&mfr_obj.acronym, name, sizeof(name)));
		if (!is_void_reference(&mfr_obj.brand_name)) printf(" - ");
		printf("%s", name);
	    }

	    if (!is_void_reference(&mfr_obj.url_list))
	    {
		url_list = nwos_malloc_reference_list(&mfr_obj.url_list);

		num_urls = url_list->common_header.num_refs;

		printf(" - ");

		for (j = 0; j < num_urls; j++)
		{
		    assert(nwos_url_to_string(&url_list->references[j], name, sizeof(name)));
		    if (j > 0) printf(", ");
		    printf("%s", name);
		}

		nwos_free_reference_list(url_list);
		url_list = NULL;
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


