/*
--          This file is part of the New World OS and Objectify projects
--         Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-09-20 12:55:11 -0600 (Sun, 20 Sep 2009) $
--   $Revision: 4371 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file, the disk_io.c file, and the
--   objectify.c file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "backup.h"
#include "bit_map.h"
#include "chunk_info.h"
#include "disk_io.h"
#include "gen_id.h"
#include "log.h"
#include "storage.h"
#include "user_config.h"



/**********************************************/
/* Function to initialize the storage modules */
/**********************************************/

void nwos_initialize_storage(AccessType type, const char* path)
{
#ifndef NO_BACKUP
    const char* backup_dir;
#endif

    /* make sure the storage is something we can deal with */
#ifdef PUBLIC_MODE
    assert(type == PUBLIC);
#else
    assert(type == PUBLIC || type == READ_ONLY || type == READ_WRITE);
#endif

    nwos_initialize_gen_id();

    nwos_initialize_disk_io(type, path);

    /* Read the chunk info file if not a compressed file */

    if (!nwos_storage_is_public_only() && !nwos_storage_is_compressed_file())
    {
	nwos_initialize_chunk_info();
    }

#ifndef NO_BACKUP
    /* Do a backup file? */

    backup_dir = nwos_get_backup_directory_path();

    if (nwos_storage_is_read_write() && backup_dir != NULL && *backup_dir != '\0')
    {
#ifdef PUBLIC_MODE
	if (!nwos_initialize_backup(backup_dir, &nwos_public_disk_header))     /* creating backup failed, shut 'er back down */
#else
	if (!nwos_initialize_backup(backup_dir, &nwos_private_disk_header))    /* creating backup failed, shut 'er back down */
#endif
	{
	    nwos_terminate_disk_io();
	    exit(1);
	}
    }
#endif
}


/********************/
/* Status functions */
/********************/

bool nwos_storage_is_public_only(void)
{
    return nwos_disk_io_is_public_only();
}

bool nwos_storage_is_compressed_file(void)
{
    return nwos_disk_io_is_compressed_file();
}

bool nwos_storage_is_read_write(void)
{
    return nwos_disk_io_is_read_write();
}

bool nwos_backup_is_enabled(void)
{
    const char *backup_dir;

    backup_dir = nwos_get_backup_directory_path();

    return backup_dir != NULL && *backup_dir != '\0';
}


/****************************/
/* Flush any caches to disk */
/****************************/

void nwos_flush_storage(void)
{
    nwos_flush_bit_maps();
}


/*********************************************/
/* Function to terminate the storage modules */
/*********************************************/

void nwos_terminate_storage(void)
{
    bool disk_was_modified;

#ifndef PUBLIC_MODE
    nwos_terminate_bit_map();

    if (!nwos_storage_is_public_only() && !nwos_disk_io_is_compressed_file())
    {
	nwos_terminate_chunk_info();
    }
#endif

    disk_was_modified = nwos_terminate_disk_io();

    nwos_terminate_gen_id();

#ifndef NO_BACKUP
#ifdef PUBLIC_MODE
    nwos_terminate_backup(disk_was_modified, &nwos_public_disk_header);
#else
    nwos_terminate_backup(disk_was_modified, &nwos_private_disk_header);
#endif
#endif
}



/*****************************************/
/* Function to read a block from storage */
/*****************************************/

bool nwos_read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    off_t offset = -1;
    int i = 0;
    int lower;
    int upper;
    uint32 ref_hash;
    uint32 mid_hash;
    uint32 hash;
    Archive_Type archive = 0;
    bool result = false;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


    if (nwos_ref_to_word(ref) <= MAXIMUM_PUBLIC_REFERENCE)
    {
	archive = Public_Archive;
	offset = (off_t) nwos_ref_to_word(ref) * FILE_BLOCK_SIZE;
    }
#ifdef PUBLIC_MODE
    else
    {
	bool private_reference_in_public_mode = false;
	assert(private_reference_in_public_mode);
    }
#else
    else if (nwos_storage_index == NULL)     /* normal storage */
    {
	archive = Private_Archive;
	hash = nwos_hash_ref(ref);

	if (hash != 0 && nwos_test_bit_in_map(hash))
	{
	    offset = (off_t) hash << 8;
	}
    }
    else
    {
	archive = Private_Archive;

	lower = 1;
	upper = nwos_storage_count;

	ref_hash = nwos_ref_to_word(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = nwos_ref_to_word(&nwos_storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	if (lower > upper)
	{
	    int i;

	    fprintf(stderr, "Can't find %08x in index table:\n", ref_hash);
	    fflush(stderr);

	    if (ref_hash < nwos_ref_to_word(&nwos_storage_index[0]))
	    {
		i = 0;
	    }
	    else
	    {
		for (i = 1; i < nwos_storage_count; i++)
		{
		    if (nwos_ref_to_word(&nwos_storage_index[i-1]) < ref_hash && ref_hash < nwos_ref_to_word(&nwos_storage_index[i])) break;
		}
	    }

	    if (i > 0)
	    {
		fprintf(stderr, "  %d: %08x\n", i-1, nwos_ref_to_word(&nwos_storage_index[i-1]));
	    }

	    if (i < nwos_storage_count)
	    {
		fprintf(stderr, "  %d: %08x\n", i, nwos_ref_to_word(&nwos_storage_index[i]));
	    }
	}
	else
	{
	    assert(is_same_object(&nwos_storage_index[i], ref));

	    offset = (off_t)i * FILE_BLOCK_SIZE;
	}
    }
#endif    // else not PUBLIC_MODE

    if (offset > 0)
    {
	result = nwos_read_block_from_offset(archive, offset, block);
    }

    /* if read was successful make sure this block contains the actual block requested */
    if (result)
    {
	if (nwos_is_public_reference(ref) || strcmp(nwos_version_string, "0030") < 0)
	{
	    assert((block[0] == 0) & (block[1] == 0) & (block[2] == 0) & (block[3] == 0));
	}
	else
	{
	    assert((block[0] == 0xff) & (block[1] == 0xff) & (block[2] == 0xff) & (block[3] == 0xfe));
	}

	assert((block[4] == ref->id[0]) & (block[5] == ref->id[1]) & (block[6] == ref->id[2]) & (block[7] == ref->id[3]));
    }

    return result;
}


/****************************************/
/* Function to write a block to storage */
/****************************************/

void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off_t offset;
    Reference_Type ref_type;
    Archive_Type archive;


    ref_type = nwos_reference_type(ref);

#ifdef PUBLIC_MODE
    assert(ref_type == Public_Reference);

    archive = Public_Archive;
#else
    assert(ref_type == Private_Reference);

    archive = Private_Archive;
#endif

    assert(!is_void_reference(ref));
    /* assert((block[0] & 0x7f) == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0); */
    assert(is_same_object((ObjRef*)block + 1, ref));

#ifdef PUBLIC_MODE
    block[0] = 0;
    block[1] = 0;
    block[2] = 0;
    block[3] = 0;
#else
    block[0] = 0xff;
    block[1] = 0xff;
    block[2] = 0xff;
    block[3] = 0xfe;
#endif

#ifdef LOG_WRITE_BLOCK
    snprintf(log_msg, sizeof(log_msg), "nwos_write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], nwos_hash_ref(ref));
    nwos_log(log_msg);
#endif

    assert(nwos_storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = nwos_ref_to_offset(ref);

    assert(offset >= 0);

    if (!nwos_write_block_to_offset(archive, offset, block))
    {
	snprintf(log_msg, sizeof(log_msg), "nwos_write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    nwos_clear_chunk_digest(ref);   /* flag the chunk as modified */

#ifndef NO_BACKUP
    nwos_backup_write_block(block);
#endif
}


/************************************************/
/* Remove an object (by writing zero's over it) */
/************************************************/

#ifndef PUBLIC_MODE

void nwos_erase_block(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off_t offset;
    uint32 hash;

    assert(nwos_reference_type(ref) == Private_Reference);

    assert(nwos_read_block(ref, block));    /* verify that there is already a block there */

    snprintf(log_msg, sizeof(log_msg), "nwos_erase_block - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    hash = nwos_hash_ref(ref);

    assert(nwos_test_bit_in_map(hash));
    assert(nwos_read_block(ref, block));
    assert(block[4] == ref->id[0] && block[5] == ref->id[1] && block[6] == ref->id[2] && block[7] == ref->id[3]);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = nwos_ref_to_offset(ref);

    assert(offset >= 0);

    nwos_write_block_to_offset(Private_Archive, offset, block);

    nwos_clear_chunk_digest(ref);   /* flag the chunk as modified */

    nwos_clear_bit_in_map(hash);

    assert(!nwos_read_block(ref, block));   /* verifty that it was actually erased */
}

#endif


