/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2004, 2005, 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-07 07:24:22 -0600 (Fri, 07 Aug 2009) $
--   $Revision: 4300 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Since this file was originally
--   copied from the file in the root directory I have left the CVS log from
--   that file below.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.49  2006/12/01 05:16:46  jsedwards
-- Added a test in get_object_class to make sure it is not a reference list
-- before trying to compute the header checksum.
--
-- Revision 1.48  2006/12/01 05:11:48  jsedwards
-- Add a strlcpy function only if compiling on Linux because it doesn't have
-- it in the library.
--
-- Revision 1.47  2006/11/27 14:28:26  jsedwards
-- Added test for unencrypted objects in read_and_decrypt and write_and_encrypt
-- functions and skip over encryption stuff if set.
--
-- Revision 1.46  2006/11/27 13:46:46  jsedwards
-- Added functions to encode and decode variable sized counts.
--
-- Revision 1.45  2006/11/19 13:44:30  jsedwards
-- Add time stamp to disk header so we can tell when it was last changed.
--
-- Revision 1.44  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.43  2006/11/18 14:26:47  jsedwards
-- Changed to read and write objects larger than FILE_BLOCK_SIZE (256 bytes).
--
-- Revision 1.42  2006/11/17 12:18:11  jsedwards
-- Added "read_only" flag to keep it from trying to write the header back when
-- read only.
--
-- Revision 1.41  2006/11/11 15:20:37  jsedwards
-- Moved all reference list code into a separate file: reference_list.c.
--
-- Revision 1.40  2006/11/11 14:24:37  jsedwards
-- Made nwos_malloc and nwos_free public (not static) and moved
-- find_class_definiton function to class_definition.c.
--
-- Revision 1.39  2006/11/11 13:58:00  jsedwards
-- Add new generate_new_public_id function.
--
-- Revision 1.38  2006/11/11 13:34:32  jsedwards
-- Made disk header a static global that is read in initialize(), updated with
-- next public reference and written back out in terminate().  Also added a
-- new global variable with the next public reference.
--
-- Revision 1.37  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.36  2006/11/10 13:39:16  jsedwards
-- Uncommented write_block in write_public_object_to_disk and removed print
-- statements.
--
-- Revision 1.35  2006/11/10 13:32:05  jsedwards
-- Moved Ref_List_First_Block and Ref_List_Extra_Block definitions from
-- objectify.c to objectify_private.h so that import_public_csv can use
-- them also.
--
-- Revision 1.34  2006/11/10 04:11:02  jsedwards
-- Add "write_public_object_to_disk" function and fix hash to deal with
-- public references.
--
-- Revision 1.33  2006/11/04 18:51:07  jsedwards
-- Made a #define in the config.h file for the log file.
--
-- Revision 1.32  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.31  2006/10/26 01:22:51  jsedwards
-- Replace the main truck version (1.30) with the latest version from the
-- alpha_05_branch.
--
-- Revision 1.24.2.57  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.24.2.56  2006/10/22 13:05:16  jsedwards
-- Changed so that blocks on disk and reserved public blocks is variable
-- based on values store in disk header.
--
-- Revision 1.24.2.55  2006/10/22 12:40:47  jsedwards
-- Corrected version string error message.
--
-- Revision 1.24.2.54  2006/10/22 00:16:43  jsedwards
-- More fixes for the changing to the separate next_block_ptr to the add to
-- reference list and write reference list.
--
-- Revision 1.24.2.53  2006/10/20 13:25:58  jsedwards
-- Change to have a separate next_block_ptr (for linked list) instead of
-- using the first 4 (unused) bytes of each reference list block as the
-- pointer to the next block.  This wasn't happy when pointers aren't 4 bytes.
--
-- Revision 1.24.2.52  2006/10/17 12:38:37  jsedwards
-- Changed formats in printf statementss to be more compatible with different
-- types (size_t, off64_t) being different sizes.  Also changed format for
-- uint32 to be unsigned instead of long unsigned (unsigned is 32 bits on
-- both 32 and 64 bit machines, whereas long unsigned is 64 on amd64).
--
-- Revision 1.24.2.51  2006/10/16 13:01:05  jsedwards
-- Add code to rewind disk after indexing a compressed file so reading the
-- header works correctly.
--
-- Revision 1.24.2.50  2006/10/14 13:13:45  jsedwards
-- Added code to check the disk header and make sure it is a nwos disk, the
-- version string matches, and the disk size matches what was compiled in.
--
-- Revision 1.24.2.49  2006/10/13 11:47:44  jsedwards
-- Comment out print statement of setting bits in block maps.
--
-- Revision 1.24.2.48  2006/10/12 12:12:22  jsedwards
-- Fixed bug in last ref for generating references in sequence.  Also added
-- function to flush dirty block maps back to disk.
--
-- Revision 1.24.2.47  2006/10/11 13:24:50  jsedwards
-- Add code to set the bits in the bit maps when a block is written.  Note:
-- this version does not use the bit maps for anything.
--
-- Revision 1.24.2.46  2006/10/10 07:30:35  jsedwards
-- Change to 8192 bytes per bit map instead of 256 bytes.
--
-- Revision 1.24.2.45  2006/10/07 22:27:02  jsedwards
-- Changed the generate id function so that it checks to see if we are
-- operating in the sequential mode and it calls the correct function.
--
-- Revision 1.24.2.44  2006/10/06 04:39:24  jsedwards
-- Changed the names of some of the public and private, blocks and id
-- definitions and added new function to generate ids that scan across
-- the disk drive, instead of totally random.
--
-- Revision 1.24.2.43  2006/10/03 12:50:16  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.24.2.42  2006/10/02 14:02:45  jsedwards
-- Changed to just read the file while indexing, instead of doing the seeks.
--
-- Revision 1.24.2.41  2006/10/02 13:29:54  jsedwards
-- Changed to use a binary search for the indexes instead of the linear.
--
-- Revision 1.24.2.40  2006/10/02 12:33:44  jsedwards
-- Split out reference hash into a separate function and initialized next_ptr
-- so that gcc doesn't worry about it possibly being uninitialized.
--
-- Revision 1.24.2.39  2006/10/02 02:15:33  jsedwards
-- Added the capability of reading blocks from a compressed archive file
-- instead of the normal disk drive or partition.
--
-- Revision 1.24.2.38  2006/10/01 11:49:47  jsedwards
-- Increased the number of tries in finding a random id.  Made the code to
-- not access the hard drive for testing purposes ifdef'd so it can be enabled
-- or disabled with a -D option.
--
-- Revision 1.24.2.37  2006/09/29 04:12:39  jsedwards
-- Fixed "object_exists" and "block_used" functions again.  Added if'd out
-- code to write to memory instead of the disk (for testing).
--
-- Revision 1.24.2.36  2006/09/28 13:38:46  jsedwards
-- Re-arranged testing of reference in block to do object exists better.
-- Added printing of offset in lseek error statements.
--
-- Revision 1.24.2.35  2006/09/26 13:22:38  jsedwards
-- Changed to use a disk drive or partition for storage instead of a file.
-- This required hashing the id to fit in the disk or partition size.
--
-- Revision 1.24.2.34  2006/09/24 13:46:14  jsedwards
-- Previous change was a mistake, added a new function that checks to see if
-- the block is used and changed "object_exists" function back to the way it
-- was.
--
-- Revision 1.24.2.33  2006/09/24 13:38:06  jsedwards
-- Add test to "object_exists" function to verify it is not just a zero block.
--
-- Revision 1.24.2.32  2006/09/19 14:10:45  jsedwards
-- Fix bug where next block wasn't zeroed when reference list sidecar filled
-- up exactly.  Added printing of num_blocks in log.
--
-- Revision 1.24.2.31  2006/09/18 02:07:04  jsedwards
-- Add asserts to make sure the reference passed into routines is not void.
--
-- Revision 1.24.2.30  2006/09/18 01:32:52  jsedwards
-- Renamed read_file_and_decrypt and write_file with "nwos" prefix and removed
-- the static so that they can be accessed by other modules (like file.c).
--
-- Revision 1.24.2.29  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.24.2.28  2006/09/08 13:28:41  jsedwards
-- Changed to use a different sequence block for objects beyond the first for
-- objects larger than one block.
--
-- Revision 1.24.2.27  2006/09/08 11:07:34  jsedwards
-- Changed so that random_sequence table is a two dimentional array so we can
-- store different sequence tables for multiple block objects.
--
-- Revision 1.24.2.26  2006/09/08 11:01:23  jsedwards
-- Changed so that the sequence table is passed into "write_object_to_disk"
-- and "read_object_from_disk_and_decrypt" routines.
--
-- Revision 1.24.2.25  2006/09/07 12:59:46  jsedwards
-- Separated the sequence table generation code into a separate function.
--
-- Revision 1.24.2.24  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.24.2.23  2006/09/05 12:59:34  jsedwards
-- Added checksums (crc32) to reference lists.
--
-- Revision 1.24.2.22  2006/09/05 01:26:41  jsedwards
-- Converted add_to_reference_list to use the new block structures.
--
-- Revision 1.24.2.21  2006/09/05 00:29:16  jsedwards
-- Converted the read_reference_list_from_disk routine to use the new block
-- structures.
--
-- Revision 1.24.2.20  2006/09/05 00:07:48  jsedwards
-- Converted the write_reference_list routine to use the new block structures.
--
-- Revision 1.24.2.19  2006/09/04 15:51:14  jsedwards
-- Change read_reference_list_into_cache routine to use new block structures.
--
-- Revision 1.24.2.18  2006/09/04 13:26:06  jsedwards
-- Added reference list block structures (for the blocks written to disk).
-- Changed cache to point to a reference list first block structure instead
-- of a reference list itself.  The code really doesn't use them yet.
--
-- Revision 1.24.2.17  2006/09/04 01:01:24  jsedwards
-- Split up the create_reference_list routine so it is possible to create a
-- reference list with a given id instead of generating a new one.  This is
-- helpful when doing the conversion from the old multiple file style (we
-- can just keep the old reference ids and not have to switch them around).
--
-- Revision 1.24.2.16  2006/09/03 13:56:35  jsedwards
-- Moved defintion of DEFAULT_FILE to objectify_private.h so the big_bang can
-- use it to create the file.
--
-- Revision 1.24.2.15  2006/09/03 13:14:55  jsedwards
-- Rearranged the order of functions in the file and added some comments to
-- mark the different sections.  NO code changes were made.
--
-- Revision 1.24.2.14  2006/09/03 12:20:40  jsedwards
-- Removed a couple more debugging print statements.
--
-- Revision 1.24.2.13  2006/09/03 12:13:57  jsedwards
-- Add code to check that all malloc'd blocks are freed during testing.
--
-- Revision 1.24.2.12  2006/09/03 11:09:14  jsedwards
-- Removed many of the debugging print statements, added a routine to allow
-- an object to be overwritted (instead of having to remove it and then write
-- it again.  Also rearranged the checksum checking code slightly.
--
-- Revision 1.24.2.11  2006/09/03 10:22:14  jsedwards
-- Initialize ivec in nwos_get_object_class function.
--
-- Revision 1.24.2.10  2006/09/02 15:08:16  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Fixed add reference routine to clean dirty flag before writing block.
--
-- Fixed remove object routine to do the seek before nuking the object.
--
-- Revision 1.24.2.9  2006/09/01 13:20:00  jsedwards
-- Fixed some bugs in new storing objects in one file.  Still NOT working!
--
-- Revision 1.24.2.8  2006/09/01 11:49:59  jsedwards
-- Merged version with file additions and version that writes objects into one
-- large file.  This vesion DOES NOT work correctly.
--
-- Revision 1.24.2.7  2006/08/25 13:06:54  jsedwards
-- Fixed daylight savings to work correctly when computing the end of time.
--
-- Revision 1.24.2.6  2006/08/25 12:43:37  jsedwards
-- Change to compute the end of time value in local time and save it.  This
-- way the time stamps on the file actually read 9/9/99...
--
-- Revision 1.24.2.5  2006/08/25 12:24:00  jsedwards
-- Added code to set the time stamps on all of the objects to 9/9/99 9:09:09.
--
-- Revision 1.24.2.4  2006/08/16 12:47:56  jsedwards
-- Changed the name of the static function "log" to "nwos_log" and added a
-- cast so that new versions of GCC don't complain.
-- doesn't complain about the type of a built in function being redefined
--
-- Revision 1.24.2.3  2006/08/13 16:36:43  jsedwards
-- Made a define for the block size of the file called FILE_BLOCK_SIZE, that
-- is the minimum size of the file for each object.  The amount of disk space
-- used for a one byte file.
--
-- Revision 1.24.2.2  2006/08/13 16:27:14  jsedwards
-- Change to compute the random sequence table only one time and save it.
--
-- Revision 1.24.2.1  2006/08/13 12:53:06  jsedwards
-- Added comments to encrypt and decrypt routines (NO code changes).
--
-- Revision 1.24  2006/01/12 02:57:28  jsedwards
-- Fixed "ask yes or no" routine to actually work.
--
-- Revision 1.23  2006/01/12 02:03:36  jsedwards
-- Added code to dump the object when the data checksum fails.
--
-- Revision 1.22  2006/01/08 15:27:28  jsedwards
-- Removed unused variable.
--
-- Revision 1.21  2006/01/03 03:21:38  jsedwards
-- Added "is compatible version" routine to determine if a particular version
-- of an object is still compatible with the current version.
--
-- Revision 1.20  2006/01/01 19:47:33  jsedwards
-- Added "is quit command" and "ask yes or no" routines.
--
-- Revision 1.19  2005/12/30 14:09:50  jsedwards
-- Commented out printing of blowfish key, was for debugging.
--
-- Revision 1.18  2005/12/29 19:08:00  jsedwards
-- Fixed some of the ifdef DISABLE_SECURITY_FEATURES that were accidentally
-- entered as #if DISABLE_SECURITY_FEATURES instead.  Also added a disable
-- security features to the unique id routine to make sure it finds a unique
-- one by incrementing (since we can punch the random number generator all
-- we want and it isn't going to fix that problem).
--
-- Revision 1.17  2005/12/29 18:30:42  jsedwards
-- Added ifdefs to disable/enable the security features.
--
-- Revision 1.16  2005/12/29 18:20:24  jsedwards
-- Split up routines, added read variable length objects and other changes to
-- make encryption work, HOWEVER, ENCRYPTION IS DISABLED IN THIS VERSION.
--
-- Revision 1.15  2005/12/28 12:53:06  jsedwards
-- First attempt at adding security.  Objects are encrypted before being
-- written to disk and decrypted when read from disk.  It is incomplete at
-- this point and doesn't work completely, for example, the read object
-- headers function won't work.  This version should only be used for testing.
--
-- Revision 1.14  2005/12/27 19:44:40  jsedwards
-- Added routine to set the root object.
--
-- Revision 1.13  2005/12/27 18:14:08  jsedwards
-- Changed ObjRefs to be 4 bytes instead of 8 bytes and made them random
-- instead of somehow based upon the contents.
--
-- Revision 1.12  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.11  2005/12/21 23:25:57  jsedwards
-- Added better error messages when the size of an object doesn't match the
-- number of bytes read.
--
-- Revision 1.10  2005/12/11 16:52:56  jsedwards
-- Added a log message when an object is added to a reference list and fixed
-- the error messages printed when adding to a reference list fails.
--
-- Revision 1.9  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.8  2005/12/05 19:03:06  jsedwards
-- Moved local read_object_from_disk (not nwos_read_object_from_disk) function
-- earlier in the file so that get_object_class function can call it instead
-- of nwos_read_object_from_disk which wasn't happy about it just reading the
-- common header.
--
-- Revision 1.7  2005/12/05 05:18:53  jsedwards
-- Ifdefd out logging each time a file is read, too much spam into log file.
--
-- Revision 1.6  2005/12/05 05:16:22  jsedwards
-- Made a separate routine 'read_reference_list_from_disk' to read reference
-- lists, that doesn't do checksums.  Changed 'read_object_from_disk' to
-- check both the header and data checksums.  Also added a log routine to
-- log each time a file is written to and read from disk.
--
-- Revision 1.5  2005/12/03 22:15:38  jsedwards
-- Added get_object_class and add_to_references functions.
--
-- Revision 1.4  2005/12/02 20:30:11  jsedwards
-- Changed so that reference list id is generated from the object that the
-- reference list is for's id.
--
-- Revision 1.3  2005/11/30 13:53:31  jsedwards
-- Added routines to get object size and read class definition.  Also added
-- 'nwos' prefix to generate_new_id routine and made it global.
--
-- Revision 1.2  2005/11/26 15:36:59  jsedwards
-- Added 'nwos' prefix to functions and added 'static' to local functions.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.5  2005/11/24 15:47:47  jsedwards
-- Changed to match new object header layouts.
--
-- Revision 1.4  2005/11/22 13:57:04  jsedwards
-- Made a common subroutine to fill in the common header.
--
-- Revision 1.3  2005/11/22 13:09:58  jsedwards
-- Added read object from disk and create simple object routines.
--
-- Revision 1.2  2005/11/12 23:02:58  jsedwards
-- Rearranged headers.
--
-- Revision 1.1  2005/11/12 14:55:12  jsedwards
-- Routines for creating objects.
--
*/

/*---------------------------------------------------------------------------*/
/* The new plan for storing objects:                                         */
/*                                                                           */
/* Instead of storing objects in separate files (as was done previously),    */
/* the plan is to store them all in one sparse file.                         */
/*                                                                           */
/* One of the major problems with the previous plan was how to cope with     */
/* block sizes of all the different file systems.  I was using Resier4       */
/* because it's block size was 512 bytes and so if you created a file that   */
/* was only say 92 bytes you only wasted 430 bytes.  However when it was     */
/* used on an ext3 file system with 4096 blocks you wasted 4004 bytes.       */
/* I also suspsect there may have been more problems when there got to be    */
/* a large number of files with exceeding the number that were possible      */
/* in a directory and running out of inodes.                                 */
/*                                                                           */
/* So the new plan is to store all of the objects in a sparse file, indexed  */
/* by their ID number.  My current theory then is to use 256 byte blocks     */
/* for each object.  The first 8 bytes of each block is it's ID number.      */
/* If the first 8 bytes of the block are zeros then that block is unused.    */
/*                                                                           */
/* Variable sized objects that are larger than 256 bytes use multiple        */
/* blocks.  The last 4 bytes of the block point to the next block.           */
/*---------------------------------------------------------------------------*/


#define _LARGEFILE64_SOURCE

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <openssl/blowfish.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "../crc32.h"
#include "../objectify_private.h"
#include "objectify_0014.h"

ObjRef nwos_class_definition_class_ref_0014;
ObjRef nwos_reference_list_class_ref_0014;

static uint8  disk_header[32];
static uint32 linear_after_first;
static uint32 serial_after_first;
static BF_KEY blowfish_key;

static int    obj_file_desc;
static uint32 blocks_on_disk;
static uint32 reserved_public_blocks;
static uint32 private_blocks_on_disk;

static bool    read_only;
static char*   storage_path;
static ObjRef* storage_index;
static int     storage_count;


/**********************************/
/* Initialization and Termination */
/**********************************/

void index_compressed_file_0014()
{
    int i;
    int size_of_index = 262144;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];

    assert(obj_file_desc > 0);

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    printf("Indexing");
    fflush(stdout);

    /* read the first block, it contains the header */
    bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

    assert(bytes_read == sizeof(buffer));

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    printf(".");
	    fflush(stdout);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += 262144;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

	if (bytes_read != sizeof(buffer)) break;

	memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));

	i++;
    }

    if (errno != 0)
    {
	perror(storage_path);
	exit(1);
    }

    assert(bytes_read == 0);

    storage_count = i;

    printf("\nblocks: %d\n", i);
}


void nwos_initialize_objectify_0014(uint8 bf_key[16], uint32 linear_seed, uint32 serial_seed, StorageType type, char* path)
{
    char log_msg[128];

    /* make sure the storage is something we can deal with */
    assert(type == Compressed_File_RO);

    assert(storage_path == NULL);   /* make sure this is only called once */
    assert(obj_file_desc == 0);

    storage_path = path;

    BF_set_key(&blowfish_key, 16, bf_key);

    linear_after_first = linear_seed;
    serial_after_first = serial_seed;

    /* No need to initialized the sequence tables here because they should be the same as the real ones */

    read_only = ((type & 1) == 0);

    assert(read_only);

    {
	obj_file_desc = open(storage_path, O_RDONLY | O_LARGEFILE);
    }

    if (type == Compressed_File_RO)
    {
	index_compressed_file_0014();   /* build an index into the compressed file */

	if (lseek64(obj_file_desc, 0LL, SEEK_SET) < 0)
	{
	    perror("rewinding after indexing");
	    exit(1);
	}
    }

    if (obj_file_desc < 0)
    {
	perror(storage_path);
	exit(1);
    }

    if (read(obj_file_desc, disk_header, sizeof(disk_header)) != sizeof(disk_header))
    {
	sprintf(log_msg, "reading disk header from: %s", storage_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(&disk_header[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&disk_header[4], VERSION_STRING_0014, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    reserved_public_blocks = ((uint32)disk_header[8] << 24) | ((uint32)disk_header[9]<< 16) | ((uint32)disk_header[10] << 8) | ((uint32)disk_header[11]);

    blocks_on_disk = ((uint32)disk_header[12] << 24) | ((uint32)disk_header[13]<< 16) | ((uint32)disk_header[14] << 8) | ((uint32)disk_header[15]);

    assert(BLOCKS_IN_CHUNK <= reserved_public_blocks && reserved_public_blocks <= NUM_PUBLIC_IDS);
    assert(reserved_public_blocks % BLOCKS_IN_CHUNK == 0);
    assert(blocks_on_disk > reserved_public_blocks);

    private_blocks_on_disk = blocks_on_disk - reserved_public_blocks;

    sprintf(log_msg, "nwos_initialize_objectify_0014() - blocks_on_disk: %u", blocks_on_disk);
    nwos_log(log_msg);
}


void nwos_set_root_object_0014(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;

    nwos_read_object_from_disk_0014(ref, &root_obj, sizeof(root_obj));
    memcpy(&nwos_class_definition_class_ref_0014, &root_obj.class_definition_class, sizeof(nwos_class_definition_class_ref_0014));
    memcpy(&nwos_reference_list_class_ref_0014, &root_obj.reference_list_class, sizeof(nwos_reference_list_class_ref_0014));

    assert(nwos_find_class_definition_0014("REFERENCE LIST", &verify_ref));
    assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref_0014));
}


void nwos_terminate_objectify_0014()
{
    if (close(obj_file_desc) < 0)
    {
	perror(storage_path);
	exit(1);
    }

    obj_file_desc = 0;
    storage_path = NULL;
}



/*******************************************************************/
/* Reference utility functions: conversion, testing and generation */
/*******************************************************************/

static inline uint32 hash_ref(ObjRef* ref)
{
    uint32 result;

    result = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
             ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	result = (result - NUM_PUBLIC_IDS) % private_blocks_on_disk + reserved_public_blocks;
    }

    return result;
}


static inline off64_t ref_to_offset(ObjRef* ref) 
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = hash_ref(ref);

    if ((ref->id[0] & 0xF0) != 0)  /* it's a private reference */
    {
	assert(BIT_MAP_BYTES == 8192);   /* if this fails need to fix the hack below */

	if ((hash & 0xffe0) == 0)     /* special hack to get around blocks that would overwrite the bit map */
	{
	    hash = (hash + 32) & 0xffffffe0;
	}
    }

    return (off64_t) hash << 8;
}


static bool block_contains_object(uint8 block[FILE_BLOCK_SIZE], ObjRef* ref)
{
    uint8 marker[8];

    marker[0] = 0;
    marker[1] = 0;
    marker[2] = 0;
    marker[3] = 0;
    marker[4] = ref->id[0];
    marker[5] = ref->id[1];
    marker[6] = ref->id[2];
    marker[7] = ref->id[3];

    return (memcmp(block, marker, sizeof(marker)) == 0);   /* return true if the object marker is there */
}


/*************************/
/* Verification routines */
/*************************/

#if 0
static bool is_compatible_version(char* header_version)
{
    static char* compatible[] = COMPATIBLE_HEADER_VERSIONS;
    const int num_compatible = sizeof(compatible) / sizeof(char*);
    int i;

    for (i = 0; i < num_compatible; i++) 
    {
	if (memcmp(header_version, compatible[i], strlen(HEADER_VERSION)) == 0) break;
    }

    return (i < num_compatible);    /* return true if found a match */

    return true;
}
#endif


static void check_common_header(ObjRef* ref, CommonHeader* common)
{
#if 0
#if 0
    printf("common header pointer: %p\n", common);
#endif

    if (memcmp(common->magic_number, MAGIC_NUMBER, sizeof(common->magic_number)))
    {
	printf("object: %s - missing magic number\n", path);
	exit(1);
    }

    if (!is_compatible_version((char*)common->version))
    {
	printf("object: %s - incompatible version: ", path);
	if (isprint(common->version[0]) && isprint(common->version[1]) &&
	    isprint(common->version[2]) && isprint(common->version[3]))
	{
	    printf("%c%c%c%c\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	else
	{
	    printf("%02x %02x %02x %02x\n", common->version[0], common->version[1], common->version[2], common->version[3]);
	}
	exit(1);
    }
#endif
}


/* ref is only needed for error messages */

static void check_object_header(ObjRef* ref, ObjectHeader* obj_header, uint8 stored[4])
{
    uint8 computed[4];
    bool checksum_ok;

    nwos_crc32_calculate((uint8*)obj_header, sizeof(*obj_header), computed);

    checksum_ok = (stored[0] == computed[0] || stored[1] == computed[1] || stored[2] == computed[2] || stored[3] == computed[3]);

    if (!checksum_ok)
    {
	printf("object: %02x%02x%02x%02x - bad header checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", sizeof(*obj_header));
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);
	assert(checksum_ok);
    }
}


static void check_object_data(ObjRef* ref, uint8* object, size_t total_size, uint8 stored[4])
{
    uint8 computed[4];
    size_t data_size;

    data_size = total_size - sizeof(EveryObject);

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)object + sizeof(EveryObject), data_size, computed);

    if (stored[0] != computed[0] || stored[1] != computed[1] || stored[2] != computed[2] || stored[3] != computed[3])
    {
	printf("object: %02x%02x%02x%02x - bad data checksum, ", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	printf("size %zd - ", data_size);
	printf("computed %02x%02x%02x%02x - ", computed[0], computed[1], computed[2], computed[3]);
	printf("stored %02x%02x%02x%02x\n", stored[0], stored[1], stored[2], stored[3]);

	printf("\n");
	{
	  int i;
	  for (i = 0; i < sizeof(EveryObject) + data_size; i++)
	  {
	      printf("%02x%c", object[i], (i % 16) == 15 ? '\n' : ' ');
	  }
	  printf("\n");
	}
	exit(1);
    }
}



/*****************************************/
/* Routines to read blocks, objects, etc */
/*****************************************/

static bool read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off64_t offset;
    int i;
    int lower;
    int upper;
    uint32 ref_hash;
    uint32 mid_hash;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else

    if (storage_index == NULL)     /* normal storage */
    {
	offset = ref_to_offset(ref);
    }
    else
    {
	lower = 1;
	upper = storage_count;

	ref_hash = hash_ref(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = hash_ref(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	assert(lower <= upper);

	assert(is_same_object(&storage_index[i], ref));

	offset = (off64_t)i * FILE_BLOCK_SIZE;
    }

    if (lseek64(obj_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	sprintf(ref_str, "read_block lseek64 ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    return read(obj_file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

#endif
}


void nwos_read_object_from_disk_and_decrypt_0014(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE])
{
    uint8 buffer1[FILE_BLOCK_SIZE];
    uint8 buffer2[FILE_BLOCK_SIZE];
    uint8* ptrobj = (uint8*)object;
    int i;
    int j;

    assert(!is_void_reference(ref));

    read_block(ref, buffer2);

    assert(block_contains_object(buffer2, ref));

#ifdef DEBUG_ENCRYPTION
    compare_bufs(buffer2, save_encrypted, "Encrypted");
#endif

#ifdef DISABLE_SECURITY_FEATURES
    if (true)
#else
    if ((buffer2[0] & UNENCRYPTED_FLAG) != 0)
#endif
    {
	memcpy(object, buffer2, size);    /* just copy the buffer into the object */
    }
    else
    {
	/* copy the first 8 bytes directly, not encrypted */
	memcpy(buffer1, buffer2, 8);

	/* decrypt the remaining bytes */
	BF_cbc_encrypt((buffer2 + 8), (buffer1 + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_DECRYPT);

#if 0
	printf("ivec: ");
	for (i = 0; i < 8; i++) printf("%02x ", ivec[i]);
	printf("\n");
#endif

#ifdef DEBUG_ENCRYPTION
	compare_bufs(buffer1, save_mixed_up, "Mixed up");
#endif

	assert(size <= FILE_BLOCK_SIZE);

	/* copy the first 8 bytes directly accross */
	for (i = 0; i < 8; i++) ptrobj[i] = buffer1[i];

	for (i = 8; i < size; i++)
	{
	    j = seq_table[i];    /* find where the next byte is stored */
	    ptrobj[i] = buffer1[j];    /* get this byte from there */
	}

    }

#ifdef DEBUG_ENCRYPTION
    compare_bufs(ptrobj, save_unencrypted, "Unencrypted");
#endif
}


#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


void nwos_read_object_from_disk_0014(ObjRef* ref, void* object, size_t size)
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    sprintf(log_msg, "nwos_read_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    assert(size > sizeof(CommonHeader));  /* make sure this wasn't called to just get the header */

    memset(ivec, 0, sizeof(ivec));

    bytes_remaining = size;

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt_0014(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt_0014(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, size, header->common.data_chksum);
}


void nwos_read_variable_sized_object_from_disk_0014(ObjRef* ref, void* object, size_t max_size, size_t (*size_function)(void*))
{
    EveryObject* header;
    uint8 ivec[IVEC_SIZE];
    uint8* ptr_obj;
    Extra_Data_Block extra;
    ObjRef next_ref;
    size_t bytes_remaining;
    int i;
    int seq = 0;

    assert(!is_void_reference(ref));

#if 0
    char log_msg[128];

    sprintf(log_msg, "nwos_read_variable_size_object_from_disk - %s size: %d", path, size);
    nwos_log(log_msg);
#endif

    memset(ivec, 0, sizeof(ivec));

    ptr_obj = object;

    nwos_read_object_from_disk_and_decrypt_0014(ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);

    assert(sizeof(ObjRef) == 4);

    *ptr_obj++ = extra.flags[0];
    *ptr_obj++ = extra.flags[1];
    *ptr_obj++ = extra.flags[2];
    *ptr_obj++ = extra.flags[3];
    *ptr_obj++ = extra.id.id[0];
    *ptr_obj++ = extra.id.id[1];
    *ptr_obj++ = extra.id.id[2];
    *ptr_obj++ = extra.id.id[3];

    bytes_remaining = (*size_function)(&extra);

    assert(bytes_remaining <= max_size);

    while (bytes_remaining > FILE_BLOCK_SIZE)
    {
	for (i = 0; i < SIZE_EXTRA_BLOCK_DATA; i++)
	{
	    *ptr_obj++ = extra.data[i];
	    bytes_remaining--;
	}

	/* save the reference because the extra block is going to get written over, we don't want it to change */
	copy_reference(&next_ref, &extra.next_block);

	seq++;

	nwos_read_object_from_disk_and_decrypt_0014(&next_ref, &extra, sizeof(extra), ivec, nwos_random_sequence[seq]);
    }

    assert(bytes_remaining <= FILE_BLOCK_SIZE);

    for (i = 0; i < bytes_remaining - 8; i++) *ptr_obj++ = extra.data[i];

#if 0
    printf("read obj path: %s\n", path);
    printf("read obj pointer: %p\n", object);
    {
      int i;
      for (i = 0; i < size; i++) printf("%02x%c", *((uint8*)object+i), i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    header = (EveryObject*) object;

    check_common_header(ref, &header->common);

    check_object_header(ref, &header->object, header->common.header_chksum);

    check_object_data(ref, (uint8*)object, (*size_function)(object), header->common.data_chksum);
}


/***************/
/* Class stuff */
/***************/

void nwos_get_object_class_0014(ObjRef* obj, ObjRef* object_class)
{
    EveryObject header;
    uint8 ivec[IVEC_SIZE];

    assert(!is_void_reference(obj));

    memset(ivec, 0, sizeof(ivec));

    nwos_read_object_from_disk_and_decrypt_0014(obj, &header, sizeof(header), ivec, nwos_random_sequence[0]);

    check_common_header(obj, &header.common);

    /* if it is not a reference list check the header checksum */

    if (!is_same_object(&header.common.class_definition, &nwos_reference_list_class_ref_0014))
    {
	check_object_header(obj, &header.object, header.common.header_chksum);
    }

    memcpy(object_class, &header.common.class_definition, sizeof(object_class));
}


void nwos_read_class_definition_0014(ObjRef* ref, C_struct_Class_Definition* class_def)
{
    assert(!is_void_reference(ref));

    nwos_read_object_from_disk_0014(ref, class_def, sizeof(*class_def));
}


