/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:13:14 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4263 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <dirent.h>
#include <limits.h>    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "objectify.h"


static void display_usage(char* prog_name)
{
    fprintf(stderr, "usage: %s [file or directory list]\n", prog_name);
    fprintf(stderr, "   or: %s --help\n", prog_name);
    fprintf(stderr, "\n");
    fprintf(stderr, "  --compressed file\tuse file for archive.\n");
    fprintf(stderr, "  --help\t\tdisplay this message.\n");
    fprintf(stderr, "  --ignore-time\t\tdon't require time stamp to match.\n");
    fprintf(stderr, "  --public\t\tverify files against public data.\n");
    fprintf(stderr, "  -r, --recursive\trecurse through directories.\n");
    fprintf(stderr, "  --remove-no-shred\terase files if they verify correctly.\n");
    fprintf(stderr, "  --remove-min-shred\twrite random data over file and then remove.\n");
    fprintf(stderr, "  --remove-max-shred\tNot implemented yet!\n");

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
    fprintf(stderr, "  PLEASE NOTE: the --remove-min-shred option performs very minor shredding.\n");
    fprintf(stderr, "  With the proper equipment it is very possible that your data could still be\n");
    fprintf(stderr, "  read by someone.  You may want to consider using the GNU shred program until\n");
    fprintf(stderr, "  I can incorporate the GNU shredding code into this program.\n");
    fprintf(stderr, "*******************************************************************************\n");
    fprintf(stderr, "\n");

    fprintf(stderr, "\n");
}


const char* check_invalid_type(mode_t mode)
{
    if (S_ISCHR(mode))
    {
	return "character device";
    }

    if (S_ISBLK(mode))
    {
	return "block device";
    }

    if (S_ISFIFO(mode))
    {
	return "FIFO (named pipe)";
    }

    if (S_ISLNK(mode))
    {
	return "symbolic link";
    }

    if (S_ISSOCK(mode))
    {
	return "socket";
    }

    if (!S_ISREG(mode) && !S_ISDIR(mode))
    {
	return "unknown file type";
    }

    return NULL;
}


#include <fcntl.h>
void shred(const char* path)
{
    struct stat stat_struct;
    int fd;
    ssize_t bytes_written;
    ssize_t total_written = 0;
    uint8 buffer[512];
    int i;

    if (lstat(path, &stat_struct) != 0)
    {
	perror(path);
	exit(1);
    }

    fd = open(path, O_WRONLY | O_SYNC);

    if (fd < 0)
    {
	perror(path);
	exit(1);
    }

    while (total_written < stat_struct.st_size)
    {
	for (i = 0; i < sizeof(buffer); i++) buffer[i] = random();

	bytes_written = write(fd, buffer, sizeof(buffer));

	if (bytes_written != sizeof(buffer))
	{
	    perror(path);
	    exit(1);
	}

	total_written += bytes_written;
    }

#ifdef HAVE_FDATASYNC
    if (fdatasync(fd) != 0)
#else
    if (fsync(fd) != 0)
#endif
    {
	perror(path);
	exit(1);
    }

    if (close(fd) != 0)
    {
	perror(path);
	exit(1);
    }
}


typedef enum { Do_Not_Remove, Just_Remove, Shred_Before_Remove } Remove_Option;

bool process_file(const char* path, MatchCode match, Remove_Option remove_option)
{
    ObjRef ref;
    struct stat stat_struct;
    const char* invalid_type;
    bool result = true;

    printf("%s: ", path);
    fflush(stdout);

    if (lstat(path, &stat_struct) != 0)
    {
	perror(path);
	exit(1);
    }

    invalid_type = check_invalid_type(stat_struct.st_mode);

    if (invalid_type != NULL)
    {
	printf("is a %s, ignored.\n", invalid_type);
    }
    else if (S_ISDIR(stat_struct.st_mode))    // it's not a normal file, investigate further
    {
	printf("is a directory and -r or --recursive was not specified, ignored.\n");
    }
    else if (nwos_find_matching_path_and_file_association(NULL, path, &ref, match))
    {
	if (remove_option != Do_Not_Remove)
	{
	    if (nwos_file_is_stored(&ref))
	    {
		if (nwos_file_is_identical(&ref, path, match))
		{
		    if (remove_option == Shred_Before_Remove)
		    {
			shred(path);
			printf("shredded and ");
		    }

		    if (unlink(path) != 0)
		    {
			perror(path);
			exit(1);
		    }

		    printf("removed\n");
		}
		else
		{
		    printf("WARNING: stored file does not match exactly, NOT removed\n");
		}
	    }
	    else if (nwos_file_has_backup(&ref))
	    {
		if (remove_option == Shred_Before_Remove)
		{
		    shred(path);
		    printf("shredded and ");
		}

		if (unlink(path) != 0)
		{
		    perror(path);
		    exit(1);
		}

		printf("removed\n");
	    }
	    else
	    {
		printf("no known backup, not removed\n");
		result = false;
	    }
	}
	else
	{
	    printf("found: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
    }
    else
    {
	printf("NOT found\n");
	result = false;
    }

    return result;
}


bool process_directory(const char* path, MatchCode match, Remove_Option remove_option)
{
    DIR* dp;
    struct dirent *dir_entry;
    const char* p;
    char full_path[PATH_MAX];
    int i;
    struct stat stat_struct;
    bool result = true;

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    // Construct the full path name

	    i = 0;

	    if (strcmp(path, ".") != 0)
	    {
		for (p = path; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

		if (i < PATH_MAX && full_path[i-1] != '/') full_path[i++] = '/';
	    }

	    for (p = dir_entry->d_name; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX)
	    {
		full_path[i] = '\0';
	    }
	    else
	    {
		fprintf(stderr, "ERROR: path %s/%s exceeds maximum size of a path (PATH_MAX = %d)\n",
			path, dir_entry->d_name, PATH_MAX);
		exit(1);
	    }

	    if (lstat(full_path, &stat_struct) != 0)
	    {
		perror(full_path);
		exit(1);
	    }

	    if (S_ISDIR(stat_struct.st_mode))
	    {
		if (!process_directory(full_path, match, remove_option)) result = false;
	    }
	    else
	    {
		if (!process_file(full_path, match, remove_option)) result = false;
	    }
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }

    if (remove_option != Do_Not_Remove)
    {
	if (strcmp(path, ".") == 0)
	{
	    printf("Warning: will not remove current working directory: '.'!\n");
	}
	else if (rmdir(path) != 0)
	{
	    perror(path);
	}
    }

    return result;
}


int main(int argc, char* argv[])
{
    int i;
    Remove_Option remove_option = Do_Not_Remove;
    bool do_public = false;
    bool do_recursive = false;
    MatchCode match = MatchAll;
    char* path = DEFAULT_FILE;
    char normalized_path[PATH_MAX];
    struct stat stat_struct;
    bool result = true;

    printf("\n");

    if (argc < 2)
    {
	display_usage(argv[0]);
	exit(1);
    }

    for (i = 1; i < argc && argv[i][0] == '-'; i++)
    {
	if (strcmp(argv[i], "--remove-no-shred") == 0)
	{
	    if (remove_option == Shred_Before_Remove)
	    {
		fprintf(stderr, "Error: both --remove-no-shred and --remove-min-shred cannot be specified.\n\n");
		exit(1);
	    }
	    else
	    {
		remove_option = Just_Remove;	// Don't shred it
	    }
	}
	else if (strcmp(argv[i], "--remove-min-shred") == 0)
	{
	    if (remove_option == Just_Remove)
	    {
		fprintf(stderr, "Error: both --remove-min-shred and --remove-no-shred cannot be specified.\n\n");
		exit(1);
	    }

	    remove_option = Shred_Before_Remove;	// Do shred it
	}
	else if (strcmp(argv[i], "--remove-max-shred") == 0)
	{
	    fprintf(stderr, "Error: the --remove-max-shred option has not been implemented yet.  Use the\n");
	    fprintf(stderr, "       --remove-min-shred for a very light shredding or use the GNU shred\n");
	    fprintf(stderr, "       program instead, for a more thorough shredding.\n\n");
	    exit(1);
	}
	else if (strcmp(argv[i], "--public") == 0)
	{
	    do_public = true;
	}
	else if (strcmp(argv[i], "--recursive") == 0 || strcmp(argv[i], "-r") == 0)
	{
	    do_recursive = true;
	}
	else if (strcmp(argv[i], "--ignore-time") == 0)
	{
	    match = IgnoreTime;
	}
	else if (strcmp(argv[i], "--compressed") == 0)
	{
	    i++;
	    path = argv[i];
	}
	else if (strcmp(argv[i], "--help") == 0)
	{
	    display_usage(argv[0]);
	    exit(1);
	}
	else
	{
	    fprintf(stderr, "Unknown option: %s\n\n", argv[i]);
	    display_usage(argv[0]);
	    exit(1);
	}
    }

    nwos_log_arguments(argc, argv);

    if (do_public && path != NULL)   /* both --public and --compressed were specified */
    {
	fprintf(stderr, "WARNING: because --public was specified --compressed file is ignored!\n");
	path = NULL;
    }

    if (do_public)
    {
	nwos_initialize_objectify(PUBLIC, NULL);
    }
    else
    {
	nwos_initialize_objectify(READ_ONLY, path);
    }

    for (/* use i as left by loop above */; i < argc; i++)
    {
	if (strcmp(argv[i], ".") == 0 && strcmp(argv[i], "./") == 0)
	{
	    if (do_recursive)
	    {
		if (!process_directory(".", match, remove_option)) result = false;
	    }
	    else
	    {
		printf("'.' is a directory and -r or --recursive was not specified, ignored.\n");
	    }
	}
	else if (strlen(argv[i]) + 1 > sizeof(normalized_path))
	{
	    fprintf(stderr, "WARNING: the path '%s' is too long for the buffer (exceeds %u characters) - ignored.\n", argv[i], (unsigned)sizeof(normalized_path));
	}
	else if (*argv[i] == '/')
	{
	    fprintf(stderr, "WARNING: the path '%s' is absolute and this version cannot process it - ignored.\n", argv[i]);
	}
	else
	{
	    nwos_normalize_path(normalized_path, argv[i], sizeof(normalized_path));

	    if (normalized_path[0] == '.' && normalized_path[1] == '.' && normalized_path[2] == '/')
	    {
		fprintf(stderr, "Cannot verify from parent directory (%s), will be skipped!\n", argv[i]);
	    }
	    else if (do_recursive)
	    {
		if (lstat(normalized_path, &stat_struct) != 0)
		{
		    perror(normalized_path);
		    exit(1);
		}

		if (S_ISDIR(stat_struct.st_mode))
		{
		    if (!process_directory(normalized_path, match, remove_option)) result = false;
		}
		else
		{
		    if (!process_file(normalized_path, match, remove_option)) result = false;
		}
	    }
	    else
	    {
		if (!process_file(normalized_path, match, remove_option)) result = false;
	    }
	}
    }

    nwos_terminate_objectify();

    return !result;    /* return 0 means success */
}


