/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 05:36:52 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4253 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "class_definition.h"
#include "crc32.h"


ObjRef nwos_public_class_definition_class_ref;
ObjRef nwos_private_class_definition_class_ref;
ObjRef nwos_reference_list_class_ref;


static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


/*********************************************************************************/
/* This routine calculates the size of an object, including variable sized ones. */
/*********************************************************************************/

#define MAX_BUILT_IN_TYPES 4

static size_t feature_size(ObjRef* class_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];
    int i;

    static struct
    {
	ObjRef class;
	size_t size;
    } dictionary[MAX_BUILT_IN_TYPES];

    /* first see if we have figured this one out before */
    for (i = 0; i < MAX_BUILT_IN_TYPES; i++)
    {
	if (is_void_reference(&dictionary[i].class)) break;
	if (is_same_object(class_ref, &dictionary[i].class)) break;
    }

    assert(i < MAX_BUILT_IN_TYPES);

    if (is_void_reference(&dictionary[i].class))
    {
	copy_reference(&dictionary[i].class, class_ref);

	assert(nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size));
	nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

	if (strcmp(name, "Object Reference") == 0)
	{
	    dictionary[i].size = sizeof(ObjRef);
	}
	else if (strcmp(name, "Byte") == 0)
	{
	    dictionary[i].size = sizeof(uint8);
	}
	else if (strcmp(name, "Character") == 0)
	{
	    dictionary[i].size = sizeof(char);
	}
	else if (strcmp(name, "Time Stamp") == 0)
	{
	    dictionary[i].size = sizeof(TimeStamp);
	}
	else
	{
	    bool Known_Built_in_Type = false;
	    assert(Known_Built_in_Type);
	}

	assert(MAX_BUILT_IN_TYPES == 4);  /* this must match how many there are */
    }

    return dictionary[i].size;
}


size_t nwos_get_object_size(void* object)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    C_struct_Feature_Definition feature_obj;
    char label[32] = "";
    EveryObject* header = (EveryObject*) object;
    size_t result = sizeof(EveryObject);
    size_t size;
    size_t prev_size = 0;
    uint8 prev_count = 0;
    int i;
    int item_count;
    static bool no_more_calls;

    if (no_more_calls) return 0;

    no_more_calls = true;

    assert(nwos_read_variable_sized_object_from_disk(&header->common.class_definition, kludge, sizeof(kludge), &get_class_object_size));

    for (i = 0; i < class_def_obj_ptr->count; i++)
    {
	assert(nwos_read_object_from_disk(&class_def_obj_ptr->feature[i], &feature_obj, sizeof(feature_obj)));

	size = feature_size(&feature_obj.class);

	/* deal with variable sized object if previous was named count */
	if (feature_obj.count == 0 && strcmp(label, "count"))
	{
	    if (prev_count == 1)   /* uint8 count */
	    {
		item_count = *((uint8*)object + result - 1);
	    }
	    else
	    {
		assert(prev_count == 4);
		item_count = nwos_decode_variable_sized_count((uint8*)object + result - 4);
	    }
	}
	else
	{
	    item_count = feature_obj.count;
	}

	result += size * item_count;

	/* deal with alignment pain in the ass */
	if (prev_size == 1 && (size == 4 || feature_obj.count == 0 || i+1 == class_def_obj_ptr->count))
	{
	    result += 3 - (prev_count - 1) % 4;
	}

	nwos_name_to_string(&feature_obj.label, label, sizeof(label));
#if 0
	printf("label: %s  size: %zd  prev_size: %zd  count: %u  item_count: %d  result: %zd\n",
	       label, size, prev_size, feature_obj.count, item_count, result);
#endif
	prev_size = size;
	prev_count = feature_obj.count;
    }

    no_more_calls = false;

    return result;
}





/*****************************************************************************/
/* This routine clones an updated public class definition for a class that   */
/* has been cloned.                                                          */
/*****************************************************************************/

void nwos_clone_updated_class_definition(ObjRef* old_private_ref, ObjRef* clone_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char msg[128];
    ObjRef public_ref;


    nwos_generate_new_id(clone_ref);

    assert(nwos_read_variable_sized_object_from_disk(old_private_ref, kludge, sizeof(kludge), &get_class_object_size));

    copy_reference(&class_def_obj_ptr->header.object.next_version, clone_ref);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_overwrite_object_to_disk(old_private_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    /* get the old public object */
    assert(nwos_read_variable_sized_object_from_disk(&class_def_obj_ptr->header.object.clone_of, kludge, sizeof(kludge), &get_class_object_size));

    assert(!is_void_reference(&class_def_obj_ptr->header.object.next_version));

    while (!is_void_reference(&class_def_obj_ptr->header.object.next_version))
    {
	copy_reference(&public_ref, &class_def_obj_ptr->header.object.next_version);
	assert(nwos_read_variable_sized_object_from_disk(&public_ref, kludge, sizeof(kludge), &get_class_object_size));
    }

    snprintf(msg, sizeof(msg), "clone_class_definition(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	   public_ref.id[0], public_ref.id[1], public_ref.id[2], public_ref.id[3],
	   clone_ref->id[0], clone_ref->id[1], clone_ref->id[2], clone_ref->id[3]);
    nwos_log(msg);

    copy_reference(&class_def_obj_ptr->header.common.id, clone_ref);

    nwos_get_time_stamp(class_def_obj_ptr->header.common.creation_time);

    copy_reference(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref);

    copy_reference(&class_def_obj_ptr->header.object.clone_of, &public_ref);
    copy_reference(&class_def_obj_ptr->header.object.prev_version, old_private_ref);

    nwos_create_reference_list(clone_ref, &class_def_obj_ptr->header.object.references);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_write_object_to_disk(clone_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    nwos_add_to_references(clone_ref, &nwos_private_class_definition_class_ref);
}



/*********************************************************************************/
/* This routine finds or creates a private class definition with the given name. */
/*********************************************************************************/

ObjCreateResult nwos_find_or_create_private_class_definition(char* name_of_class, ObjRef* ref)
{
    ObjRef public_ref;
    ObjRef old_private_ref;
    ObjRef new_private_ref;
    EveryObject header;
    ObjCreateResult result = FOUND_EXISTING;

    if (nwos_find_private_class_definition(name_of_class, ref))   /* see if public is updated */
    {
	nwos_read_object_headers_from_disk(ref, &header);

	nwos_read_object_headers_from_disk(&header.object.clone_of, &header);

	if (!is_void_reference(&header.object.next_version))  /* the public class has been updated */
	{
	    copy_reference(&old_private_ref, ref);

	    nwos_clone_updated_class_definition(&old_private_ref, ref);
	    assert(!is_same_object(&old_private_ref, ref));

	    assert(nwos_find_private_class_definition(name_of_class, &new_private_ref));
	    assert(is_same_object(&new_private_ref, ref));

	    nwos_read_object_headers_from_disk(ref, &header);

	    nwos_read_object_headers_from_disk(&header.object.clone_of, &header);

	    assert(is_void_reference(&header.object.next_version));

	    result = CREATED_NEW;
	}
    }
    else
    {
	/* if there isn't a private one there certainly had better be a public one */
	assert(nwos_find_public_class_definition(name_of_class, &public_ref));

	nwos_clone_class_definition(&public_ref, ref);

	result = CREATED_NEW;
    }

    return result;
}


/*****************************************************************************/
/* This routine clones a public class definition.                            */
/*****************************************************************************/

void nwos_clone_class_definition(ObjRef* class_to_clone_ref, ObjRef* clone_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char msg[128];

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(class_to_clone_ref, kludge, sizeof(kludge), &get_class_object_size));

    nwos_generate_new_id(clone_ref);

    snprintf(msg, sizeof(msg), "clone_class_definition(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
	   class_to_clone_ref->id[0], class_to_clone_ref->id[1], class_to_clone_ref->id[2], class_to_clone_ref->id[3],
	   clone_ref->id[0], clone_ref->id[1], clone_ref->id[2], clone_ref->id[3]);
    nwos_log(msg);

    copy_reference(&class_def_obj_ptr->header.common.id, clone_ref);

    nwos_get_time_stamp(class_def_obj_ptr->header.common.creation_time);

    copy_reference(&class_def_obj_ptr->header.common.class_definition, &nwos_private_class_definition_class_ref);

    copy_reference(&class_def_obj_ptr->header.object.clone_of, class_to_clone_ref);

    nwos_create_reference_list(clone_ref, &class_def_obj_ptr->header.object.references);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->header.object, sizeof(ObjectHeader), class_def_obj_ptr->header.common.header_chksum);

    nwos_write_object_to_disk(clone_ref, class_def_obj_ptr, get_class_object_size(class_def_obj_ptr));

    nwos_add_to_references(clone_ref, &nwos_private_class_definition_class_ref);
}



/*****************************************************************************/
/* This routine finds the class definition with the given name.              */
/*****************************************************************************/

bool nwos_find_public_class_definition(char* name, ObjRef* class_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ReferenceList* ref_list;
    int num_classes;
    ObjRef object_class;
    char buffer[128];
    int i;
    bool result = false;

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_public_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_classes = ref_list->common_header.num_refs;

    /* printf("num_classes (in find_class_definition): %d\n", num_classes); */

    for (i = 0; i < num_classes; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &nwos_public_class_definition_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_class_object_size));

	    if (is_void_reference(&class_def_obj_ptr->header.object.next_version))
	    {
		nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));  /* get it's name */

		if (strcasecmp(name, buffer) == 0)
		{
		    memcpy(class_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


bool nwos_find_private_class_definition(char* name, ObjRef* class_ref)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ReferenceList* ref_list;
    int num_classes;
    ObjRef object_class;
    char buffer[128];
    int i;
    bool result = false;

    memset(kludge, 0, sizeof(kludge));

    assert(nwos_read_variable_sized_object_from_disk(&nwos_private_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size));

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_classes = ref_list->common_header.num_refs;

    /* printf("num_classes (in find_class_definition): %d\n", num_classes); */

    for (i = 0; i < num_classes; i++)
    {
	/* find out what kind of object it is without update to keep from getting into a recursive mess */
	nwos_get_object_class_without_update(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &nwos_private_class_definition_class_ref))   /* it is a class definition object */
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_class_object_size));

	    if (is_void_reference(&class_def_obj_ptr->header.object.next_version))
	    {
		nwos_name_to_string(&class_def_obj_ptr->name, buffer, sizeof(buffer));   /* get it's name */

		if (strcasecmp(name, buffer) == 0)
		{
		    memcpy(class_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


#if 0
bool nwos_find_class_definition(char* name, ObjRef* class_ref)
{
    bool result = true;

    if (is_void_reference(&nwos_private_class_definition_class_ref) || !nwos_find_private_class_definition(name, class_ref))
    {
	result = nwos_find_public_class_definition(name, class_ref);
    }

    return result;
}
#endif


void nwos_read_class_definition(ObjRef* ref, C_struct_Class_Definition* class_def)
{
    uint8 kludge[MAX_CLASS_DEFINITION_OBJ_SIZE];
    assert(!is_void_reference(ref));

    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), get_class_object_size));

    /* Holy shit Batman this is a mess! */
    /* memcpy(class_def, kludge, get_class_object_size(kludge)); */

    memcpy(class_def, kludge, sizeof(C_struct_Class_Definition));  /* for now just copy the first part of the class without features*/
}


static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);

    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}


bool nwos_find_public_feature_definition(ObjRef* type_ref, char* label, int count, ObjRef* ref)
{
    ObjRef label_ref;
    ObjRef feature_class_ref;
    ObjRef object_class;
    uint8 kludge[MAX_FEATURE_NAME_OBJ_SIZE];
    C_struct_Name* ptr_name_obj = (C_struct_Name*)kludge;
    C_struct_Feature_Definition feature_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    if (nwos_find_public_name(label, &label_ref))
    {
	printf("Found public name: %s\n", label);

	assert(nwos_find_public_class_definition("FEATURE DEFINITION", &feature_class_ref));

	assert(nwos_read_variable_sized_object_from_disk(&label_ref, kludge, sizeof(kludge), &get_name_object_size));
	ref_list = nwos_malloc_reference_list(&ptr_name_obj->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&feature_class_ref, &object_class))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &feature_obj, sizeof(feature_obj)));

		assert(is_same_object(&feature_obj.label, &label_ref));

		if (is_same_object(&feature_obj.class, type_ref) && feature_obj.count == count)
		{
		    copy_reference(ref, &ref_list->references[i]);
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


#ifdef PUBLIC_MODE
void nwos_find_or_create_public_feature_definition(ObjRef* type_ref, char* label, int count, ObjRef* ref)
{
    C_struct_Feature_Definition feature_obj;
    ObjRef feature_class_ref;

    if (!nwos_find_public_feature_definition(type_ref, label, count, ref))
    {
	assert(nwos_find_public_class_definition("FEATURE DEFINITION", &feature_class_ref));

	nwos_generate_new_id(ref);

	memset(&feature_obj, 0, sizeof(feature_obj));

	nwos_fill_in_common_header(&feature_obj.header.common, ref, &feature_class_ref);

	copy_reference(&feature_obj.class, type_ref);
	nwos_create_name(label, &feature_obj.label);
	feature_obj.count = count;

	nwos_create_reference_list(ref, &feature_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &feature_obj.header.object, sizeof(ObjectHeader), feature_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &feature_obj.inherit, sizeof(feature_obj) - sizeof(EveryObject), feature_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &feature_obj, sizeof(feature_obj));

        nwos_add_to_references(ref, &feature_class_ref);
        nwos_add_to_references(ref, &feature_obj.label);
    }
}
#endif


/*****************************************************************************/
/* Functions to deal with the various file class definitions                 */
/*****************************************************************************/

#ifndef USE_PREDEFINED_STRUCTS

/* These should only be referenced by using the get_file_XXX_reference functions below! */
static ObjRef public_file_references[3];
static ObjRef file_references[3];

static void get_file_references()
{
    static volatile bool done = false;
    ObjRef class_ref;
    EveryObject header;
    int i;

    /* can't do anything until everything is initialized */
    if (nwos_is_ready() && !done)
    {
	done = true;   /* in case we get called recursively */

	/* we know that as of version 0027 there are 3 versions of the file class */
	nwos_find_public_class_definition("FILE", &public_file_references[2]);

	nwos_read_object_headers_from_disk(&public_file_references[2], &header);
	assert(!is_void_reference(&header.object.prev_version));
	assert(is_void_reference(&header.object.next_version));

	copy_reference(&public_file_references[1], &header.object.prev_version);

	nwos_read_object_headers_from_disk(&public_file_references[1], &header);
	assert(!is_void_reference(&header.object.prev_version));
	assert(is_same_object(&header.object.next_version, &public_file_references[2]));

	copy_reference(&public_file_references[0], &header.object.prev_version);

	nwos_read_object_headers_from_disk(&public_file_references[0], &header);
	assert(is_void_reference(&header.object.prev_version));
	assert(is_same_object(&header.object.next_version, &public_file_references[1]));

	copy_reference(&file_references[0], &public_file_references[0]);
	copy_reference(&file_references[1], &public_file_references[1]);
	copy_reference(&file_references[2], &public_file_references[2]);

#ifndef PUBLIC_MODE
	if (nwos_find_private_class_definition("FILE", &class_ref))       /* are there any private files? */
	{
	    nwos_read_object_headers_from_disk(&class_ref, &header);

	    assert(is_void_reference(&header.object.next_version));
	    assert(!is_void_reference(&header.object.clone_of));

	    for (i = 2; i >= 0; i--)
	    {
		if (is_same_object(&public_file_references[i], &header.object.clone_of))
		{
		    copy_reference(&file_references[i], &class_ref);
		    break;
		}
	    }

	    while (i > 0)
	    {
		assert(!is_void_reference(&header.object.prev_version));

		copy_reference(&class_ref, &header.object.prev_version);

		if (nwos_reference_type(&class_ref) == Public_Reference) break;

		nwos_read_object_headers_from_disk(&class_ref, &header);

		assert(is_same_object(&header.object.next_version, &file_references[i]));

		i--;

		assert(is_same_object(&public_file_references[i], &header.object.clone_of));

		copy_reference(&file_references[i], &class_ref);
	    }
	}
#endif
    }

#if 0
    printf("public_file_reference[0]: %08x\n", nwos_ref_to_word(&public_file_references[0]));
    printf("public_file_reference[1]: %08x\n", nwos_ref_to_word(&public_file_references[1]));
    printf("public_file_reference[2]: %08x\n", nwos_ref_to_word(&public_file_references[2]));
    printf("\n");
    printf("file_reference[0]: %08x\n", nwos_ref_to_word(&file_references[0]));
    printf("file_reference[1]: %08x\n", nwos_ref_to_word(&file_references[1]));
    printf("file_reference[2]: %08x\n", nwos_ref_to_word(&file_references[2]));
#endif
}


ObjRef* nwos_get_file_001_reference()
{
    if (is_void_reference(&file_references[0]))
    {
	get_file_references();
    }

    return &file_references[0];
}


ObjRef* nwos_get_file_002_reference()
{
    if (is_void_reference(&file_references[1]))
    {
	get_file_references();
    }

    return &file_references[1];
}


ObjRef* nwos_get_file_class_reference()
{
    if (is_void_reference(&file_references[2]))
    {
	get_file_references();
    }
#ifndef PUBLIC_MODE
    /* if storage is writable check to see if an updated file class has been created since this was first initialized */
    else if (nwos_is_writable() && nwos_reference_type(&file_references[2]) == Public_Reference)
    {
	ObjRef class_ref;
	EveryObject header;

	if (nwos_find_private_class_definition("FILE", &class_ref))       /* see if one has been created since */
	{
	    nwos_read_object_headers_from_disk(&class_ref, &header);

	    if (is_same_object(&header.object.clone_of, &public_file_references[2]))
	    {
		copy_reference(&file_references[2], &class_ref);
		printf("Updating file class reference 2: %08x\n", nwos_ref_to_word(&file_references[2]));
	    }
	}
    }
#endif

    return &file_references[2];
}
#endif


