/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-01 08:34:47 -0600 (Sat, 01 Aug 2009) $
--   $Revision: 4227 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file, the disk_io.c file (which these
--   functions were created from), and the objectify.c file (which the
--   disk_io.c file was created from).
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <stdio.h>
#include <string.h>

#include "chunk_info.h"     /* define BLOCKS_IN_CHUNK */
#include "config.h"
#include "header.h"


Disk_Header nwos_public_disk_header;    /* these should only be accessed by storage.c to pass to backup */
Disk_Header nwos_private_disk_header;

uint64 nwos_used_public_blocks;
uint64 nwos_used_private_blocks;

uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_chunks = 0;
uint32 nwos_used_private_chunks;

char   nwos_version_string[5];   /* this is the version of the archive being read, can be older if it is a compressed file. */


static const char* check_disk_header(Disk_Header* disk_header, bool compressed_file, const char* type_code_1, const char* type_code_2, const char* type_code_3)
{
    assert(type_code_1 != NULL);

    if (memcmp(disk_header->magic_number, MAGIC_NUMBER, 4) != 0)
    {
	return "Missing magic number in disk header";
    }

    if (memcmp(disk_header->version_string, VERSION_STRING, 4) != 0)
    {
	// allow more compressed files to be other compatible versions

	if ((compressed_file && memcmp(disk_header->version_string, OLDEST_COMPATIBLE_COMPRESSED_VERSION, 4 < 0)) ||
	    (!compressed_file && memcmp(disk_header->version_string, VERSION_STRING, 4) != 0))
	{
	    return "Incorrect version string in disk header";
	}

	if (compressed_file && memcmp(disk_header->version_string, "0030", 4) < 0)
	{
	    return NULL;
	}
    }

    if (memcmp(disk_header->type_code, type_code_1, 4) != 0)
    {
	if (type_code_2 == NULL || memcmp(disk_header->type_code, type_code_2, 4) != 0)
	{
	    if (type_code_3 == NULL || memcmp(disk_header->type_code, type_code_3, 4) != 0)
            {
		return "Incorrect file type";
	    }
	}
    }

    return NULL;   /* no errors */
}


const char* nwos_check_disk_header(Disk_Header* disk_header, bool compressed_file, const char* type_code)
{
    return check_disk_header(disk_header, compressed_file, type_code, NULL, NULL);
}


const char* nwos_load_public_data(void* buffer, size_t buffer_size)
{
    const char* error_msg;

    assert(buffer_size >= sizeof(nwos_public_disk_header));

    memcpy(&nwos_public_disk_header, buffer, sizeof(nwos_public_disk_header));

    error_msg = nwos_check_disk_header(&nwos_public_disk_header, false, TYPE_CODE_PUBLIC);   /* don't allow older versions */

    if (error_msg == NULL)
    {
	nwos_8_uint8_to_uint64(nwos_public_disk_header.used_blocks, &nwos_used_public_blocks);
    }

    return error_msg;
}


#ifdef PUBLIC_MODE
void nwos_store_public_data(void* buffer, size_t buffer_size)
{
    nwos_used_public_blocks = nwos_next_public_ref;

    nwos_uint64_to_8_uint8(&nwos_used_public_blocks, &nwos_public_disk_header.used_blocks);

    nwos_get_time_stamp(nwos_public_disk_header.last_change);

    assert(buffer_size >= sizeof(nwos_public_disk_header));

    memcpy(buffer, &nwos_public_disk_header, sizeof(nwos_public_disk_header));
}
#endif


const char* nwos_load_private_data(void* buffer, size_t buffer_size, bool compressed_file)
{
    const char* error_msg;

    assert(buffer_size >= sizeof(nwos_private_disk_header));

    memcpy(&nwos_private_disk_header, buffer, sizeof(nwos_private_disk_header));

    error_msg = check_disk_header(&nwos_private_disk_header, compressed_file, TYPE_CODE_DISK, TYPE_CODE_FILE, TYPE_CODE_COMPRESSED);

    if (error_msg == NULL)
    {
	assert(sizeof(nwos_version_string) == 5);
	memcpy(nwos_version_string, nwos_private_disk_header.version_string, 4);
	nwos_version_string[4] = '\0';

	nwos_4_uint8_to_uint32(nwos_private_disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(nwos_private_disk_header.total_chunks, &nwos_total_private_chunks);
	nwos_4_uint8_to_uint32(nwos_private_disk_header.used_chunks, &nwos_used_private_chunks);
	nwos_8_uint8_to_uint64(nwos_private_disk_header.used_blocks, &nwos_used_private_blocks);

	assert(nwos_total_private_chunks > 0);
    }

    return error_msg;
}


void nwos_store_private_data(void* buffer, size_t buffer_size, bool archive_is_file)
{
    uint32 total_private_chunks;

    nwos_uint64_to_8_uint8(&nwos_used_private_blocks, nwos_private_disk_header.used_blocks);
    nwos_uint32_to_4_uint8(&nwos_used_private_chunks, nwos_private_disk_header.used_chunks);

    if (archive_is_file)   /* totals could have changed */
    {
	nwos_uint32_to_4_uint8(&nwos_total_private_chunks, nwos_private_disk_header.total_chunks);
    }
    else                   /* it's a disk, make sure it didn't */
    {
	nwos_4_uint8_to_uint32(nwos_private_disk_header.total_chunks, &total_private_chunks);
	assert(nwos_total_private_chunks == total_private_chunks);
    }

    nwos_get_time_stamp(nwos_private_disk_header.last_change);

    assert(buffer_size >= sizeof(nwos_private_disk_header));

    memcpy(buffer, &nwos_private_disk_header, sizeof(nwos_private_disk_header));
}


