/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:34:21 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4265 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "header.h"


uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


static int version_string_to_int(char string[4])
{
    int i;
    int result = 0;

    for (i = 0; i < 4; i++) if (string[i] != '0') break;

    while (i < 4)
    {
	if (!isdigit(string[i])) return -1;

	result = result * 10 + string[i] - '0';

	i++;
    }

    return result;
}


int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    FILE* fp3 = NULL;
    unsigned char buf1a[FILE_BLOCK_SIZE];
    unsigned char buf2a[FILE_BLOCK_SIZE];
    int block1;
    int block2;
    int changed = 0;
    int added = 0;
    size_t read1;
    size_t read2;
    size_t write3;
    Disk_Header header1;
    Disk_Header header2;
    Disk_Header_0022_to_0029 old_header1;
    Disk_Header_0022_to_0029 old_header2;
    bool wrapped_file = false;     /* true if older file that wasn't sequencial */
    bool bad_upper_ref;
    int header1_version;
    int header2_version;
    uint32 header1_chunks;
    uint32 header2_chunks;
    const char* error_msg;

    if (argc != 3 && argc != 4)
    {
	fprintf(stderr, "usage: %s old_file new_file [outfile]\n", argv[0]);
	exit(1);
    }


    /* Open the first file and check it */

    fp1 = fopen(argv[1], "r");
    if (fp1 == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1a, sizeof(header1));

    error_msg = nwos_check_disk_header(&header1, true, TYPE_CODE_COMPRESSED);  /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Open the second file and check it */

    fp2 = fopen(argv[2], "r");
    if (fp2 == NULL)
    {
	perror(argv[2]);
	exit(1);
    }

    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    memcpy(&header2, buf2a, sizeof(header2));

    error_msg = nwos_check_disk_header(&header2, true, TYPE_CODE_COMPRESSED);  /* allow older versions */

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Verify they are the same version and disk size */

    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
	sleep(5);
    }

    header1_version = version_string_to_int(header1.version_string);

    if (header1_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[1]);
	exit(1);
    }

    header2_version = version_string_to_int(header2.version_string);

    if (header2_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[2]);
	exit(1);
    }

    if (header1_version > header2_version)
    {
	fprintf(stderr, "ERROR: '%s' is a newer version than '%s', cannot diff!\n", argv[1], argv[2]);
	exit(1);
    }

    if (header1_version < 23 && header2_version < 23)
    {
	wrapped_file = true;   /* it's an old file */
    }
    else if (header1_version < 23)
    {
	fprintf(stderr, "ERROR: '%s' is an old wrapped file, but file '%s' is not, cannot diff!\n", argv[1], argv[2]);
	exit(1);
    }

    if (header1_version < 30)
    {
	memcpy(&old_header1, buf1a, sizeof(old_header1));
	header1_chunks = convert_4_uint8_to_uint32(old_header1.total_blocks) / 65536; /* old chunks were 65536 blocks including bit map */
    }
    else
    {
	header1_chunks = convert_4_uint8_to_uint32(header1.total_chunks);
    }

    if (header2_version < 30)
    {
	memcpy(&old_header2, buf2a, sizeof(old_header2));
	header2_chunks = convert_4_uint8_to_uint32(old_header2.total_blocks) / 65536; /* old chunks were 65536 blocks including bit map */
    }
    else
    {
	header2_chunks = convert_4_uint8_to_uint32(header2.total_chunks);
    }

    if (header1_chunks != header2_chunks)
    {
	fprintf(stderr, "Warning: disk total chunks size different - %s: %u  %s: %u\n", argv[1], header1_chunks, argv[2], header2_chunks);
    }

    if (argc == 4)
    {
	fp3 = fopen(argv[3], "w");

	if (fp3 == NULL)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    exit(1);
	}
    }

    if (fp3 != NULL)   /* write both headers into the patch file */
    {
	if (header1_version >= 30)
	{
	    memcpy(&(((Disk_Header*)buf1a)->type_code), TYPE_CODE_DIFF, sizeof((((Disk_Header*)buf1a)->type_code)));
	}
	    
	write3 = fwrite(buf1a, 1, sizeof(buf1a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	if (header2_version >= 30)
	{
	    memcpy(&(((Disk_Header*)buf2a)->type_code), TYPE_CODE_DIFF, sizeof((((Disk_Header*)buf2a)->type_code)));
	}
	    
	write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}
    }

    block1 = 1;
    block2 = 1;

    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
    read2 = fread(buf2a, 1, sizeof(buf2a), fp2);

    while (!feof(fp1) && !feof(fp2) && read1 == FILE_BLOCK_SIZE && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header1_version < 30)
	{
	    if (buf1a[0] != 0 || buf1a[1] != 0 || buf1a[2] != 0 || buf1a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[1], block1, buf1a[0], buf1a[1], buf1a[2], buf1a[3]);
		bad_upper_ref = true;
	    }

	    if (header2_version >= 30)
	    {
		buf1a[0] = 0xff;
		buf1a[1] = 0xff;
		buf1a[2] = 0xff;
		buf1a[3] = 0xfe;
	    }
	}
	else 
	{
	    if (buf1a[0] != 0xff || buf1a[1] != 0xff || buf1a[2] != 0xff || buf1a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[1], block1, buf1a[0], buf1a[1], buf1a[2], buf1a[3]);
		bad_upper_ref = true;
	    }
	}

	if (header2_version < 30)
	{
	    if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2a[0] != 0xff || buf2a[1] != 0xff || buf2a[2] != 0xff || buf2a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    if (fp3 != NULL) fclose(fp3);

	    exit(1);
	}

	if (buf1a[4] == buf2a[4] && buf1a[5] == buf2a[5] && buf1a[6] == buf2a[6] && buf1a[7] == buf2a[7])
	{
	    /* blocks have the same id number, just compare them */

	    if (memcmp(buf1a, buf2a, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1a[4], buf1a[5], buf1a[6], buf1a[7]);

		if (fp3 != NULL)   /* write the changed block */
		{
		    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		    if (write3 != FILE_BLOCK_SIZE)
		    {
			perror(argv[2]);
			fclose(fp1);
			fclose(fp2);
			fclose(fp3);
			exit(1);
		    }
		}

		changed++;
	    }

	    read1 = fread(buf1a, 1, sizeof(buf1a), fp1);
	    block1++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    printf("%02x%02x%02x%02x: added\n",
		   buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	    if (fp3 != NULL)   /* write the new block */
	    {
		if (wrapped_file)   /* write the id of the next old block into bytes 0-3 of the new block */
		{
		    buf2a[0] = buf1a[4];
		    buf2a[1] = buf1a[5];
		    buf2a[2] = buf1a[6];
		    buf2a[3] = buf1a[7];
		}

		write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

		if (write3 != FILE_BLOCK_SIZE)
		{
		    perror(argv[3]);
		    fclose(fp1);
		    fclose(fp2);
		    fclose(fp3);
		    exit(1);
		}
	    }

	    added++;

	    /* Change the upper 4 bytes back because we are going through the loop again */
	    if (header1_version < 30 && header2_version >= 30)
	    {
		buf1a[0] = 0;
		buf1a[1] = 0;
		buf1a[2] = 0;
		buf1a[3] = 0;
	    }
	}

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    if (feof(fp2) && !feof(fp1))
    {
	fprintf(stderr, "WARNING: end of file reached on %s before %s\n",
		argv[2], argv[1]);
    }

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	bad_upper_ref = false;

	if (header2_version < 30)
	{
	    if (buf2a[0] != 0 || buf2a[1] != 0 || buf2a[2] != 0 || buf2a[3] != 0)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}
	else
	{
	    if (buf2a[0] != 0xff || buf2a[1] != 0xff || buf2a[2] != 0xff || buf2a[3] != 0xfe)
	    {
		fprintf(stderr, "\n%s block %d - first four bytes not fffffffe: %02x%02x%02x%02x\n",
			argv[2], block2, buf2a[0], buf2a[1], buf2a[2], buf2a[3]);
		bad_upper_ref = true;
	    }
	}

	if (bad_upper_ref)
	{
	    fclose(fp1);
	    fclose(fp2);
	    if (fp3 != NULL) fclose(fp3);

	    exit(1);
	}

	printf("%02x%02x%02x%02x: added\n",
	       buf2a[4], buf2a[5], buf2a[6], buf2a[7]);

	if (fp3 != NULL)   /* write the new block */
	{
	    write3 = fwrite(buf2a, 1, sizeof(buf2a), fp3);

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}

	added++;

	read2 = fread(buf2a, 1, sizeof(buf2a), fp2);
	block2++;
    }

    printf("\n");


    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }

    printf("Added: %d\n", added);
    printf("Changed: %d\n", changed);

    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(argv[1]);
    }

    fclose(fp1);

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[2]);
    }

    fclose(fp2);
    
    if (fp3 != NULL)   /* close the output file */
    {
	if (fclose(fp3) != 0)
	{
	    perror(argv[3]);
	}
    }

    return 0;
}
