/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"



static size_t get_url_object_size(void* url_obj)
{
    assert(((C_struct_Uniform_Resource_Locator*)url_obj)->count > 0);

    return sizeof(C_struct_Uniform_Resource_Locator) + ((C_struct_Uniform_Resource_Locator*)url_obj)->count;
}


bool nwos_find_public_url(char* url, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_URL_OBJ_SIZE];
    C_struct_Uniform_Resource_Locator* ptr_url_obj = (C_struct_Uniform_Resource_Locator*)kludge;
    ObjRef public_url_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;


    length = strlen(url);

    assert(length <= 255);
    
    nwos_find_public_class_definition("UNIFORM RESOURCE LOCATOR", &public_url_class_ref);

    nwos_read_class_definition(&public_url_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &public_url_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_url_object_size));

	    /* remember ptr_url_obj points to the kludge buffer */

	    if (ptr_url_obj->count == length && strncasecmp((char*)ptr_url_obj->storage, url, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing url or create new */

#ifdef PUBLIC_MODE
ObjCreateResult nwos_create_url(char* url, ObjRef* ref)
{
    uint8 kludge[MAX_URL_OBJ_SIZE];
    C_struct_Uniform_Resource_Locator* ptr_url_obj = (C_struct_Uniform_Resource_Locator*)kludge;
    size_t length;
    ObjRef public_url_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    C_struct_Class_Definition class_def_obj;

    length = strlen(url);

    assert(length <= 255);

    /* first find out if we already have this url */

    if (!nwos_find_public_url(url, ref))   /* didn't find it */
    {
	nwos_find_public_class_definition("UNIFORM RESOURCE LOCATOR", &public_url_class_ref);

	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_url_obj points to the kludge buffer */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_url_obj->header.common, ref, &public_url_class_ref);

	nwos_read_class_definition(&public_url_class_ref, &class_def_obj);

	ptr_url_obj->count = length;
	for (i = 0; i < length; i++) ptr_url_obj->storage[i] = tolower(url[i]);

	nwos_create_reference_list(ref, &ptr_url_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_url_obj->header.object, sizeof(ObjectHeader), ptr_url_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_url_obj->start_date, sizeof(C_struct_Uniform_Resource_Locator) + length - sizeof(EveryObject), ptr_url_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Uniform_Resource_Locator) + length);

	nwos_add_to_references(ref, &public_url_class_ref);

#ifdef VERIFY_WRITE
	ptr_url_obj = malloc(sizeof(C_struct_Uniform_Resource_Locator) + length);
	assert(nwos_read_object_from_disk(ref, ptr_url_obj, sizeof(C_struct_Uniform_Resource_Locator) + length));
	assert(memcmp(kludge, ptr_url_obj, sizeof(C_struct_Uniform_Resource_Locator) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_url_object_size));  /* read the other way */
	assert(memcmp(ptr_url_obj, kludge, sizeof(C_struct_Uniform_Resource_Locator) + length) == 0);

	free(ptr_url_obj);
	ptr_url_obj = NULL;
#endif

	result = CREATED_NEW;
    }

    return result;
}
#endif


bool nwos_url_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[MAX_URL_OBJ_SIZE];
    C_struct_Uniform_Resource_Locator* ptr_url_obj = (C_struct_Uniform_Resource_Locator*)kludge;


    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_url_object_size));

    /* remember ptr_url_obj points to the kludge buffer */

    if (size < (ptr_url_obj->count + 1))
    {
	return false;
    }

    memcpy(string, ptr_url_obj->storage, ptr_url_obj->count);
    string[ptr_url_obj->count] = '\0';

    return true;
}


