/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include "crc32.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



static char* file_is_used_in_disc_list(ObjRef* disc_list_class_ref, C_struct_Path_And_File_Association* assoc_obj)
{
    ObjRef object_class;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    static char id[sizeof(ptr_list_obj->id) + 1];
    char* result = NULL;


    ref_list = nwos_malloc_reference_list(&assoc_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, disc_list_class_ref))
	{
	    assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

	    memcpy(id, ptr_list_obj->id, sizeof(id));
	    id[sizeof(ptr_list_obj->id)] = '\0';

	    result = id;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return result;
}


/* NOTE: assoc_obj is modified by this function! */

void create_new_path(char* new_path, C_struct_Path_And_File_Association *assoc_obj)
{
#ifdef VERIFY_WRITE
    C_struct_Path_And_File_Association temp;
#endif
    ObjRef assoc_class_ref;
    ObjRef new_path_ref;
    ObjRef old_assoc_ref;
    ObjRef new_assoc_ref;


    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    assert(nwos_create_file_path(new_path, &new_path_ref) == CREATED_NEW);

    copy_reference(&old_assoc_ref, &assoc_obj->header.common.id);

    nwos_generate_new_id(&new_assoc_ref);


    /***********************************************************************/
    /* Point to the new association object from the old association object */
    /***********************************************************************/

    assert(is_void_reference(&assoc_obj->header.object.next_version));   /* above code checks this */

    copy_reference(&assoc_obj->header.object.next_version, &new_assoc_ref);

    nwos_crc32_calculate((uint8*) &assoc_obj->header.object, sizeof(ObjectHeader), assoc_obj->header.common.header_chksum);

    nwos_overwrite_object_to_disk(&old_assoc_ref, assoc_obj, sizeof(*assoc_obj));

#ifdef VERIFY_WRITE
    /* read back to verify */
    assert(nwos_read_object_from_disk(&old_assoc_ref, &temp, sizeof(temp)));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);
#endif

    void_reference(&assoc_obj->header.object.next_version);   /* remove it before creating the new */


    /*****************************************/
    /* Now create the new association object */
    /*****************************************/

    nwos_fill_in_common_header(&assoc_obj->header.common, &new_assoc_ref, &assoc_class_ref);

    copy_reference(&assoc_obj->header.object.prev_version, &old_assoc_ref);

    copy_reference(&assoc_obj->path, &new_path_ref);

    nwos_create_reference_list(&new_assoc_ref, &assoc_obj->header.object.references);

    nwos_crc32_calculate((uint8*) &assoc_obj->header.object, sizeof(ObjectHeader), assoc_obj->header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &assoc_obj->path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj->header.common.data_chksum);

    nwos_write_object_to_disk(&new_assoc_ref, assoc_obj, sizeof(*assoc_obj));

    nwos_add_to_references(&new_assoc_ref, &assoc_class_ref);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj->path);
    nwos_add_to_references(&new_assoc_ref, &assoc_obj->file);

#ifdef VERIFY_WRITE
    /* read back to verify */
    assert(nwos_read_object_from_disk(&new_assoc_ref, &temp, sizeof(temp)));
    assert(memcmp(&temp, assoc_obj, sizeof(temp)) == 0);
#endif
}


int main(int argc, char* argv[])
{
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[MAX_PATH_OBJ_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef list_class_ref;
    ObjRef old_path_ref;
    ObjRef dummy_ref;
    ObjRef old_assoc_ref;
    size_t length;
    char *id;
    char* old_name = "-";
    char* new_name = "-";
    bool keep_history = false;


    if (argc == 4 && strcmp(argv[1], "--keep-history") == 0)
    {
	keep_history = true;
	old_name = argv[2];
	new_name = argv[3];
    }
    else if (argc == 4 && strcmp(argv[1], "--no-history") == 0)
    {
	old_name = argv[2];
	new_name = argv[3];
    }

    if (*old_name == '-' || *new_name == '-')
    {
	fprintf(stderr, "usage: %s --no-history | --keep-history old-file-name new-file-name\n", argv[0]);
	exit(1);
    }

    length = strlen(new_name);

    if (length > 255)
    {
	fprintf(stderr, "Error: maximum length of file name is 255\n");
	exit(1);
    }

    nwos_log_arguments(argc, argv);

    printf("\n");

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

#ifdef CHANGE_SECURITY_TO_DENSITY
    nwos_set_security_level(Security_Very_Low);
#endif

    if (keep_history && !nwos_check_blocks_available(4))
    {
	fprintf(stderr, "Cannot rename with history (could rename with --no-history option).\n");
	nwos_terminate_objectify();
	exit(1);
    }


    /**************************/
    /* Check the names are ok */
    /**************************/

    if (!nwos_find_file_path(old_name, &old_path_ref))
    {
	fprintf(stderr, "Could not find: %s in system.\n", old_name);
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_number_of_files_for_path(&old_path_ref) > 1)
    {
	fprintf(stderr, "This version cannot change a file name if there is more than one file referencing it.\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_file_path(new_name, &dummy_ref))
    {
	fprintf(stderr, "Error: %s already exists in the system\n", new_name);
	nwos_terminate_objectify();
	exit(1);
    }


    assert(nwos_read_variable_sized_object_from_disk(&old_path_ref, kludge, sizeof(kludge), &get_path_object_size));

    assert(nwos_file_path_to_path_and_file_association(&old_path_ref, 0, &old_assoc_ref));

    assert(nwos_read_object_from_disk(&old_assoc_ref, &assoc_obj, sizeof(assoc_obj)));

    if (!is_void_reference(&assoc_obj.header.object.next_version))
    {
	fprintf(stderr, "The file that had the name %s has already been renamed.\n", old_name);
	nwos_terminate_objectify();
	exit(1);
    }


    /******************************************************/
    /* before changing verify it is NOT in any disc lists */
    /******************************************************/

    if (nwos_find_private_class_definition("DISC LIST", &list_class_ref))
    {
	id = file_is_used_in_disc_list(&list_class_ref, &assoc_obj);

	if (id != NULL)   /* the file was found in disc_list with this id */
	{
	    fprintf(stderr, "\nError: found file %s in disc list: %s\n\n", old_name, id);
	    fprintf(stderr, "           *** Name NOT changed! ***\n\n");

	    nwos_terminate_objectify();
	    exit(1);
	}
    }

    assert(is_same_object(&ptr_path_obj->header.common.id, &old_path_ref));
    assert(is_same_object(&assoc_obj.header.common.id, &old_assoc_ref));

    if (keep_history)
    {
	create_new_path(new_name, &assoc_obj);   /* assoc_obj is modified by this function */
    }
    else
    {
	ptr_path_obj->count = length;

	memcpy(ptr_path_obj->storage, new_name, length);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->count, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

	nwos_overwrite_object_to_disk(&old_path_ref, ptr_path_obj, get_path_object_size(ptr_path_obj));
    }

    nwos_terminate_objectify();

    return 0;
}


