/*
--          This file is part of the New World OS and Objectify projects
--                        Copyright (C) 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "header.h"
#include "user_config.h"


int main(int argc, char* argv[])
{
    const char* obj_file_path;
    int obj_file_desc;
    struct flock lock;
    Disk_Header disk_header;
    Disk_Header_0022_to_0029 old_disk_header;
    char ref_str[128];
    uint8 block[FILE_BLOCK_SIZE];
    uint32 chunk_offset;
    uint32 total_chunks;
    uint32 used_chunks;
    uint32 total_blocks_32_bits;
    uint32 used_blocks_32_bits;
    uint64 total_blocks;
    uint64 used_blocks;
    TimeStamp last_prep_disk;
    TimeStamp last_change;
    size_t disk_header_size;
    int i;

    printf("\n");

    if (argc > 2)
    {
	fprintf(stderr, "usage: %s [archive]\n", argv[0]);
	exit(1);
    }

    if (argc == 1)   /* no archive specified, use the default */
    {
	obj_file_path = nwos_get_private_objects_path();
    }
    else
    {
	obj_file_path = argv[1];
    }

    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    if (read(obj_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", obj_file_path);
	perror(ref_str);
	exit(1);
    }

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	exit(1);
    }

    printf("Magic number:   %c%c%c%c\n", block[0], block[1], block[2], block[3]);


    if (!isdigit(block[4]) || !isdigit(block[5]) || !isdigit(block[6]) || !isdigit(block[7]))
    {
	fprintf(stderr, "Invalid version string in disk header: %s\n", obj_file_path);
	exit(1);
    }

    printf("Version string: %c%c%c%c\n", block[4], block[5], block[6], block[7]);


    if (block[4] != '0' || block[5] != '0' || (block[6] != '2' && block[6] != '3'))
    {
	fprintf(stderr, "This version of dump_header can only deal with versions Alpha_2x and Alpha_3x.\n");
	exit(1);
    }

    if (block[6] == '2')    /* it is an old Alpha_2x file */
    {
	memcpy(&old_disk_header, block, sizeof(old_disk_header));

	printf("File type:      <undefined in Alpha_2x>\n");

	memcpy(last_prep_disk, old_disk_header.last_prep_disk, sizeof(last_prep_disk));
	memcpy(last_change, old_disk_header.last_change, sizeof(last_change));

	nwos_4_uint8_to_uint32(old_disk_header.total_blocks, &total_blocks_32_bits);
	nwos_4_uint8_to_uint32(old_disk_header.used_blocks, &used_blocks_32_bits);
	nwos_4_uint8_to_uint32(old_disk_header.block_offset_to_chunks, &chunk_offset);
	nwos_4_uint8_to_uint32(old_disk_header.used_chunks, &used_chunks);

	total_chunks = total_blocks_32_bits / 65536;

	if (total_chunks * 65536 != total_blocks_32_bits)
	{
	    printf("WARNING: total blocks is not a multiple of 65504!\n");
	}

	total_blocks = total_blocks_32_bits;
	used_blocks = used_blocks_32_bits;

	disk_header_size = sizeof(old_disk_header);
    }
    else
    {
	memcpy(&disk_header, block, sizeof(disk_header));

	if (!isalpha(block[8]) || !isalpha(block[9]) || !isalpha(block[10]) || !isalpha(block[11]))
	{
	    fprintf(stderr, "Invalid file type code in disk header: %s\n", obj_file_path);
	    exit(1);
	}

	printf("File type:      %c%c%c%c\n", disk_header.type_code[0], disk_header.type_code[1], disk_header.type_code[2], disk_header.type_code[3]);

	memcpy(last_prep_disk, disk_header.last_prep_disk, sizeof(last_prep_disk));
	memcpy(last_change, disk_header.last_change, sizeof(last_change));

	nwos_4_uint8_to_uint32(disk_header.total_chunks, &total_chunks);
	nwos_8_uint8_to_uint64(disk_header.used_blocks, &used_blocks);
	nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_offset);
	nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

	total_blocks = total_chunks * 65536;

	disk_header_size = sizeof(old_disk_header);
    }

    if (last_prep_disk[0] == 0)
    {
	printf("Formatted:      <empty>\n");
    }
    else
    {
	printf("Formatted:      %u-%02u-%02u %02u:%02u:%02u\n",
	       nwos_extract_year_from_time_stamp(last_prep_disk),
	       nwos_extract_month_from_time_stamp(last_prep_disk),
	       nwos_extract_day_of_month_from_time_stamp(last_prep_disk),
	       nwos_extract_hour_from_time_stamp(last_prep_disk),
	       nwos_extract_minute_from_time_stamp(last_prep_disk),
	       nwos_extract_second_from_time_stamp(last_prep_disk));
    }

    printf("Last change:    %u-%02u-%02u %02u:%02u:%02u\n",
	   nwos_extract_year_from_time_stamp(last_change),
	   nwos_extract_month_from_time_stamp(last_change),
	   nwos_extract_day_of_month_from_time_stamp(last_change),
	   nwos_extract_hour_from_time_stamp(last_change),
	   nwos_extract_minute_from_time_stamp(last_change),
	   nwos_extract_second_from_time_stamp(last_change));

    printf("Total chunks:   %u\n", total_chunks);
    printf("Used chunks:    %u\n", used_chunks);
    printf("Total blocks:   %llu\n", total_blocks);
    printf("Used blocks:    %llu\n", used_blocks);

    for (i = disk_header_size; i < FILE_BLOCK_SIZE; i++)
    {
	if (block[i] != 0) break;
    }

    if (i < FILE_BLOCK_SIZE)
    {
	printf("\nWARNING: unused portions of header block are non-zero\n");
    }

    printf("\n");


    /***********************************************/
    /* Dump the second header if it is a .dif file */
    /***********************************************/

    if (block[6] == '3' && memcmp(disk_header.type_code, TYPE_CODE_DIFF, sizeof(disk_header.type_code)) == 0)    /* it is an Alpha_3x diff file */
    {
	if (read(obj_file_desc, block, sizeof(block)) != sizeof(block))
	{
	    snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", obj_file_path);
	    perror(ref_str);
	    exit(1);
	}

	if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
	{
	    fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	    exit(1);
	}

	printf("Magic number:   %c%c%c%c\n", block[0], block[1], block[2], block[3]);


	if (!isdigit(block[4]) || !isdigit(block[5]) || !isdigit(block[6]) || !isdigit(block[7]))
	{
	    fprintf(stderr, "Invalid version string in disk header: %s\n", obj_file_path);
	    exit(1);
	}

	printf("Version string: %c%c%c%c\n", block[4], block[5], block[6], block[7]);


	memcpy(&disk_header, block, sizeof(disk_header));

	if (!isalpha(block[8]) || !isalpha(block[9]) || !isalpha(block[10]) || !isalpha(block[11]))
	{
	    fprintf(stderr, "Invalid file type code in disk header: %s\n", obj_file_path);
	    exit(1);
	}

	printf("File type:      %c%c%c%c\n", disk_header.type_code[0], disk_header.type_code[1], disk_header.type_code[2], disk_header.type_code[3]);

	memcpy(last_prep_disk, disk_header.last_prep_disk, sizeof(last_prep_disk));
	memcpy(last_change, disk_header.last_change, sizeof(last_change));

	nwos_4_uint8_to_uint32(disk_header.total_chunks, &total_chunks);
	nwos_8_uint8_to_uint64(disk_header.used_blocks, &used_blocks);
	nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_offset);
	nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

	total_blocks = total_chunks * 65536;

	disk_header_size = sizeof(old_disk_header);

	if (last_prep_disk[0] == 0)
	{
	    printf("Formatted:      <empty>\n");
	}
	else
	{
	    printf("Formatted:      %u-%02u-%02u %02u:%02u:%02u\n",
		   nwos_extract_year_from_time_stamp(last_prep_disk),
		   nwos_extract_month_from_time_stamp(last_prep_disk),
		   nwos_extract_day_of_month_from_time_stamp(last_prep_disk),
		   nwos_extract_hour_from_time_stamp(last_prep_disk),
		   nwos_extract_minute_from_time_stamp(last_prep_disk),
		   nwos_extract_second_from_time_stamp(last_prep_disk));
	}

	printf("Last change:    %u-%02u-%02u %02u:%02u:%02u\n",
	       nwos_extract_year_from_time_stamp(last_change),
	       nwos_extract_month_from_time_stamp(last_change),
	       nwos_extract_day_of_month_from_time_stamp(last_change),
	       nwos_extract_hour_from_time_stamp(last_change),
	       nwos_extract_minute_from_time_stamp(last_change),
	       nwos_extract_second_from_time_stamp(last_change));

	printf("Total chunks:   %u\n", total_chunks);
	printf("Used chunks:    %u\n", used_chunks);
	printf("Total blocks:   %llu\n", total_blocks);
	printf("Used blocks:    %llu\n", used_blocks);

	for (i = disk_header_size; i < FILE_BLOCK_SIZE; i++)
	{
	    if (block[i] != 0) break;
	}

	if (i < FILE_BLOCK_SIZE)
	{
	    printf("\nWARNING: unused portions of header block are non-zero\n");
	}

	printf("\n");
    }

    return 0;
}


