/*
--          This file is part of the New World OS and Objectify projects
--            Copyright (C) 2005, 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"


static uint8 days_per_month[12] = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };




static bool find_year_from_cardinal(ObjRef* year_class_ref, ObjRef* cardinal_ref, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Year year_obj;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_years;
    int i;

    nwos_read_class_definition(year_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_years = ref_list->common_header.num_refs;

    /* printf("num_years: %d\n", num_years); */

    for (i = 0; i < num_years; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, year_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &year_obj, sizeof(year_obj)));

	    if (is_same_object(&year_obj.number, cardinal_ref))   /* found a match */
	    {
		copy_reference(ref, &ref_list->references[i]);
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_years;
}


bool nwos_find_public_year(uint16 year, ObjRef* ref)
{
    char year_ascii[5];
    ObjRef year_class_ref;
    ObjRef cardinal_ref;
    bool result = false;

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    assert(nwos_find_public_class_definition("YEAR", &year_class_ref));

    year_ascii[0] = '0' +  (year / 1000);
    year_ascii[1] = '0' + ((year / 100) % 10);
    year_ascii[2] = '0' + ((year / 10)  % 10);
    year_ascii[3] = '0' +  (year        % 10);
    year_ascii[4] = '\0';

    if (nwos_find_cardinal_number(year_ascii, &cardinal_ref))
    {
	result = find_year_from_cardinal(&year_class_ref, &cardinal_ref, ref);
    }

    return result;
}


bool nwos_find_private_year(uint16 year, ObjRef* ref)
{
    char year_ascii[5];
    ObjRef year_class_ref;
    ObjRef cardinal_ref;
    bool result = false;

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    if (nwos_find_private_class_definition("YEAR", &year_class_ref))
    {
	year_ascii[0] = '0' +  (year / 1000);
	year_ascii[1] = '0' + ((year / 100) % 10);
	year_ascii[2] = '0' + ((year / 10)  % 10);
	year_ascii[3] = '0' +  (year        % 10);
	year_ascii[4] = '\0';

	if (nwos_find_cardinal_number(year_ascii, &cardinal_ref))
	{
	    result = find_year_from_cardinal(&year_class_ref, &cardinal_ref, ref);
	}
    }

    return result;
}


#ifdef PUBLIC_MODE
ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref)
{
    C_struct_Year year_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    char year_ascii[5];
    ObjCreateResult result;

    assert(1000 <= year && year <= 9999);   /* for now limit it to 4 digit years */

    year_ascii[0] = '0' +  (year / 1000);
    year_ascii[1] = '0' + ((year / 100) % 10);
    year_ascii[2] = '0' + ((year / 10)  % 10);
    year_ascii[3] = '0' +  (year        % 10);
    year_ascii[4] = '\0';

    /* first find out if we already have this year */

    assert(nwos_find_public_class_definition("YEAR", &class_ref));

    result = nwos_create_cardinal_number(year_ascii, &cardinal_ref);

    if (result == CREATED_NEW || !nwos_find_year_from_cardinal(&cardinal_ref, ref))
    {
	memset(&year_obj, 0, sizeof(year_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&year_obj.header.common, ref, &class_ref);

	copy_reference(&year_obj.number, &cardinal_ref);

	nwos_create_reference_list(ref, &year_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &year_obj.header.object, sizeof(ObjectHeader), year_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &year_obj.number, sizeof(year_obj) - sizeof(EveryObject), year_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &year_obj, sizeof(year_obj));

        nwos_add_to_references(ref, &class_ref);
	nwos_add_to_references(ref, &cardinal_ref);

	result = CREATED_NEW;
    }

    return result;
}
#endif



/**************************************************************************************/
/* This routine finds a private year object or creates one by cloning the public one. */
/**************************************************************************************/

void nwos_find_or_create_private_year(uint16 year, ObjRef* ref)
{
    ObjRef public_year_ref;
    ObjRef private_year_class_ref;
    C_struct_Year year_obj;
    char msg[128];

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    if (!nwos_find_private_year(year, ref))
    {
	assert(nwos_find_public_year(year, &public_year_ref));

	nwos_find_or_create_private_class_definition("YEAR", &private_year_class_ref);

	assert(nwos_read_object_from_disk(&public_year_ref, &year_obj, sizeof(year_obj)));

	nwos_generate_new_id(ref);

	snprintf(msg, sizeof(msg), "clone_year(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_year_ref.id[0], public_year_ref.id[1], public_year_ref.id[2], public_year_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&year_obj.header.common.id, ref);

	nwos_get_time_stamp(year_obj.header.common.creation_time);

	copy_reference(&year_obj.header.common.class_definition, &private_year_class_ref);

	copy_reference(&year_obj.header.object.clone_of, &public_year_ref);

	nwos_create_reference_list(ref, &year_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &year_obj.header.object, sizeof(ObjectHeader), year_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &year_obj, sizeof(year_obj));

	nwos_add_to_references(ref, &private_year_class_ref);
    }
}


/* The months should always exist (all twelve of them) */

static bool find_month(ObjRef* month_class_ref, uint8 month, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Month month_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ReferenceList* ref_list;
    int num_months;
    int i;
    char month_ascii[3];

    assert(1 <= month && month <= 12);

    month_ascii[0] = '0' + (month / 10);
    month_ascii[1] = '0' + (month % 10);
    month_ascii[2] = '\0';

    assert(nwos_find_cardinal_number(month_ascii, &cardinal_ref));
    assert(nwos_find_ordinal_number(&cardinal_ref, &ordinal_ref));

    nwos_read_class_definition(month_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_months = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_months; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, month_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &month_obj, sizeof(month_obj)));

	    if (is_same_object(&month_obj.number, &ordinal_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_months;
}


void nwos_find_public_month(uint8 month, ObjRef* ref)
{
    ObjRef month_class_ref;

    assert(1 <= month && month <= 12);

    assert(nwos_find_public_class_definition("MONTH", &month_class_ref));

    assert(find_month(&month_class_ref, month, ref));
}


/* See if this private month exists */

bool nwos_find_private_month(uint8 month, ObjRef* ref)
{
    ObjRef month_class_ref;
    bool result = false;

    assert(1 <= month && month <= 12);

    if (nwos_find_private_class_definition("MONTH", &month_class_ref))
    {
	result = find_month(&month_class_ref, month, ref);
    }

    return result;
}



/***************************************************************************************/
/* This routine finds a private month object or creates one by cloning the public one. */
/***************************************************************************************/

void nwos_find_or_create_private_month(uint8 month, ObjRef* ref)
{
    ObjRef public_month_ref;
    ObjRef private_month_class_ref;
    C_struct_Month month_obj;
    char msg[128];

    assert(1 <= month && month <= 12);

    if (!nwos_find_private_month(month, ref))
    {
	nwos_find_public_month(month, &public_month_ref);

	nwos_find_or_create_private_class_definition("MONTH", &private_month_class_ref);

	assert(nwos_read_object_from_disk(&public_month_ref, &month_obj, sizeof(month_obj)));

	nwos_generate_new_id(ref);

	snprintf(msg, sizeof(msg), "clone_month(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_month_ref.id[0], public_month_ref.id[1], public_month_ref.id[2], public_month_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&month_obj.header.common.id, ref);

	nwos_get_time_stamp(month_obj.header.common.creation_time);

	copy_reference(&month_obj.header.common.class_definition, &private_month_class_ref);

	copy_reference(&month_obj.header.object.clone_of, &public_month_ref);

	nwos_create_reference_list(ref, &month_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &month_obj.header.object, sizeof(ObjectHeader), month_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &month_obj, sizeof(month_obj));

	nwos_add_to_references(ref, &private_month_class_ref);
    }
}


/* All of the month_and_day objects should always exist (all 366 of them) */

static bool find_month_and_day(ObjRef* month_and_day_class_ref, ObjRef* month_ref, uint8 day, ObjRef* ref)
{
    EveryObject month_header;
    C_struct_Month_And_Day month_and_day_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char ascii[3];

    ascii[0] = '0' + (day / 10);
    ascii[1] = '0' + (day % 10);
    ascii[2] = '\0';

    assert(nwos_find_cardinal_number(ascii, &cardinal_ref));
    assert(nwos_find_ordinal_number(&cardinal_ref, &ordinal_ref));

    nwos_read_object_headers_from_disk(month_ref, &month_header);

    ref_list = nwos_malloc_reference_list(&month_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, month_and_day_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &month_and_day_obj, sizeof(month_and_day_obj)));

	    if (is_same_object(&month_and_day_obj.day, &ordinal_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_refs;   /* we had to have found it */
}


void nwos_find_public_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef month_and_day_class_ref;
    ObjRef month_ref;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    assert(nwos_find_public_class_definition("MONTH AND DAY", &month_and_day_class_ref));

    nwos_find_public_month(month, &month_ref);

    assert(find_month_and_day(&month_and_day_class_ref, &month_ref, day, ref));
}


bool nwos_find_private_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef month_and_day_class_ref;
    ObjRef month_ref;
    bool result = false;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    if (nwos_find_private_class_definition("MONTH AND DAY", &month_and_day_class_ref))
    {
	if (nwos_find_private_month(month, &month_ref))
	{
	    result = find_month_and_day(&month_and_day_class_ref, &month_ref, day, ref);
	}
    }

    return result;
}


/***********************************************************************************************/
/* This routine finds a private month_and_day object or creates one by cloning the public one. */
/***********************************************************************************************/

void nwos_find_or_create_private_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef public_month_and_day_ref;
    ObjRef private_month_and_day_class_ref;
    ObjRef private_month_ref;
    C_struct_Month_And_Day month_and_day_obj;
    char msg[128];

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    if (!nwos_find_private_month_and_day(month, day, ref))
    {
	nwos_find_public_month_and_day(month, day, &public_month_and_day_ref);

	nwos_find_or_create_private_class_definition("MONTH AND DAY", &private_month_and_day_class_ref);

	nwos_find_or_create_private_month(month, &private_month_ref);

	assert(nwos_read_object_from_disk(&public_month_and_day_ref, &month_and_day_obj, sizeof(month_and_day_obj)));

	nwos_generate_new_id(ref);

	snprintf(msg, sizeof(msg), "clone_month_and_day(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_month_and_day_ref.id[0], public_month_and_day_ref.id[1], public_month_and_day_ref.id[2], public_month_and_day_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&month_and_day_obj.header.common.id, ref);

	nwos_get_time_stamp(month_and_day_obj.header.common.creation_time);

	copy_reference(&month_and_day_obj.header.common.class_definition, &private_month_and_day_class_ref);

	copy_reference(&month_and_day_obj.header.object.clone_of, &public_month_and_day_ref);

	copy_reference(&month_and_day_obj.month, &private_month_ref);

	nwos_create_reference_list(ref, &month_and_day_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &month_and_day_obj.header.object, sizeof(ObjectHeader), month_and_day_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &month_and_day_obj.month, sizeof(month_and_day_obj) - sizeof(EveryObject), month_and_day_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &month_and_day_obj, sizeof(month_and_day_obj));

	nwos_add_to_references(ref, &private_month_and_day_class_ref);
	nwos_add_to_references(ref, &private_month_ref);
    }
}


bool nwos_is_leap_year(int year)
{
    return ((year % 4 == 0) && (year % 100 != 0)) || (year % 400 == 0);
}
    


static bool find_date(ObjRef* date_class_ref, ObjRef* year_ref, ObjRef* month_and_day_ref, ObjRef* ref)
{
    C_struct_Date date_obj;
    EveryObject year_header;

    ObjRef object_class;

    ReferenceList* year_ref_list;
    int num_refs;
    int i;

    bool result = false;

    nwos_read_object_headers_from_disk(year_ref, &year_header);

    year_ref_list = nwos_malloc_reference_list(&year_header.object.references);

    num_refs = year_ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&year_ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, date_class_ref))   /* it is a date object */
	{
	    assert(nwos_read_object_from_disk(&year_ref_list->references[i], &date_obj, sizeof(date_obj)));

	    if (is_same_object(&date_obj.month_and_day, month_and_day_ref))
	    {
		memcpy(ref, &year_ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}
    }

    nwos_free_reference_list(year_ref_list);
    year_ref_list = NULL;

    return result;
}


bool nwos_find_public_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }


    assert(nwos_find_public_class_definition("DATE", &date_class_ref));

    assert(nwos_find_public_year(year, &year_ref));
    nwos_find_public_month_and_day(month, day, &month_and_day_ref);

    return find_date(&date_class_ref, &year_ref, &month_and_day_ref, ref);
}


bool nwos_find_private_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;
    bool result = false;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }


    if (nwos_find_private_class_definition("DATE", &date_class_ref))
    {
	if (nwos_find_private_year(year, &year_ref))
	{
	    if (nwos_find_private_month_and_day(month, day, &month_and_day_ref))
	    {
		result = find_date(&date_class_ref, &year_ref, &month_and_day_ref, ref);
	    }
	}
    }

    return result;
}


ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    C_struct_Date date_obj;

    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;

    ObjCreateResult result = FOUND_EXISTING;

    assert(1 <= month && month <= 12);
    assert(1 <= day && month <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }

#ifdef PUBLIC_MODE
    if (!nwos_find_public_date(year, month, day, ref))   /* didn't find it */
    {
	assert(nwos_find_public_class_definition("DATE", &date_class_ref));

	assert(nwos_find_public_year(year, &year_ref));
	nwos_find_public_month_and_day(month, day, &month_and_day_ref);
#else
    if (!nwos_find_private_date(year, month, day, ref))   /* didn't find it */
    {
	nwos_find_or_create_private_class_definition("DATE", &date_class_ref);
	nwos_find_or_create_private_year(year, &year_ref);
	nwos_find_or_create_private_month_and_day(month, day, &month_and_day_ref);
#endif
	memset(&date_obj, 0, sizeof(date_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&date_obj.header.common, ref, &date_class_ref);

	copy_reference(&date_obj.year, &year_ref);
	copy_reference(&date_obj.month_and_day, &month_and_day_ref);

	nwos_create_reference_list(ref, &date_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &date_obj.header.object, sizeof(ObjectHeader), date_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), date_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &date_obj, sizeof(date_obj));

        nwos_add_to_references(ref, &date_class_ref);
	nwos_add_to_references(ref, &year_ref);
	nwos_add_to_references(ref, &month_and_day_ref);
    }

    return result;
}


bool nwos_year_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Year year_obj;

    assert(size >= 5);

    assert(nwos_read_object_from_disk(ref, &year_obj, sizeof(year_obj)));

    nwos_cardinal_number_to_string(&year_obj.number, string, size);

    return true;
}


bool nwos_month_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Month month_obj;

    assert(size >= 3);

    assert(nwos_read_object_from_disk(ref, &month_obj, sizeof(month_obj)));

    nwos_ordinal_number_to_string(&month_obj.number, string, size);

    return true;
}


bool nwos_date_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Date date_obj;
    C_struct_Month_And_Day month_and_day_obj;

    assert(nwos_read_object_from_disk(ref, &date_obj, sizeof(date_obj)));
    assert(nwos_read_object_from_disk(&date_obj.month_and_day, &month_and_day_obj, sizeof(month_and_day_obj)));

    assert(size >= 11);

    nwos_month_number_to_string(&month_and_day_obj.month, string, size);

    if (string[1] == '\0')
    {
	string[2] = '\0';
	string[1] = string[0];
	string[0] = '0';
    }
    assert(string[2] == '\0');

    string[2] = '/';

    nwos_ordinal_number_to_string(&month_and_day_obj.day, &string[3], size-3);

    if (string[4] == '\0')
    {
	string[5] = '\0';
	string[4] = string[3];
	string[3] = '0';
    }
    assert(string[5] == '\0');

    string[5] = '/';

    nwos_year_to_string(&date_obj.year, &string[6], size-6);

    return true;
}



