/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revisions of this file and the disk_io.c file which this file was
--   taken from.  (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "backup.h"
#include "disk_io.h"
#include "log.h"
#include "time_stamp.h"


#define SORT_BLOCKS (1048576 / FILE_BLOCK_SIZE)  /* sort file in 1 megabyte chunks */

char*  nwos_backup_path;

static int    backup_file_desc = -1;


typedef struct {
    uint8 block[4];
    ObjRef ref;
    uint8 body[FILE_BLOCK_SIZE - sizeof(ObjRef) - 4];
} Dummy_Block;


bool nwos_initialize_backup(const char* backup_location, Disk_Header* disk_header)
{
    char* username;
    TimeStamp time_stamp;
    int path_length;
    char* backup_file_suffix = ".dif";
    Disk_Header copy_header;
    uint8 buffer[FILE_BLOCK_SIZE];
    size_t bytes_written();
    int retry_count = 0;
    char buf[128];
    bool result = true;

    assert(backup_location != NULL);

#ifdef USE_USERNAME
    username = getenv("USER");

    if (username == NULL)
    {
	fprintf(stderr, "\nUSER environment variable not defined, NO backup file will be created\n\n");
    }
    else
#else
    username = "objectify";   /* for now just use this instead of the username */
#endif
    {
	path_length = strlen(backup_location) + 1 + strlen(username) + 1 + strlen(backup_file_suffix) + 14 + 1;
	nwos_backup_path = malloc(path_length);

	while (1)
	{
	    nwos_get_time_stamp(time_stamp);

	    snprintf(nwos_backup_path, path_length, "%s/%s-%d%02d%02d%02d%02d%02d%s",
		     backup_location,
		     username,
		     nwos_extract_year_from_time_stamp(time_stamp),
		     nwos_extract_month_from_time_stamp(time_stamp),
		     nwos_extract_day_of_month_from_time_stamp(time_stamp),
		     nwos_extract_hour_from_time_stamp(time_stamp),
		     nwos_extract_minute_from_time_stamp(time_stamp),
		     nwos_extract_second_from_time_stamp(time_stamp),
		     backup_file_suffix);

	    snprintf(buf, sizeof(buf), "Opening backup file: %s", nwos_backup_path);
	    nwos_log(buf);

	    errno = 0;
	    backup_file_desc = open(nwos_backup_path, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);

	    /* exit the loop if a file by that name didn't already exist or we have retried 5 times already */
	    if (errno != EEXIST || retry_count > 5) break;

	    fprintf(stderr, "Warning: backup file %s already existed.\n", nwos_backup_path);

	    sleep(1);                /* wait one second so that the file name will be different next time */

	    retry_count++;
	}

	if (backup_file_desc < 0)
	{
	    fprintf(stderr, "Error creating backup file ");
	    perror(nwos_backup_path);
	    result = false;
	}
	else
	{
	    /* copy the header and then change the type_code to "diff" */

	    memcpy(&copy_header, disk_header, sizeof(copy_header));
	    memcpy(&copy_header.type_code, TYPE_CODE_DIFF, sizeof(copy_header.type_code));

	    assert(sizeof(copy_header) <= sizeof(buffer));

	    memset(buffer, 0, sizeof(buffer));
	    memcpy(buffer, &copy_header, sizeof(copy_header));

	    if (write(backup_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
	    {
		perror(nwos_backup_path);
		close(backup_file_desc);
		backup_file_desc = -1;
		result = false;
	    }

	    memset(buffer, 0, sizeof(buffer));

	    if (result && write(backup_file_desc, buffer, sizeof(buffer)) != sizeof(buffer))
	    {
		perror(nwos_backup_path);
		close(backup_file_desc);
		backup_file_desc = -1;
		result = false;
	    }
	}
    }

    return result;
}


void nwos_backup_write_block(uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];

    if (backup_file_desc >= 0 && write(backup_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "write_block write ref:%02x%02x%02x%02x to %s",
		 block[4], block[5], block[6], block[7], nwos_backup_path);

	perror(log_msg);

	close(backup_file_desc);

	backup_file_desc = -1;
    }
}




/* returns negative if a < b, 0 if a == b, and positive if a > b */
 
static int compare_id(ObjRef* a, ObjRef* b)
{
    uint32 ua = nwos_ref_to_word(a);
    uint32 ub = nwos_ref_to_word(b);

    if (ua < ub)
    {
	if ((ub - ua) > INT_MAX)
	{
	    return INT_MIN;
	}
    }

    if (ua > ub)
    {
	if ((ua - ub) > INT_MAX)
	{
	    return INT_MAX;
	}
    }

    return ua - ub;
}


/*****************************************************************************/
/* The idea is to read N blocks, sorting as we go.  Then when the buffer is  */
/* full, scan the remainder of the file and insert any blocks in that range  */
/* into the buffer.  Everytime a block is found in the range in the buffer,  */
/* the last block is written back in place of it.                            */
/*****************************************************************************/

void sort_file()
{
    Dummy_Block* buffer;
    Dummy_Block  block;
    uint32 current_chunk;
    uint32 offset;
    uint32 read_offset = 0;
    size_t write_size;
    int    lower_top;
    int    upper_bottom;
    int    ldiff;
    int    udiff;
    int    i;
    uint32 num_duplicates = 0;
    uint32 last_ref;
    char log_msg[128];


    buffer = malloc(sizeof(Dummy_Block) * SORT_BLOCKS);

    lower_top = -1;
    upper_bottom = SORT_BLOCKS;

    current_chunk = 2;
    offset = current_chunk;

    errno = 0;  /* in case this was set when we came in */

    while (1)
    {
	assert(lower_top < upper_bottom);

	if (lseek(backup_file_desc, (off_t)offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
	{
	    break;
	}

	assert(errno == 0);

	if (read(backup_file_desc, &block, sizeof(block)) != sizeof(block))
	{
	    if (errno != 0 || lower_top + 1 != upper_bottom)   /* error or buffer is not full */
	    {
		break;   /* break out of loop and finish up */
	    }
	    else
	    {
		if (lseek(backup_file_desc, (off_t)current_chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
		{
		    break;
		}

		/* {
		  int i;
		  for (i = 0; i < SORT_BLOCKS; i++)
		    {
		      printf("write: %08x - %08x\n", current_chunk + i, nwos_ref_to_word(&buffer[i].ref));

		    }
		} */

		write_size = SORT_BLOCKS * FILE_BLOCK_SIZE;

		if (write(backup_file_desc, &buffer[0], write_size) != write_size)
		{
		    break;
		}

		lower_top = -1;
		upper_bottom = SORT_BLOCKS;

		current_chunk += SORT_BLOCKS;

		assert(read_offset > 0);
		offset = read_offset;   /* go back to where we were reading */

		if (lseek(backup_file_desc, (off_t)offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
		{
		    break;
		}

		if (read(backup_file_desc, &block, sizeof(block)) != sizeof(block))
		{
		    break;   /* break out of loop and finish up */
		}
	    }
	}

	/* printf("read: %d - %08x\n", offset, nwos_ref_to_word(&block.ref)); */

	if (is_void_reference(&block.ref))   /* was a duplicate block */
	{
	    offset++;   /* just skip over it */
	}
	else if (lower_top == -1)  /* this is the first block */
	{
	    lower_top++;
	    memcpy(&buffer[0].block, &block, FILE_BLOCK_SIZE);
	    offset++;
	}
	else if (upper_bottom == SORT_BLOCKS)  /* this is the second block */
	{
	    ldiff = compare_id(&block.ref, &buffer[0].ref);

	    if (ldiff < 0)  /* the second one is below than the first */
	    {
		upper_bottom--;
		memcpy(&buffer[upper_bottom], &buffer[0], sizeof(*buffer)); /* copy first to last */
		memcpy(&buffer[0].block, &block, FILE_BLOCK_SIZE);
	    }
	    else if (ldiff > 0)  /* the second one is above the first */
	    {
		upper_bottom--;
		memcpy(&buffer[upper_bottom].block, &block, FILE_BLOCK_SIZE);
	    }
	    else  /* equal, found first duplicate */
	    {
		memcpy(&buffer[0].block, &block, FILE_BLOCK_SIZE);
		/* printf("Duplicate A: %d - %08x\n", offset, nwos_ref_to_word(&block.ref)); */
		num_duplicates++;
	    }

	    offset++;
	}
	else if (lower_top + 1 != upper_bottom)   /* buffer isn't full yet */
	{
	    ldiff = compare_id(&block.ref, &buffer[lower_top].ref);
	    udiff = compare_id(&block.ref, &buffer[upper_bottom].ref);

	    if (ldiff <= 0)    /* goes in the lower section */
	    {
		while (ldiff < 0)
		{
		    upper_bottom--;
		    memcpy(&buffer[upper_bottom], &buffer[lower_top], sizeof(*buffer)); /* copy lower to upper */
		    lower_top--;

		    if (lower_top == -1) break;

		    ldiff = compare_id(&block.ref, &buffer[lower_top].ref);
		}

		if (ldiff == 0)
		{
		    memcpy(&buffer[lower_top].block, &block, FILE_BLOCK_SIZE);
		    /* printf("Duplicate B: %d - %08x\n", offset, nwos_ref_to_word(&block.ref)); */
		    num_duplicates++;
		}
		else
		{
		    lower_top++;
		    memcpy(&buffer[lower_top].block, &block, FILE_BLOCK_SIZE);
		}
	    }
	    else if (udiff >= 0)    /* goes in the upper section */
	    {
		while (udiff > 0)
		{
		    lower_top++;
		    memcpy(&buffer[lower_top], &buffer[upper_bottom], sizeof(*buffer)); /* copy lower to upper */
		    upper_bottom++;

		    if (upper_bottom == SORT_BLOCKS) break;

		    udiff = compare_id(&block.ref, &buffer[upper_bottom].ref);
		}

		if (udiff == 0)
		{
		    memcpy(&buffer[upper_bottom].block, &block, FILE_BLOCK_SIZE);
		    /* printf("Duplicate C: %08x  offset: %d  upper_bottom: %d\n",
		       nwos_ref_to_word(&block.ref), offset, upper_bottom); */
		    num_duplicates++;
		}
		else
		{
		    upper_bottom--;
		    memcpy(&buffer[upper_bottom].block, &block, FILE_BLOCK_SIZE);
		}
	    }
	    else   /* falls somewhere in the middle */
	    {
		if (ldiff <= -udiff)
		{
		    lower_top++;
		    memcpy(&buffer[lower_top].block, &block, FILE_BLOCK_SIZE);
		}
		else
		{
		    upper_bottom--;
		    memcpy(&buffer[upper_bottom].block, &block, FILE_BLOCK_SIZE);
		}
	    }

	    offset++;

	    if (lower_top + 1 == upper_bottom)  /* buffer is now full, need to save the offset */
	    {
		read_offset = offset;
	    }
	}
	else   /* buffer is full */
	{
	    udiff = compare_id(&block.ref, &buffer[SORT_BLOCKS - 1].ref);

	    if (udiff < 0)
	    {
		/* first see if it is a duplicate of one of the blocks in the buffer */
		for (i = SORT_BLOCKS - 2; i >= 0; i--)
		{
		    udiff = compare_id(&block.ref, &buffer[i].ref);

		    if (udiff >= 0) break;
		}

		if (udiff == 0)     /* it's a duplicate */
		{
		    memcpy(&buffer[i].block, &block, FILE_BLOCK_SIZE);
		    /* printf("Duplicate D: %d - %08x\n", offset, nwos_ref_to_word(&block.ref)); */
		    num_duplicates++;

		    if (lseek(backup_file_desc, (off_t)offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
		    {
			break;
		    }

		    memset(&block, 0, sizeof(block));

		    if (write(backup_file_desc, &block, sizeof(block)) != sizeof(block))
		    {
			break;
		    }
		}
		else
		{
		    if (lseek(backup_file_desc, (off_t)offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
		    {
			break;
		    }

		    i = SORT_BLOCKS - 1;

		    if (write(backup_file_desc, &buffer[i], sizeof(*buffer)) != sizeof(*buffer))
		    {
			break;
		    }

		    while (i > 0)
		    {
			udiff = compare_id(&block.ref, &buffer[i-1].ref);

			assert(udiff != 0);  /* searched for duplicates above, this shouldn't happen */

			if (udiff > 0) break;

			memcpy(&buffer[i].block, &buffer[i-1].block, FILE_BLOCK_SIZE);

			i--;
		    }

		    memcpy(&buffer[i].block, &block.block, FILE_BLOCK_SIZE);
		}

		offset++;
	    }
	    else if (udiff > 0)   /* above range currently in buffer */
	    {
		offset++;              /* ignore it */
	    }
	    else                  /* duplicate of last block */
	    {
		memcpy(&buffer[SORT_BLOCKS - 1].block, &block, FILE_BLOCK_SIZE);
		/* printf("Duplicate E: %d - %08x\n", offset, nwos_ref_to_word(&block.ref)); */
		num_duplicates++;

		if (lseek(backup_file_desc, (off_t)offset * FILE_BLOCK_SIZE, SEEK_SET) < 0)
		{
		    break;
		}

		memset(&block, 0, sizeof(block));

		if (write(backup_file_desc, &block, sizeof(block)) != sizeof(block))
		{
		    break;
		}

		offset++;
	    }
	}
    }

    if (errno != 0)   /* exited loop because of read error */
    {
	perror(nwos_backup_path);
    }
    else  /* normal end of file exit */
    {
	/* we are done, write the last chunk back out */

	if (lseek(backup_file_desc, (off_t)current_chunk * FILE_BLOCK_SIZE, SEEK_SET) < 0)
	{
	    perror(nwos_backup_path);
	    lower_top = -1;                /* stop it from writing anything */
	    upper_bottom = SORT_BLOCKS;
	}

	if (lower_top > -1)
	{
	    write_size = sizeof(*buffer) * (lower_top + 1);

	    if (write(backup_file_desc, &buffer[0], write_size) != write_size)
	    {
		perror(nwos_backup_path);
		upper_bottom = SORT_BLOCKS;
	    }
	}

	if (upper_bottom < SORT_BLOCKS)
	{
	    write_size = sizeof(*buffer) * (SORT_BLOCKS - upper_bottom);

	    if (write(backup_file_desc, &buffer[upper_bottom], write_size) != write_size)
	    {
		perror(nwos_backup_path);
	    }
	}

	/* printf("Number of duplicates: %u\n", num_duplicates); */

	if (num_duplicates > 0)
	{
	    snprintf(log_msg, sizeof(log_msg), "Backup file duplicates: %u\n", num_duplicates);
	    nwos_log(log_msg);
	}

	if (num_duplicates > 0)   /* need to shorten the file */
	{
	    /* fprintf(stderr, "Truncating file - offset: %u  num_duplicates: %u\n", offset, num_duplicates); */

	    if (truncate(nwos_backup_path, (off_t)(offset - num_duplicates) * FILE_BLOCK_SIZE) != 0)
	    {
		perror(nwos_backup_path);
	    }
	}
    }

    free(buffer);

    /* last verify there are no out of order blocks in the .dif file */

    if (lseek(backup_file_desc, (off_t)2 * FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	perror(nwos_backup_path);
    }
    else
    {
	last_ref = 0;

	assert(errno == 0);

	while (read(backup_file_desc, &block, sizeof(block)) == sizeof(block))
	{
	    if (last_ref != 0 && nwos_ref_to_word(&block.ref) <= last_ref)
	    {
		fprintf(stderr, "ERROR: blocks are out of order in backup file: %s\n", nwos_backup_path);
		break;
	    }

	    last_ref = nwos_ref_to_word(&block.ref);
	}

	if (errno != 0)
	{
	    perror(nwos_backup_path);
	}
    }
}


/**********************************************/
/* Function to finish writing the backup file */
/**********************************************/

#include <time.h>

void nwos_terminate_backup(bool modified, Disk_Header* disk_header)
{
    time_t start_time;
    Disk_Header copy_header;

    if (backup_file_desc < 0)
    {
	if (modified)
	{
	    fprintf(stderr, "WARNING: no backup file was made!\n");
	}
    }
    else
    {
	if (modified)
	{
	    /* copy the header and then change the type_code to "diff" */

	    memcpy(&copy_header, disk_header, sizeof(copy_header));
	    memcpy(&copy_header.type_code, TYPE_CODE_DIFF, sizeof(copy_header.type_code));

	    if (lseek(backup_file_desc, (off_t)FILE_BLOCK_SIZE, SEEK_SET) < 0)
	    {
		perror(nwos_backup_path);
	    }
	    else if (write(backup_file_desc, &copy_header, sizeof(copy_header)) != sizeof(copy_header))
	    {
		perror(nwos_backup_path);
	    }
	    else
	    {
		start_time = time(NULL);
		sort_file();
		fprintf(stderr, "Sort time: %ld seconds\n", time(NULL) - start_time);
	    }

	    if (close(backup_file_desc) < 0)
	    {
		perror(nwos_backup_path);
	    }
	}
	else  /* not modified, no need for a backup */
	{
	    if (close(backup_file_desc) < 0)
	    {
		perror(nwos_backup_path);
	    }

	    if (unlink(nwos_backup_path) < 0)
	    {
		perror(nwos_backup_path);
	    }
	}

	backup_file_desc = -1;
    }

    if (nwos_backup_path != NULL)
    {
	free(nwos_backup_path);
	nwos_backup_path = NULL;
    }
}


