/*
--          This file is part of the New World OS and Objectify projects
--                  Copyright (C) 2006, 2007, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-07-25 17:25:15 -0600 (Sat, 25 Jul 2009) $
--   $Revision: 4184 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  I have retained the CVS log from
--   this file before it was moved to the attic.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
--
-- Revision 1.7  2006/12/05 04:22:56  jsedwards
-- Changed to create a root object (00000001).
--
-- Revision 1.6  2006/11/11 14:00:04  jsedwards
-- Deleted inc_next_ref function and called new generate_new_public_id instead.
--
-- Revision 1.5  2006/11/11 13:31:19  jsedwards
-- Move "next_ref" variable to objectify.c and rename "nwos_next_public_ref".
--
-- Revision 1.4  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/10 13:45:57  jsedwards
-- Removed debugging print statements.
--
-- Revision 1.2  2006/11/10 13:40:18  jsedwards
-- Fixed reference lists to not use encrypted versions in objectify.c.
--
-- Revision 1.1  2006/11/10 04:13:33  jsedwards
-- Program to import public objects (classes, features, names, and spellings)
-- into the disk.
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../crc32.h"
#include "../objectify_private.h"


char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;

#define MAX_LINE_LENGTH 256
#define MAX_REF_LIST_LINE_LENGTH 8192


#if 0
ObjRef nwos_root_object_ref;
#endif



static size_t get_spelling_object_size(void* spelling_obj)
{
    assert(((C_struct_Spelling*)spelling_obj)->count > 0);
    return sizeof(C_struct_Spelling) + ((C_struct_Spelling*)spelling_obj)->count;
}

static size_t get_name_object_size(void* name_obj)
{
    assert(((C_struct_Name*)name_obj)->count > 0);
    return sizeof(C_struct_Name) + (((C_struct_Name*)name_obj)->count * sizeof(ObjRef));
}

static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


void shift_line(char* line, int index)
{
    int i;

    assert(strlen(line) < (MAX_REF_LIST_LINE_LENGTH - index));

    for (i = 0; line[i + index] != '\0'; i++)
    {
	line[i] = line[i + index];
    }

    line[i] = '\0';
}


void fput_char(char c, FILE* fp)
{
    if (fputc(c, fp) == EOF)
    {
	perror(current_filename);
	exit(1);
    }
}


void check_comma(char line[MAX_LINE_LENGTH])
{
    if (line[0] != ',')
    {
	printf("%s: comma\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}


void check_double_quote(char line[MAX_LINE_LENGTH])
{
    if (line[0] != '"')
    {
	printf("%s: missing double quote\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}


void check_newline(char line[MAX_LINE_LENGTH])
{
    if (line[0] != '\n')
    {
	printf("%s: missing end of line\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);
}

void get_string(char line[MAX_LINE_LENGTH], char string[], int size)
{
    int i;

    check_double_quote(line);

    for (i = 0; line[i] != '"' && line[i] != '\0'; i++)
    {
	if (i >= (size - 1))
	{
	    printf("%s: string too long\n", current_filename);
	    printf("remainder of line: %s\n", line);
	    exit(1);
	}

	string[i] = line[i];
    }

    string[i] = '\0';

    shift_line(line, i);

    check_double_quote(line);
}


void get_uint8(char line[MAX_LINE_LENGTH], uint8* ptr_uint8)
{
    uint8 digit;

    if (!isdigit(line[0]))
    {
	printf("no digit for reading uint8\n");
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    *ptr_uint8 = line[0] - '0';

    shift_line(line, 1);

    if (isdigit(line[0]))
    {
	digit = line[0] - '0';

	shift_line(line, 1);

	*ptr_uint8 = *ptr_uint8 * 10 + digit;
    }

    if (isdigit(line[0]))
    {
	digit = line[0] - '0';

	shift_line(line, 1);

	if (*ptr_uint8 > 25 || (*ptr_uint8 == 25 && line[0] > '6'))
	{
	    printf("value too large for uint8: %u%u\n", *ptr_uint8, digit);
	    printf("remainder of line: %s\n", line);
	    exit(1);
	}

	*ptr_uint8 = *ptr_uint8 * 10 + digit;
    }

    if (isdigit(line[0]))
    {
	printf("too many digits for uint8: %u\n", *ptr_uint8);
	printf("remainder of line: %s\n", line);
	exit(1);
    }
}


int hex_char_to_value(char line[MAX_LINE_LENGTH])
{
    char hex_digit[17] = "0123456789ABCDEF";
    char* p = strchr(hex_digit, line[0]);

    if (p == NULL)
    {
	printf("%s: invalid hexadecimal character\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 1);

    return p - hex_digit;
}
    

void get_hex4(char line[MAX_LINE_LENGTH], uint8 hex[4])
{
    int i;
    uint8 upper;
    uint8 lower;

    if (line[0] != '0' || line[1] != 'x')
    {
	printf("%s: missing 0x for hexadecimal value\n", current_filename);
	printf("remainder of line: %s\n", line);
	exit(1);
    }

    shift_line(line, 2);

    for (i = 0; i < 4; i++)
    {
	upper = hex_char_to_value(line);
	lower = hex_char_to_value(line);
	hex[i] = (upper << 4) | lower;
    }
}


void get_objref(char line[MAX_LINE_LENGTH], ObjRef* ref)
{
    int i;
    ObjRef dummy;

    get_hex4(line, ref->id);

    for (i = 0; i < sizeof(ObjRef); i++)
    {
	if (ref->id[i] != nwos_next_public_ref.id[i])
	{
	    break;
	}
    }

    if (ref->id[i] > nwos_next_public_ref.id[i])
    {
	copy_reference(&nwos_next_public_ref, ref);
    }

    if (is_same_object(ref, &nwos_next_public_ref))
    {
	nwos_generate_new_public_id(&dummy);   /* increment it */
	assert(is_same_object(&dummy, ref));
    }
}


void get_timestamp(char line[MAX_LINE_LENGTH], TimeStamp ts)
{
    char string[32];
    uint year;
    uint month;
    uint day_of_month;
    uint day_of_week;
    uint hour;
    uint minute;
    uint second;
    uint microsecond;
    int count;

    get_string(line, string, sizeof(string));


    count = sscanf(string, "%04u-%02u-%02u (%u) %02u:%02u:%02u.%06u",
		   &year, &month, &day_of_month,  &day_of_week,  &hour, &minute, &second, &microsecond);

    if (count != 8)
    {
	printf("count mismatch when reading time stamp: %s - %d\n", string, count);
	exit(1);
    }

    nwos_insert_year_into_time_stamp(year, ts);
    nwos_insert_month_into_time_stamp(month, ts);
    nwos_insert_day_of_month_into_time_stamp(day_of_month, ts);

    nwos_insert_day_of_week_into_time_stamp(day_of_week, ts);

    nwos_insert_hour_into_time_stamp(hour, ts);
    nwos_insert_minute_into_time_stamp(minute, ts);
    nwos_insert_second_into_time_stamp(second, ts);
    nwos_insert_microsecond_into_time_stamp(microsecond, ts);
}



void import_common_header(char line[MAX_LINE_LENGTH], CommonHeader* ch)
{
    get_timestamp(line, ch->creation_time);

    check_comma(line);

    get_objref(line, &ch->class_definition);
}


void import_object_header(char line[MAX_LINE_LENGTH], ObjectHeader* oh)
{
    get_objref(line, &oh->references);
}


void import_every_object(char line[MAX_LINE_LENGTH], EveryObject* eo)
{
    get_objref(line, &eo->common.id);

    check_comma(line);

    import_common_header(line, &eo->common);

    check_comma(line);

    import_object_header(line, &eo->object);

    check_comma(line);

    nwos_crc32_calculate((uint8*) &eo->object, sizeof(eo->object), eo->common.header_chksum);
}
    

void import_class_definition(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*) kludge;
    int i;

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &class_def_obj_ptr->header);

    printf("Class Def %02x%02x%02x%02x: ", 
	   class_def_obj_ptr->header.common.id.id[0],
	   class_def_obj_ptr->header.common.id.id[1],
	   class_def_obj_ptr->header.common.id.id[2],
	   class_def_obj_ptr->header.common.id.id[3]);

    get_objref(line, &class_def_obj_ptr->name);

    printf("%02x%02x%02x%02x\n",
	   class_def_obj_ptr->name.id[0],
	   class_def_obj_ptr->name.id[1],
	   class_def_obj_ptr->name.id[2],
	   class_def_obj_ptr->name.id[3]);

    i = 0;

    while (line[0] == ',')
    {
	check_comma(line);

	get_objref(line, &class_def_obj_ptr->feature[i]);

	printf(" %02x%02x%02x%02x\n", 
	   class_def_obj_ptr->feature[i].id[0],
	   class_def_obj_ptr->feature[i].id[1],
	   class_def_obj_ptr->feature[i].id[2],
	   class_def_obj_ptr->feature[i].id[3]);

	i++;
    }

    class_def_obj_ptr->count = i;

    check_newline(line);

    nwos_crc32_calculate((uint8*) &class_def_obj_ptr->name,
			 get_class_object_size(class_def_obj_ptr) - sizeof(EveryObject), 
			 class_def_obj_ptr->header.common.data_chksum);

    nwos_write_public_object_to_disk(&class_def_obj_ptr->header.common.id, kludge, get_class_object_size(class_def_obj_ptr));

    /* kludgy thing to find reference list class */
    if (is_void_reference(&class_def_obj_ptr->header.object.references))   /* this must be the reference list class */
    {
	assert(is_void_reference(&nwos_reference_list_class_ref));
	memcpy(&nwos_reference_list_class_ref, &class_def_obj_ptr->header.common.id, sizeof(ObjRef));
    }
}


void import_feature_definition(char line[MAX_LINE_LENGTH])
{
    C_struct_Feature_Definition feature_def_obj;


    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(&feature_def_obj, 0, sizeof(feature_def_obj));

    import_every_object(line, &feature_def_obj.header);

    printf("Feature Def %02x%02x%02x%02x: ", 
	   feature_def_obj.header.common.id.id[0],
	   feature_def_obj.header.common.id.id[1],
	   feature_def_obj.header.common.id.id[2],
	   feature_def_obj.header.common.id.id[3]);

    get_objref(line, &feature_def_obj.class);

    check_comma(line);

    printf("%02x%02x%02x%02x\n",
	   feature_def_obj.class.id[0],
	   feature_def_obj.class.id[1],
	   feature_def_obj.class.id[2],
	   feature_def_obj.class.id[3]);

    get_objref(line, &feature_def_obj.label);

    printf("%02x%02x%02x%02x\n",
	   feature_def_obj.label.id[0],
	   feature_def_obj.label.id[1],
	   feature_def_obj.label.id[2],
	   feature_def_obj.label.id[3]);

    check_comma(line);

    get_uint8(line, &feature_def_obj.count);

    check_newline(line);

    nwos_crc32_calculate((uint8*) &feature_def_obj.inherit,
			 sizeof(feature_def_obj) - sizeof(EveryObject), 
			 feature_def_obj.header.common.data_chksum);

    nwos_write_public_object_to_disk(&feature_def_obj.header.common.id, &feature_def_obj, sizeof(feature_def_obj));
}


void import_spelling(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Spelling* spelling_obj_ptr = (C_struct_Spelling*) kludge;
    int i;

    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &spelling_obj_ptr->header);

    printf("Spelling %02x%02x%02x%02x: ", 
	   spelling_obj_ptr->header.common.id.id[0],
	   spelling_obj_ptr->header.common.id.id[1],
	   spelling_obj_ptr->header.common.id.id[2],
	   spelling_obj_ptr->header.common.id.id[3]);

    check_double_quote(line);

    for (i = 0; line[i] != '"' && line[i] != '\0'; i++)
    {
	assert(i < 100);                          /* there shouldn't be any words this long */
	spelling_obj_ptr->storage[i] = line[i];
	printf("%c", spelling_obj_ptr->storage[i]);
    }

    shift_line(line, i);

    spelling_obj_ptr->count = i;

    check_double_quote(line);

    check_newline(line);

    printf("\n");

    nwos_crc32_calculate((uint8*) &spelling_obj_ptr->character_set, 
			 get_spelling_object_size(spelling_obj_ptr) - sizeof(EveryObject), 
			 spelling_obj_ptr->header.common.data_chksum);

    nwos_write_public_object_to_disk(&spelling_obj_ptr->header.common.id, spelling_obj_ptr, get_spelling_object_size(spelling_obj_ptr));
}
    

void import_name(char line[MAX_LINE_LENGTH])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Name* name_obj_ptr = (C_struct_Name*) kludge;
    int i;


    assert(strlen(line) < MAX_LINE_LENGTH);

    memset(kludge, 0, sizeof(kludge));

    import_every_object(line, &name_obj_ptr->header);

    printf("Name %02x%02x%02x%02x:", 
	   name_obj_ptr->header.common.id.id[0],
	   name_obj_ptr->header.common.id.id[1],
	   name_obj_ptr->header.common.id.id[2],
	   name_obj_ptr->header.common.id.id[3]);

    printf("(Ref: %02x%02x%02x%02x)", 
	   name_obj_ptr->header.object.references.id[0],
	   name_obj_ptr->header.object.references.id[1],
	   name_obj_ptr->header.object.references.id[2],
	   name_obj_ptr->header.object.references.id[3]);

    i = 0;
    do
    {
	if (i > 0) check_comma(line);

	get_objref(line, &name_obj_ptr->spelling[i]);

	printf(" %02x%02x%02x%02x", 
	   name_obj_ptr->spelling[i].id[0],
	   name_obj_ptr->spelling[i].id[1],
	   name_obj_ptr->spelling[i].id[2],
	   name_obj_ptr->spelling[i].id[3]);

	i++;
    }
    while (line[0] == ',');

    name_obj_ptr->count = i;

    check_newline(line);

    printf("\n");

    nwos_crc32_calculate((uint8*) &name_obj_ptr->count,
			 get_name_object_size(name_obj_ptr) - sizeof(EveryObject), 
			 name_obj_ptr->header.common.data_chksum);

    nwos_write_public_object_to_disk(&name_obj_ptr->header.common.id, name_obj_ptr, get_name_object_size(name_obj_ptr));
}
    

void import_reference_list(char *line)
{
    ObjRef ref;
    int num_on_this_line;
    Ref_List_First_Block first_block;
    Ref_List_Extra_Block extra_block;
    bool first;
    int block_count;
    char msg[128];

    memset(&first_block, 0, sizeof(first_block));
    memset(&extra_block, 0, sizeof(extra_block));

    get_objref(line, &first_block.list.common_header.id);

    check_comma(line);

    import_common_header(line, &first_block.list.common_header);

    /* note there may not be a comma after the class definition because the reference list could be empty */

    printf("Reference List %02x%02x%02x%02x:", 
	   first_block.list.common_header.id.id[0],
	   first_block.list.common_header.id.id[1],
	   first_block.list.common_header.id.id[2],
	   first_block.list.common_header.id.id[3]);

    num_on_this_line = 1;

    first = true;
    block_count = 0;

    while (line[0] == ',')
    {
	check_comma(line);

	if (first)
	{
	    if (block_count == MAX_REFS_IN_REF_LIST)
	    {
		block_count = 0;

		snprintf(msg, sizeof(msg), "Creating second block: %02x%02x%02x%02x", 
			 nwos_next_public_ref.id[0], nwos_next_public_ref.id[1],
			 nwos_next_public_ref.id[2], nwos_next_public_ref.id[3]);
		nwos_log(msg);

		nwos_generate_new_public_id(&first_block.next_block_ref);

		copy_reference(&extra_block.id, &first_block.next_block_ref);

		first = false;
	    }
	}
	else
	{
	    if (block_count == MAX_REFS_IN_SIDECAR)
	    {
		copy_reference(&extra_block.next_block_ref, &nwos_next_public_ref);

		snprintf(msg, sizeof(msg), "Creating another block: %02x%02x%02x%02x", 
			 nwos_next_public_ref.id[0], nwos_next_public_ref.id[1],
			 nwos_next_public_ref.id[2], nwos_next_public_ref.id[3]);
		nwos_log(msg);

		nwos_crc32_calculate((uint8*) &extra_block.refs, 
				     (MAX_REFS_IN_SIDECAR + 1) * sizeof(ObjRef),  /* include next_block_ref */
				     extra_block.checksum);

		nwos_write_public_object_to_disk(&extra_block.id, &extra_block.dirty, FILE_BLOCK_SIZE);

		memset(&extra_block, 0, sizeof(extra_block));

		/* NOTE: the new id was read directly (without increment) and used above */
		nwos_generate_new_public_id(&extra_block.id);

		block_count = 0;
	    }
	}

	get_objref(line, &ref);

	printf(" %02x%02x%02x%02x", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);

	if (first)
	{
	    copy_reference(&first_block.refs[block_count], &ref);

	}
	else
	{
	    copy_reference(&extra_block.refs[block_count], &ref);
	}

	block_count++;

	if (num_on_this_line > 10)
	{
	    printf("\n");
	    num_on_this_line = 0;
	}
    }

    if (num_on_this_line > 0) printf("\n");

    check_newline(line);

    if (first)  /* only one block */
    {
	nwos_crc32_calculate((uint8*) &first_block.refs, 
			     (block_count + 1) * sizeof(ObjRef),               /* include void ptr */
			     first_block.list.common_header.data_chksum);
    }
    else
    {
	nwos_crc32_calculate((uint8*) &first_block.refs, 
			     (MAX_REFS_IN_REF_LIST + 1) * sizeof(ObjRef),          /* include next_block_ref */
			     first_block.list.common_header.data_chksum);

	nwos_crc32_calculate((uint8*) &extra_block.refs, 
			     (block_count + 1) * sizeof(ObjRef),       /* include void ptr */
			     extra_block.checksum);

	nwos_write_public_object_to_disk(&extra_block.id, &extra_block.dirty, FILE_BLOCK_SIZE);
    }

    nwos_write_public_object_to_disk(&first_block.list.common_header.id, &first_block.list, FILE_BLOCK_SIZE);
}
    

void import_class(char* name)
{
    char filename[36];
    char line[MAX_LINE_LENGTH];      /* this is large enough for all files except reference_lists */
    int i;
    FILE* fp = NULL;
    char* ref_list_line;
    char msg[128];

    snprintf(msg, sizeof(msg), "Import: %s", name);
    nwos_log(msg);


    for (i = 0; name[i] != '\0'; i++)
    {
	if (name[i] == ' ')
	{
	    filename[i] = '_';
	}
	else
	{
	  filename[i] = tolower(name[i]);
	}
    }

    filename[i++] = '.';
    filename[i++] = 'c';
    filename[i++] = 's';
    filename[i++] = 'v';
    filename[i] = '\0';

    fp = fopen(filename, "r");

    if (fp == NULL)
    {
	perror(filename);
	exit(1);
    }

    current_filename = filename;

    if (strcmp(name, "Class Definition") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_class_definition(line);
    }
    if (strcmp(name, "Feature Definition") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_feature_definition(line);
    }
    else if (strcmp(name, "Spelling") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_spelling(line);
    }
    else if (strcmp(name, "Name") == 0)
    {
	while (fgets(line, sizeof(line), fp) != NULL) import_name(line);
    }
    else if (strcmp(name, "Reference List") == 0)
    {
	ref_list_line = malloc(MAX_REF_LIST_LINE_LENGTH);
	assert(ref_list_line != NULL);
	while (fgets(ref_list_line, MAX_REF_LIST_LINE_LENGTH, fp) != NULL) import_reference_list(ref_list_line);
	free(ref_list_line);
    }
}


int main(int argc, char* argv[])
{
    ObjRef root_ref;

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE, DEFAULT_FILE);

    nwos_set_encryption_level(Encryption_None);

    import_class("Spelling");
    import_class("Name");
    import_class("Feature Definition");
    import_class("Class Definition");

    printf("next_reference: %02x%02x%02x%02x\n", 
	   nwos_next_public_ref.id[0], nwos_next_public_ref.id[1], nwos_next_public_ref.id[2], nwos_next_public_ref.id[3]);

    import_class("Reference List");

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_public_class_definition_class_ref.id[0] = 0;
    nwos_public_class_definition_class_ref.id[1] = 0;
    nwos_public_class_definition_class_ref.id[2] = 0;
    nwos_public_class_definition_class_ref.id[3] = 2;

    nwos_create_root(&root_ref);

    nwos_terminate_objectify();

    return 0;
}

