/*
--         This file is part of the New World OS and Objectify projects
--                       Copyright (C) 2009  QRW Software
--              J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                         http://www.qrwsoftware.com
--                         http://nwos.sourceforge.com
--                      http://objectify.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
-- This program repairs the headers in a .dif file with the older 0030
-- header format.  No one should really need this except me.
--
-- $Log: fix_0030_dif_file.c,v $
-- Revision 1.1  2009/07/12 19:36:40  jsedwards
-- Initial version.
--
*/

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "../header.h"
#include "../user_config.h"


typedef struct {
    char magic_number[4];
    char version_string[4];
    char type_code[4];
    uint8 reserved[4];

    TimeStamp last_prep_disk;
    uint8 total_chunks[8];               /* this does NOT include the chunk used by the system */
    uint8 block_offset_to_chunks[4];
    uint8 unused[4];

    TimeStamp last_change;
    uint8 used_chunks[8];               /* this does NOT include the chunk used by the system */
    uint8 used_blocks[8];

} Old_0030_Disk_Header;


const char* convert_header(uint8* block)
{
    Disk_Header disk_header;
    Old_0030_Disk_Header old_disk_header;
    uint32 chunk_offset;
    uint32 total_chunks;
    uint32 used_chunks;
    uint32 used_blocks_32_bits;
    uint64 used_blocks;
    TimeStamp last_prep_disk;
    TimeStamp last_change;
    int i;


    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	return "Missing magic number in disk header";
    }

    printf("Magic number:   %c%c%c%c\n", block[0], block[1], block[2], block[3]);


    if (!isdigit(block[4]) || !isdigit(block[5]) || !isdigit(block[6]) || !isdigit(block[7]))
    {
	return "Invalid version string in disk header";
    }

    printf("Version string: %c%c%c%c\n", block[4], block[5], block[6], block[7]);


    if (block[4] != '0' || block[5] != '0' || block[6] != '3' || block[7] != '0')
    {
	return "This program only works on old Alpha_30 files";
    }

    memcpy(&old_disk_header, block, sizeof(old_disk_header));

    memcpy(last_prep_disk, old_disk_header.last_prep_disk, sizeof(last_prep_disk));
    memcpy(last_change, old_disk_header.last_change, sizeof(last_change));

    nwos_4_uint8_to_uint32(old_disk_header.total_chunks, &total_chunks);
    nwos_4_uint8_to_uint32(old_disk_header.used_blocks, &used_blocks_32_bits);
    nwos_4_uint8_to_uint32(old_disk_header.block_offset_to_chunks, &chunk_offset);
    nwos_4_uint8_to_uint32(old_disk_header.used_chunks, &used_chunks);

    used_blocks = used_blocks_32_bits;

    if (last_prep_disk[0] == 0)
    {
	printf("Formatted:      <empty>\n");
    }
    else
    {
	printf("Formatted:      %u-%02u-%02u %02u:%02u:%02u\n",
	       nwos_extract_year_from_time_stamp(last_prep_disk),
	       nwos_extract_month_from_time_stamp(last_prep_disk),
	       nwos_extract_day_of_month_from_time_stamp(last_prep_disk),
	       nwos_extract_hour_from_time_stamp(last_prep_disk),
	       nwos_extract_minute_from_time_stamp(last_prep_disk),
	       nwos_extract_second_from_time_stamp(last_prep_disk));
    }

    printf("Last change:    %u-%02u-%02u %02u:%02u:%02u\n",
	   nwos_extract_year_from_time_stamp(last_change),
	   nwos_extract_month_from_time_stamp(last_change),
	   nwos_extract_day_of_month_from_time_stamp(last_change),
	   nwos_extract_hour_from_time_stamp(last_change),
	   nwos_extract_minute_from_time_stamp(last_change),
	   nwos_extract_second_from_time_stamp(last_change));

    printf("Total chunks:   %u\n", total_chunks);
    printf("Used chunks:    %u\n", used_chunks);
    printf("Used blocks:    %llu\n", used_blocks);

    for (i = sizeof(Old_0030_Disk_Header); i < FILE_BLOCK_SIZE; i++)
    {
	if (block[i] != 0) break;
    }

    if (i < FILE_BLOCK_SIZE)
    {
	printf("\nWARNING: unused portions of header block are non-zero\n");
    }

    printf("\n");


    memset(block, 0, FILE_BLOCK_SIZE);

    memcpy(&disk_header.magic_number, &old_disk_header.magic_number, 4);
    memcpy(&disk_header.version_string, &old_disk_header.version_string, 4);
    memcpy(&disk_header.type_code, TYPE_CODE_DIFF, 4);

    memcpy(&disk_header.last_prep_disk, &old_disk_header.last_prep_disk, 8);
    memcpy(&disk_header.last_change, &old_disk_header.last_change, 8);

    nwos_uint32_to_4_uint8(&total_chunks, disk_header.total_chunks);
    nwos_uint32_to_4_uint8(&used_chunks, disk_header.used_chunks);
    nwos_uint64_to_8_uint8(&used_blocks, disk_header.used_blocks);
    nwos_uint32_to_4_uint8(&chunk_offset, disk_header.block_offset_to_chunks);

    memcpy(block, &disk_header, sizeof(disk_header));

    for (i = 0; i < FILE_BLOCK_SIZE; i++)
    {
	printf("%02x%c", block[i], i % 16 == 15 ? '\n' : ' ');
    }

    return NULL;
}



int main(int argc, char* argv[])
{
    const char* old_file_path;
    int old_file_desc;
    int new_file_desc;
    char ref_str[128];
    uint8 block[FILE_BLOCK_SIZE];
    const char* error_msg;

    printf("\n");

    if (argc != 3)
    {
	fprintf(stderr, "usage: %s old.dif new.dif\n", argv[0]);
	exit(1);
    }

    old_file_path = argv[1];

    old_file_desc = open(old_file_path, O_RDONLY);

    if (old_file_desc < 0)
    {
	perror(old_file_path);
	exit(1);
    }

    new_file_desc = open(argv[2], O_WRONLY | O_CREAT);

    if (new_file_desc < 0)
    {
	perror(argv[2]);
	close(old_file_desc);
	exit(1);
    }

    if (read(old_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", old_file_path);
	perror(ref_str);
	exit(1);
    }

    error_msg = convert_header(block);

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, old_file_path);
	exit(1);
    }

    if (write(new_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "writting disk header to: %s", argv[2]);
	perror(ref_str);
	exit(1);
    }


    /* second header */

    if (read(old_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", old_file_path);
	perror(ref_str);
	exit(1);
    }

    error_msg = convert_header(block);

    if (error_msg != NULL)
    {
	fprintf(stderr, "%s: %s\n", error_msg, old_file_path);
	exit(1);
    }

    if (write(new_file_desc, block, sizeof(block)) != sizeof(block))
    {
	snprintf(ref_str, sizeof(ref_str), "writting disk header to: %s", argv[2]);
	perror(ref_str);
	exit(1);
    }

    while (true)
    {
	if (read(old_file_desc, block, sizeof(block)) != sizeof(block)) break;

	if (write(new_file_desc, block, sizeof(block)) != sizeof(block))
	{
	    snprintf(ref_str, sizeof(ref_str), "writting disk header to: %s", argv[2]);
	    perror(ref_str);
	    exit(1);
	}
    }

    close(old_file_desc);
    close(new_file_desc);

    return 0;
}


