/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- This is a one time use program to change the serial number in the DEVICE
-- class to just be an object reference.
--
-- $Log: fix_device.c,v $
-- Revision 1.4  2008/09/01 00:22:59  jsedwards
-- Fix year in copyright.  NO code changes.
--
-- Revision 1.3  2008/08/31 16:47:43  jsedwards
-- Added assert around calls to nwos_read_variable_sized_object_from_disk and
-- nwos_read_object_from_disk because they now return false if they fail,
-- whereas before they asserted themselves.
--
-- Revision 1.2  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.1  2007/05/20 17:20:35  jsedwards
-- Initial version.
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "../crc32.h"
#include "../objectify_private.h"


static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


ObjRef* byte_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("BYTE", &result))
	{
	    fprintf(stderr, "Could not locate BYTE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* character_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("CHARACTER", &result))
	{
	    fprintf(stderr, "Could not locate CHARACTER class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("byte_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


ObjRef* objref_class_ref()
{
    static ObjRef result;

    if (is_void_reference(&result))
    {
	if (!nwos_find_public_class_definition("OBJECT REFERENCE", &result))
	{
	    fprintf(stderr, "Could not locate OBJECT REFERENCE class definition\n");
	    nwos_terminate_objectify();
	    exit(1);
	}

	printf("object_reference_class_ref: %02x%02x%02x%02x\n",
	       result.id[0],
	       result.id[1],
	       result.id[2],
	       result.id[3]);
    }

    return &result;
}


void add_feature(C_struct_Class_Definition* class_obj_ptr, ObjRef* type, char* name, int num_elements)
{
    int count;
    ObjRef feature_ref;

    assert(get_class_object_size(class_obj_ptr) < FILE_BLOCK_SIZE);

    count = class_obj_ptr->count;

    nwos_find_or_create_public_feature_definition(type, name, num_elements, &feature_ref);

    copy_reference(&class_obj_ptr->feature[count], &feature_ref);

    nwos_add_to_references(&class_obj_ptr->header.common.id, &feature_ref);

    class_obj_ptr->count = count + 1;

    printf("feature: %d - %s  new size: %d\n", count, name, (int)get_class_object_size(class_obj_ptr));
}



int main(int argc, char* argv[])
{
    ObjRef root_ref;
    ObjRef class_ref;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_obj_ptr = (C_struct_Class_Definition*)kludge;
    C_struct_Feature_Definition feature_obj;


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(NULL, 0, 0, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_encryption_level(Encryption_None);

    printf("Fixing DEVICE class on: %s\n", DEFAULT_FILE);

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);

    memset(kludge, 0, sizeof(kludge));

    nwos_fill_in_common_header(&class_obj_ptr->header.common, &class_ref, &nwos_public_class_definition_class_ref);

    nwos_find_public_class_definition("DEVICE", &class_ref);

    printf("class_ref: %02x%02x%02x%02x\n",
	   class_ref.id[0],
	   class_ref.id[1],
	   class_ref.id[2],
	   class_ref.id[3]);

    assert(nwos_read_variable_sized_object_from_disk(&class_ref, kludge, sizeof(kludge), &get_class_object_size));

    printf("  number of features: %d\n", class_obj_ptr->count);

    if (class_obj_ptr->count != 5)
    {
	fprintf(stderr, "Device class doesn't have 5 features\n");
	exit(1);
    }

    copy_reference(&class_obj_ptr->feature[3], &class_obj_ptr->feature[4]);
    void_reference(&class_obj_ptr->feature[4]);

    class_obj_ptr->count = 4;

    nwos_crc32_calculate((uint8*) &class_obj_ptr->name,
			 get_class_object_size(class_obj_ptr) - sizeof(EveryObject),
			 class_obj_ptr->header.common.data_chksum);

    nwos_overwrite_object_to_disk(&class_ref, class_obj_ptr, get_class_object_size(class_obj_ptr));

    printf("  number of features: %d\n", class_obj_ptr->count);


    assert(nwos_read_object_from_disk(&class_obj_ptr->feature[3], &feature_obj, sizeof(feature_obj)));

    copy_reference(&feature_obj.class, objref_class_ref());

    feature_obj.count = 1;

    nwos_crc32_calculate((uint8*) &feature_obj.inherit,
			 sizeof(feature_obj) - sizeof(EveryObject),
			 feature_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(&class_obj_ptr->feature[3], &feature_obj, sizeof(feature_obj));

    nwos_terminate_objectify();

    return 0;
}

