/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: reference_list.c,v $
-- Revision 1.45  2009/03/14 20:25:52  jsedwards
-- Fix bug introduced in previous check-in where seq was not set to block_num.
--
-- Revision 1.44  2009/03/14 19:28:34  jsedwards
-- Added include of security.h file and renamed read and write object and
-- encrypt or decrypt functions to read and write block.
--
-- Revision 1.43  2009/03/14 03:31:24  jsedwards
-- Added include of bit_map.h file.
--
-- Revision 1.42  2009/03/13 13:10:15  jsedwards
-- Added includes for log.h and reference_list.h files.  Also moved
-- MAX_REFS_IN_REF_LIST, MAX_REFS_IN_SIDECAR, Ref_List_Extra_Block, and
-- Ref_List_First_Block.
--
-- Revision 1.41  2009/03/08 19:21:49  jsedwards
-- Surrounded nwos_remove_from_reference_list and nwos_remove_from_references
-- with ifndef PUBLIC_MODE so they won't be compiled in public mode.
--
-- Revision 1.40  2009/03/08 00:52:42  jsedwards
-- Changed include objectify_private.h to disk_io.h and added include
-- class_definition.h.
--
-- Revision 1.39  2009/01/24 18:16:04  jsedwards
-- Added test to nwos_remove_from_reference_list to verify the last_block
-- pointer is valid after changing the reference list.  Also undefined
-- VERIFY_LAST_BLOCK_POINTER so that verification is normally disabled.
--
-- Revision 1.38  2009/01/24 12:35:49  jsedwards
-- Fix Bug #2392531 - Added new verify_last_block_pointer function and removed
-- loop to find last block from nwos_add_to_reference_list.
--
-- Revision 1.37  2009/01/22 12:36:31  jsedwards
-- Added last_block to cache.
--
-- Revision 1.36  2008/08/02 04:31:25  jsedwards
-- Finished changes to read a reference list that has the old encryption at
-- the beginning and later on has blocks with the new encryption.
--
-- Revision 1.35  2008/08/01 22:10:53  jsedwards
-- Changed to check both the class definition and the checksum instead of just
-- the class definition.  Also fixed to not re-read the block if it is a public
-- reference list.
--
-- Revision 1.34  2008/08/01 00:13:18  jsedwards
-- Changed to try old decryption algorithm if class definition doesn't match
-- nwos_reference_list_class_ref after reading with new decryption algorithm.
--
-- Revision 1.33  2008/07/19 13:59:15  jsedwards
-- Changed nwos_add_to_references to call nwos_get_object_class_without_update
-- instead of nwos_get_object_class.  This fixed a problem where cloning the
-- updated file class definition called this function to add the new reference
-- to the class definition reference list.  But when this called
-- nwos_get_object_class, it called get_file_00x_reference, which called
-- get_file_references, which called find_private_class_defintion, which reads
-- the class defintion reference list.  Unfortunately that was what was being
-- added with this function so it wasn't in the list yet so it could not find
-- the "FILE" class definition, then later puked because it didn't have the
-- private FILE class definitions in the file_references array.
--
-- Revision 1.32  2008/05/06 13:27:26  jsedwards
-- Added assert around calls to nwos_read_object_from_disk_and_decrypt because
-- it now returns true or false if it read the object okay.  Before it asserted
-- itself if it could not read the block.
--
-- Revision 1.31  2007/09/16 15:00:48  jsedwards
-- Fixed case where entry to be removed is in first block and there are two
-- or more blocks in the list.
--
-- Revision 1.30  2007/09/16 13:30:50  jsedwards
-- Added function header for remove_from_reference_list that explains the
-- different cases, added and changed some comments in the code.  NO code
-- changes.
--
-- Revision 1.29  2007/09/16 03:53:56  jsedwards
-- Added code to handle moving entries down one and rewriting intermediate
-- blocks.
--
-- Revision 1.28  2007/09/15 20:53:45  jsedwards
-- Changes to handle removing a block when an entry is removed in the previous
-- block and there is only one entry in the last block.
--
-- Revision 1.27  2007/09/15 19:36:56  jsedwards
-- Added code to deal with removing block because there was only one entry in
-- it, for blocks beyond the second block.
--
-- Revision 1.26  2007/09/15 18:59:26  jsedwards
-- Moved ivec out of Block_Cache and into First_Block and Extra_Block
-- structures so we have the ivec for every block.  Needed when removing
-- entries from a reference list and need to write blocks other than the last.
--
-- Revision 1.25  2007/09/15 16:32:59  jsedwards
-- Added code in remove_from_reference_list to deal with special case where
-- there is two blocks in the list, there is only one entry in the second
-- list and that entry is the one that has to be removed.
--
-- Revision 1.24  2007/09/15 14:04:36  jsedwards
-- Added code to remove_from_reference_list to deal with the entry being in
-- the first block if there is only one entry after the first block and so
-- the second block can be eliminated.
--
-- Revision 1.23  2007/09/14 13:27:32  jsedwards
-- Removed unused ivec variable in write_reference_list.
--
-- Revision 1.22  2007/09/14 12:56:12  jsedwards
-- Changed so that when removing entry from reference list that only has one
-- block, it moves them down by one instead of copying the last over it.
--
-- Revision 1.21  2007/09/14 00:07:07  jsedwards
-- Added code to nwos_remove_from_reference_list to remove entry as long as
-- it is in last block.
--
-- Revision 1.20  2007/08/30 03:58:01  jsedwards
-- Moved #endif to the correct place (mistake in previous check-in).
--
-- Revision 1.19  2007/08/29 04:20:17  jsedwards
-- Put #ifdefs around two more log messages of add to references so normally
-- the are not printed.
--
-- Revision 1.18  2007/07/11 02:45:08  jsedwards
-- Fix bug in create_reference_list where not enough memory was allocated for
-- the first block.
--
-- Revision 1.17  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.16  2007/06/30 14:14:01  jsedwards
-- Added #ifdef around logging of add_to_reference_list, so it is off by default.
--
-- Revision 1.15  2007/05/28 16:03:25  jsedwards
-- Added 'remove_from_references' and 'remove_from_reference_list' functions.
--
-- Revision 1.14  2007/05/28 09:06:48  jsedwards
-- Fixed the comment added in previous revision.  Still NO code changes.
--
-- Revision 1.13  2007/05/28 09:04:53  jsedwards
-- Added comment, NO code changes.
--
-- Revision 1.12  2007/02/11 14:32:39  jsedwards
-- Changed calls to 'sprintf' to 'snprintf' so OpenBSD won't whine so much.
--
-- Revision 1.11  2007/01/09 03:02:34  jsedwards
-- Fixed bug where if there were exactly 54 entries in the reference list it
-- wrote a random value into the next block pointer (which has to be null in
-- this case).
--
-- Revision 1.10  2007/01/05 13:44:15  jsedwards
-- Added asserts to verify the reference_list is being created in the correct
-- space (public/private).
--
-- Revision 1.9  2006/12/29 01:49:45  jsedwards
-- Fix compile error.
--
-- Revision 1.8  2006/12/28 23:17:23  jsedwards
-- Removed line feed from messages to go into the log.
--
-- Revision 1.7  2006/12/25 12:18:34  jsedwards
-- Fix error in printing reference to log in add_reference.
--
-- Revision 1.6  2006/12/20 13:12:48  jsedwards
-- Added malloc_class_reference_list function (untested).
--
-- Revision 1.5  2006/12/15 15:01:38  jsedwards
-- Changed for only two disk areas (Public and Private-Encrypted).
--
-- Revision 1.4  2006/12/05 13:56:33  jsedwards
-- Fix so that it doesn't fill unencrypted blocks with random data.
--
-- Revision 1.3  2006/12/04 04:42:05  jsedwards
-- Changed to call new generate_new_id_of_same_type function.
--
-- Revision 1.2  2006/12/01 14:24:31  jsedwards
-- Added malloc and free reference list functions.
--
-- Revision 1.1  2006/11/11 15:21:40  jsedwards
-- Moved all reference list code out of objectify.c and into this file.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bit_map.h"
#include "class_definition.h"
#include "crc32.h"
#include "disk_io.h"
#include "log.h"
#include "reference_list.h"
#include "security.h"


#undef VERIFY_LAST_BLOCK_POINTER   /* disable tests that verify the last block pointer is correct */


#define MAX_REFS_IN_REF_LIST (((FILE_BLOCK_SIZE - sizeof(CommonHeader)) / sizeof(ObjRef)) - 1)
#define MAX_REFS_IN_SIDECAR  (((FILE_BLOCK_SIZE - 12) / sizeof(ObjRef)) - 1)

typedef struct ref_list_extra_block {
    struct ref_list_extra_block* next_block_ptr;
    int dirty;
    ObjRef id;
    uint8 checksum[4];
    ObjRef refs[MAX_REFS_IN_SIDECAR];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_Extra_Block;

typedef struct {
    struct ref_list_extra_block* next_block_ptr;
    ReferenceList list;
    ObjRef refs[MAX_REFS_IN_REF_LIST];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_First_Block;


#define REF_LIST_CACHE_SIZE 64

typedef struct {
  ObjRef ref;
  int num_refs;
  Ref_List_First_Block* first_block;
  Ref_List_Extra_Block* last_block;        /* null if only one block in list */
  int age;
} Block_Cache;


static int ref_list_tick = 0;

static Block_Cache ref_list_cache[REF_LIST_CACHE_SIZE];


static bool ref_list_checksum_matches(uint8* data, size_t data_size, uint8 stored[4])
{
    uint8 computed[4];

#if 0
    printf("Total size: %d\n", total_size);
    printf("ObjectHeader: %d\n", sizeof(EveryObject));
    printf("object pointer: %p\n", object);
    {
      int i;
      uint8* ptr = (uint8*)object + sizeof(EveryObject);
      printf("data pointer: %p\n", ptr);
      printf("data: ");
      for (i = 0; i < data_size; i++) printf("%02x%c", *ptr++, i%16==15?'\n':' ');
      printf("\n");
    }
#endif

    nwos_crc32_calculate((uint8*)data, data_size, computed);

    return stored[0] == computed[0] && stored[1] == computed[1] && stored[2] == computed[2] && stored[3] == computed[3];
}


static void read_ref_list_into_cache(Block_Cache* cache)
{
    uint8 ivec[IVEC_SIZE];
    Ref_List_Extra_Block* prev_ptr;            /* points to the previous block */
    Ref_List_Extra_Block* next_ptr;            /* points to the next block */
    int i;
    int seq;
    bool use_old_decryption = false;
    int save_num_refs;

    assert(sizeof(Ref_List_First_Block) == FILE_BLOCK_SIZE + sizeof(void*) + IVEC_SIZE);
    assert(sizeof(Ref_List_Extra_Block) == FILE_BLOCK_SIZE + sizeof(void*) + IVEC_SIZE);

    cache->first_block = nwos_malloc(sizeof(Ref_List_First_Block));
    assert(cache->first_block != NULL);

    memset(ivec, 0, sizeof(ivec));
    memcpy(cache->first_block->ivec, ivec, sizeof(cache->first_block->ivec));

    cache->first_block->next_block_ptr = NULL;

    cache->last_block = NULL;

    while (1)
    {
	memset(ivec, 0, sizeof(ivec));

	if (use_old_decryption)
	{
	    assert(nwos_read_block_from_disk_and_old_decrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0));
	}
	else
	{
	    assert(nwos_read_block_from_disk_and_decrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0));
	}

	if (nwos_reference_type(&cache->ref) == Public_Reference || is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref))
	{
	    cache->num_refs = 0;

	    while (cache->num_refs < MAX_REFS_IN_REF_LIST)
	    {
		if (is_void_reference(&cache->first_block->refs[cache->num_refs])) break;
		cache->num_refs++;
	    }

	    if (ref_list_checksum_matches((uint8*) cache->first_block->refs, (cache->num_refs + 1) * sizeof(ObjRef), cache->first_block->list.common_header.data_chksum))
	    {
		break;   /* found correct decryption, exit loop */
	    }

	    if (nwos_reference_type(&cache->ref) == Public_Reference || use_old_decryption)
	    {
		printf("\n");
		fflush(stdout);
		fprintf(stderr, "Unable to decrypt reference list: %08x\n", nwos_ref_to_word(&cache->ref));
		nwos_terminate_objectify();
		exit(1);
	    }
	}

	use_old_decryption = true;
    }


    /* use the flags as a pointer to the next block */
    prev_ptr = (Ref_List_Extra_Block*) cache->first_block;

    if (cache->num_refs < MAX_REFS_IN_REF_LIST)
    {
	prev_ptr->next_block_ref.word = 0;
    }

    assert(prev_ptr->next_block_ptr == NULL);

    seq = 1;    /* use different sequence tables for each block in turn */

    while (!is_void_reference(&prev_ptr->next_block_ref))    /* more blocks in this list */
    {
	next_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	assert(next_ptr != NULL);

	next_ptr->next_block_ptr = NULL;

	memcpy(next_ptr->ivec, ivec, sizeof(next_ptr->ivec));  /* save this ivec in case this block has to be written back to disk */

	save_num_refs = cache->num_refs;

	while (1)
	{
	    memcpy(ivec, next_ptr->ivec, sizeof(ivec));  /* get the saved ivec back */

	    if (use_old_decryption)
	    {
		assert(nwos_read_block_from_disk_and_old_decrypt(&prev_ptr->next_block_ref, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq));
	    }
	    else
	    {
		assert(nwos_read_block_from_disk_and_decrypt(&prev_ptr->next_block_ref, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq));
	    }

	    cache->num_refs = save_num_refs;

	    for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
	    {
		if (is_void_reference(&next_ptr->refs[i])) break;
		cache->num_refs++;
	    }

	    if (ref_list_checksum_matches((uint8*) next_ptr->refs, (i + 1) * sizeof(ObjRef), next_ptr->checksum))
	    {
		break;
	    }

	    if (nwos_reference_type(&cache->ref) == Public_Reference || !use_old_decryption)
	    {
		printf("\n");
		fflush(stdout);
		fprintf(stderr, "Unable to decrypt reference list: %08x  seq: %d\n", nwos_ref_to_word(&cache->ref), seq);
		nwos_terminate_objectify();
		exit(1);
	    }

	    use_old_decryption = false;   /* could have encountered a new one */
	}

	if (i < MAX_REFS_IN_SIDECAR)    /* the link will be garbage, make it void */
	{
	    next_ptr->next_block_ref.word = 0;
	}
	else
	{
	    assert(!is_same_object(&prev_ptr->next_block_ref, &next_ptr->next_block_ref));
	}

	prev_ptr->next_block_ptr = next_ptr;

	prev_ptr = next_ptr;

	cache->last_block = prev_ptr;        /* save it in cache in case this is the last block */

	seq++;
    }
}



static void write_reference_list(Block_Cache* cache)
{
    Ref_List_Extra_Block *block = NULL;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* prev_ptr = NULL;
    Ref_List_Extra_Block* next_ptr = NULL;
    ObjRef ref;
    int i;
    int seq;
    bool encrypted;

    encrypted = (nwos_reference_type(&cache->ref) == Private_Reference);

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	if (cache->first_block->list.common_header.flags == 0xffffffff)    /* block is dirty, needs to be written to disk */
	{
	    for (i = cache->num_refs + 1; i < MAX_REFS_IN_REF_LIST; i++)
	    {
		if (encrypted)
		{
		    cache->first_block->refs[i].word = (random() << 1) ^ (random() >> 1);
		}
		else
		{
		    cache->first_block->refs[i].word = 0;
		}
	    }

	    if (encrypted && cache->num_refs < MAX_REFS_IN_REF_LIST)
	    {
		cache->first_block->next_block_ref.word = (random() << 1) ^ (random() >> 1);
	    }
	    else
	    {
		cache->first_block->next_block_ref.word = 0;
	    }

	    nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				 (cache->num_refs + 1) * sizeof(ObjRef),               /* include void ptr or next_block_ref */
				 cache->first_block->list.common_header.data_chksum);

	    block = (Ref_List_Extra_Block*)cache->first_block;     /* flag that it needs to be written */

	    seq = 0;
	}
	else                    /* we don't need to write this block so we can get rid of it now */
	{
	    nwos_free(cache->first_block);
	}
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);  /* the next block pointer should not be null */

	running_total = MAX_REFS_IN_REF_LIST;
	prev_ptr = (Ref_List_Extra_Block*)cache->first_block;

	seq = 0;

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    next_ptr = prev_ptr->next_block_ptr;

	    assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	    nwos_free(prev_ptr);   /* won't be needing this anymore */

	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));

		if (next_ptr->dirty == -1)    /* block is dirty, needs to be written to disk */
		{

		    for (i = refs_remaining + 1; i < MAX_REFS_IN_SIDECAR; i++)
		    {
			if (encrypted)
			{
			    next_ptr->refs[i].word = (random() << 1) ^ (random() >> 1);
			}
			else
			{
			    next_ptr->refs[i].word = 0;
			}
		    }

		    if (encrypted && refs_remaining < MAX_REFS_IN_SIDECAR)
		    {
			next_ptr->next_block_ref.word = (random() << 1) ^ (random() >> 1);    /* fill next with random junk */
		    }
		    else
		    {
			next_ptr->next_block_ref.word = 0;    /* mark the end */
		    }

		    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
					 (refs_remaining + 1) * sizeof(ObjRef),        /* include void ptr or next_block_ref */
					 next_ptr->checksum);

		    block = next_ptr;     /* flag that it needs to be written */
		}

		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL); /* the next block pointer should not be null */
		running_total += MAX_REFS_IN_SIDECAR;
	    }
	    
	    prev_ptr = next_ptr;

	    seq++;
	}


	if (block == NULL)      /* we don't need to write this block so we can get rid of it now */
	{
	    assert(next_ptr != NULL);

	    nwos_free(next_ptr);
	}

	assert(running_total == cache->num_refs);
	assert(cache->last_block == prev_ptr);
    }

    if (block != NULL)    /* it needs to be written */
    {
	assert(block->dirty == 0xffffffff);

	/* write object expects these to be null right now */
	block->dirty = 0;

	copy_reference(&ref, &block->id);

	/* printf("writing block: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]); */

	nwos_write_block_to_disk_and_encrypt(&ref, &block->dirty, FILE_BLOCK_SIZE, block->ivec, seq);

	nwos_free(block);
    }

    /* just in case, nuke these */
    cache->first_block = NULL;
    cache->last_block = NULL;
    memset(&cache->ref, 0, sizeof(cache->ref));
}


static Block_Cache* find_ref_list_in_cache(ObjRef* ref)
{
    int i;
    Block_Cache* result;

    assert(!is_void_reference(ref));

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (is_same_object(&ref_list_cache[i].ref, ref)) break;
    }

    if (i < REF_LIST_CACHE_SIZE)   /* found it */
    {
	result = &ref_list_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = ref_list_cache;

	for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
	{
	    if (is_void_reference(&ref_list_cache[i].ref))
	    {
		result = &ref_list_cache[i];
		break;
	    }

	    if (ref_list_cache[i].age < result->age)
	    {
		result = &ref_list_cache[i];
	    }
	}

	if (i == REF_LIST_CACHE_SIZE)    /* didn't find an empty one, write the oldest one out */
	{
	    write_reference_list(result);
	}

	memcpy(&result->ref, ref, sizeof(result->ref));

	read_ref_list_into_cache(result);
    }

    ref_list_tick++;
    result->age = ref_list_tick;

    return result;
}


/* WARNING: this only works on reference lists */

size_t nwos_reference_list_size(ObjRef* ref)
{
    Block_Cache* cache;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    return sizeof(ReferenceList) + (cache->num_refs * sizeof(ObjRef));
}


void nwos_read_reference_list_from_disk(ObjRef* ref, ReferenceList* object, size_t size)
{
    Block_Cache* cache;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;

    assert(!is_void_reference(ref));

    cache = find_ref_list_in_cache(ref);

    assert(size >= sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));

    /* copy the first block (which is a reference list object */

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs])); /* this should always be null */

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + cache->num_refs * sizeof(ObjRef));
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
	assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

	memcpy(object, &cache->first_block->list, sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef));

	running_total = MAX_REFS_IN_REF_LIST;
	next_ptr = *(Ref_List_Extra_Block**)cache->first_block;        /* get pointer to first sidecar */

	assert(is_same_object(&cache->first_block->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

	while (running_total < cache->num_refs)    /* read more blocks */
	{
	    refs_remaining = cache->num_refs - running_total;

	    if (refs_remaining <= MAX_REFS_IN_SIDECAR)
	    {
		assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
		assert(is_void_reference(&next_ptr->refs[refs_remaining]));
		memcpy(&object->references[running_total], next_ptr->refs, refs_remaining * sizeof(ObjRef));
		running_total += refs_remaining;
	    }
	    else
	    {
		assert(next_ptr->next_block_ptr != NULL);  /* the next block pointer should not be null */
		memcpy(&object->references[running_total], next_ptr->refs, MAX_REFS_IN_SIDECAR * sizeof(ObjRef));
		running_total += MAX_REFS_IN_SIDECAR;

		/* next pointer ref matches */
		assert(is_same_object(&next_ptr->next_block_ref, &next_ptr->next_block_ptr->id));
	    }

	    next_ptr = next_ptr->next_block_ptr;
	}
	assert(running_total == cache->num_refs);
    }
}


void nwos_create_reference_list_with_existing_id(ObjRef* for_obj, ObjRef* ref_list)
{
    Block_Cache* cache;
    int i;

    assert(!is_void_reference(ref_list));

    cache = ref_list_cache;

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (is_void_reference(&ref_list_cache[i].ref))
	{
	    cache = &ref_list_cache[i];
	    break;
	}

	if (ref_list_cache[i].age < cache->age)
	{
	    cache = &ref_list_cache[i];
	}
    }

    if (i == REF_LIST_CACHE_SIZE)    /* didn't find an empty one, write the oldest one out */
    {
	write_reference_list(cache);
    }

    copy_reference(&cache->ref, ref_list);

    ref_list_tick++;
    cache->age = ref_list_tick;

    cache->first_block = nwos_malloc(sizeof(Ref_List_First_Block));

    assert(cache->first_block != NULL);

    memset(cache->first_block, 0, sizeof(Ref_List_First_Block));  /* zero it out */

    cache->last_block = NULL;

    cache->num_refs = 0;

    memset(cache->first_block->ivec, 0, sizeof(cache->first_block->ivec));

    nwos_fill_in_common_header(&cache->first_block->list.common_header, &cache->ref, &nwos_reference_list_class_ref);

    cache->first_block->list.common_header.flags = 0xffffffff;  /* mark it as dirty */

    copy_reference(&cache->first_block->list.common_header.id, &cache->ref);


#if 0
    printf("sizeof: %u chksum: %02x %02x %02x %02x\n", sizeof(ReferenceList),
	   rl.common_header.chksum[0], rl.common_header.chksum[1], rl.common_header.chksum[2], rl.common_header.chksum[3]);
#endif

    assert(cache->num_refs == 0);
    assert(cache->first_block != NULL);
    assert(cache->first_block->next_block_ptr == NULL);
    assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
}


void nwos_create_reference_list(ObjRef* for_obj, ObjRef* ref_list)
{
    nwos_generate_new_id(ref_list);

    nwos_create_reference_list_with_existing_id(for_obj, ref_list);
}


void nwos_add_to_references(ObjRef* ref, ObjRef* obj)
{
    EveryObject header;
    ObjRef class_ref;

    /* first verify we are didn't pass in a reference list as the object */
    /* I.E. meant to call add_to_reference_list instead. */

    nwos_get_object_class_without_update(obj, &class_ref);
    assert(!is_same_object(&class_ref, &nwos_reference_list_class_ref));

    nwos_read_object_headers_from_disk(obj, &header);

    nwos_add_to_reference_list(ref, &header.object.references);
}



/* Verify the last_block pointer and offset are correct, only used in testing.  */

#ifdef VERIFY_LAST_BLOCK_POINTER

static void verify_last_block_pointer(Block_Cache* cache, int remaining, int calc_seq)
{
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;
    int seq;

    assert(cache->num_refs > MAX_REFS_IN_REF_LIST);    /* there is more than one block */
    assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
    assert(cache->last_block != NULL);                          /* the last block pointer should not be null */
    assert(cache->last_block->next_block_ptr == NULL);          /* but it's next block pointer should be */
    assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

    next_ptr = cache->first_block->next_block_ptr;        /* get pointer to first sidecar */
    assert(is_same_object(&cache->first_block->next_block_ref, &next_ptr->id));  /* next pointer ref matches */

    seq = 1;

    running_total = MAX_REFS_IN_REF_LIST;

    refs_remaining = cache->num_refs - running_total;

    while (refs_remaining > MAX_REFS_IN_SIDECAR)    /* read more blocks */
    {
	assert(running_total < cache->num_refs);

	assert(next_ptr->next_block_ptr != NULL);  /* the next block pointer should not be null */

	/* next pointer ref matches */
	assert(is_same_object(&next_ptr->next_block_ref, &next_ptr->next_block_ptr->id));

	next_ptr = next_ptr->next_block_ptr;

	seq++;

	running_total += MAX_REFS_IN_SIDECAR;

	refs_remaining = cache->num_refs - running_total;
    }

    assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
    assert(is_void_reference(&next_ptr->refs[refs_remaining]));

    assert(next_ptr == cache->last_block);                         /* verify last_block is correct */
    assert(running_total == ((cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR) * MAX_REFS_IN_SIDECAR + MAX_REFS_IN_REF_LIST);

    assert(refs_remaining == remaining);
    assert(seq == calc_seq);
}
#endif


/* Add a new reference to the reference list specified (ref_list is the id of the reference list itself). */

void nwos_add_to_reference_list(ObjRef* ref, ObjRef* ref_list)
{
#ifdef LOG_REFERENCE_LIST
    char log_msg[128];
#endif
    Block_Cache* cache;
    int running_total;
    int refs_remaining;
    Ref_List_Extra_Block* next_ptr;
    int block_num;
    uint8 ivec[IVEC_SIZE];
    int seq;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
    assert(nwos_reference_type(ref_list) == Public_Reference);
#else
    assert(nwos_reference_type(ref) == Private_Reference);
    assert(nwos_reference_type(ref_list) == Private_Reference);
#endif

    cache = find_ref_list_in_cache(ref_list);

    /* first make sure it is a reference list */

    assert(is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref));

#ifdef LOG_REFERENCE_LIST
    snprintf(log_msg, sizeof(log_msg), "nwos_add_to_reference_list - object: %02x%02x%02x%02x - ref list: %02x%02x%02x%02x - num_refs: %d", 
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3],
	    ref_list->id[0], ref_list->id[1], ref_list->id[2], ref_list->id[3],
	    cache->num_refs);
    nwos_log(log_msg);
#endif

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* there is just one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	if (cache->num_refs < MAX_REFS_IN_REF_LIST)    /* there is room for more */
	{
	    memcpy(&cache->first_block->refs[cache->num_refs], ref, sizeof(ObjRef));
	    cache->num_refs++;
	    memset(&cache->first_block->refs[cache->num_refs], 0, sizeof(ObjRef));
	    cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */
	}
	else       /* no room for more, must expand to 2 blocks */
	{
	    assert(cache->num_refs == MAX_REFS_IN_REF_LIST);

	    nwos_generate_new_id(&cache->first_block->refs[MAX_REFS_IN_REF_LIST]);    /* id of new list */

	    memset(ivec, 0, sizeof(ivec));

	    cache->first_block->list.common_header.flags = 0;    /* remove dirty flag */

	    nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				 (cache->num_refs + 1) * sizeof(ObjRef),               /* include void ptr or next_block_ref */
				 cache->first_block->list.common_header.data_chksum);

	    nwos_write_block_to_disk_and_encrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0);

	    next_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	    assert(next_ptr != NULL);

	    cache->first_block->next_block_ptr = next_ptr;

	    next_ptr->next_block_ptr = NULL;

	    memcpy(next_ptr->ivec, ivec, sizeof(next_ptr->ivec));

#ifdef LOG_REFERENCE_LIST
	    snprintf(log_msg, sizeof(log_msg), "add_reference - first_block ref: %02x%02x%02x%02x obj: %p  next_block ref: %02x%02x%02x%02x obj: %p",
		   cache->ref.id[0], cache->ref.id[2], cache->ref.id[2], cache->ref.id[3], cache->first_block,
		   cache->first_block->next_block_ref.id[0], cache->first_block->next_block_ref.id[1], 
		   cache->first_block->next_block_ref.id[2], cache->first_block->next_block_ref.id[3], 
		   cache->first_block->next_block_ptr);
	    nwos_log(log_msg);
#endif

	    next_ptr->dirty = -1;     /* mark it as dirty */
	    memcpy(&next_ptr->id, &cache->first_block->next_block_ref, sizeof(ObjRef));
	    memcpy(&next_ptr->refs[0], ref, sizeof(ObjRef));
	    next_ptr->refs[1].word = 0;              /* void reference to mark the end of the list */

	    cache->last_block = next_ptr;

	    cache->num_refs++;
	}
    }
    else
    {
	assert(cache->first_block->next_block_ptr != NULL);         /* the next block pointer should not be null */
	assert(cache->last_block != NULL);                          /* the last block pointer should not be null */
	assert(cache->last_block->next_block_ptr == NULL);          /* but it's next block pointer should be */
	assert(sizeof(ReferenceList) + MAX_REFS_IN_REF_LIST * sizeof(ObjRef) == FILE_BLOCK_SIZE - 4);

	next_ptr = cache->last_block;                         /* go right to the last block */

	block_num = (cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR;
	seq = block_num + 1;
	running_total = MAX_REFS_IN_REF_LIST + block_num * MAX_REFS_IN_SIDECAR;

	assert(running_total < cache->num_refs);
	assert(cache->num_refs - running_total <= MAX_REFS_IN_SIDECAR);

	refs_remaining = cache->num_refs - running_total;

	assert(refs_remaining <= MAX_REFS_IN_SIDECAR);
	assert(next_ptr->next_block_ptr == NULL);  /* the next block pointer should be null */
	assert(is_void_reference(&next_ptr->refs[refs_remaining]));

#ifdef VERIFY_LAST_BLOCK_POINTER
	verify_last_block_pointer(cache, refs_remaining, seq);     /* verify last_block is correct */
#endif

	if (refs_remaining < MAX_REFS_IN_SIDECAR)    /* there is room for more */
	{
	    memcpy(&next_ptr->refs[refs_remaining], ref, sizeof(ObjRef));
	    next_ptr->refs[refs_remaining + 1].word = 0;  /* set the next ref to null */
	    next_ptr->dirty = -1;                   /* mark it as dirty */
	}
	else       /* no room for more, must add new block */
	{
	    assert(refs_remaining == MAX_REFS_IN_SIDECAR);

	    nwos_generate_new_id(&next_ptr->next_block_ref);    /* id of new list */

	    next_ptr->dirty = 0;   /* remove dirty flag */

	    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
				 (refs_remaining + 1) * sizeof(ObjRef),        /* include void ptr or next_block_ref */
				 next_ptr->checksum);

	    memcpy(ivec, next_ptr->ivec, sizeof(ivec));

	    nwos_write_block_to_disk_and_encrypt(&next_ptr->id, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq);

	    next_ptr->next_block_ptr = nwos_malloc(sizeof(Ref_List_Extra_Block));

	    assert(next_ptr->next_block_ptr != NULL);

	    next_ptr->next_block_ptr->next_block_ptr = NULL;

	    memcpy(next_ptr->next_block_ptr->ivec, ivec, sizeof(next_ptr->next_block_ptr->ivec));

#ifdef LOG_REFERENCE_LIST
	    snprintf(log_msg, sizeof(log_msg), "add_reference - next_block ref: %02x%02x%02x%02x obj: %p  next_block ref: %02x%02x%02x%02x obj: %p",
		   next_ptr->id.id[0], next_ptr->id.id[2], next_ptr->id.id[2], next_ptr->id.id[3], 
		   next_ptr,
		   next_ptr->next_block_ref.id[0], next_ptr->next_block_ref.id[1], 
		   next_ptr->next_block_ref.id[2], next_ptr->next_block_ref.id[3], 
		   next_ptr->next_block_ptr);
	    nwos_log(log_msg);
#endif

	    next_ptr->next_block_ptr->dirty = -1;     /* mark it as dirty */
	    memcpy(&next_ptr->next_block_ptr->id, &next_ptr->next_block_ref, sizeof(ObjRef));
	    memcpy(&next_ptr->next_block_ptr->refs[0], ref, sizeof(ObjRef));
	    next_ptr->next_block_ptr->refs[1].word = 0;              /* void reference to mark the end of the list */

	    cache->last_block = next_ptr->next_block_ptr;
	}

	cache->num_refs++;

	assert(running_total + refs_remaining + 1 == cache->num_refs);
    }
}


#ifndef PUBLIC_MODE

void nwos_remove_from_references(ObjRef* ref, ObjRef* obj)
{
    EveryObject header;
    ObjRef class_ref;

    /* first verify we are didn't pass in a reference list as the object */
    /* I.E. meant to call remove_from_reference_list instead. */

    nwos_get_object_class(obj, &class_ref);
    assert(!is_same_object(&class_ref, &nwos_reference_list_class_ref));

    nwos_read_object_headers_from_disk(obj, &header);

    nwos_remove_from_reference_list(ref, &header.object.references);
}



/*-------------------------------------------------------------------------------------------------------
 Remove a reference from the reference list specified (ref_list is the id of the reference list itself).
 
  Since the first block of a reference list is different than the subsequent blocks,
  there are some special cases which arise:

    1. There is only one block.
    2. There are two blocks, but the second block only has 1 entry
    3. There are multiple blocks but the entry to be deleted is in the first.
    4. There are two blocks, the entry is in the first, and the second only has one entry.

  After we are past the first blocker there are four cases:

    5. The entry is in the last block, no other blocks need to be modified.
    6. The entry is the only entry in the last block, the last block needs to be eliminated.
    7. The entry is not in the last block, blocks from the entry to the end need to be written.
    8. The entry is not in the last, and the last block has only one entry and needs to be removed.
-------------------------------------------------------------------------------------------------------*/

void nwos_remove_from_reference_list(ObjRef* ref, ObjRef* ref_list)
{
    char log_msg[128];
    Block_Cache* cache;
    Ref_List_Extra_Block* next_ptr;
    Ref_List_Extra_Block* prev_ptr = NULL;
    int count;
    int i;
    int seq;
    uint8 ivec[IVEC_SIZE];

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
    assert(nwos_reference_type(ref_list) == Public_Reference);
#else
    assert(nwos_reference_type(ref) == Private_Reference);
    assert(nwos_reference_type(ref_list) == Private_Reference);
#endif

    cache = find_ref_list_in_cache(ref_list);

    /* first make sure it is a reference list */

    assert(is_same_object(&cache->first_block->list.common_header.class_definition, &nwos_reference_list_class_ref));

    snprintf(log_msg, sizeof(log_msg), "nwos_remove_from_reference_list - object: %02x%02x%02x%02x - ref list: %02x%02x%02x%02x - num_refs: %d", 
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3],
	    ref_list->id[0], ref_list->id[1], ref_list->id[2], ref_list->id[3],
	    cache->num_refs);
    nwos_log(log_msg);

    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)    /* case #1 - there is only one block */
    {
	assert(cache->first_block->next_block_ptr == NULL);         /* the next block pointer should be null */
	assert(is_void_reference(&cache->first_block->refs[cache->num_refs]));
	assert(cache->last_block == NULL);

	cache->num_refs--;

	if (!is_same_object(ref, &cache->first_block->refs[cache->num_refs]))    /* not the last one in the list */
	{
	    for (i = 0; i < cache->num_refs; i++)
	    {
		if (is_same_object(ref, &cache->first_block->refs[i])) break;
	    }
		
	    assert(i < cache->num_refs);    /* make sure it was in the list */

	    while ((i + 1) < MAX_REFS_IN_REF_LIST && !is_void_reference(&cache->first_block->refs[i+1]))
	    {
		copy_reference(&cache->first_block->refs[i], &cache->first_block->refs[i+1]);
		i++;
	    }

	    assert(i == cache->num_refs);
	}

	void_reference(&cache->first_block->refs[cache->num_refs]);
	cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */
    }
    else   /* cases 2 through 8 - more than one block */
    {
	assert(!is_void_reference(&cache->first_block->next_block_ref));
	assert(cache->last_block != NULL);

	for (i = 0; i < MAX_REFS_IN_REF_LIST; i++)
	{
	    if (is_same_object(ref, &cache->first_block->refs[i])) break;
	}
		
	next_ptr = cache->first_block->next_block_ptr;
	assert(next_ptr != NULL);         /* the next block pointer should not be null */

	count = i;

	if (i < MAX_REFS_IN_REF_LIST)  /* case 2 or 3 - is in the first and there are blocks after it */
	{
	    assert(is_same_object(ref, &cache->first_block->refs[i]));

	    while (i + 1 < MAX_REFS_IN_REF_LIST)
	    {
		copy_reference(&cache->first_block->refs[i], &cache->first_block->refs[i + 1]);

		count++;
		i++;
	    }

	    assert (i == MAX_REFS_IN_REF_LIST - 1);

	    copy_reference(&cache->first_block->refs[i], &next_ptr->refs[0]);
	    count++;

	    if (cache->num_refs == (MAX_REFS_IN_REF_LIST + 1)) /* case 2 - don't need extra block anymore */
	    {
		assert(next_ptr == cache->last_block);

		nwos_clear_bit_in_map(nwos_hash_ref(&cache->first_block->next_block_ref));
		void_reference(&cache->first_block->next_block_ref);

		cache->first_block->next_block_ptr = NULL;
		cache->last_block = NULL;

		nwos_free(next_ptr);
		next_ptr = NULL;    /* flag that we don't need to move anymore */

		cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */

		/* should not need to change ivec for first block, should always be zero */
	    }
	    else   /* case 3 - is in first block, and we need to deal with following blocks */
	    {
		nwos_crc32_calculate((uint8*) &cache->first_block->refs, 
				     (MAX_REFS_IN_REF_LIST + 1) * sizeof(ObjRef),  /* include next_block_ref */
				     cache->first_block->list.common_header.data_chksum);

		memcpy(ivec, cache->first_block->ivec, sizeof(ivec));  /* should be all zeros */
		nwos_write_block_to_disk_and_encrypt(&cache->ref, &cache->first_block->list, FILE_BLOCK_SIZE, ivec, 0);

		memcpy(next_ptr->ivec, ivec, sizeof(ivec));

		i = 0;
		seq = 1;
	    }
	}
	else if (cache->num_refs == (MAX_REFS_IN_REF_LIST + 1))  /* case 4, two blocks with 1 entry in 2nd */
	{
	    assert(next_ptr == cache->last_block);
	    assert(is_same_object(ref, &next_ptr->refs[0]));
	    assert(is_void_reference(&next_ptr->refs[1]));

	    nwos_clear_bit_in_map(nwos_hash_ref(&cache->first_block->next_block_ref));
	    void_reference(&cache->first_block->next_block_ref);

	    cache->first_block->next_block_ptr = NULL;
	    cache->last_block = NULL;

	    nwos_free(next_ptr);
	    next_ptr = NULL;    /* flag that we don't need to move anymore */

	    cache->first_block->list.common_header.flags = 0xffffffff;     /* mark it as dirty */

	    /* should not need to change ivec for first block, should always be zero */
	}
	else   /* cases 5 through 8 */
	{
	    seq = 1;

	    while (true)
	    {
		assert (next_ptr != NULL);

		for (i = 0; i < MAX_REFS_IN_SIDECAR; i++)
		{
		    assert(!is_void_reference(&next_ptr->refs[i]));

		    if (is_same_object(ref, &next_ptr->refs[i])) break;

		    count++;
		}

		if (i < MAX_REFS_IN_SIDECAR) break;

		prev_ptr = next_ptr;

		next_ptr = next_ptr->next_block_ptr;

		seq++;
	    }
	}	

	if (next_ptr != NULL)   /* cases 3 and 5-8 */
	{
	    assert(i < MAX_REFS_IN_SIDECAR);

	    if (i == 0 && is_void_reference(&next_ptr->refs[1]))  /* case 6 - only entry in this block */
	    {
		assert(next_ptr == cache->last_block);
		assert(is_same_object(ref, &next_ptr->refs[0]));
		assert(prev_ptr != NULL);
		assert(prev_ptr->next_block_ptr == next_ptr);
		assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));

		nwos_clear_bit_in_map(nwos_hash_ref(&next_ptr->id));
		void_reference(&prev_ptr->next_block_ref);

		prev_ptr->next_block_ptr = NULL;

		nwos_free(next_ptr);

		prev_ptr->dirty = -1;

		cache->last_block = prev_ptr;
	    }
	    else if (next_ptr->next_block_ptr == NULL)   /* case 5 - just need to remove it from the last block */
	    {
		assert(next_ptr == cache->last_block);

		while (!is_void_reference(&next_ptr->refs[i]))
		{
		    copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

		    count++;
		    i++;
		}

		next_ptr->dirty = -1;     /* mark it as dirty */
	    }
	    else   /* case 7 or 8 */
	    {
		while (next_ptr->next_block_ptr != NULL)
		{
		    while ((i + 1) < MAX_REFS_IN_SIDECAR)
		    {
			copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

			count++;
			i++;
		    }

		    copy_reference(&next_ptr->refs[i], &next_ptr->next_block_ptr->refs[0]);

		    nwos_crc32_calculate((uint8*) &next_ptr->refs, 
					 (MAX_REFS_IN_SIDECAR + 1) * sizeof(ObjRef),  /* include next_block_ref */
					 next_ptr->checksum);

		    memcpy(ivec, next_ptr->ivec, sizeof(ivec));
		    nwos_write_block_to_disk_and_encrypt(&next_ptr->id, &next_ptr->dirty, FILE_BLOCK_SIZE, ivec, seq);

		    memcpy(next_ptr->next_block_ptr->ivec, ivec, sizeof(ivec));

		    prev_ptr = next_ptr;

		    next_ptr = next_ptr->next_block_ptr;

		    seq++;

		    i = 0;
		}

		assert(prev_ptr->next_block_ptr == next_ptr);
		assert(is_same_object(&prev_ptr->next_block_ref, &next_ptr->id));
		assert(next_ptr == cache->last_block);

		if (!is_void_reference(&next_ptr->refs[0]) && is_void_reference(&next_ptr->refs[1])) /* case 8 - only one entry in last block */
		{
		    assert(prev_ptr != NULL);

		    nwos_clear_bit_in_map(nwos_hash_ref(&prev_ptr->next_block_ref));
		    void_reference(&prev_ptr->next_block_ref);

		    nwos_free(prev_ptr->next_block_ptr);
		    prev_ptr->next_block_ptr = NULL;

		    prev_ptr->dirty = -1;     /* mark it as dirty */

		    cache->last_block = prev_ptr;
		}
		else  /* case 7 - just need to finish last block */
		{
		    assert(!is_void_reference(&next_ptr->refs[i + 1]));

		    while (!is_void_reference(&next_ptr->refs[i + 1]))
		    {
			copy_reference(&next_ptr->refs[i], &next_ptr->refs[i+1]);

			count++;
			i++;
		    }

		    void_reference(&next_ptr->refs[i]);

		    next_ptr->dirty = -1;
		}
	    }

	    next_ptr = NULL;

#ifdef VERIFY_LAST_BLOCK_POINTER
	    if (cache->num_refs <= MAX_REFS_IN_REF_LIST)
	    {
		assert(cache->last_block == NULL);
	    }
	    else
	    {
		int block_num;
		int running_total;
		int refs_remaining;

		assert(cache->last_block != NULL);

		block_num = (cache->num_refs - MAX_REFS_IN_REF_LIST - 1) / MAX_REFS_IN_SIDECAR;
		seq = block_num + 1;
		running_total = MAX_REFS_IN_REF_LIST + block_num * MAX_REFS_IN_SIDECAR;

		refs_remaining = cache->num_refs - running_total;

		verify_last_block_pointer(cache, refs_remaining, seq);     /* verify we didn't screw up the last_block pointer */
	    }
#endif
	}

	assert(next_ptr == NULL);

	printf("num_refs: %d  count: %d\n", cache->num_refs, count);

	cache->num_refs--;
    }
}


/* flush any reference lists that are dirty out to disk */

void nwos_flush_dirty_ref_lists()
{
    int i;

    for (i = 0; i < REF_LIST_CACHE_SIZE; i++)
    {
	if (!is_void_reference(&ref_list_cache[i].ref))
	{
	    write_reference_list(&ref_list_cache[i]);
	}
    }
}
#endif

ReferenceList* nwos_malloc_reference_list(ObjRef* ref)
{
    size_t ref_list_size;
    ReferenceList* ref_list;

    ref_list_size = nwos_reference_list_size(ref);

    ref_list = nwos_malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(ref, ref_list, ref_list_size);

    ref_list->common_header.num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    return ref_list;
}


/* This creates a reference list of all objects of a class including both public and private */

ReferenceList* nwos_malloc_class_reference_list(char* class_name)
{
    size_t ref_list_size;
    ReferenceList* ref_list;
    ReferenceList* public_ref_list;
    ReferenceList* private_ref_list = NULL;
    C_struct_Class_Definition class_def;
    ObjRef public_ref;
    ObjRef private_ref;
    ObjRef obj_class_ref;
    int i;
    int num_refs = 0;

    assert(nwos_find_public_class_definition(class_name, &public_ref));
    nwos_read_class_definition(&public_ref, &class_def);
    public_ref_list = nwos_malloc_reference_list(&class_def.header.object.references);
    ref_list_size = public_ref_list->common_header.num_refs;

    if (nwos_find_private_class_definition(class_name, &private_ref))
    {
	nwos_read_class_definition(&private_ref, &class_def);
	private_ref_list = nwos_malloc_reference_list(&class_def.header.object.references);
	ref_list_size += private_ref_list->common_header.num_refs;
    }

    assert(ref_list_size > 0);

    ref_list = nwos_malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading reference list");
	exit(1);
    }

    if (private_ref_list != NULL)
    {
	for (i = 0; i < private_ref_list->common_header.num_refs; i++)
	{
	    nwos_get_object_class(&private_ref_list->references[i], &obj_class_ref);   /* find out what kind of object it is */

	    if (is_same_object(&obj_class_ref, &nwos_private_class_definition_class_ref))   /* it is a class definition object */
	    {
		copy_reference(&ref_list->references[num_refs], &private_ref_list->references[i]);
		num_refs++;
	    }
	}

	nwos_free_reference_list(private_ref_list);
	private_ref_list = NULL;
    }

    for (i = 0; i < public_ref_list->common_header.num_refs; i++)
    {
	nwos_get_object_class(&public_ref_list->references[i], &obj_class_ref);   /* find out what kind of object it is */

	if (is_same_object(&obj_class_ref, &nwos_public_class_definition_class_ref))   /* it is a class definition object */
	{
	    copy_reference(&ref_list->references[num_refs], &public_ref_list->references[i]);
	    num_refs++;
	}
    }

    nwos_free_reference_list(public_ref_list);
    public_ref_list = NULL;

    ref_list->common_header.num_refs = num_refs;

    return ref_list;
}


void nwos_free_reference_list(ReferenceList* ref_list)
{
    nwos_free(ref_list);
}

