/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: log_disc.c,v $
-- Revision 1.40  2008/09/01 03:13:43  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.39  2008/08/31 21:53:53  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.38  2008/08/30 12:46:14  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.37  2008/08/03 17:31:34  jsedwards
-- Add --ignore-empty-directories and --ignore-empty-files options.
--
-- Revision 1.36  2008/08/03 15:44:36  jsedwards
-- Changed to handle new error values returned by nwos_read_files_disc_list.
--
-- Revision 1.35  2008/07/25 12:11:03  jsedwards
-- Changed to allow it to scan subdirectories recursively.
--
-- Revision 1.34  2008/03/25 14:18:40  jsedwards
-- Added an error message if the disk is empty so the user doesn't get some
-- bogus assert.
--
-- Revision 1.33  2008/03/25 14:11:39  jsedwards
-- Fix so that it can deal with more than 9 storage locations (2 digits).
--
-- Revision 1.32  2008/03/25 12:59:08  jsedwards
-- Fixed a typo in a couple of comments and add a couple of new comments.
-- NO code changes!
--
-- Revision 1.31  2008/02/03 01:00:53  jsedwards
-- Changed DEFAULT_TYPE_RW to READ_WRITE.
--
-- Revision 1.30  2007/10/07 03:43:50  jsedwards
-- Renamed 'nwos_set_block_estimate' to 'nwos_check_blocks_available' and
-- changed to exit gracefully if not enough space available.
--
-- Revision 1.29  2007/09/02 19:47:25  jsedwards
-- Added call to set the block estimate.
--
-- Revision 1.28  2007/08/12 20:45:00  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.27  2007/07/12 16:00:04  jsedwards
-- Give 'binder' an illegal initial value so compiler on FreeBSD won't complain.
--
-- Revision 1.26  2007/07/07 20:33:01  jsedwards
-- Changed to also print disc_list info to log.
--
-- Revision 1.25  2007/07/06 12:26:36  jsedwards
-- Change so that storage locations are not hard coded in the program but can
-- be added on the fly by the user.
--
-- Revision 1.24  2007/07/05 18:21:55  jsedwards
-- Changed assert(result == CREATED_NEW) to if statement that says if a new
-- disc_copy was created or it was updated.
--
-- Revision 1.23  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.22  2007/06/13 21:15:01  jsedwards
-- Comment out debugging print statement accidentally left in.
--
-- Revision 1.21  2007/06/13 20:53:50  jsedwards
-- Changed to take a directory as a parameter instead of a list of files.
--
-- Revision 1.20  2007/06/04 13:44:05  jsedwards
-- Add 'match' paramter to find_matchine_path_and_file_association.
--
-- Revision 1.19  2007/06/03 22:03:55  jsedwards
-- Changed so that 'location' can be a Location object or a Person object.
--
-- Revision 1.18  2007/04/28 12:40:56  jsedwards
-- Added printing of each file name so the printing from finding the name and
-- file isn't so weird.
--
-- Revision 1.17  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.16  2007/01/27 01:43:51  jsedwards
-- Change storage options to be numbered from 1 to 6 instead of 0 to 5.
--
-- Revision 1.15  2007/01/20 13:36:58  jsedwards
-- Change encryption level to "very low".
--
-- Revision 1.14  2007/01/13 14:36:48  jsedwards
-- Added "failed" storage location for disk copies that didn't write, verify,
-- or some other problem occured.
--
-- Revision 1.13  2007/01/10 13:15:59  jsedwards
-- Added "missing" as a storage location for discs that are misplaced.
--
-- Revision 1.12  2007/01/07 19:13:41  jsedwards
-- Add call to log the arguments.
--
-- Revision 1.11  2007/01/06 20:12:06  jsedwards
-- Changed binder names and error if no files specified on command line.
--
-- Revision 1.10  2007/01/05 13:51:16  jsedwards
-- Change to work with new file association.
--
-- Revision 1.9  2006/12/07 14:10:25  jsedwards
-- Removed call to file_setup, no longer needed.
--
-- Revision 1.8  2006/12/02 17:23:32  jsedwards
-- Removed start_time that wasn't used and added SafeDepositBox location.
--
-- Revision 1.7  2006/11/19 16:40:22  jsedwards
-- Moved creation of disc list to separate program (disc_list) and made this
-- program just verify the disc list create the disc_copy object.
--
-- Revision 1.6  2006/11/18 14:34:58  jsedwards
-- Changed so that discs can be stored in one of three different binders.
--
-- Revision 1.5  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/11/07 14:03:35  jsedwards
-- Add assert to verify file_path didn't come back void.
--
-- Revision 1.3  2006/11/06 13:54:07  jsedwards
-- Changed wording in print statement because disc_list now stores the
-- references to the file_path object instead of the file object.
--
-- Revision 1.2  2006/11/05 21:32:18  jsedwards
-- Add code to create storage_location and disc_copy objects.
--
-- Revision 1.1  2006/11/04 18:56:22  jsedwards
-- Program to scan all the files on a CD or DVD and make a disk_list object
-- of it.
--
*/


#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"

#define MAX_STORAGE_LOCATIONS 48  /* for now */

static char* file_names[MAX_FILES_PER_DISC_LIST];
static ObjRef files[MAX_FILES_PER_DISC_LIST];


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}



int main(int argc, char* argv[])
{
    ObjRef ref;
    int i;
    char id[13];
    char copy_num[3] = "";
    char binder_num[4] = "";
    char name[24];
    char *p;
    bool ok = false;
    int binder = -2;
    uint32 num_files;
    ObjRef location_class_ref;
    ObjRef object_class;
    C_struct_Class_Definition class_def_obj;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef location_ref[MAX_STORAGE_LOCATIONS];
    int num_locations = 0;
    C_struct_Storage_Location loc_obj;
    ObjRef disc_list_ref;
    ObjRef person_ref;
    bool missing_files = false;
    char msg[128];
#if 0
    bool bad_checksum = false;
    CheckFileResult check_file_result;
#endif
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;
    uint32 error_mask = DISC_LIST_ERROR_MASK;
    int argi = 1;

    while (argi < argc - 1)
    {
	if (strcmp(argv[argi], "--ignore-empty-directories") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_DIR_FLAG;
	    argi++;
	}
	else if (strcmp(argv[argi], "--ignore-empty-files") == 0)
	{
	    error_mask &= ~DISC_LIST_EMPTY_FILE_FLAG;
	    argi++;
	}
    }

    if (argi != argc - 1)
    {
	fprintf(stderr, "usage: %s [--ignore-empty-directories] [--ignore-empty-files] directory\n", argv[0]);
	fprintf(stderr, "    --ignore-empty-directories  ignore any empty directories\n");
	fprintf(stderr, "    --ignore-empty-files        ignore any empty files\n");
	exit(1);
    }

    printf("\n");


    /******************************************************/
    /* First open the disc and read all of the file names */
    /******************************************************/

    num_files = nwos_read_files_disc_list(argv[argi], NULL, file_names, 0);    /* NULL for subdirectory indicates this is the root */

    if ((num_files & DISC_LIST_NUM_FILES_MASK) == 0)
    {
	fprintf(stderr, "\nERROR: Disc is empty!  Perhaps it is not mounted?\n\n");

	exit(16);  /* 16 means directory is empty */
    }

    if ((num_files & error_mask) != 0)
    {
	fprintf(stderr, "\n");

	exit(num_files >> DISC_LIST_ERROR_SHIFT);  /* exit code is error code returned by nwos_read_files_disc_list 1-15 */
    }

    num_files &= DISC_LIST_NUM_FILES_MASK;

    printf("\n");

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

    nwos_set_security_level(Security_Very_Low);


    if (!nwos_check_blocks_available(14))   /* 14 if new storage location is created, normally just 2 */
    {
	fprintf(stderr, "Cannot create new storage location!\n");
	nwos_terminate_objectify();
	exit(1);
    }

    if (nwos_find_private_class_definition("STORAGE LOCATION", &location_class_ref))
    {
	nwos_read_class_definition(&location_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &location_class_ref))
	    {
		copy_reference(&location_ref[num_locations], &ref_list->references[i]);

		num_locations++;

		if (num_locations == MAX_STORAGE_LOCATIONS)
		{
		    fprintf(stderr, "WARNING: more than 40 storage locations, remainder skipped\n");
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
    else
    {
	fprintf(stderr, "Note: no storage locations are defined!\n");
    }

    /* first verify that all the listed files are already in the system */

    memset(files, 0, sizeof(files));

    for (i = 0; i < num_files; i++)
    {
	printf("%s: ", file_names[i]);
	fflush(stdout);

	if (!nwos_find_matching_path_and_file_association(argv[argi], file_names[i], &files[i], MatchAll))
	{
	    missing_files = true;
	    printf("  <--- NOT found");
	}

	printf("\n");
    }

    if (missing_files)
    {
	printf("The following files are not in the system:\n\n");

	for (i = 0; i < num_files; i++)
	{
	    if (is_void_reference(&files[i]))
	    {
		printf("   %s\n", file_names[i]);
	    }
	}

	printf("\n");
	exit(1);
    }


    /* find and verify the disc */

    if (!nwos_find_matching_disc_list(files, &disc_list_ref))
    {
	printf("No matching disc list found!\n");
	exit(1);
    }

    assert(nwos_read_variable_sized_object_from_disk(&disc_list_ref, kludge, sizeof(kludge), &nwos_get_disc_list_object_size));

    memcpy(id, ptr_list_obj->id, sizeof(id) - 1);
    id[sizeof(id) - 1] = '\0';

    snprintf(msg, sizeof(msg), "disc list %02x%02x%02x%02x: %s",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3], id);
    nwos_log(msg);
    printf("%s\n", msg);

    /* create record of this disc */

    ok = false;
    while (!ok)
    {
	printf("Copy number: ");
	fflush(stdout);

	fgets(copy_num, sizeof(copy_num), stdin);

	p = strchr(copy_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was too long */
	{
	    while (p == NULL)   /* keep reading until we find the newline char */
	    {
		fgets(copy_num, sizeof(copy_num), stdin);
		p = strchr(copy_num, '\n');   /* find the newline char */
	    }
	    printf("copy number can only be 1 digit in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(copy_num) != 1)
	    {
		printf("copy number can only be 1 digit in this version\n");
	    }
	    else if (copy_num[0] < '1' || copy_num[0] > '9')
	    {
		printf("copy number must be a 1 digit number in this version\n");
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }

    ok = false;
    while (!ok)
    {
	printf("Available locations:\n\n");

	printf("   0=create new\n");

	for (i = 0; i < num_locations; i++)
	{
	    assert(nwos_read_object_from_disk(&location_ref[i], &loc_obj, sizeof(loc_obj)));
	    nwos_name_to_string(&loc_obj.name, name, sizeof(name));
	    printf("   %d=%s\n", i + 1, name);
	}

	printf("   P=person\n\n");

	printf("Selection: ");
	fflush(stdout);

	fgets(binder_num, sizeof(binder_num), stdin);

	p = strchr(binder_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was too long */
	{
	    while (p == NULL)    /* keep reading until we find the newline char */
	    {
		fgets(binder_num, sizeof(binder_num), stdin);
		p = strchr(binder_num, '\n');   /* find the newline char */
	    }
	    printf("storage binder can only be 1 or 2 digits in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (toupper(binder_num[0]) == 'P' && binder_num[1] == '\0')
	    {
		binder = -1;   /* flag it is a person */
		break;
	    }
	    else if (binder_num[0] == '0' && binder_num[1] == '\0')      /* create a new name */
	    {
		get_input("name", name, sizeof(name));

		if (nwos_create_storage_location(name, &location_ref[num_locations]) == CREATED_NEW)
		{
		    binder = num_locations;
		    num_locations++;
		    break;
		}
		else
		{
		    fprintf(stderr, "Note: a storage location with that name already exists.\n");
		}
	    }
	    else if (isdigit(binder_num[0]) && binder_num[1] == '\0')    /* single non-zero digit entered */
	    {
		binder = binder_num[0] - '1';

		ok = true;   /* we should be good to go */
	    }
	    else if (isdigit(binder_num[0]) && binder_num[0] != '0' && isdigit(binder_num[1]) && binder_num[2] == '\0')   /* two digits entered */
	    {
		binder = (binder_num[0] - '0') * 10 + binder_num[1] - '0' - 1;

		if (binder < num_locations)
		{
		    ok = true;   /* we should be good to go */
		}
		else
		{
		    printf("Please enter a number less than or equal to %d\n", num_locations);
		}
	    }
	    else
	    {
		printf("Please enter a number between 0 and %d or 'P'.\n", num_locations);
	    }
	}
    }

    assert(-1 <= binder && binder < num_locations);

    if (binder == -1)
    {
	get_input("name", name, sizeof(name));

	if (nwos_find_person(name, &person_ref))
	{
	    assert(nwos_create_disc_copy(&disc_list_ref, copy_num[0] - '0', &person_ref, &ref) == CREATED_NEW);
	}
	else
	{
	    fprintf(stderr, "%s not found, terminating without creating record\n", name);
	}
    }
    else
    {
	if (nwos_create_disc_copy(&disc_list_ref, copy_num[0] - '0', &location_ref[binder], &ref) == CREATED_NEW)
	{
	    printf("created new: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
	else
	{
	    printf("updated: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
    }

    for (i = 0; i < num_files; i++)
    {
	free(file_names[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


