/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: sync_archive.c,v $
-- Revision 1.4  2008/12/30 14:10:59  jsedwards
-- Fixed usage statement.
--
-- Revision 1.3  2008/12/30 13:46:24  jsedwards
-- Changed to call new set_private_objects_path function in user_config.c to
-- change to write the destination archive.
--
-- Revision 1.2  2008/12/30 13:41:56  jsedwards
-- Changed to set the private path environment variable.  Still doesn't work
-- because the environment variable has already been read by that point in time.
--
-- Revision 1.1  2008/12/30 13:36:35  jsedwards
-- Initial version created by combining compress_sparse.c and expand_sparce.c.
-- Does not work!
--
*/


#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

extern void nwos_set_private_objects_path(const char* path);


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s destination-archive\n", program);
}


#define SIZE_COUNTS 16

int main(int argc, char* argv[])
{
    int obj_file_desc;
    const char* obj_file_path;
    off_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block[FILE_BLOCK_SIZE];
    int i;
    int j;
    size_t bytes_read;
    int num_blocks;
    uint32 total_private_blocks;
    uint32 used_private_blocks;
    uint32 chunk_block_offset;
    uint32 blocks_on_disk;
    char answer[8];
    char msg[128];
    Disk_Header disk_header;
    uint32 counts[SIZE_COUNTS];
    uint32 ref;
    uint32 hash;
    int chunk_num;
    uint32 used_chunks;
    Chunk_Info* chunk_info;
    int blocks_with_bad_ids = 0;


    for (i = 0; i < SIZE_COUNTS; i++) counts[i] = 0;

    if (argc != 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }


    nwos_log_arguments(argc, argv);

    obj_file_path = nwos_get_private_objects_path();    /* must do this before changing env variable */

    printf("obj_file_path: %s\n", obj_file_path);

    nwos_set_private_objects_path(argv[1]);

    assert(strcmp(nwos_get_private_objects_path(), argv[1]) == 0);

    nwos_initialize_disk_io(READ_WRITE, DEFAULT_FILE);

    nwos_terminate_backup(false, NULL);   /* don't create a backup file */

    /* verify that the number of used blocks is consistent with empty disk */
    if (nwos_used_private_blocks != 0 || nwos_used_private_chunks != 0)
    {
	fprintf(stderr, "ERROR: disk not empty, chunks used: %u  blocks used: %u!.\n", 
		nwos_used_private_blocks, nwos_used_private_chunks);

	nwos_terminate_disk_io();
	exit(1);
    }

    /* Verify that is what s/he really wants to do */

    fprintf(stderr, "\n");
    fprintf(stderr, "WARNING: the contents of %s will be OVERWRITTEN!!!\n", nwos_private_path);
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you want to continue? (enter `yes' to write %s) ", nwos_private_path);

    fflush(stderr);

    fgets(answer, sizeof(answer), stdin);

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }

    /* Open the storage drive and verify the header info */

    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block, sizeof(block));

    if (bytes_read != sizeof(block))
    {
	perror("reading first block");
	exit(1);
    }

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	exit(1);
    }

    if (memcmp(&block[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", obj_file_path);
	exit(1);
    }

    memcpy(&disk_header, block, sizeof(disk_header));

    nwos_4_uint8_to_uint32(disk_header.total_blocks, &total_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &chunk_block_offset);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

    assert(used_chunks > 0);

    chunk_info = malloc(used_chunks * sizeof(Chunk_Info));
    assert(chunk_info != NULL);

    bytes_read = read(obj_file_desc, chunk_info, used_chunks * sizeof(Chunk_Info));

    if (bytes_read != used_chunks * sizeof(Chunk_Info))
    {
	perror("reading chunk info");
	exit(1);
    }

    /* fix the byte order on little endian machines */
#ifndef WORDS_BIGENDIAN
	{
	  int i;
	  for (i = 0; i < used_chunks; i++)
	  {
	      chunk_info[i].ref = byteswap_uint32(chunk_info[i].ref);
	      chunk_info[i].used = byteswap_uint16(chunk_info[i].used);
	      chunk_info[i].index = byteswap_uint16(chunk_info[i].index);
	  }
	}
#endif

    blocks_on_disk = total_private_blocks; 


    printf("header: %c%c%c%c %c%c%c%c\n",
	   disk_header.magic_number[0], disk_header.magic_number[1], disk_header.magic_number[2], disk_header.magic_number[3], 
	   disk_header.version_string[0], disk_header.version_string[1], disk_header.version_string[2], disk_header.version_string[3]);

    printf("total blocks on disk: %08u  chunks_used: %d\n", blocks_on_disk, used_chunks);
    fflush(stdout);

    if (used_chunks > nwos_total_private_chunks)
    {
	fprintf(stderr, "used chunks: %u greater than disk space: %u\n",
		used_chunks, nwos_total_private_chunks);
	exit(1);
    }


    num_blocks = 0;

    nwos_start_progress_bar();

    // for now we can skip over public blocks because they should always stay the same
    for (chunk_num = 0; chunk_num < used_chunks; chunk_num++)
    {
	nwos_update_progress_bar((float)num_blocks / (float)used_private_blocks);

	chunk = chunk_block_offset + chunk_info[chunk_num].index * BLOCKS_IN_CHUNK;

	if (lseek(obj_file_desc, chunk << 8, SEEK_SET) < 0)
	{
	    snprintf(msg, sizeof(msg), "lseek chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	if (bytes_read != sizeof(block_map))
	{
	    snprintf(msg, sizeof(msg), "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	/* scan block map (skip over the blocks for the block map itself) */
	for (i = BIT_MAP_BLOCKS; i < BLOCKS_IN_CHUNK; i++)
	{
	    if ((block_map[i/8] & (0x80 >> (i%8))) != 0)
	    {
		if (lseek(obj_file_desc, (chunk + i) << 8, SEEK_SET) < 0)
		{
		    snprintf(msg, sizeof(msg), "lseek block:%08x", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}

		bytes_read = read(obj_file_desc, block, sizeof(block));

		if (bytes_read != sizeof(block))
		{
		    snprintf(msg, sizeof(msg), "reading block: %u", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}
		
		ref = (uint32)block[4] << 24 | (uint32)block[5] << 16 | (uint32)block[6] << 8 | (uint32)block[7];

		if (ref == 0)
		{
		    blocks_with_bad_ids++;

		    printf("WARNING: reference in block is zero, not written:\n");

		    for (j = 0; j < FILE_BLOCK_SIZE; j += 16)
		    {
			printf("%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x\n",
			       block[j+0],  block[j+1],  block[j+2],  block[j+3],
			       block[j+4],  block[j+5],  block[j+6],  block[j+7],
			       block[j+8],  block[j+9],  block[j+10], block[j+11],
			       block[j+12], block[j+13], block[j+14], block[j+15]);
		      }
		}
		else
		{
#if 0
		    /* save this for verbose mode? */
		    printf("id: %08x  block: %08x\n", ref, (uint32)chunk+i);
		    /* printf("id: %08x\n", ref); */
		    fflush(stdout);
#endif
		    counts[ref >> 28]++;

		    nwos_4_uint8_to_uint32(&block[4], &ref);

		    hash = nwos_hash_uint32_ref(ref);

		    if (hash == 0)
		    {
			nwos_allocate_new_chunk(ref);

			hash = nwos_hash_uint32_ref(ref);

			assert(hash != 0);
		    }

		    if (nwos_test_bit_in_map(hash))
		    {
			fprintf(stderr, "ERROR: block is already used!\n");
			exit(1);
		    }

		    nwos_set_bit_in_map(hash);

		    nwos_write_block((ObjRef*)&block[4], block);

		    num_blocks++;

		    assert(num_blocks == nwos_used_private_blocks);
		}
	    }
	}
    }

    nwos_finish_progress_bar();

    printf("Number of blocks: %d\n", num_blocks);

    if (blocks_with_bad_ids > 0)
    {
	printf("WARNING: %d blocks had a reference ID of zero and weren't written!\n", blocks_with_bad_ids);
    }

    close(obj_file_desc);

    nwos_terminate_objectify();

    snprintf(msg, sizeof(msg), "Number of blocks written: %d  chunks_used: %u",
	     num_blocks, nwos_used_private_chunks);
    nwos_log(msg);
    printf("%s\n", msg);

    if (used_private_blocks != nwos_used_private_blocks)
    {
	snprintf(msg, sizeof(msg), "WARNING: used_private_blocks mismatch, disk: %u  compressed %u\n",
		nwos_used_private_blocks, used_private_blocks);
	fprintf(stderr, "%s\n", msg);
    }

    for (i = 0; i < SIZE_COUNTS; i++)
    {
	if (counts[i] > 0) printf("  %d: %u\n", i, counts[i]);
    }

    return 0;
}



