/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: export_file.c,v $
-- Revision 1.24  2008/12/27 15:00:29  jsedwards
-- Merged from alpha_29_5_branch back into main trunk.
--
-- Revision 1.23.2.2  2008/12/21 14:12:52  jsedwards
-- Changed to exit with code 1 when export_file subroutine fails.
--
-- Revision 1.23.2.1  2008/12/17 03:29:41  jsedwards
-- Changed to exit with failure status when file doesn't exist.
--
-- Revision 1.23  2008/09/01 03:13:42  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.22  2008/08/31 21:53:52  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.21  2008/08/30 12:46:10  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.20  2008/06/05 11:04:14  jsedwards
-- Fixed to find the latest version of the path and file association object.
--
-- Revision 1.19  2008/03/09 15:41:12  jsedwards
-- Added code to create subdirectories if they don't exist.  (File changed on
-- laptop on 2008/02/29 in Columbia, South Carolina but could not check in
-- because didn't have Internet access).
--
-- Revision 1.18  2008/02/03 01:04:28  jsedwards
-- Changed DEFAULT_TYPE_RO to READ_ONLY and changed path variable to const char
-- instead of char.
--
-- Revision 1.17  2007/09/18 01:22:29  jsedwards
-- Rearrange the order of testing for an old (depreciated) file name, so that
-- it is only done if the file name matches.  That way the association object
-- only has to be read if we have found a match instead of for every single
-- file.
--
-- Revision 1.16  2007/08/26 18:41:26  jsedwards
-- Added code so that if an old (renamed) file name is specifically given it
-- will be exported, but old (renamed) file names will not match a wild card
-- search.
--
-- Revision 1.15  2007/08/23 13:50:22  jsedwards
-- Added file name globbing.
--
-- Revision 1.14  2007/07/24 17:27:09  jsedwards
-- Modified to deal with files that have more than one revision.
--
-- Revision 1.13  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.12  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.11  2007/01/14 17:17:48  jsedwards
-- Changed to use path_and_file_association objects instead of file objects.
--
-- Revision 1.10  2007/01/07 20:31:55  jsedwards
-- Added call to log arguments.
--
-- Revision 1.9  2007/01/07 14:18:10  jsedwards
-- Changed so that file_path_to_file has to succeed before reading the file
-- (before it ignored the return value of file_path_to_file because it always
-- returned true).
--
-- Revision 1.8  2006/12/21 12:59:47  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.7  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.6  2006/11/29 19:03:45  jsedwards
-- Added code to handle the case where the requested file is not stored in
-- the system.
--
-- Revision 1.5  2006/11/29 18:52:04  jsedwards
-- Change so that this function finds the file reference and passes it along
-- with the path to the restore_file function.
--
-- Revision 1.4  2006/11/18 15:09:09  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.3  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.5  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.4  2006/10/07 12:35:41  jsedwards
-- Changed to allow an '*' to specify exporting all files.
--
-- Revision 1.1.2.3  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.1.2.2  2006/10/02 02:27:03  jsedwards
-- Added option to read the file from a compressed archive file.
--
-- Revision 1.1.2.1  2006/09/18 01:45:12  jsedwards
-- Program to export a file stored in objectify.
--
*/

#include <errno.h>
#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "objectify.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}



bool export_file(char* name)
{
    bool result = false;
    ObjRef path_ref;
    ObjRef assoc_ref;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    int num_files;
    char* p;

    printf("%s: ", name);
    fflush(stdout);

    assert(nwos_find_file_path(name, &path_ref));

    num_files = nwos_number_of_files_for_path(&path_ref);

    if (num_files > 0)
    {
	if (num_files > 1)
	{
	    printf(" WARNING: this version cannot deal with multiple files - skipping\n");
	}
	else
	{
	    assert(nwos_file_path_to_path_and_file_association(&path_ref, 0, &assoc_ref));

	    assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the assoc. object */

	    while (!is_void_reference(&assoc_obj.header.object.next_version))
	    {
		assert(nwos_read_object_from_disk(&assoc_obj.header.object.next_version, &next_assoc_obj, sizeof(next_assoc_obj)));

		if (!is_same_object(&next_assoc_obj.path, &path_ref))   /* it has a different name */
		{
		    assert(is_same_object(&next_assoc_obj.file, &assoc_obj.file));
		    break;
		}

		copy_reference(&assoc_ref, &assoc_obj.header.object.next_version);
		memcpy(&assoc_obj, &next_assoc_obj, sizeof(assoc_obj));
	    }

	    assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));  /* read the file object */

	    if (is_void_reference(&file_obj.block_list))
	    {
		printf("not stored in system\n");
	    }
	    else
	      {
		p = strchr(name, '/');
		while (p != NULL)
		{
		    *p = '\0';
		    if (mkdir(name, 0755) != 0 && errno != EEXIST)
		    {
			perror(p);
		    }
		    *p = '/';

		    p = strchr(p + 1, '/');
		}

		result = nwos_restore_file(&assoc_ref, name);

		if (result)
		{
		    printf("OK\n");
		}
		else
		{
		    printf("failed\n");
		}
	    }
	}
    }
    else
    {
	printf("no file associated with that name in system\n");
    }

    return result;
}


int main(int argc, char* argv[])
{
    ObjRef path_class_ref;
    ObjRef object_class;
    ObjRef path_ref;
    ObjRef assoc_ref;
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    int argn;
    int i;
    int j;
    int k;
    const char* path = DEFAULT_FILE;
    char name[256];
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    bool match;
    bool result = true;


    if (argc > 1 && *argv[1] == '-')
    {
	if (strcmp(argv[1], "--compressed") != 0 || argc < 3)
	{
	    fprintf(stderr, "usage: %s [--compressed compressed-file] files to extract\n", argv[0]);
	    exit(1);
	}

	path = argv[2];
	argn = 3;
    }
    else
    {
	argn = 1;
    }

    printf("\n");

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_ONLY, path);


    if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref))
    {
	printf("No files are stored in the system!\n");
	exit(1);
    }

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = argn; i < argc; i++)
    {
	match = false;

	for (j = 0; j < num_refs; j++)
	{
	    nwos_get_object_class(&ref_list->references[j], &object_class);

	    if (is_same_object(&object_class, &path_class_ref))
	    {
		assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[j], kludge, sizeof(kludge), &get_path_object_size));

		/* remember ptr_path_obj points to the kludge buffer */

		/* assert(ptr_path_obj->count < sizeof(name)); <<- count is uint8 so always true */

		for (k = 0; k < ptr_path_obj->count; k++) name[k] = ptr_path_obj->storage[k];
		name[k] = '\0';

		if (strcmp(argv[i], name) == 0)   /* found an exact match */
		{
		    if (export_file(name))
		    {
			num_files++;
		    }
		    else
		    {
			result = false;
		    }

		    match = true;
		}
		else
		{
		    if (fnmatch(argv[i], name, 0) == 0)   /* found a match */
		    {
			assert(nwos_find_file_path(name, &path_ref));
			assert(nwos_file_path_to_path_and_file_association(&path_ref, 0, &assoc_ref));

			assert(nwos_read_object_from_disk(&assoc_ref, &assoc_obj, sizeof(assoc_obj)));  /* read the assoc. object */

			while (!is_void_reference(&assoc_obj.header.object.next_version))  /* find latest version */
			{
			    assert(nwos_read_object_from_disk(&assoc_obj.header.object.next_version, &next_assoc_obj, sizeof(next_assoc_obj)));

			    if (!is_same_object(&next_assoc_obj.path, &path_ref))   /* it has a different name */
			    {
				assert(is_same_object(&next_assoc_obj.file, &assoc_obj.file));
				break;    /* don't match old file names to wildcards */
			    }

			    copy_reference(&assoc_ref, &assoc_obj.header.object.next_version);
			    memcpy(&assoc_obj, &next_assoc_obj, sizeof(assoc_obj));
			}

			if (is_void_reference(&assoc_obj.header.object.next_version))
			{
			    if (export_file(name))
			    {
				num_files++;
			    }
			    else
			    {
				result = false;
			    }

			    match = true;
			}
		    }
		}
	    }
	}

	if (!match)
	{
	    printf("No file(s) matching '%s' found.\n", argv[i]);

	    result = false;
	}
    }

    printf("Number of files: %d\n", num_files);

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return !result;    /* return 0 means success */
}


