/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: date.c,v $
-- Revision 1.40  2008/09/01 00:10:49  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.39  2008/08/31 20:33:17  jsedwards
-- Add assert around calls to nwos_read_object_from_disk because it now
-- returns false when it fails, whereas before it would assert itself.
--
-- Revision 1.38  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.37  2007/04/01 17:39:41  jsedwards
-- Added code for private mode to create_date.
--
-- Revision 1.36  2007/04/01 17:25:24  jsedwards
-- Added find_or_create_private_month_and_day function.
--
-- Revision 1.35  2007/04/01 16:07:50  jsedwards
-- Added find_or_create_private_year and find_or_create_public_year functions.
--
-- Revision 1.34  2007/04/01 14:58:15  jsedwards
-- Added find_public_date function.
--
-- Revision 1.33  2007/04/01 14:45:04  jsedwards
-- Changed incorrect calls to find_or_create_private_class to find_private_class
-- in find routines.
--
-- Revision 1.32  2007/04/01 14:32:56  jsedwards
-- Added find_private_month_and_day function.
--
-- Revision 1.31  2007/04/01 14:21:59  jsedwards
-- Added find_private_month function.
--
-- Revision 1.30  2007/04/01 14:11:49  jsedwards
-- Added find_private_year function.
--
-- Revision 1.29  2007/04/01 14:00:19  jsedwards
-- Added 'public' to the names of the find functions because they currently
-- only work on public dates.
--
-- Revision 1.28  2007/04/01 12:04:48  jsedwards
-- Added check for valid day in find_month_and_day and fixed bug in find_date
-- assertion on day.
--
-- Revision 1.27  2007/04/01 11:53:57  jsedwards
-- Removed unused result variables.
--
-- Revision 1.26  2006/12/28 23:13:15  jsedwards
-- Added ifdef to create_year function so that it is only compiled when in
-- public mode because all years should already be in the public data.  Also
-- changed create date to just call find_year instead of creating the year.
--
-- Revision 1.25  2006/12/25 12:13:00  jsedwards
-- Ifdef out code to find date in private area and switch find class to public
-- or private.
--
-- Revision 1.24  2006/12/21 13:10:51  jsedwards
-- Attempt to fix for new split public and private objects, but turned into a
-- hack to make it compile, NON-FUNCTIONAL!
--
-- Revision 1.23  2006/12/10 19:27:09  jsedwards
-- Converted to new date format with month_and_day objects.
--
-- Revision 1.22  2006/12/10 18:26:35  jsedwards
-- Split find_date out of create_date routine (untested).
--
-- Revision 1.21  2006/12/10 14:21:55  jsedwards
-- Split find year code out of create_year routine (untested).
--
-- Revision 1.20  2006/12/07 14:08:46  jsedwards
-- Moved setup routine and associated functions to attic/big_bang.c.
--
-- Revision 1.19  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.18  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.17  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.16.2.3  2006/10/25 12:22:27  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.16.2.2  2006/09/10 20:43:30  jsedwards
-- Changed to use sizeof instead of hard coded 4 (not sure 4 was correct).
--
-- Revision 1.16.2.1  2006/09/02 01:11:19  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.16  2006/01/09 03:19:12  jsedwards
-- Changed the upgrade routine to search through the month class reference
-- list itself, to find all of the months instead of using the "get month ref"
-- routine.  The "get month ref" routine reads the month objects to find the
-- right one and since they are of the wrong size the checksum fails.
--
-- Revision 1.15  2006/01/06 14:08:52  jsedwards
-- Added code to upgrade a 0004 month object to a 0005 month object.
--
-- Revision 1.14  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.13  2006/01/01 00:41:23  jsedwards
-- Removed count from month object.  Since it was always 2, why have it?
-- Added date, month, and year to string routines.
--
-- Revision 1.12  2005/12/30 14:07:03  jsedwards
-- Added is_leap_year function.
--
-- Revision 1.11  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.10  2005/12/29 17:47:58  jsedwards
-- Changed to make Year object variable sized (although in reality at this
-- time it is fixed at 4 digits in the year).  This allows the write object
-- to disk routine to fill in the remainder of the 512 bytes with random data.
--
-- Revision 1.9  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.8  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.7  2005/12/21 03:52:27  jsedwards
-- Moved US state objects creation from 'create_class_def.c' into 'date.c'.
--
-- Revision 1.6  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.5  2005/12/05 05:22:46  jsedwards
-- Change to call read_reference_list_from_disk for reference lists instead
-- of read_object_from_disk.  Also moved calculating checksums down to just
-- before writing object, so that no changes occur after computing them.
--
-- Revision 1.4  2005/12/04 14:09:10  jsedwards
-- Added checks when creating a date that it is a valid date.
--
-- Revision 1.3  2005/12/04 04:13:02  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.2  2005/12/04 00:37:46  jsedwards
-- Added get_month function and made create_date function work.
--
-- Revision 1.1  2005/12/03 23:22:09  jsedwards
-- Initial version extracted from create_person.c.
--
*/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


static uint8 days_per_month[12] = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };




static bool find_year_from_cardinal(ObjRef* year_class_ref, ObjRef* cardinal_ref, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Year year_obj;
    ObjRef class_ref;
    ReferenceList* ref_list;
    int num_years;
    int i;

    nwos_read_class_definition(year_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_years = ref_list->common_header.num_refs;

    /* printf("num_years: %d\n", num_years); */

    for (i = 0; i < num_years; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, year_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &year_obj, sizeof(year_obj)));

	    if (is_same_object(&year_obj.number, cardinal_ref))   /* found a match */
	    {
		copy_reference(ref, &ref_list->references[i]);
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_years;
}


bool nwos_find_public_year(uint16 year, ObjRef* ref)
{
    char year_ascii[5];
    ObjRef year_class_ref;
    ObjRef cardinal_ref;
    bool result = false;

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    assert(nwos_find_public_class_definition("YEAR", &year_class_ref));

    year_ascii[0] = '0' +  (year / 1000);
    year_ascii[1] = '0' + ((year / 100) % 10);
    year_ascii[2] = '0' + ((year / 10)  % 10);
    year_ascii[3] = '0' +  (year        % 10);
    year_ascii[4] = '\0';

    if (nwos_find_cardinal_number(year_ascii, &cardinal_ref))
    {
	result = find_year_from_cardinal(&year_class_ref, &cardinal_ref, ref);
    }

    return result;
}


bool nwos_find_private_year(uint16 year, ObjRef* ref)
{
    char year_ascii[5];
    ObjRef year_class_ref;
    ObjRef cardinal_ref;
    bool result = false;

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    if (nwos_find_private_class_definition("YEAR", &year_class_ref))
    {
	year_ascii[0] = '0' +  (year / 1000);
	year_ascii[1] = '0' + ((year / 100) % 10);
	year_ascii[2] = '0' + ((year / 10)  % 10);
	year_ascii[3] = '0' +  (year        % 10);
	year_ascii[4] = '\0';

	if (nwos_find_cardinal_number(year_ascii, &cardinal_ref))
	{
	    result = find_year_from_cardinal(&year_class_ref, &cardinal_ref, ref);
	}
    }

    return result;
}


#ifdef PUBLIC_MODE
ObjCreateResult nwos_create_year(uint16 year, ObjRef* ref)
{
    C_struct_Year year_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    char year_ascii[5];
    ObjCreateResult result;

    assert(1000 <= year && year <= 9999);   /* for now limit it to 4 digit years */

    year_ascii[0] = '0' +  (year / 1000);
    year_ascii[1] = '0' + ((year / 100) % 10);
    year_ascii[2] = '0' + ((year / 10)  % 10);
    year_ascii[3] = '0' +  (year        % 10);
    year_ascii[4] = '\0';

    /* first find out if we already have this year */

    assert(nwos_find_public_class_definition("YEAR", &class_ref));

    result = nwos_create_cardinal_number(year_ascii, &cardinal_ref);

    if (result == CREATED_NEW || !nwos_find_year_from_cardinal(&cardinal_ref, ref))
    {
	memset(&year_obj, 0, sizeof(year_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&year_obj.header.common, ref, &class_ref);

	copy_reference(&year_obj.number, &cardinal_ref);

	nwos_create_reference_list(ref, &year_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &year_obj.header.object, sizeof(ObjectHeader), year_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &year_obj.number, sizeof(year_obj) - sizeof(EveryObject), year_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &year_obj, sizeof(year_obj));

        nwos_add_to_references(ref, &class_ref);
	nwos_add_to_references(ref, &cardinal_ref);

	result = CREATED_NEW;
    }

    return result;
}
#endif



/**************************************************************************************/
/* This routine finds a private year object or creates one by cloning the public one. */
/**************************************************************************************/

void nwos_find_or_create_private_year(uint16 year, ObjRef* ref)
{
    ObjRef public_year_ref;
    ObjRef private_year_class_ref;
    C_struct_Year year_obj;
    char msg[128];

    assert(1582 <= year && year <= 2020);   /* for now limit it to 4 digit years */

    if (!nwos_find_private_year(year, ref))
    {
	assert(nwos_find_public_year(year, &public_year_ref));

	nwos_find_or_create_private_class_definition("YEAR", &private_year_class_ref);

	assert(nwos_read_object_from_disk(&public_year_ref, &year_obj, sizeof(year_obj)));

	nwos_generate_new_completely_random_id(ref);

	snprintf(msg, sizeof(msg), "clone_year(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_year_ref.id[0], public_year_ref.id[1], public_year_ref.id[2], public_year_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&year_obj.header.common.id, ref);

	nwos_get_time_stamp(year_obj.header.common.creation_time);

	copy_reference(&year_obj.header.common.class_definition, &private_year_class_ref);

	copy_reference(&year_obj.header.object.clone_of, &public_year_ref);

	nwos_create_reference_list(ref, &year_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &year_obj.header.object, sizeof(ObjectHeader), year_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &year_obj, sizeof(year_obj));

	nwos_add_to_references(ref, &private_year_class_ref);
    }
}


/* The months should always exist (all twelve of them) */

static bool find_month(ObjRef* month_class_ref, uint8 month, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Month month_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ReferenceList* ref_list;
    int num_months;
    int i;
    char month_ascii[3];

    assert(1 <= month && month <= 12);

    month_ascii[0] = '0' + (month / 10);
    month_ascii[1] = '0' + (month % 10);
    month_ascii[2] = '\0';

    assert(nwos_find_cardinal_number(month_ascii, &cardinal_ref));
    assert(nwos_find_ordinal_number(&cardinal_ref, &ordinal_ref));

    nwos_read_class_definition(month_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_months = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_months; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, month_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &month_obj, sizeof(month_obj)));

	    if (is_same_object(&month_obj.number, &ordinal_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_months;
}


void nwos_find_public_month(uint8 month, ObjRef* ref)
{
    ObjRef month_class_ref;

    assert(1 <= month && month <= 12);

    assert(nwos_find_public_class_definition("MONTH", &month_class_ref));

    assert(find_month(&month_class_ref, month, ref));
}


/* See if this private month exists */

bool nwos_find_private_month(uint8 month, ObjRef* ref)
{
    ObjRef month_class_ref;
    bool result = false;

    assert(1 <= month && month <= 12);

    if (nwos_find_private_class_definition("MONTH", &month_class_ref))
    {
	result = find_month(&month_class_ref, month, ref);
    }

    return result;
}



/***************************************************************************************/
/* This routine finds a private month object or creates one by cloning the public one. */
/***************************************************************************************/

void nwos_find_or_create_private_month(uint8 month, ObjRef* ref)
{
    ObjRef public_month_ref;
    ObjRef private_month_class_ref;
    C_struct_Month month_obj;
    char msg[128];

    assert(1 <= month && month <= 12);

    if (!nwos_find_private_month(month, ref))
    {
	nwos_find_public_month(month, &public_month_ref);

	nwos_find_or_create_private_class_definition("MONTH", &private_month_class_ref);

	assert(nwos_read_object_from_disk(&public_month_ref, &month_obj, sizeof(month_obj)));

	nwos_generate_new_completely_random_id(ref);

	snprintf(msg, sizeof(msg), "clone_month(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_month_ref.id[0], public_month_ref.id[1], public_month_ref.id[2], public_month_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&month_obj.header.common.id, ref);

	nwos_get_time_stamp(month_obj.header.common.creation_time);

	copy_reference(&month_obj.header.common.class_definition, &private_month_class_ref);

	copy_reference(&month_obj.header.object.clone_of, &public_month_ref);

	nwos_create_reference_list(ref, &month_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &month_obj.header.object, sizeof(ObjectHeader), month_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &month_obj, sizeof(month_obj));

	nwos_add_to_references(ref, &private_month_class_ref);
    }
}


/* All of the month_and_day objects should always exist (all 366 of them) */

static bool find_month_and_day(ObjRef* month_and_day_class_ref, ObjRef* month_ref, uint8 day, ObjRef* ref)
{
    EveryObject month_header;
    C_struct_Month_And_Day month_and_day_obj;
    ObjRef class_ref;
    ObjRef cardinal_ref;
    ObjRef ordinal_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char ascii[3];

    ascii[0] = '0' + (day / 10);
    ascii[1] = '0' + (day % 10);
    ascii[2] = '\0';

    assert(nwos_find_cardinal_number(ascii, &cardinal_ref));
    assert(nwos_find_ordinal_number(&cardinal_ref, &ordinal_ref));

    nwos_read_object_headers_from_disk(month_ref, &month_header);

    ref_list = nwos_malloc_reference_list(&month_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_months: %d\n", num_months); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &class_ref);

	if (is_same_object(&class_ref, month_and_day_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &month_and_day_obj, sizeof(month_and_day_obj)));

	    if (is_same_object(&month_and_day_obj.day, &ordinal_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return i < num_refs;   /* we had to have found it */
}


void nwos_find_public_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef month_and_day_class_ref;
    ObjRef month_ref;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    assert(nwos_find_public_class_definition("MONTH AND DAY", &month_and_day_class_ref));

    nwos_find_public_month(month, &month_ref);

    assert(find_month_and_day(&month_and_day_class_ref, &month_ref, day, ref));
}


bool nwos_find_private_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef month_and_day_class_ref;
    ObjRef month_ref;
    bool result = false;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    if (nwos_find_private_class_definition("MONTH AND DAY", &month_and_day_class_ref))
    {
	if (nwos_find_private_month(month, &month_ref))
	{
	    result = find_month_and_day(&month_and_day_class_ref, &month_ref, day, ref);
	}
    }

    return result;
}


/***********************************************************************************************/
/* This routine finds a private month_and_day object or creates one by cloning the public one. */
/***********************************************************************************************/

void nwos_find_or_create_private_month_and_day(uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef public_month_and_day_ref;
    ObjRef private_month_and_day_class_ref;
    ObjRef private_month_ref;
    C_struct_Month_And_Day month_and_day_obj;
    char msg[128];

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);

    if (!nwos_find_private_month_and_day(month, day, ref))
    {
	nwos_find_public_month_and_day(month, day, &public_month_and_day_ref);

	nwos_find_or_create_private_class_definition("MONTH AND DAY", &private_month_and_day_class_ref);

	nwos_find_or_create_private_month(month, &private_month_ref);

	assert(nwos_read_object_from_disk(&public_month_and_day_ref, &month_and_day_obj, sizeof(month_and_day_obj)));

	nwos_generate_new_completely_random_id(ref);

	snprintf(msg, sizeof(msg), "clone_month_and_day(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_month_and_day_ref.id[0], public_month_and_day_ref.id[1], public_month_and_day_ref.id[2], public_month_and_day_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&month_and_day_obj.header.common.id, ref);

	nwos_get_time_stamp(month_and_day_obj.header.common.creation_time);

	copy_reference(&month_and_day_obj.header.common.class_definition, &private_month_and_day_class_ref);

	copy_reference(&month_and_day_obj.header.object.clone_of, &public_month_and_day_ref);

	copy_reference(&month_and_day_obj.month, &private_month_ref);

	nwos_create_reference_list(ref, &month_and_day_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &month_and_day_obj.header.object, sizeof(ObjectHeader), month_and_day_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &month_and_day_obj.month, sizeof(month_and_day_obj) - sizeof(EveryObject), month_and_day_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &month_and_day_obj, sizeof(month_and_day_obj));

	nwos_add_to_references(ref, &private_month_and_day_class_ref);
	nwos_add_to_references(ref, &private_month_ref);
    }
}


bool nwos_is_leap_year(int year)
{
    return ((year % 4 == 0) && (year % 100 != 0)) || (year % 400 == 0);
}
    


static bool find_date(ObjRef* date_class_ref, ObjRef* year_ref, ObjRef* month_and_day_ref, ObjRef* ref)
{
    C_struct_Date date_obj;
    EveryObject year_header;

    ObjRef object_class;

    ReferenceList* year_ref_list;
    int num_refs;
    int i;

    bool result = false;

    nwos_read_object_headers_from_disk(year_ref, &year_header);

    year_ref_list = nwos_malloc_reference_list(&year_header.object.references);

    num_refs = year_ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&year_ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, date_class_ref))   /* it is a date object */
	{
	    assert(nwos_read_object_from_disk(&year_ref_list->references[i], &date_obj, sizeof(date_obj)));

	    if (is_same_object(&date_obj.month_and_day, month_and_day_ref))
	    {
		memcpy(ref, &year_ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}
    }

    nwos_free_reference_list(year_ref_list);
    year_ref_list = NULL;

    return result;
}


bool nwos_find_public_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }


    assert(nwos_find_public_class_definition("DATE", &date_class_ref));

    assert(nwos_find_public_year(year, &year_ref));
    nwos_find_public_month_and_day(month, day, &month_and_day_ref);

    return find_date(&date_class_ref, &year_ref, &month_and_day_ref, ref);
}


bool nwos_find_private_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;
    bool result = false;

    assert(1 <= month && month <= 12);
    assert(1 <= day && day <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }


    if (nwos_find_private_class_definition("DATE", &date_class_ref))
    {
	if (nwos_find_private_year(year, &year_ref))
	{
	    if (nwos_find_private_month_and_day(month, day, &month_and_day_ref))
	    {
		result = find_date(&date_class_ref, &year_ref, &month_and_day_ref, ref);
	    }
	}
    }

    return result;
}


ObjCreateResult nwos_create_date(uint16 year, uint8 month, uint8 day, ObjRef* ref)
{
    C_struct_Date date_obj;

    ObjRef date_class_ref;
    ObjRef year_ref;
    ObjRef month_and_day_ref;

    ObjCreateResult result = FOUND_EXISTING;

    assert(1 <= month && month <= 12);
    assert(1 <= day && month <= days_per_month[month-1]);
    assert(1582 <= year && year <= 2099);
    if (month == 2 && day == 29)  /* more checking required */
    {
	assert(nwos_is_leap_year(year));
    }

#ifdef PUBLIC_MODE
    if (!nwos_find_public_date(year, month, day, ref))   /* didn't find it */
    {
	assert(nwos_find_public_class_definition("DATE", &date_class_ref));

	assert(nwos_find_public_year(year, &year_ref));
	nwos_find_public_month_and_day(month, day, &month_and_day_ref);
#else
    if (!nwos_find_private_date(year, month, day, ref))   /* didn't find it */
    {
	nwos_find_or_create_private_class_definition("DATE", &date_class_ref);
	nwos_find_or_create_private_year(year, &year_ref);
	nwos_find_or_create_private_month_and_day(month, day, &month_and_day_ref);
#endif
	memset(&date_obj, 0, sizeof(date_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&date_obj.header.common, ref, &date_class_ref);

	copy_reference(&date_obj.year, &year_ref);
	copy_reference(&date_obj.month_and_day, &month_and_day_ref);

	nwos_create_reference_list(ref, &date_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &date_obj.header.object, sizeof(ObjectHeader), date_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &date_obj.year, sizeof(date_obj) - sizeof(EveryObject), date_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &date_obj, sizeof(date_obj));

        nwos_add_to_references(ref, &date_class_ref);
	nwos_add_to_references(ref, &year_ref);
	nwos_add_to_references(ref, &month_and_day_ref);
    }

    return result;
}


bool nwos_year_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Year year_obj;

    assert(size >= 5);

    assert(nwos_read_object_from_disk(ref, &year_obj, sizeof(year_obj)));

    nwos_cardinal_number_to_string(&year_obj.number, string, size);

    return true;
}


bool nwos_month_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Month month_obj;

    assert(size >= 3);

    assert(nwos_read_object_from_disk(ref, &month_obj, sizeof(month_obj)));

    nwos_ordinal_number_to_string(&month_obj.number, string, size);

    return true;
}


bool nwos_date_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Date date_obj;
    C_struct_Month_And_Day month_and_day_obj;

    assert(nwos_read_object_from_disk(ref, &date_obj, sizeof(date_obj)));
    assert(nwos_read_object_from_disk(&date_obj.month_and_day, &month_and_day_obj, sizeof(month_and_day_obj)));

    assert(size >= 11);

    nwos_month_number_to_string(&month_and_day_obj.month, string, size);

    if (string[1] == '\0')
    {
	string[2] = '\0';
	string[1] = string[0];
	string[0] = '0';
    }
    assert(string[2] == '\0');

    string[2] = '/';

    nwos_ordinal_number_to_string(&month_and_day_obj.day, &string[3], size-3);

    if (string[4] == '\0')
    {
	string[5] = '\0';
	string[4] = string[3];
	string[3] = '0';
    }
    assert(string[5] == '\0');

    string[5] = '/';

    nwos_year_to_string(&date_obj.year, &string[6], size-6);

    return true;
}



