/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: security.c,v $
-- Revision 1.25  2009/03/14 19:08:10  jsedwards
-- Moved security related variables and functions from objectify.c file.
--
-- Revision 1.24  2009/03/13 13:14:57  jsedwards
-- Removed include of objectify.h file and added includes for strlcxx.h and
-- types.h files.
--
-- Revision 1.23  2009/03/07 20:17:19  jsedwards
-- Removed #include of objectify_private.h, not needed.
--
-- Revision 1.22  2008/12/27 15:00:29  jsedwards
-- Merged from alpha_29_5_branch back into main trunk.
--
-- Revision 1.21.2.1  2008/12/23 11:57:48  jsedwards
-- Fix Bug #2026708 - change to use strlcpy instead of strncpy.
--
-- Revision 1.21  2008/09/13 15:09:27  jsedwards
-- Fix Bug #2103915 - pass phrase now obscured in "show" mode.
--
-- Revision 1.20  2008/09/07 14:42:46  jsedwards
-- Fixed so pass phrase from environment variable works correctly.
--
-- Revision 1.19  2008/09/07 12:38:40  jsedwards
-- Made display_help function, changed Escape to not exit, and simplified the
-- backspace code.
--
-- Revision 1.18  2008/09/07 12:28:01  jsedwards
-- Added 'last' and 'none' display modes and 'quit' or 'exit' to exit the
-- program.
--
-- Revision 1.17  2008/09/07 11:22:24  jsedwards
-- Changed to have 'show' and 'hide' modes for pass phrase.
--
-- Revision 1.16  2008/09/07 03:18:55  jsedwards
-- First attempt at reading pass phrase in non-canonical mode.
--
-- Revision 1.15  2008/09/06 15:34:24  jsedwards
-- Changed to use TEST_ENVIRONMENT_VARIABLE.
--
-- Revision 1.14  2008/09/06 13:36:50  jsedwards
-- Change pass phrase from TEST_PASS_PHRASE define to environment variable.
--
-- Revision 1.13  2008/03/12 04:06:44  jsedwards
-- Changed to use the GNU MD5 context and functions instead of the RSA context
-- and functions.
--
-- Revision 1.12  2007/09/15 14:31:58  jsedwards
-- Added conditional code so that if TEST_PASS_PHRASE is defined, it will use
-- it instead of asking for pass phrase from user.
--
-- Revision 1.11  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.10  2007/01/16 14:12:02  jsedwards
-- Changed to use MD5 and SHA1 to generate key from short pass phrases.
--
-- Revision 1.9  2007/01/16 13:37:22  jsedwards
-- Eliminated get_short_key_from_password and made get_key_from_password call
-- convert_short_pass_phrase_to_reference if the password entered was too
-- short by itself.
--
-- Revision 1.8  2007/01/16 02:27:04  jsedwards
-- Added new routine for short pass phrases.
--
-- Revision 1.7  2006/12/01 14:37:23  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.6  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.5  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.4.2.4  2006/09/09 13:03:33  jsedwards
-- Moved "create_root_object" routine from security.c to objectify.c so that
-- the security module didn't have references to so many other parts of the
-- system.
--
-- Revision 1.4.2.3  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.4.2.2  2006/09/02 15:04:30  jsedwards
-- Add reference list class reference to root object because now it has to
-- read the reference list to get it's size and we can't easily read it
-- without the read routine verifying it's class.
--
-- Revision 1.4.2.1  2006/09/02 01:12:25  jsedwards
-- Add root object reference in call to fill_in_common_header so it can put
-- it into the header.
--
-- Revision 1.4  2005/12/30 03:16:12  jsedwards
-- Changed "Password" prompt to "Pass phrase".
--
-- Revision 1.3  2005/12/28 13:03:24  jsedwards
-- Added sequence generator (from Fine random_number_generator).  Changed the
-- get reference from password routine to get a variable length key from the
-- password.  It's actually more like a pass phrase now because it requires
-- the input to be long enough to get enough bits to fill up the key.
--
-- Revision 1.2  2005/12/27 19:45:27  jsedwards
-- Added routine to create a root object.
--
-- Revision 1.1  2005/12/27 18:02:42  jsedwards
-- Initial version converts a pass phrase into an object reference (ObjRef).
--
*/

#include <assert.h>
#include <openssl/blowfish.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>

#include "crc32.h"
#include "disk_io.h"           /* define FILE_BLOCK_SIZE and read and write block functions */
#include "security.h"
#include "strlcxx.h"           /* in case strlcpy and strlcat are not provided by the system */
#include "types.h"

#include "gnu/md5.h"
#include "gnu/sha1.h"

#define ESC 0x1b

#define MAX_NUMBER_SIZE 512    /* number can be up to 4096 bits */
#define SHORT_NUMBER_SIZE 36

#define MULTIPLIER 3141592621U
#define ADDEND     101U

#define NUM_STORED_SEQ 4096     /* number of sequence tables that are stored, 4096 = 1 MB worth and uses 1 MB worth of RAM */


#ifndef PUBLIC_MODE
static uint32 linear_after_first;
static uint32 serial_after_first;

static uint8  nwos_random_sequence[NUM_STORED_SEQ][FILE_BLOCK_SIZE];     /* this will only work for blocks <= 256 bytes */

static BF_KEY blowfish_key;
#endif


static uint32 nwos_next_sequence(uint32* linear, uint32* serial)
{
    uint32 feedback;

    *linear = (*linear * MULTIPLIER) + ADDEND;

    /* feedback_mask: BIT 32 is 0x80004000 */
    /* exclusive or of bits 31 and 14 */
    feedback = ((*serial >> 31) ^ (*serial >> 14)) & 1;

    *serial = (*serial << 1) | feedback;

    return *linear ^ *serial;
}


void convert_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
    int i;
    int j;
    uint8 number[MAX_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, MAX_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = MAX_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < MAX_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < MAX_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = key_size - 1;
    for (i = MAX_NUMBER_SIZE; i > 0; i--)
    {
	key[j] = key[j] ^ number[i-1];
	j--;
	if (j < 0) j = key_size - 1;
    }
}


void convert_short_pass_phrase_to_reference(char* pass_phrase, uint8 key[], size_t key_size)
{
#ifndef PUBLIC_MODE
    int i;
    int j;
    uint8 number[SHORT_NUMBER_SIZE];
    uint8 mult;
    uint32 acc;
    char *p;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[16];
    uint8 sha1_digest[20];

    mult = '~' - ' ' + 2;

#if 0
    printf("mult: %02x\n", mult);
#endif

    memset(&number, 0, SHORT_NUMBER_SIZE);

    for (p = pass_phrase; *p != '\0'; p++)
    {
	acc = *p;

	if (' ' <= acc && acc <= '~')
	{
	    acc = acc - ' ' + 1;
	}
	else
	{
	    acc = 0;
	}

	/* multiply and add big endian ([0] is most significant byte) */
	for (i = SHORT_NUMBER_SIZE; i > 0; i--)
	{
	    acc = (number[i-1] * mult) + acc;
	    number[i-1] = acc;
	    acc = acc >> 8;
	}
#if 0
	for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

	while (i < SHORT_NUMBER_SIZE)
	  {
	    printf("%02x", number[i++]);
	  }
	printf("\n");
#endif
    }

    for (i = 0; i < SHORT_NUMBER_SIZE; i++) if (number[i] != 0) break;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    md5_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &md5_context);    /* include this data in the md5 checksum */
    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */

    sha1_init_ctx(&sha1_context);
    sha1_process_bytes(&number[i], SHORT_NUMBER_SIZE-i, &sha1_context);    /* include this data in the sha1 checksum */
    sha1_finish_ctx(&sha1_context, sha1_digest);

    /* exclusive or each reference size chunk into the result */

    memset(key, 0, key_size);

    j = 0;
    for (i = 0; i < 20; i++)
    {
	key[j] = key[j] ^ sha1_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
    for (i = 0; i < 16; i++)
    {
	key[j] = key[j] ^ md5_digest[i];
	j++;
	if (j == key_size) j = 0;
    }
#endif
}


static void display_help()
{
    fputc('\n', stderr);
    fprintf(stderr, "Valid characters are A-Z, a-z, 0-9, @#$%%& *~^ +-= ;:,.!? /|\\ `'\" _ ()[]{}<>.\n");
    fputc('\n', stderr);
    fprintf(stderr, "Enter 'show' to display the characters typed, 'hide' to display asterisks,\n");
    fprintf(stderr, "'last' to display the last character typed, 'none' for no display at all,\n");
    fprintf(stderr, "and 'quit', 'exit', or Control-C to exit back to the shell without running.\n");
    fputc('\n', stderr);
}


void nwos_get_key_from_password(uint8 key[], size_t key_size)
{
    bool ok = false;
    char buffer[MAX_NUMBER_SIZE];
    char *test_phrase;
    int chars_needed;
    int num_chars = 0;
    int chars_read;
    int i;
    struct termios oldtio, newtio;
    static enum { None, Hide, Last, Show } display_mode = Hide;


    chars_needed = (key_size * 80) / 65; /* we get 6.5 bits for each character entered, we need 8 for each output byte */

    while (chars_needed * 65 < key_size * 80) chars_needed++;   /* adjust up for truncation errors */

    test_phrase = getenv(TEST_ENVIRONMENT_VARIABLE);

    if (test_phrase != NULL)
    {
	num_chars = strlcpy(buffer, test_phrase, sizeof(buffer));   /* copy environment variable value to buffer */

	assert(num_chars < sizeof(buffer));  
    }
    else
    {
	tcgetattr(STDIN_FILENO, &oldtio); /* save current port settings */

	memcpy(&newtio, &oldtio, sizeof(newtio));

	newtio.c_lflag &= ~(ICANON | ECHO | ISIG);   /* by default turn off canonical, echo, and signals */

	while (!ok)
	{
	    fprintf(stderr, "Pass phrase: ");
	    fflush(stderr);

	    tcflush(STDIN_FILENO, TCIFLUSH);
	    tcsetattr(STDIN_FILENO, TCSANOW, &newtio);

	    num_chars = 0;
	    chars_read = 0;

	    while (chars_read == 0 && num_chars < sizeof(buffer))
	    {
		chars_read = read(STDIN_FILENO, buffer + num_chars, sizeof(buffer) - num_chars);

		if (chars_read == 0) break;

		while (chars_read > 0)
		{
		    if (buffer[num_chars] == newtio.c_cc[VERASE])  /* backspace */
		    {
			if (num_chars > 0)
			{
			    num_chars--;

			    if (display_mode != None)
			    {
				fputc('\b', stderr);
				fputc(' ', stderr);
				fputc('\b', stderr);
			    }

			    if (display_mode == Last && num_chars > 0)
			    {
				fputc('\b', stderr);
				fputc(buffer[num_chars - 1], stderr);
			    }
			}
		    }
		    else if (buffer[num_chars] == newtio.c_cc[VKILL])  /* erase line */
		    {
			num_chars = 0;
		    }
		    else if (' ' <= buffer[num_chars] && buffer[num_chars] <= '~')
		    {
			switch (display_mode)
			{
			  case None:
			    /* don't display anything */
			    break;

			  case Hide:
			    fputc('*', stderr);
			    break;

			  case Last:
			    if (num_chars > 0)
			    {
				fputc('\b', stderr);
				fputc('*', stderr);
			    }
			    fputc(buffer[num_chars], stderr);
			    break;

			  case Show:
			    fputc(buffer[num_chars], stderr);
			    break;
			}
			num_chars++;
		    }
		    else   /* anything else */
		    {
			break;   /* we are outta here */
		    }

		    chars_read--;
		}

		if (buffer[num_chars] == '\n')
		{
		    if (display_mode == Show)
		    {
			for (i = 0; i < num_chars; i++) fputc('\b', stderr);
			for (i = 0; i < num_chars; i++) fputc('*', stderr);
		    }
		    else if (display_mode == Last)
		    {
			fputc('\b', stderr);
			fputc('*', stderr);
		    }
		    fputc('\n', stderr);
		    chars_read--;
		    break;
		}
	    }

	    tcsetattr(STDIN_FILENO, TCSANOW, &oldtio);    /* restore to original settings */

	    if (chars_read > 0)
	    {
		if (buffer[num_chars] == newtio.c_cc[VINTR] || buffer[num_chars] == newtio.c_cc[VQUIT])
		{
		    fprintf(stderr, "\n");
		    exit(1);
		}
		else
		{
		    fprintf(stderr, "\nUnknown input while reading pass phrase: ");
		    for (i = 0; i < chars_read; i++)
		    {
			fprintf(stderr, "%02x ", buffer[num_chars]);
			num_chars++;
		    }
		    fprintf(stderr, "\n");

		    display_help();
		}
	    }
	    else if (num_chars == sizeof(buffer))    /* pass phrase is too long */
	    {
		fprintf(stderr, "password was too long, must be less than %d characters\n", MAX_NUMBER_SIZE);
	    }
	    else if (num_chars < 10)            /* pass phrase is too short */
	    {
		buffer[num_chars] = '\0';

		if (strcasecmp(buffer, "show") == 0)
		{
		    display_mode = Show;
		}
		else if (strcasecmp(buffer, "hide") == 0)
		{
		    display_mode = Hide;
		}
		else if (strcasecmp(buffer, "last") == 0)
		{
		    display_mode = Last;
		}
		else if (strcasecmp(buffer, "none") == 0)
		{
		    display_mode = None;
		}
		else if (strcasecmp(buffer, "quit") == 0 || strcasecmp(buffer, "exit") == 0)
		{
		    exit(1);
		}
		else if (strcasecmp(buffer, "help") == 0)
		{
		    display_help();
		}
		else
		{
		    fprintf(stderr, "password was too short, must be at least 10 characters for\n"
			    "short password or %d characters for higher security password\n", chars_needed);
		}
	    }
	    else
	    {
		buffer[num_chars] = '\0';   /* eliminate the newline character */

		ok = true;   /* we should be good to go */
	    }
	}
    }

    assert(num_chars >= 10);
    assert(strlen(buffer) == num_chars);

    if (num_chars < chars_needed)
    {
	convert_short_pass_phrase_to_reference(buffer, key, key_size);
    }
    else
    {
	convert_pass_phrase_to_reference(buffer, key, key_size);
    }
}



/**********************/
/* Sequence generator */
/**********************/

#ifndef PUBLIC_MODE
static void generate_sequence_table(uint32* linear, uint32* serial, uint8 table[FILE_BLOCK_SIZE])
{
    int i;
    uint8 j;
    uint8 filled[FILE_BLOCK_SIZE];    /* remember which slots are already filled */

    /* mark all slots as empty */
    for (i = 0; i < FILE_BLOCK_SIZE; i++) filled[i] = 0;

    /* then mark the first 8 as used */
    for (i = 0; i < 8; i++) filled[i] = 1;

    /* start at 0 because it is used already, so while loop will fail the first time. */
    j = 0;
    assert(filled[j] != 0);

    for (i = 8; i < FILE_BLOCK_SIZE; i++)
    {
	/* find where the next byte is stored */
	while(filled[j] != 0) j = nwos_next_sequence(linear, serial) % FILE_BLOCK_SIZE;
	table[i] = j;   /* store it in the table */
	filled[j] = 1;            /* mark it as used */
    }
}
#endif


#ifndef PUBLIC_MODE
void nwos_setup_from_big_key(uint8 big_key[16 + 8 + 4], ObjRef* root_object_ref)
{
    int i;
    int j;
    int k;
    int tries;

    BF_set_key(&blowfish_key, 16, big_key);

    linear_after_first = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    serial_after_first = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    /* Create all of the sequence tables from the linear and serial parameters */
    for (i = 0; i < NUM_STORED_SEQ; i++)
    {
	tries = 0;

	do
	{
	    tries++;

	    assert(tries < 4);

	    generate_sequence_table(&linear_after_first, &serial_after_first, nwos_random_sequence[i]);

	    for (j = 0; j < i; j++)
	    {
		for (k = 8; k < FILE_BLOCK_SIZE; k++) if (nwos_random_sequence[i][k] != nwos_random_sequence[j][k]) break;

		if (k == FILE_BLOCK_SIZE) break;
	    }
	}
	while (j < i);   /* if j is less than i it means one of the tables matched and needs to be regenerated */
    }

    assert(sizeof(ObjRef) == 4);   /* the following code assumes 4 byte IDs */

    memcpy(root_object_ref->id, big_key+20, 4);
}
#endif


bool nwos_read_block_from_disk_and_old_decrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], int seq)
{
    uint8 encrypted[FILE_BLOCK_SIZE];
#ifndef PUBLIC_MODE
    uint8 scrambled[FILE_BLOCK_SIZE];
    int i;
    int j;
#endif
    bool result = false;

    assert(!is_void_reference(ref));

    if (nwos_read_block(ref, encrypted))
    {
#ifdef PUBLIC_MODE
	memcpy(object, encrypted, size);
#else

#ifdef DISABLE_SECURITY_FEATURES
	if (true)
#else
	if (nwos_reference_type(ref) == Public_Reference)
#endif
	{
	    memcpy(object, encrypted, size);    /* just copy the buffer into the object */
	}
	else
	{
	    /* copy the first 8 bytes directly, not encrypted */
	    memcpy(object, encrypted, 8);

	    /* decrypt the remaining bytes */
	    BF_cbc_encrypt((encrypted + 8), (scrambled + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_DECRYPT);

	    assert(size <= FILE_BLOCK_SIZE);

	    for (i = 8; i < size; i++)
	    {
		j = nwos_random_sequence[seq % NUM_STORED_SEQ][i];    /* find where the next byte is stored */
		((uint8*)object)[i] = scrambled[j];    /* get this byte from there */
	    }
	}

#endif
	result = true;
    }

    return result;
}


bool nwos_read_block_from_disk_and_decrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], int seq)
{
    uint8 scrambled[FILE_BLOCK_SIZE];
#ifndef PUBLIC_MODE
    uint8 encrypted[FILE_BLOCK_SIZE];
    uint8 decrypted[FILE_BLOCK_SIZE];
    int i;
    int j;
#endif
    bool result = false;

    assert(!is_void_reference(ref));

    if (nwos_read_block(ref, scrambled))
    {
#ifdef PUBLIC_MODE
	memcpy(object, scrambled, size);
#else

#ifdef DISABLE_SECURITY_FEATURES
	if (true)
#else
	if (nwos_reference_type(ref) == Public_Reference)
#endif
	{
	    memcpy(object, scrambled, size);    /* just copy the buffer into the object */
	}
	else
	{
	    /* copy the first 8 bytes directly, not encrypted */
	    memcpy(object, scrambled, 8);

	    /* descramble */
	    for (i = 8; i < FILE_BLOCK_SIZE; i++)
	    {
		j = nwos_random_sequence[seq % NUM_STORED_SEQ][i];    /* find where the next byte is stored */
		encrypted[i] = scrambled[j];    /* get this byte from there */
	    }

	    assert(size <= FILE_BLOCK_SIZE);

	    /* decrypt */
	    BF_cbc_encrypt((encrypted + 8), (decrypted + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_DECRYPT);

	    memcpy((object + 8), (decrypted + 8), (size - 8));
	}

#endif
	result = true;
    }

    return result;
}


void nwos_write_block_to_disk_and_encrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], int seq)
{
    uint8 scrambled[FILE_BLOCK_SIZE];
#ifndef PUBLIC_MODE
    uint8 decrypted[FILE_BLOCK_SIZE];
    uint8 encrypted[FILE_BLOCK_SIZE];
    int i;
    int j;
#endif

    assert(!is_void_reference(ref));
    assert(size <= FILE_BLOCK_SIZE);

#ifdef PUBLIC_MODE
    memset(scrambled, 0, FILE_BLOCK_SIZE);
    memcpy(scrambled, object, size);
#else

#ifdef DISABLE_SECURITY_FEATURES    /* always do the unencryped code */
    if (true)
#else
    if (nwos_reference_type(ref) == Public_Reference)
#endif
    {
	memset(scrambled, 0, FILE_BLOCK_SIZE);    /* just copy the object into the buffer */
	memcpy(scrambled, object, size);
    }
    else
    {
	/* copy the first 8 bytes straight across */
	memcpy(scrambled, object, 8);

	memcpy((decrypted + 8), (object + 8), size - 8);

	/* fill any empty slots with random junk */
	for (i = size; i < FILE_BLOCK_SIZE; i++) decrypted[i] = random();

	/* and encrypt the remainder */
	BF_cbc_encrypt((decrypted + 8), (encrypted + 8), (FILE_BLOCK_SIZE - 8), &blowfish_key, ivec, BF_ENCRYPT);

	for (i = 8; i < FILE_BLOCK_SIZE; i++)
	{
	    /* find where the next byte is stored */
	    j = nwos_random_sequence[seq % NUM_STORED_SEQ][i];    /* find where the next byte is stored */
	    scrambled[j] = encrypted[i];    /* save this byte there */
	}
    }
#endif

    nwos_write_block(ref, scrambled);
}


#if 0
int main(int argc, char* argv[])
{
  ObjRef key;
  int i;

  key = convert_pass_phrase_to_reference(argv[1]);

  for (i = 0; i < sizeof(ObjRef); i++) printf("%02x", key.id[i]);
  printf("\n");

  return 0;
}
#endif

#if 0
int main(int argc, char* argv[])
{
  uint8 big_key[28];
  int i;

  nwos_get_key_from_password(big_key, sizeof(big_key));

  for (i = 0; i < sizeof(big_key); i++)
  {
      printf("%02x", big_key[i]);
  }

  printf("\n");

  return 0;
}
#endif


