/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: check_block_maps.c,v $
-- Revision 1.24  2009/03/14 23:14:04  jsedwards
-- Added include of the new chunk_info.h file.
--
-- Revision 1.23  2009/03/14 11:46:07  jsedwards
-- Added include of user_config.h file.
--
-- Revision 1.22  2009/03/13 12:10:26  jsedwards
-- Added include config.h file.
--
-- Revision 1.21  2009/03/08 00:43:49  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.20  2008/08/10 15:23:24  jsedwards
-- Added path to "Missing magic number" and "Incorrect version in header" error
-- messages.
--
-- Revision 1.19  2008/04/02 03:06:21  jsedwards
-- Added file locking.
--
-- Revision 1.18  2008/02/03 01:15:40  jsedwards
-- Change to use nwos_get_private_objects_path function instead of DEFAULT_FILE.
--
-- Revision 1.17  2007/08/10 00:03:35  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Also removed using O_LARGEFILE from open call.
--
-- Revision 1.16  2007/07/03 01:24:44  jsedwards
-- Added code to check new "used" block counts in layout 0024.
--
-- Revision 1.15  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.14  2007/06/28 13:51:51  jsedwards
-- Modified for 0023 to only check block maps that are used, instead of all
-- block maps on the disk.
--
-- Revision 1.13  2007/06/27 15:44:45  jsedwards
-- Changed two more hard coded constants to defines (that I missed the first
-- time).
--
-- Revision 1.12  2007/06/27 15:30:15  jsedwards
-- Changed hard coded constants to use defines (BIT_MAP_BYTES, FILE_BLOCK_SIZE)
-- and deleted old commented out line of code.
--
-- Revision 1.11  2007/06/21 01:33:02  jsedwards
-- Change to deal with odd size disk partitions after old 0021 compressed file
-- was imported into a 0022 disk partition and you have an odd sized chunk.
--
-- Revision 1.10  2007/06/19 18:55:58  jsedwards
-- Removed all stuff related to public objects which are now stored separately
-- from the private objects.
--
-- Revision 1.9  2007/04/15 16:54:34  jsedwards
-- Fixed percentage print out.
--
-- Revision 1.8  2007/03/27 11:32:09  jsedwards
-- Change time_last_change in disk_header to private_last_change.
--
-- Revision 1.7  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.6  2006/12/16 13:10:24  jsedwards
-- Make sure last_total is always initialized.
--
-- Revision 1.5  2006/12/16 11:35:59  jsedwards
-- Change for new disk layout and check public blocks too.
--
-- Revision 1.4  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:22  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.9  2006/10/22 16:02:58  jsedwards
-- Add print statement to print used in final blocks.
--
-- Revision 1.1.2.8  2006/10/22 14:19:23  jsedwards
-- Changed so that blocks_on_disk and reserved_public_blocks are read off
-- disk instead of defined constants.
--
-- Revision 1.1.2.7  2006/10/18 13:18:51  jsedwards
-- Changed printf formats for off64_t to 'z' instead of 'll'.
--
-- Revision 1.1.2.6  2006/10/15 12:30:02  jsedwards
-- Change to print blocks used in each print instead of a running total.
--
-- Revision 1.1.2.5  2006/10/15 12:13:14  jsedwards
-- Changed to correctly account for the first chunk on the disk as all used.
--
-- Revision 1.1.2.4  2006/10/12 12:09:57  jsedwards
-- Added kludge to add in stats for the section of the disk (reserved public)
-- that it currently skips over.  Also included the block maps themselves.
--
-- Revision 1.1.2.3  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.2  2006/10/09 13:13:14  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.1  2006/10/08 13:44:09  jsedwards
-- Program to verify the 256 byte block maps are correct on the disk.
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "chunk_info.h"
#include "config.h"
#include "disk_io.h"
#include "user_config.h"


static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s\n", program);
}


bool block_all_zeros(uint8* block)
{
    int i;

    for (i = 0; i < FILE_BLOCK_SIZE; i++)
    {
	if (block[i] != 0) break;
    }

    return i == FILE_BLOCK_SIZE;
}


bool reference_matches(uint32 ref, uint8 *block)
{
    return (uint8)(ref >> 24) == block[4] && (uint8)(ref >> 16) == block[5] && (uint8)(ref >> 8) == block[6] && (uint8)ref == block[7];
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    struct flock lock;
    const char * obj_file_path;
    char ref_str[128];
    uint8 *chunk;
    int i;
    int j;
    size_t bytes_read;
    int bit_num;
    int byte_num;
    uint32 private_blocks_used = 0;
    uint32 private_blocks_free = 0;
    uint32 private_blocks_total;
    uint32 chunk_num;
    uint32 last_total = 0;
    uint32 total_private_blocks;
    uint32 used_private_blocks;
    uint32 odd_blocks;
    uint32 used_blocks;
    uint32 used_chunks;
    uint32 ref;
    Chunk_Info* chunk_info;
    Disk_Header disk_header;

    if (argc != 1)
    {
	print_usage(argv[0]);
	exit(1);
    }

    obj_file_path = nwos_get_private_objects_path();

    obj_file_desc = open(obj_file_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(obj_file_path);
	exit(1);
    }

    lock.l_type = F_RDLCK;
    lock.l_whence = SEEK_SET;
    lock.l_start = 0;
    lock.l_len = 0;

    if (fcntl(obj_file_desc, F_SETLK, &lock) != 0)
    {
	perror(obj_file_path);
	exit(1);
    }


    /* allocate buffer for reading chunks */

    chunk = malloc(CHUNK_SIZE);

    assert(chunk != NULL);

    if (read(obj_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	snprintf(ref_str, sizeof(ref_str), "reading first chunk from: %s", obj_file_path);
	perror(ref_str);
	close(obj_file_desc);
	exit(1);
    }

    memcpy(&disk_header, chunk, sizeof(disk_header));

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header: %s\n", obj_file_path);
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header: %s\n", obj_file_path);
	exit(1);
    }

    printf("Last change: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c\n",
	   nwos_extract_year_from_time_stamp(disk_header.last_change),
	   nwos_extract_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_day_of_month_from_time_stamp(disk_header.last_change),
	   nwos_extract_hour_from_time_stamp(disk_header.last_change),
	   nwos_extract_minute_from_time_stamp(disk_header.last_change),
	   nwos_extract_second_from_time_stamp(disk_header.last_change),
	   disk_header.version_string[0], disk_header.version_string[1],
	   disk_header.version_string[2], disk_header.version_string[3]);

    /* get what the disk thinks */
    nwos_4_uint8_to_uint32(disk_header.total_blocks, &total_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_blocks, &used_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_chunks, &used_chunks);

    assert(total_private_blocks % BLOCKS_IN_CHUNK == 0);

    printf("total_blocks: %10u  total_chunks: %5u\n", total_private_blocks, total_private_blocks / BLOCKS_IN_CHUNK);
    printf("used_blocks:  %10u  used_chunks:  %5u\n", used_private_blocks, used_chunks);

    assert(used_chunks > 0);

    chunk_info = malloc(used_chunks * sizeof(Chunk_Info));
    assert(chunk_info != NULL);

    for (i = 0; i < used_chunks; i++)
    {
	nwos_4_uint8_to_uint32(&chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info)], &chunk_info[i].ref);
	nwos_2_uint8_to_uint16(&chunk[FILE_BLOCK_SIZE + i * sizeof(Chunk_Info) + 4], &chunk_info[i].used);
    }

    /*****************************/
    /* now do the private blocks */
    /*****************************/

    for (chunk_num = 0; chunk_num < used_chunks; chunk_num++)
    {
	bytes_read = read(obj_file_desc, chunk, CHUNK_SIZE);
	used_blocks = 0;

	for (i = BIT_MAP_BYTES; i < bytes_read; i += FILE_BLOCK_SIZE)
	{
	    for (j = 0; j < FILE_BLOCK_SIZE; j++)
	    {
		if (chunk[i+j] != 0) break;
	    }

	    byte_num = (i / FILE_BLOCK_SIZE) / 8;
	    bit_num = (i / FILE_BLOCK_SIZE) % 8;

	    if (j < FILE_BLOCK_SIZE)   /* block not empty */
	    {
		private_blocks_used++;
		used_blocks++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) == 0)
		{
		    printf("bit clear that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);
		    for (j = 0; j < BIT_MAP_BYTES; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}

		nwos_4_uint8_to_uint32(&chunk[i+4], &ref);

		if (ref < chunk_info[chunk_num].ref || ref > chunk_info[chunk_num].ref + USABLE_BLOCKS_PER_CHUNK)
		{
		    printf("block_reference: %08x  outside of index: %08x to %08x\n",
			   ref, chunk_info[chunk_num].ref, chunk_info[chunk_num].ref + USABLE_BLOCKS_PER_CHUNK);
		}
	    }
	    else
	    {
		private_blocks_free++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) != 0)
		{
		    printf("bit set that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", chunk_num, i, byte_num, bit_num);
		    for (j = 0; j < BIT_MAP_BYTES; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < FILE_BLOCK_SIZE; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	}

	if (bytes_read != CHUNK_SIZE)
	{
	    printf("last chunk read: %08zx\n", chunk_num + (bytes_read / FILE_BLOCK_SIZE));
	    break;
	}

	if (used_blocks != chunk_info[chunk_num].used)
	{
	    printf("chunk: %d  used blocks (%u) doesn't match map count (%u)\n", 
		   chunk_num, chunk_info[chunk_num].used, used_blocks);
	    break;
	}


	if (chunk_num % 64 == 0)
	{
	    printf("%u - %u  used: %d\n", chunk_num / 64, chunk_num * BLOCKS_IN_CHUNK, private_blocks_used - last_total);
	    fflush(stdout);
	    last_total = private_blocks_used;
	}
    }

    printf("%u - %u  used: %d\n", chunk_num / 64, chunk_num * BLOCKS_IN_CHUNK, private_blocks_used - last_total);
    fflush(stdout);

    close(obj_file_desc);

    if (total_private_blocks % BLOCKS_IN_CHUNK != 0)    /* odd sized block */
    {
	odd_blocks = BLOCKS_IN_CHUNK - total_private_blocks % BLOCKS_IN_CHUNK;
	printf("Warning: storage contains an odd sized chunk at the end,\n");
	printf("         subracting %u from free block count.\n", odd_blocks);

	private_blocks_free = private_blocks_free - odd_blocks;
    }

    private_blocks_total = private_blocks_used + private_blocks_free;

    printf("private blocks used: %u  stored: %u - %s\n", private_blocks_used, used_private_blocks,
	   private_blocks_used == used_private_blocks ? "OK" : "MISMATCH");

    printf("private blocks free: %u  total: %u  stored: %u - %s\n",
	   private_blocks_free, private_blocks_total, total_private_blocks,
	   private_blocks_total == total_private_blocks ? "OK" : "MISMATCH");

    printf("usage: %.1f%%\n", ((float)private_blocks_used / (float)total_private_blocks) * 100.0f );

    return 0;
}

