/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: log.c,v $
-- Revision 1.16  2008/04/26 11:36:13  jsedwards
-- Added 'const' to char* file_name parameter in the four log_*sum functions.
--
-- Revision 1.15  2008/03/14 13:11:55  jsedwards
-- Added nwos_log_sha512sum function.
--
-- Revision 1.14  2008/03/12 14:43:59  jsedwards
-- Added code to create subdirectories in the path if they don't already
-- exist.
--
-- Revision 1.13  2008/03/12 11:40:48  jsedwards
-- Added code to disable logging if nwos_log_arguments is called with argc=0
-- and argv=NULL.
--
-- Revision 1.12  2008/02/05 04:23:44  jsedwards
-- Added include of objectify_private.h.  Removed forward references for
-- strlcat and strlcpy because they are declared in objectify.h.  Changed to
-- get the log file path from the new nwos_get_log_file_path function.
-- Added 'const' specifier to strlcat and strlcpy.
--
-- Revision 1.11  2008/01/13 20:09:06  jsedwards
-- Added nwos_log_md5sum, nwos_log_sha1sum, and nwos_log_sha256sum functions.
--
-- Revision 1.10  2007/10/27 14:55:21  jsedwards
-- Changed to use #ifndef STRLCAT and #ifndef STRLCPY instead of #ifdef linux.
--
-- Revision 1.9  2007/10/03 13:37:13  jsedwards
-- Bug #1806945 - Fixed so least significant digit of month isn't goofed up
-- in October, November, and December.
--
-- Revision 1.8  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.7  2007/03/07 13:10:45  jsedwards
-- Fixed bug so appending date is null terminated.
-- Changed to use file name defined in config.h so debug versions can have a
-- different file name.
--
-- Revision 1.6  2007/02/23 13:27:02  jsedwards
-- Move the strlcpy and strlcat functions from objectify.c to log.c so that
-- log.o can be used without having to drag in the whole objecitfy library.
--
-- Revision 1.5  2007/02/11 18:04:22  jsedwards
-- Added code to dynamically create the log file path using the users home
-- directory and the date.
--
-- Revision 1.4  2007/02/11 14:43:45  jsedwards
-- Cast the time value for ctime so the compiler won't complain.
--
-- Revision 1.3  2007/01/07 20:30:56  jsedwards
-- Increased size of msg buffer for logging arguments.
--
-- Revision 1.2  2006/12/28 23:24:39  jsedwards
-- Added include of sys/time.h to declare gettimeofday.
--
-- Revision 1.1  2006/12/27 14:15:24  jsedwards
-- Move log functions out of objectify.c and into a new file log.c.
--
*/


#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>

#include "objectify.h" /* include this for config.h the log file path and because these functions prototypes are there */
#include "objectify_private.h"  /* include this to get the nwos_get_log_file_path function */


static bool log_disabled = false;


/******************************************/
/* Log file - THIS NEEDS TO GO AWAY SOON. */
/******************************************/

void append_date_from_time_string(char* dst, char* time_str, size_t length)
{
    int i;
    int month;
    char* months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

    for (i = 0; i < length; i++) if (dst[i] == '\0') break;

    assert((i + 9) < length);

    dst[i++] = time_str[20];
    dst[i++] = time_str[21];
    dst[i++] = time_str[22];
    dst[i++] = time_str[23];

    for (month = 0; month < 12; month++) if (memcmp(months[month], &time_str[4], 3) == 0) break;

    assert(month < 12);

    month++;

    if (month < 10)
    {
	dst[i++] = '0';
	dst[i++] = '0' + month;
    }
    else
    {
	dst[i++] = '1';
	dst[i++] = '0' + month - 10;
    }

    if (time_str[8] == ' ')
    {
	dst[i++] = '0';
    }
    else
    {
	dst[i++] = time_str[8];
    }
    dst[i++] = time_str[9];

    dst[i++] = '\0';
}


void nwos_log(char* str)
{
    struct timeval tv;
    FILE* fp;
    char time_str[32];
    static char* log_file_path;
    size_t path_len;
    const char* path;
    char* p;

    if (log_disabled) return;    /* if logging has been disabled, just return */

    gettimeofday(&tv, NULL);
    strncpy(time_str, ctime((time_t*)&tv.tv_sec), 32);
    *strchr(time_str, '\n') = '\0';

    if (log_file_path == NULL)
    {
	path = nwos_get_log_file_path();

	/* NOTE: this is a bad thing to do, it temporarily modifies */
	/* the string returned by nwos_get_log_file_path.           */
        /* We cheat the system by calling strchr.                   */

	p = strchr(path, '/');
	assert(p != NULL);
	p = strchr(p + 1, '/');
	while (p != NULL)
	{
	    *p = '\0';
	    if (mkdir(path, 0755) != 0 && errno != EEXIST)
	    {
		perror(p);
		exit(1);
	    }
	    *p = '/';

	    p = strchr(p + 1, '/');
	}

	path_len = strlen(path) + 1 + 8 + 4 + 1;

	log_file_path = malloc(path_len);
	assert(log_file_path != NULL);

	strlcpy(log_file_path, path, path_len);
	strlcat(log_file_path, "-", path_len);
	append_date_from_time_string(log_file_path, time_str, path_len);
	strlcat(log_file_path, ".txt", path_len);

	/* printf("LOG FILE: %s\n", log_file_path); */
    }

    fp = fopen(log_file_path, "a");
    if (fp == NULL)
    {
	perror(log_file_path);
	exit(1);
    }
    fprintf(fp, "%s: %s\n", time_str, str);
    fclose(fp);
}


void nwos_log_arguments(int argc, char* argv[])
{
    int i;
    int arg;
    char* src;
    char msg[16384] = "Program:";

    if (argc == 0 || argv == NULL)
    {
	assert(argc == 0 && argv == NULL);    /* if one is zero, they should both be */
	log_disabled = true;
	return;
    }

    assert(argc > 0);
    assert(argv != NULL);

    i = strlen(msg);

    assert(msg[i] == '\0');

    for (arg = 0; arg < argc; arg++)
    {
	assert(i < sizeof(msg));

	msg[i] = ' ';
	i++;

	src = argv[arg];

	while (*src != '\0')
	{
	    assert(i < sizeof(msg));

	    msg[i] = *src;
	    src++;
	    i++;
	}
    }

    assert(i < sizeof(msg));

    msg[i] = '\0';

    nwos_log(msg);
}


static char hexdigit[16] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

void nwos_log_md5sum(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE])
{
    char msg[12 + FILENAME_MAX + MD5_DIGEST_SIZE * 2] = "MD5 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + MD5_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < MD5_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[md5_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[md5_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha1sum(const char* file_name, uint8 sha1_digest[SHA1_DIGEST_SIZE])
{
    char msg[12 + FILENAME_MAX + SHA1_DIGEST_SIZE * 2] = "SHA1 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA1_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA1_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha1_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha1_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha256sum(const char* file_name, uint8 sha256_digest[SHA256_DIGEST_SIZE])
{
    char msg[16 + FILENAME_MAX + SHA256_DIGEST_SIZE * 2] = "SHA256 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA256_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA256_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha256_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha256_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}


void nwos_log_sha512sum(const char* file_name, uint8 sha512_digest[SHA512_DIGEST_SIZE])
{
    char msg[16 + FILENAME_MAX + SHA512_DIGEST_SIZE * 2] = "SHA512 (";
    int prefix_len;
    int i;

    strlcat(msg, file_name, sizeof(msg));
    strlcat(msg, ") = ", sizeof(msg));

    prefix_len = strlen(msg);

    assert(prefix_len + SHA512_DIGEST_SIZE < sizeof(msg));

    for (i = 0; i < SHA512_DIGEST_SIZE; i++)
    {
	msg[prefix_len + i * 2] = hexdigit[sha512_digest[i] >> 4];
	msg[prefix_len + i * 2 + 1] = hexdigit[sha512_digest[i] & 0xf];
    }

    msg[prefix_len + i * 2] = '\0';

    nwos_log(msg);
}



/* since neither Linux or GNU seem to have these functions */

#ifndef HAVE_STRLCPY
size_t strlcpy(char* dst, const char* src, size_t len)
{
    int i;

    for (i = 0; src[i] != '\0'; i++)
    {
	if (i < len)
	{
	    dst[i] = src[i];
	}
    }

    if (i < len)
    {
	dst[i] = src[i];
    }
    else
    {
	dst[len-1] = src[i];
    }

    return i;
}
#endif


#ifndef HAVE_STRLCAT
size_t strlcat(char* dst, const char* src, size_t len)
{
    int i, j;

    /* find the end of the destination string */
    /* if we don't find the null before len, don't write a zero */
    for (i = 0; dst[i] != '\0'; i++)
    {
	if (i >= len)
	{
	    return len + 1 + strlen(src);
	}
    }

    for (j = 0; src[j] != '\0'; j++)
    {
	if (i < len)
	{
	    dst[i] = src[j];
	}
	i++;
    }

    if (i < len)
    {
	dst[i] = src[j];
    }
    else
    {
	dst[len-1] = src[j];
    }

    return i;
}

#endif


