/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: file.c,v $
-- Revision 1.124  2008/07/26 15:06:20  jsedwards
-- Added test to nwos_read_files_disc_list for empty files and fixed invalid
-- test so it would merge -4 instead of writing over any existing errors.
--
-- Revision 1.123  2008/07/26 14:12:10  jsedwards
-- Changed error processing in nwos_read_files_disc_list to make combining
-- errors more generic (or'ing them).  Fixed too many files test to include
-- result value and changed to return -8 for too many files instead of exiting.
--
-- Revision 1.122  2008/07/26 01:51:55  jsedwards
-- Comment out print statement I forgot to take out in the last check in.
--
-- Revision 1.121  2008/07/24 16:36:34  jsedwards
-- Moved read_file_names function from disc_list.c, renamed it to
-- nwos_read_files_disc_list and modified it to pass in file names array.
--
-- Revision 1.120  2008/07/17 13:05:11  jsedwards
-- Changed update_file_00x_object_to_current functions to always change the
-- class, if an updated private FILE class doesn't exist yet then change the
-- class to the public FILE class.  Before it would leave the class unchanged
-- if an updated private FILE class didn't exist yet, then if the object was
-- overwritten to disk it had the wrong class.
--
-- Revision 1.119  2008/07/14 00:25:35  jsedwards
-- Added code to check for missing checksums when importing the file data and
-- include them now.
--
-- Revision 1.118  2008/07/13 14:16:36  jsedwards
-- Feature Request #1856369 - Updated table to match previous change (code to
-- import data into file that already existed from disc_list).
--
-- Revision 1.117  2008/07/13 14:09:36  jsedwards
-- Added code to import data for file if it already existed from a disc_list.
--
-- Revision 1.116  2008/07/13 05:23:03  jsedwards
-- Changed nwos_create_file to test both path and file before making any
-- decisions about how to proceed.
--
-- Revision 1.115  2008/07/13 04:35:53  jsedwards
-- Changed table created in previous version to match this version exactly.
-- It doesn't check for an association, the time, or if the data is stored.
-- Also put it in the order the decisions are made in the code.
--
-- Revision 1.114  2008/07/13 00:05:26  jsedwards
-- Added table in comments for nwos_create_file that shows all possible cases.
--
-- Revision 1.113  2008/06/22 18:51:22  jsedwards
-- Changed nwos_normalize_path to return void instead of bool, added assert
-- to make sure strlen(src) is less than size.  Also fixed to handle all of
-- the ./../subdir/./.. stuff I can think of.
--
-- Revision 1.112  2008/06/21 20:56:11  jsedwards
-- Fixed nwos_normalize_path to eliminate any /./ or /../ in the path.
--
-- Revision 1.111  2008/06/21 16:21:11  jsedwards
-- Moved cleanup_path function from import_file.c. renamed to nwos_normalize_path,
-- and added size parameter.
--
-- Revision 1.110  2008/06/19 13:10:21  jsedwards
-- Added 'allow_multiple_paths' parameter to nwos_create_file function, so it
-- will create more than one path to the same file.
--
-- Revision 1.109  2008/06/12 03:27:05  jsedwards
-- Moved 'check_invalid_type' and 'check_directory_is_all_ok' functions from
-- import_file.c and added 'nwos_' prefix.
--
-- Revision 1.108  2008/06/05 10:55:20  jsedwards
-- Changed nwos_number_of_files_for_path to get next version link from
-- path and file association object instead of file object.
--
-- Revision 1.107  2008/05/21 14:28:07  jsedwards
-- Changed to put previous and next links for revised file in PATH AND FILE
-- ASSOCIATION objects instead of FILE objects.
--
-- Revision 1.106  2008/05/06 13:27:26  jsedwards
-- Added assert around calls to nwos_read_object_from_disk_and_decrypt because
-- it now returns true or false if it read the object okay.  Before it asserted
-- itself if it could not read the block.
--
-- Revision 1.105  2008/04/26 11:32:39  jsedwards
-- Added 'const' to char* parameters to functions.
--
-- Revision 1.104  2008/03/24 13:54:30  jsedwards
-- Changed to pass public parameter along to find_md5, find_sha1, find_sha256,
-- and find_sha512 so that they will search in the correct space.
--
-- Revision 1.103  2008/03/24 01:15:22  jsedwards
-- Change path and file association code to deal with public objects and
-- set clone_of if found.
--
-- Revision 1.102  2008/03/23 15:32:58  jsedwards
-- Change find_matching_file_from_size_md5_sha1 function and
-- create_file_without_saving_data function to deal with public objects
-- and set clone_of field if one is found.
--
-- Revision 1.101  2008/03/22 23:50:06  jsedwards
-- Change file path functions to deal with public objects and set clone_of
-- field if one is found.
--
-- Revision 1.100  2008/03/22 23:30:33  jsedwards
-- Changed md5, sha1, sha256, sha512 functions to set clone_of field if they
-- are a clone of a public object.
--
-- Revision 1.99  2008/03/20 14:00:09  jsedwards
-- Added tests to determine if running in public mode or not.
--
-- Revision 1.98  2008/03/15 22:12:27  jsedwards
-- Added checking SHA256 and SHA512 to nwos_check_file_checksums.
--
-- Revision 1.97  2008/03/15 00:35:35  jsedwards
-- Fix a bug in all of the checksum (MD5, SHA1, SHA256, SHA512) comparisions,
-- they all used a sizeof which return 4 instead of the size, so only the
-- first four bytes of each checksum was being compared.
--
-- Revision 1.96  2008/03/14 14:24:22  jsedwards
-- Added sha512 paramter to 'read_file' function.
--
-- Revision 1.95  2008/03/14 14:18:52  jsedwards
-- Added more SHA512 stuff.
--
-- Revision 1.94  2008/03/14 14:04:50  jsedwards
-- Added sha512 to nwos_find_matching_file_from_size_md5_sha1 function.
--
-- Revision 1.93  2008/03/14 13:49:29  jsedwards
-- It turns out that I made a mistake in the previous revision (1.92), instead
-- of removing the test for md5sum matching, I should have fixed the test to
-- make sure the sha256sum matched.  All this time it wasn't testing to see if
-- the SHA256 checksum matched.
--
-- Revision 1.92  2008/03/14 13:40:21  jsedwards
-- Remove duplicate test for md5sum matching.
--
-- Revision 1.91  2008/03/14 13:21:21  jsedwards
-- Added nwos_find_sha512 and nwos_create_sha512 functions.  Modified
-- checksum_file to calculate the SHA512 checksum in addition to the others.
-- Changed calls to checksum_file to do sha512_digest as well.
--
-- Revision 1.90  2008/03/14 12:25:24  jsedwards
-- Added code to nwos_update_file_001_object_to_current to void new SHA512
-- reference and added nwos_update_file_002_object_to_current function to
-- update a FILE revision 002 object to 003 with a SHA512 reference.
--
-- Revision 1.89  2008/03/12 04:01:22  jsedwards
-- Changed to use the GNU MD5 context and functions instead of the RSA context
-- and functions.
--
-- Revision 1.88  2008/02/16 17:13:24  jsedwards
-- Added new nwos_add_info_to_existing_file function that adds time_stamp,
-- size, sha1 and sha256 checksums to to file with md5 already read in.
--
-- Revision 1.87  2008/02/16 14:06:15  jsedwards
-- Took all of the changes made to nwos_create_file_with_only_md5 in the
-- previous revision back out (this revision is the same as revision 1.85).
--
-- Revision 1.86  2008/02/16 14:00:55  jsedwards
-- Changed nwos_create_file_with_only_md5 to read the file and compute all the
-- other things at the same time.
--
-- Revision 1.85  2008/02/16 13:14:21  jsedwards
-- Added new nwos_create_file_with_only_md5 function.
--
-- Revision 1.84  2008/01/13 21:08:13  jsedwards
-- Changed to print MD5, SHA1, and SHA256 sums to log file instead of stdout.
--
-- Revision 1.83  2008/01/13 19:51:13  jsedwards
-- Changed to use new MD5_DIGEST_SIZE, SHA1_DIGEST_SIZE, and SHA256_DIGEST_SIZE
-- instead of hard coded values.
--
-- Revision 1.82  2007/10/26 13:57:35  jsedwards
-- Changed ifdef test for "linux" to not having HAVE_STRUCT_STAT_ST_MTIMESPEC.
--
-- Revision 1.81  2007/09/02 19:39:37  jsedwards
-- Added estimate_blocks_for_file function to estimate the number blocks that
-- will be needed to store a file.
--
-- Revision 1.80  2007/08/25 21:50:06  jsedwards
-- Corrected an incorrect comment for the nwos_number_of_files_for_path
-- function.  NO code changes!
--
-- Revision 1.79  2007/08/17 16:36:21  jsedwards
-- Added SHA256 checksums in addition to MD5 and SHA1.  Still works with files
-- which were created before, that only have MD5 and SHA1 sums.
--
-- Revision 1.78  2007/08/17 11:59:37  jsedwards
-- Added conditional code to allow creating_file_without_storing_data to
-- create public objects, when compiled in PUBLIC_MODE.
--
-- Revision 1.77  2007/08/12 19:24:13  jsedwards
-- Removed code that set security level to Extreme before calling
-- 'create_file_path' so that if running minimum security (or whatever) the
-- file will be created following the previous one.
--
-- Revision 1.76  2007/08/10 00:09:10  jsedwards
-- Removed defintion of _LARGEFILE64_SOURCE, now using _FILE_OFFSET_BITS=64.
-- Added LL suffix to long long constants.
--
-- Revision 1.75  2007/08/01 14:29:20  jsedwards
-- Added code to recalculate data checksum when File object is updated and to
-- make sure an updated private File class defintion exists before reading any
-- old File objects in add_revision function.
--
-- Revision 1.74  2007/07/30 18:49:31  jsedwards
-- Changed to NOT attempt to create the updated File private class definition
-- when an old object is updated.
--
-- Revision 1.73  2007/07/29 19:37:51  jsedwards
-- Bug #1763165 - Fixed bug in nwos_find_matching_path_and_file_association
-- where it would return ERROR_OCCURRED (non-zero) when an error occurred.
-- Since the return value is boolean and ERROR_OCCURRED is non-zero, it
-- appeared to return true even when an error occurred.
--
-- Revision 1.72  2007/07/28 04:01:35  jsedwards
-- Converted to use uint64 for file size instead of uint32 and added function
-- to convert an object in the old format to the new.
--
-- Revision 1.71  2007/07/24 16:57:11  jsedwards
-- Modified nwos_number_of_files_for_path and nwos_file_path_to_path_and_file_
-- association to skip over files that have a later revision.
--
-- Revision 1.70  2007/07/24 00:16:55  jsedwards
-- Added nwos_add_new_revision_of_file function to create a new revision of a
-- file that is already in the system.
--
-- Revision 1.69  2007/07/21 14:55:00  jsedwards
-- Added 'match' parameter to nwos_file_is_identical so that it can ignore the
-- time stamp on a file.
--
-- Revision 1.68  2007/07/05 17:59:07  jsedwards
-- Change nwos_create_disc_copy to move disc to a new location if the disc
-- (number) already exists and location is different.
--
-- Revision 1.67  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.66  2007/06/18 13:16:17  jsedwards
-- Fix so create_file_without_storing correctly adds file path separator.
--
-- Revision 1.65  2007/06/13 21:19:47  jsedwards
-- Change find_matching_path_and_file function to accept a directory path that
-- is null and just ignore it.
--
-- Revision 1.64  2007/06/13 21:12:56  jsedwards
-- Change to correctly append a '/' to the directory path if it is missing.
--
-- Revision 1.63  2007/06/13 20:49:36  jsedwards
-- Added directory parameter to find_matching_path_and_file_association.
--
-- Revision 1.62  2007/06/13 20:16:50  jsedwards
-- Added a directory parameter to create_file_without_storing_data.
--
-- Revision 1.61  2007/06/05 23:49:05  jsedwards
-- Modify to allow option to ignore time stamp on files.
--
-- Revision 1.60  2007/04/07 02:17:46  jsedwards
-- Added code to print if matching path and file objects were printed.
--
-- Revision 1.59  2007/03/18 19:54:20  jsedwards
-- Added file_has_backup routine.
--
-- Revision 1.58  2007/03/18 18:17:01  jsedwards
-- Moved freeing reference list outside of if so it was at the same level as
-- the malloc.
--
-- Revision 1.57  2007/02/28 14:04:44  jsedwards
-- Added "file_is_stored" function.
--
-- Revision 1.56  2007/01/27 17:10:36  jsedwards
-- Added file_is_identical function that compares the actual data stored in
-- the system with the file, not the checksums.
--
-- Revision 1.55  2007/01/27 15:46:08  jsedwards
-- Change name of check_file_md5sum to check_file_checksums and correct bytes
-- printed for SHA1 sum.
--
-- Revision 1.54  2007/01/27 15:38:34  jsedwards
-- Change to check SHA1 sum as well as MD5 when restoring file.
--
-- Revision 1.53  2007/01/27 15:24:22  jsedwards
-- Change to check SHA1 sum as well as MD5 when verifying file.
--
-- Revision 1.52  2007/01/27 14:55:45  jsedwards
-- Added code to handle the case where a duplicate file with a different name
-- already exists in the system.
--
-- Revision 1.51  2007/01/27 14:26:36  jsedwards
-- Added code to set reference to 'void' in all of the find routines in the
-- case that they don't find what they are looking for.
--
-- Revision 1.50  2007/01/27 01:56:13  jsedwards
-- Changed two print statements in find_matching_file_from_size_md5_sha1 to
-- only print if 'vebose' global variable is set.
--
-- Revision 1.49  2007/01/20 13:42:53  jsedwards
-- Change to use the SHA1 sum to search for the file instead of the MD5
-- (less likely to be two files with the same checksum?).  Then just use
-- the MD5 to verify.
--
-- Revision 1.48  2007/01/19 15:06:11  jsedwards
-- Fix bug in creating sha1 checksum for files without storing data.
--
-- Revision 1.47  2007/01/19 13:14:10  jsedwards
-- Fix bug in compare time stamp.
--
-- Revision 1.46  2007/01/17 13:11:36  jsedwards
-- Move push and pop encryption level so that if it finds an existing file
-- (and returns an error) it doesn't leave it pushed.
--
-- Revision 1.45  2007/01/14 22:20:00  jsedwards
-- Modifications to do SHA1 checksum in addition to MD5 checksum.
--
-- Revision 1.44  2007/01/14 17:10:39  jsedwards
-- Changed file_path_to_file to file_path_to_path_and_file_association and
-- changed restore_file to take a path_and_file_assocation object as input
-- instead of a file object.
--
-- Revision 1.43  2007/01/14 13:35:51  jsedwards
-- Change nwos_number_of_files_for_path to take a path object instead of a
-- character path so it doesn't have to search for it (redundant).
--
-- Revision 1.42  2007/01/14 13:23:33  jsedwards
-- Added number_of_file_for_path and find_path_and_file_association_by_index
-- functions.
--
-- Revision 1.41  2007/01/14 13:19:44  jsedwards
-- Changed find_path_and_file_association so mod_time cannot be null (mod_time
--  has to match) because otherwise it would only return the first one found
-- (no way to find subsequent matches).
--
-- Revision 1.40  2007/01/14 04:03:10  jsedwards
-- Move modification time from 'file' object to 'path_and_file_association'
-- object.
--
-- Revision 1.39  2007/01/13 20:14:10  jsedwards
-- Changed so that file id isn't generated unless file is going to be created.
-- Also changed so 'seq' variable in 'read_file' keeps incrementing instead of
-- being mod'ed itself, and it is moded when used to index into sequence table.
--
-- Revision 1.38  2007/01/12 13:58:27  jsedwards
-- Add code to print read_file messages to the log too.
--
-- Revision 1.37  2007/01/10 14:55:16  jsedwards
-- Separated out routine to read file from create_file.
--
-- Revision 1.36  2007/01/07 14:16:47  jsedwards
-- Changed file_path_to_file to deal with new path_and_file_association object,
-- however it can currently only deal with a path to one file relationship.
--
-- Revision 1.35  2007/01/07 03:19:39  jsedwards
-- Fixed create_file routine to work with new file association object.
--
-- Revision 1.34  2007/01/06 20:13:32  jsedwards
-- Changed to find_private_name instead of find_name.
--
-- Revision 1.33  2007/01/05 13:47:46  jsedwards
-- Fix uninitialized variables in find_matching_file_from_size_time_md5.
--
-- Revision 1.32  2007/01/05 13:37:38  jsedwards
-- Add find_matching_path_and_file_association function.
--
-- Revision 1.31  2007/01/04 19:17:09  jsedwards
-- Moved path and file association routines lower in file so they can use
-- routines that are declared farther down.
--
-- Revision 1.30  2007/01/04 19:08:26  jsedwards
-- Extracted routine to find a file from the size, time, and md5sum from the
-- create_file_without_storing_data function.
--
-- Revision 1.29  2007/01/04 18:18:32  jsedwards
-- Extracted loop from create_file_without_storing_data that checksums file
-- into a separate function.
--
-- Revision 1.28  2007/01/04 17:32:26  jsedwards
-- In find routines change to return false if private class doesn't exist and
-- in create routines change to find or create the private class.
--
-- Revision 1.27  2007/01/04 04:05:59  jsedwards
-- Fixed mispelled function names.
--
-- Revision 1.26  2007/01/03 15:44:17  jsedwards
-- Fixed path and file association stuff to compile (UNTESTED!)
--
-- Revision 1.25  2006/12/24 15:05:12  jsedwards
-- Added code to clear md5sum, disc_copy, and location objects to zero when
-- they are created.  This was somehow left out earlier and objects of these
-- types had garbage in the unused fields.
--
-- Revision 1.24  2006/12/21 13:14:28  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.23  2006/12/19 14:52:16  jsedwards
-- Kludged version to compile with new 'file path' structure without 'file'
-- feature.  DOES NOT function correctly!
--
-- Revision 1.22  2006/12/19 14:42:21  jsedwards
-- Kludged version to compile with new File_Path structure that doesn't have
-- the 'file' variable.  This version is BROKEN!
--
-- Revision 1.21  2006/12/07 14:08:46  jsedwards
-- Moved setup routine and associated functions to attic/big_bang.c.
--
-- Revision 1.20  2006/12/02 03:41:49  jsedwards
-- Changed create_file_path to return created_new when it added the file to
-- an existing path name.
--
-- Revision 1.19  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.18  2006/12/01 05:19:14  jsedwards
-- Added code that updates the data checksum and re-writes the object to disk
-- when changing from a single file to a file list.
--
-- Revision 1.17  2006/11/30 14:59:44  jsedwards
-- Change to allow 'file' in a file path object to point to a reference list
-- of files, if there is more than one file with that name.
--
-- Revision 1.16  2006/11/29 18:49:49  jsedwards
-- Change so both path and file object reference are passed to restore_file.
-- Also changed file error handling to exit on errors (this is probably bad).
--
-- Revision 1.15  2006/11/28 13:13:29  jsedwards
-- Changed to use BSD name for modification time in stat struct and added a
-- define to map it to the Linux name when compiling on Linux.
--
-- Revision 1.14  2006/11/27 13:48:08  jsedwards
-- Changed to use variable sized counts for disc lists (so number of files can
-- be larger than 127).
--
-- Revision 1.13  2006/11/19 16:38:18  jsedwards
-- Added find_matching_disc_list function.
--
-- Revision 1.12  2006/11/19 15:28:20  jsedwards
-- Made get_disc_list_object_size a global function.
--
-- Revision 1.11  2006/11/19 14:48:47  jsedwards
-- Change check_file_md5sum function to return one of three possible results
-- (file not found, md5sum match, or md5sum mismatch) instead of just a
-- boolean saying file found or not found.
--
-- Revision 1.10  2006/11/18 15:09:09  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.9  2006/11/18 14:33:03  jsedwards
-- Change size of kludge buffer for reading and writing disc lists to max size
-- of disc list instead of file block size, because now disc list objects can
-- be larger than one file block.
--
-- Revision 1.8  2006/11/07 14:07:36  jsedwards
-- Completely rearranged create_file_without_storing_data function to deal
-- with files that have duplicate md5sums (and are possibly exact duplicates).
-- Fixed bug where all of disc id was not being compared.
--
-- Revision 1.7  2006/11/06 13:51:24  jsedwards
-- Change so create_file_without_storing_data returns the reference for the
-- file_path instead of the file.  Also fixed bug in class name.
--
-- Revision 1.6  2006/11/05 21:29:51  jsedwards
-- Add functions for finding and creating disc_copy and storage_location
-- objects.
--
-- Revision 1.5  2006/11/04 18:59:21  jsedwards
-- Added routines to find or create a Disc_List.
--
-- Revision 1.4  2006/11/02 11:45:59  jsedwards
-- Change maximum file size from 4GB - 1 to 2GB - 1 because the old compiler
-- isn't happy about the 32 bit constant.  It needs to be addressed sometime
-- in the future anyway.
--
-- Revision 1.3  2006/10/29 13:28:40  jsedwards
-- Added routines to just record a files md5sum without storing the file and
-- to check the md5sum of a file.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.24  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.23  2006/10/22 13:11:55  jsedwards
-- Changed to pass approximate number of blocks needed for a file to
-- set_sequential_blocks instead of spacing.  It now computes the
-- spacing from the blocks needed because the blocks on disk is variable.
--
-- Revision 1.1.2.22  2006/10/18 13:10:32  jsedwards
-- Changed printf formats for uint32 that is an integer instead of long.
--
-- Revision 1.1.2.21  2006/10/07 22:25:21  jsedwards
-- Changed so that sequential blocks are handled within objectify when the
-- 'set_sequential_blocks' function is called first.
--
-- Revision 1.1.2.20  2006/10/06 04:42:49  jsedwards
-- Changed to use new id generator that generates ids that scan across the
-- disk drive instead of totally random so that files don't cause so much
-- repositioning of the heads and hopefully will be faster.
--
-- Revision 1.1.2.19  2006/09/18 11:38:12  jsedwards
-- Fix so block sequence number wraps correctly.
--
-- Revision 1.1.2.18  2006/09/18 01:34:23  jsedwards
-- Changed for 256 byte blocks and use a "reference_list" to point to them.
--
-- Revision 1.1.2.17  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.1.2.16  2006/08/26 15:30:11  jsedwards
-- Added routines to find and create MD5 objects and modified file routines
-- to use them instead of having the md5sum embedded in the file object.
-- Also removed the "tolower" for path names, it was a mistake.
--
-- Revision 1.1.2.15  2006/08/25 01:37:47  jsedwards
-- Added code to test md5sum on restoring the file.
--
-- Revision 1.1.2.14  2006/08/24 13:01:18  jsedwards
-- Added code to compute the MD5 checksum while reading the file.
--
-- Revision 1.1.2.13  2006/08/24 12:58:08  jsedwards
-- Changed the <utime.h> include to <sys/time.h> because although the man page
-- says "utimes()" is defined in <utime.h>, gcc still gives a warning.  I only
-- see "utimes()" in <sys/time.h>.
--
-- Revision 1.1.2.12  2006/08/24 11:57:21  jsedwards
-- Added code to restore the modification time of the file when it is restored.
--
-- Revision 1.1.2.11  2006/08/23 13:23:09  jsedwards
-- Added code to save the files time stamp.
--
-- Revision 1.1.2.10  2006/08/23 12:41:56  jsedwards
-- Changed to use fread instead of fgetc for every byte.
--
-- Revision 1.1.2.9  2006/08/20 15:35:27  jsedwards
-- Removed many of the debugging printf statements.
--
-- Revision 1.1.2.8  2006/08/20 15:13:43  jsedwards
-- Fix MAX_BLOCKS define.
--
-- Revision 1.1.2.7  2006/08/20 14:54:01  jsedwards
-- Fixed bug in num_blocks calculation and reading small (< 28 bytes) files.
--
-- Revision 1.1.2.6  2006/08/20 02:17:53  jsedwards
-- Fixed several bugs in the create_file function and added the restore_file
-- function.
--
-- Revision 1.1.2.5  2006/08/19 20:02:55  jsedwards
-- Fix comment about create_file routine.
--
-- Revision 1.1.2.4  2006/08/19 18:58:18  jsedwards
-- Added code to store a file in objects.
--
-- Revision 1.1.2.3  2006/08/19 18:51:59  jsedwards
-- Changed so checksum uses new "file" element instead of "count" for it's base.
--
-- Revision 1.1.2.2  2006/08/19 14:34:29  jsedwards
-- Added creation of MD5sum class (moved here from big_bang.c).
--
-- Revision 1.1.2.1  2006/08/18 13:04:03  jsedwards
-- Initial version.
--
*/

#include <assert.h>
#include <dirent.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "gnu/md5.h"
#include "gnu/sha1.h"
#include "gnu/sha256.h"
#include "gnu/sha512.h"

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"
#include "time_stamp.h"

/* Map linux file modification time name in stat struct */
#ifndef HAVE_STRUCT_STAT_ST_MTIMESPEC
#define st_mtimespec st_mtim
#endif

#define DATA_STORAGE_SIZE (FILE_BLOCK_SIZE - 12)   /* 4 bytes each for flags, id, checksum */

typedef struct file_block {
    uint8 flags[4];
    ObjRef id;
    uint8 checksum[4];
    uint8 storage[DATA_STORAGE_SIZE];
} DataObject;



static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


static inline uint64 file_size_to_uint64(uint8 size[5])
{
    return ((uint64)size[0] << 32) |
	   ((uint64)size[1] << 24) |
	   ((uint64)size[2] << 16) |
	   ((uint64)size[3] << 8)  |
	    (uint64)size[4];
}


uint32 nwos_estimate_blocks_for_file(const char* path)
{
    struct stat stat_struct;
    uint32 result = 0;
    off_t num_blocks;

    /* if file fails to open just return zero */

    if (stat(path, &stat_struct) == 0)
    {
	num_blocks = stat_struct.st_size / DATA_STORAGE_SIZE + 14;

	num_blocks += num_blocks / 60;

	if (num_blocks < 0xffffffffLL)
	{
	    result = (uint32) num_blocks;
	}
	else
	{
	    result = 0xffffffff;
	}
    }

    return result;
}


/* This function copies a path string and removes any extra slashes, so there are no "//" or "." in the path.         */
/* It will not return with a trailing slash.  If there are more ".."'s in the path than subdirectories it will return */
/* the result.  Examples:                                                                                             */
/*                           "subdir1/./subdir2/"          -> "subdir1/subdir2"                                       */
/*                           "subdir1/subdir2/.."          -> "subdir1"                                               */
/*                           "subdir1/subdir2/../../"      -> "."                                                     */
/*                           "subdir1/subdir2/../../.."    -> ".."                                                    */
/*                           "subdir1/subdir2/../../../.." -> "../.."                                                 */

void nwos_normalize_path(char* dst, char* src, size_t size)
{
    int i = 0;

    assert(src[0] != '/');                                      /* cannot be an absolute path */
    assert(strlen(src) + 1 < size);

    while (src[0] == '.' && src[1] == '/') src += 2;

    while (*src != '\0')
    {
	assert(i < size);

	dst[i++] = *src++;

	while (dst[i-1] == '/' && *src == '/') src++;

	if (i == 2 && dst[0] == '.' && dst[1] == '/') i = 0;  /* eliminate any "./" in the path */ 

	if (i > 2 && dst[i-3] == '/' && dst[i-2] == '.' && dst[i-1] == '/') i = i - 2;  /* eliminate any "/./" in the path */ 

	if (i > 3 && dst[i-4] == '/' && dst[i-3] == '.' && dst[i-2] == '.' && dst[i-1] == '/') /* eliminate "/../" in the path */ 
	{
	    if (i < 6 || (i == 6 && memcmp(dst, "..", 2) != 0) || (i > 6 && memcmp(&dst[i-7], "/..", 3) != 0))
	    {
		i = i - 4;
		while (i > 0 && dst[i-1] != '/') i--;
	    }
	}

	assert(i < 2 || dst[0] != '.' || dst[1] != '/');
    }

    if (i > 0 && dst[i-2] == '/' && dst[i-1] == '.') i = i - 2;   /* eliminate "/." ending the path */ 

    if (i > 2 && dst[i-3] == '/' && dst[i-2] == '.' && dst[i-1] == '.')  /* eliminate "/.." ending the path */ 
    {
	if (i < 5 || (i == 5 && memcmp(dst, "..", 2) != 0) || (i > 5 && memcmp(&dst[i-6], "/..", 3) != 0))
	{
	    i = i - 3;
	    while (i > 0 && dst[i-1] != '/') i--;
	}
    }

    if (dst[i-1] == '/')
    {
	i--;
    }

    if (i == 0)
    {
	dst[i++] = '.';
    }

    assert(i < size);

    dst[i] = '\0';

    assert(dst[0] != '\0');      /* should not be an empty string */

    assert(dst[i-1] != '/');     /* should not end with a '/' */

    assert(dst[0] != '.' || dst[1] != '/');     /* if the first char is a dot it should not be ./ */

    char *p = dst;

    while (strlen(p) > 3 && memcmp(p, "../", 3) == 0) p += 3;

    if (strlen(p) > 3)
    {
	assert(strstr(p, "/../") == NULL);
    }

    assert(i < 2 || dst[i-1] != '.' || dst[i-2] != '/');   /* should not end with a "/." */
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* File path (name) object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_file_path(bool public, const char* path, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ObjRef path_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;

    length = strlen(path);

    assert(length <= 255);

    void_reference(ref);   /* in case we don't find it */


    /* for now return false if file path private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE PATH", &path_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("FILE PATH", &path_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("FILE PATH", &path_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&path_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &path_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_path_object_size);

	    /* remember ptr_path_obj points to the kludge buffer */

	    if (ptr_path_obj->count == length && strncasecmp((char*)ptr_path_obj->storage, path, length) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_file_path(const char* path, ObjRef* ref)
{
    return find_file_path(true, path, ref);
}


bool nwos_find_file_path(const char* path, ObjRef* ref)
{
    return find_file_path(nwos_in_public_mode(), path, ref);
}


/* returns the number of path and file association objects that refer to this path */
int nwos_number_of_files_for_path(ObjRef* path_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int result = 0;


    if (nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	nwos_read_variable_sized_object_from_disk(path_ref, kludge, sizeof(kludge), &get_path_object_size);

	/* remember ptr_path_obj points to the kludge buffer */

	ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &assoc_class_ref))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

		assert(is_same_object(&assoc_obj.path, path_ref));

		if (is_void_reference(&assoc_obj.header.object.next_version))
		{
		    result++;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;   /* return number of path_and_file_associations we found that refer to this path */
}



/* Find existing path or create new */

ObjCreateResult nwos_create_file_path(const char* path, ObjRef* ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    size_t length;
    ObjRef path_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];
#endif

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    if (!nwos_find_file_path(path, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("FILE PATH", &path_class_ref));
#else
	nwos_find_or_create_private_class_definition("FILE PATH", &path_class_ref);
#endif

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_path_obj->header.common, ref, &path_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_file_path(path, &public_ref))
	{
	    copy_reference(&ptr_path_obj->header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "file path: %s  ref: %08x  clone_of: %08x",
		     path, nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	ptr_path_obj->count = length;
	for (i = 0; i < length; i++) ptr_path_obj->storage[i] = path[i];

	nwos_create_reference_list(ref, &ptr_path_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->header.object, sizeof(ObjectHeader), ptr_path_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_path_obj->count, sizeof(C_struct_File_Path) + length - sizeof(EveryObject), ptr_path_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_File_Path) + length);

	nwos_add_to_references(ref, &path_class_ref);

	ptr_path_obj = malloc(sizeof(C_struct_File_Path) + length);
	nwos_read_object_from_disk(ref, ptr_path_obj, sizeof(C_struct_File_Path) + length);
	assert(memcmp(kludge, ptr_path_obj, sizeof(C_struct_File_Path) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);  /* read the other way */
	assert(memcmp(ptr_path_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);

	free(ptr_path_obj);
	ptr_path_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



bool nwos_file_path_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj;
    int i;
    size_t path_obj_size;

    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_path_object_size);

    /* point to the kludge buffer temporarily to get the size of the object */
    ptr_path_obj = (C_struct_File_Path*)kludge;

    path_obj_size = sizeof(C_struct_File_Path) + ptr_path_obj->count;
    /* printf("path_obj_size: %d\n", path_obj_size); */
    assert(path_obj_size > sizeof(C_struct_File_Path));

    /* now we have the size allocate memory for it and copy it there */
    ptr_path_obj = malloc(path_obj_size);
    memcpy(ptr_path_obj, kludge, path_obj_size);

    for (i = 0; i < ptr_path_obj->count; i++) 
    {
	assert(i < size);
	string[i] = ptr_path_obj->storage[i];
    }
    string[i] = '\0';

    free(ptr_path_obj);

    return true;
}


bool nwos_file_path_to_path_and_file_association(ObjRef* path_ref, int index, ObjRef* assoc_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    ReferenceList* ref_list;
    int num_refs;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;
    int i;
    int count;

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;


    /* read the path object into the kludge buffer and then after we know what size it is malloc space and copy it there */
    nwos_read_variable_sized_object_from_disk(path_ref, kludge, sizeof(kludge), &get_path_object_size);

    ref_list = nwos_malloc_reference_list(&ptr_path_obj->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    count = 0;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    assert(is_same_object(&assoc_obj.path, path_ref));

	    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

	    /* don't count any that have newer versions */

	    if (is_void_reference(&file_obj.header.object.next_version))
	    {
		if (count == index)    /* this is the one we want */
		{
		    copy_reference(assoc_ref, &ref_list->references[i]);
		    break;
		}
		else
		{
		    count++;
		}
	    }
	}
    }

    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_refs);
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* MD5 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_md5(bool public, uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef md5_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_MD5sum md5_object;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the md5sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("MD5SUM", &md5_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&md5_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &md5_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &md5_object, sizeof(md5_object));

	    if (memcmp(md5_object.md5sum, md5sum, MD5_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(true, md5sum, ref);
}


/* find an md5 based upon what mode we are in */

bool nwos_find_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    return find_md5(nwos_in_public_mode(), md5sum, ref);
}


/* Find existing MD5sum or create new */

ObjCreateResult nwos_create_md5(uint8 md5sum[MD5_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef md5_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_MD5sum md5_object;
    C_struct_MD5sum* ptr_md5_obj;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this md5 */

    if (!nwos_find_md5(md5sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("MD5SUM", &md5_class_ref));
#else
	nwos_find_or_create_private_class_definition("MD5SUM", &md5_class_ref);
#endif

	memset(&md5_object, 0, sizeof(md5_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&md5_object.header.common, ref, &md5_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_md5(md5sum, &public_ref))
	{
	    copy_reference(&md5_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "md5sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(md5_object.md5sum, md5sum, sizeof(md5_object.md5sum));

	nwos_create_reference_list(ref, &md5_object.header.object.references);

	nwos_crc32_calculate((uint8*) &md5_object.header.object, sizeof(ObjectHeader), md5_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &md5_object.md5sum, sizeof(C_struct_MD5sum) - sizeof(EveryObject), md5_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &md5_object, sizeof(md5_object));

	nwos_add_to_references(ref, &md5_class_ref);

	ptr_md5_obj = malloc(sizeof(C_struct_MD5sum));
	nwos_read_object_from_disk(ref, ptr_md5_obj, sizeof(C_struct_MD5sum));
	assert(memcmp(&md5_object, ptr_md5_obj, sizeof(C_struct_MD5sum)) == 0);
	free(ptr_md5_obj);
	ptr_md5_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA1 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha1(bool public, uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha1_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA1sum sha1_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha1sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA1SUM", &sha1_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha1_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha1_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &sha1_object, sizeof(sha1_object));

	    if (memcmp(sha1_object.sha1sum, sha1sum, SHA1_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(true, sha1sum, ref);
}


/* find an sha1 based upon what mode we are in */

bool nwos_find_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha1(nwos_in_public_mode(), sha1sum, ref);
}



/* Find existing SHA1sum or create new */

ObjCreateResult nwos_create_sha1(uint8 sha1sum[SHA1_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha1_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA1sum sha1_object;
    C_struct_SHA1sum* ptr_sha1_obj;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha1 */

    if (!nwos_find_sha1(sha1sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA1SUM", &sha1_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA1SUM", &sha1_class_ref);
#endif

	memset(&sha1_object, 0, sizeof(sha1_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha1_object.header.common, ref, &sha1_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha1(sha1sum, &public_ref))
	{
	    copy_reference(&sha1_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha1sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha1_object.sha1sum, sha1sum, sizeof(sha1_object.sha1sum));

	nwos_create_reference_list(ref, &sha1_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha1_object.header.object, sizeof(ObjectHeader), sha1_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha1_object.sha1sum, sizeof(C_struct_SHA1sum) - sizeof(EveryObject), sha1_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha1_object, sizeof(sha1_object));

	nwos_add_to_references(ref, &sha1_class_ref);

	ptr_sha1_obj = malloc(sizeof(C_struct_SHA1sum));
	nwos_read_object_from_disk(ref, ptr_sha1_obj, sizeof(C_struct_SHA1sum));
	assert(memcmp(&sha1_object, ptr_sha1_obj, sizeof(C_struct_SHA1sum)) == 0);
	free(ptr_sha1_obj);
	ptr_sha1_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA256 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha256(bool public, uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha256_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA256sum sha256_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha256sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA256SUM", &sha256_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha256_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha256_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &sha256_object, sizeof(sha256_object));

	    if (memcmp(sha256_object.sha256sum, sha256sum, SHA256_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(true, sha256sum, ref);
}


/* find an sha256 based upon what mode we are in */

bool nwos_find_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha256(nwos_in_public_mode(), sha256sum, ref);
}




/* Find existing SHA256sum or create new */

ObjCreateResult nwos_create_sha256(uint8 sha256sum[SHA256_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha256_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA256sum sha256_object;
    C_struct_SHA256sum* ptr_sha256_obj;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha256 */

    if (!nwos_find_sha256(sha256sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA256SUM", &sha256_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA256SUM", &sha256_class_ref);
#endif

	memset(&sha256_object, 0, sizeof(sha256_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha256_object.header.common, ref, &sha256_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha256(sha256sum, &public_ref))
	{
	    copy_reference(&sha256_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha256sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha256_object.sha256sum, sha256sum, sizeof(sha256_object.sha256sum));

	nwos_create_reference_list(ref, &sha256_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha256_object.header.object, sizeof(ObjectHeader), sha256_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha256_object.sha256sum, sizeof(C_struct_SHA256sum) - sizeof(EveryObject), sha256_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha256_object, sizeof(sha256_object));

	nwos_add_to_references(ref, &sha256_class_ref);

	ptr_sha256_obj = malloc(sizeof(C_struct_SHA256sum));
	nwos_read_object_from_disk(ref, ptr_sha256_obj, sizeof(C_struct_SHA256sum));
	assert(memcmp(&sha256_object, ptr_sha256_obj, sizeof(C_struct_SHA256sum)) == 0);
	free(ptr_sha256_obj);
	ptr_sha256_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* SHA512 object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static bool find_sha512(bool public, uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    ObjRef sha512_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    C_struct_SHA512sum sha512_object;
    int i;

    void_reference(ref);  /* in case we don't find it */

    /* for now return false if the sha512sum private class doesn't exist yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("SHA512SUM", &sha512_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&sha512_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &sha512_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &sha512_object, sizeof(sha512_object));

	    if (memcmp(sha512_object.sha512sum, sha512sum, SHA512_DIGEST_SIZE) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(true, sha512sum, ref);
}


/* find an sha512 based upon what mode we are in */

bool nwos_find_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_sha512(nwos_in_public_mode(), sha512sum, ref);
}




/* Find existing SHA512sum or create new */

ObjCreateResult nwos_create_sha512(uint8 sha512sum[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef sha512_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    C_struct_SHA512sum sha512_object;
    C_struct_SHA512sum* ptr_sha512_obj;
#ifndef PUBLIC_MODE
    ObjRef public_ref;
    char msg[128];

    assert(!nwos_in_public_mode());
#endif

    /* first find out if we already have this sha512 */

    if (!nwos_find_sha512(sha512sum, ref))   /* didn't find it */
    {
#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("SHA512SUM", &sha512_class_ref));
#else
	nwos_find_or_create_private_class_definition("SHA512SUM", &sha512_class_ref);
#endif

	memset(&sha512_object, 0, sizeof(sha512_object));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&sha512_object.header.common, ref, &sha512_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_sha512(sha512sum, &public_ref))
	{
	    copy_reference(&sha512_object.header.object.clone_of, &public_ref);
	    snprintf(msg, sizeof(msg), "sha512sum - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_ref));
	    nwos_log(msg);
	}
#endif
	memcpy(sha512_object.sha512sum, sha512sum, sizeof(sha512_object.sha512sum));

	nwos_create_reference_list(ref, &sha512_object.header.object.references);

	nwos_crc32_calculate((uint8*) &sha512_object.header.object, sizeof(ObjectHeader), sha512_object.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &sha512_object.sha512sum, sizeof(C_struct_SHA512sum) - sizeof(EveryObject), sha512_object.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &sha512_object, sizeof(sha512_object));

	nwos_add_to_references(ref, &sha512_class_ref);

	ptr_sha512_obj = malloc(sizeof(C_struct_SHA512sum));
	nwos_read_object_from_disk(ref, ptr_sha512_obj, sizeof(C_struct_SHA512sum));
	assert(memcmp(&sha512_object, ptr_sha512_obj, sizeof(C_struct_SHA512sum)) == 0);
	free(ptr_sha512_obj);
	ptr_sha512_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* File object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

static uint64 checksum_file(const char* path, uint8 md5_digest[MD5_DIGEST_SIZE],
			    uint8 sha1_digest[SHA1_DIGEST_SIZE],
			    uint8 sha256_digest[SHA256_DIGEST_SIZE],
			    uint8 sha512_digest[SHA512_DIGEST_SIZE])
{
    FILE* fp;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    size_t bytes_read;
    uint8 buffer[4096];
    uint64 file_length;

    fp = fopen(path, "rb");

    file_length = 0;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    sha1_init_ctx(&sha1_context);
    sha256_init_ctx(&sha256_context);
    sha512_init_ctx(&sha512_context);

    bytes_read = sizeof(buffer);   /* go through loop at least once */

    while (bytes_read == sizeof(buffer))
    {
	bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	if (bytes_read < sizeof(buffer))    /* eof or error */
	{
	    if (ferror(fp))
	    {
		perror(path);
		exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
	    }
	}

	file_length = file_length + bytes_read;

	if (bytes_read > 0)   /* we have data to store */
	{
	    md5_process_bytes(buffer, bytes_read, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(buffer, bytes_read, &sha1_context);    /* include this data in the sha1 checksum */
	    sha256_process_bytes(buffer, bytes_read, &sha256_context); /* include this data in the sha256 checksum */
	    sha512_process_bytes(buffer, bytes_read, &sha512_context); /* include this data in the sha512 checksum */
	}
    }

    fclose(fp);

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
    sha1_finish_ctx(&sha1_context, sha1_digest);
    sha256_finish_ctx(&sha256_context, sha256_digest);
    sha512_finish_ctx(&sha512_context, sha512_digest);

    return file_length;
}



/* find a matching file from the md5sum */

static bool find_matching_file_from_size_md5_sha1(bool public, uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    ObjRef file_class_ref;
    ObjRef object_class;
    ObjRef md5sum_ref;
    ObjRef sha1sum_ref;
    ObjRef sha256sum_ref;
    ObjRef sha512sum_ref;
    uint64 size;
    C_struct_SHA1sum sha1_obj;
    C_struct_File file_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    void_reference(ref);  /* in case we don't find it */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    if (public)
    {
	if (!nwos_find_public_class_definition("FILE", &file_class_ref)) return false;
    }
    else
    {
	if (!nwos_find_private_class_definition("FILE", &file_class_ref)) return false;
    }
#endif

    if (find_md5(public, md5_digest, &md5sum_ref) && find_sha1(public, sha1_digest, &sha1sum_ref))
    {
	if (verbose)
	{
	    printf("Found existing MD5: %02x%02x%02x%02x and SHA1: %02x%02x%02x%02x - ", 
		   md5sum_ref.id[0], md5sum_ref.id[1], md5sum_ref.id[2], md5sum_ref.id[3],
		   sha1sum_ref.id[0], sha1sum_ref.id[1], sha1sum_ref.id[2], sha1sum_ref.id[3]);
	}

	nwos_read_object_from_disk(&sha1sum_ref, &sha1_obj, sizeof(sha1_obj));
	ref_list = nwos_malloc_reference_list(&sha1_obj.header.object.references);
	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &file_class_ref))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &file_obj, sizeof(file_obj));

		assert(is_same_object(&sha1sum_ref, &file_obj.sha1sum));

		size = file_size_to_uint64(file_obj.size);

		if (verbose) printf("size: %llu\n", size);

		if (size == file_size && is_same_object(&file_obj.md5sum, &md5sum_ref))
		{
		    if (is_void_reference(&file_obj.sha256sum))   /* no SHA256 sum for this file */
		    {
			copy_reference(ref, &ref_list->references[i]);   /* so just go with it */
			break;
		    }
		    else   /* there is a SHA256 checksum for this file */
		    {
			if (find_sha256(public, sha256_digest, &sha256sum_ref) &&
			    is_same_object(&file_obj.sha256sum, &sha256sum_ref)) /* make sure it matches too */
			{
			    if (is_void_reference(&file_obj.sha512sum))   /* no SHA512 sum for this file */
			    {
				copy_reference(ref, &ref_list->references[i]);   /* so just go with it */
				break;
			    }
			    else   /* there is a SHA512 checksum for this file */
			    {
				if (find_sha512(public, sha512_digest, &sha512sum_ref) &&
				    is_same_object(&file_obj.sha512sum, &sha512sum_ref)) /* make sure it matches too */
				{
				    copy_reference(ref, &ref_list->references[i]);
				    break;
				}
			    }
			}
		    }
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;

	result = i < num_refs;
    }

    return result;
}


bool nwos_find_public_matching_file_from_size_md5_sha1(uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_matching_file_from_size_md5_sha1(true, file_size, md5_digest, sha1_digest, sha256_digest, sha512_digest, ref);
}


/* find an sha512 based upon what mode we are in */

bool nwos_find_matching_file_from_size_md5_sha1(uint64 file_size, uint8 md5_digest[MD5_DIGEST_SIZE], uint8 sha1_digest[SHA1_DIGEST_SIZE], uint8 sha256_digest[SHA256_DIGEST_SIZE], uint8 sha512_digest[SHA512_DIGEST_SIZE], ObjRef* ref)
{
    return find_matching_file_from_size_md5_sha1(nwos_in_public_mode(), file_size, md5_digest, sha1_digest, sha256_digest, sha512_digest, ref);
}



/* check to see if the file given has the data stored in the system */

bool nwos_file_is_stored(ObjRef* assoc_ref)
{
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_File file_obj;

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));

    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

    /* if the block_list is not empty then the data is stored */

    return !is_void_reference(&file_obj.block_list);
}


/* Create a file record with just an MD5 checksum, for Public Objects */

#ifdef PUBLIC_MODE
ObjCreateResult nwos_create_file_with_only_md5(const char* file_name, uint8 md5_digest[MD5_DIGEST_SIZE], ObjRef* ref)
{
    C_struct_File file_obj;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    TimeStamp mod_time;

    assert(nwos_find_public_class_definition("FILE", &file_class_ref));

    assert(nwos_create_file_path(file_name, &path_ref) == CREATED_NEW);

    nwos_generate_new_id(&file_ref);

    memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

    nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

    nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

    if (nwos_create_md5(md5_digest, &file_obj.md5sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: MD5 already existed: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
    }

    nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
    nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

    nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */

    nwos_add_to_references(&file_ref, &file_class_ref);

    memset(mod_time, 0, sizeof(mod_time));

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


ObjCreateResult nwos_add_info_to_existing_file(const char* file_name, ObjRef* assoc_ref)
{
    struct stat stat_struct;
    C_struct_Class_Definition class_def_obj;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_MD5sum md5_obj;
    ObjRef assoc_class_ref;
    ObjRef path_ref;
    ObjRef object_class;
    uint64 file_length;
    TimeStamp mod_time;
    ReferenceList* ref_list;
    int num_refs;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    int i;

    if (stat(file_name, &stat_struct) != 0)
    {
	perror(file_name);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }

    if (!nwos_find_file_path(file_name, &path_ref))
    {
	fprintf(stderr, "%s is not in the public objects", file_name);
	return ERROR_OCCURRED;
    }

    assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("association num_refs: %d\n", num_refs);

    void_reference(assoc_ref);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    if (is_same_object(&assoc_obj.path, &path_ref))
	    {
		assert(is_void_reference(assoc_ref));   /* error if there is more than one for now */

		copy_reference(assoc_ref, &ref_list->references[i]);
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    printf("Association: %08x\n", nwos_ref_to_word(assoc_ref));

    assert(!is_void_reference(assoc_ref));   /* path found but no association?? */

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));

    assert(is_same_object(&assoc_obj.path, &path_ref));

    memset(mod_time, 0, sizeof(mod_time));
    assert(memcmp(assoc_obj.modification_time, mod_time, sizeof(mod_time)) == 0);

    /* first put the modification time into the association object */

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, assoc_obj.modification_time);

    nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));

    /* now do the file stuff */
    
    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

    assert(!is_void_reference(&file_obj.md5sum));
    assert(is_void_reference(&file_obj.sha1sum));
    assert(is_void_reference(&file_obj.sha256sum));
    assert(is_void_reference(&file_obj.sha512sum));
    assert((file_obj.size[0] | file_obj.size[1] | file_obj.size[2] | file_obj.size[3] | file_obj.size[4]) == 0);


    printf("reading file: %s ", file_name);
    fflush(stdout);

    file_length = checksum_file(file_name, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    printf("length: %llu\n", file_length);

    nwos_log_md5sum(file_name, md5_digest);
    nwos_log_sha1sum(file_name, sha1_digest);
    nwos_log_sha256sum(file_name, sha256_digest);
    nwos_log_sha512sum(file_name, sha512_digest);

    nwos_read_object_from_disk(&file_obj.md5sum, &md5_obj, sizeof(md5_obj));

    if (memcmp(md5_obj.md5sum, md5_digest, sizeof(md5_digest)) != 0)
    {
	fprintf(stderr, "Error: md5sum of file doesn't match MD5 in file:\n");

	fprintf(stderr, "   ");
	for (i = 0; i < MD5_DIGEST_SIZE; i++) fprintf(stderr, "%02x", md5_obj.md5sum[i]);
	fprintf(stderr, " from MD5SUMS file\n");

	fprintf(stderr, "   ");
	for (i = 0; i < MD5_DIGEST_SIZE; i++) fprintf(stderr, "%02x", md5_digest[i]);
	fprintf(stderr, " %s\n", file_name);

	return ERROR_OCCURRED;
    }

    /* store file size as 4 bytes in big endian order */
    file_obj.size[0] = file_length >> 32;
    file_obj.size[1] = file_length >> 24;
    file_obj.size[2] = file_length >> 16;
    file_obj.size[3] = file_length >> 8;
    file_obj.size[4] = file_length;

    if (nwos_create_sha1(sha1_digest, &file_obj.sha1sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA1 already existed: %08x\n", nwos_ref_to_word(&file_obj.sha1sum));
    }

    if (nwos_create_sha256(sha256_digest, &file_obj.sha256sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA256 already existed: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
    }

    if (nwos_create_sha512(sha512_digest, &file_obj.sha512sum) != CREATED_NEW)
    {
	fprintf(stderr, "WARNING: SHA512 already existed: %08x\n", nwos_ref_to_word(&file_obj.md5sum));
    }

    nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

    nwos_overwrite_object_to_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));

    nwos_add_to_references(&assoc_obj.file, &file_obj.sha1sum);   /* add to this sha1sum object's reference list */
    nwos_add_to_references(&assoc_obj.file, &file_obj.sha256sum);  /* add to this sha256sum object's reference list */
    nwos_add_to_references(&assoc_obj.file, &file_obj.sha512sum);  /* add to this sha512sum object's reference list */

    return FOUND_EXISTING;
}
#endif


/* Create a file record (doesn't store data of file, just time, md5, etc.) */

ObjCreateResult nwos_create_file_without_storing_data(const char* directory, const char* file_name, ObjRef* ref)
{
    size_t length;
    struct stat stat_struct;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    uint64 file_length;
    C_struct_File file_obj;
    TimeStamp mod_time;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    char path[PATH_MAX];
#ifndef PUBLIC_MODE
    ObjRef public_file_ref;
    char msg[128];
#endif

    strncpy(path, directory, PATH_MAX);
    if (path[strlen(path) - 1] != '/')   /* the directory didn't end with a '/' so add one */
    {
	strncat(path, "/", PATH_MAX);
    }
    strncat(path, file_name, PATH_MAX);


    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }


    length = strlen(file_name);

    assert(length <= 255);
    
    printf("creating file record: %s ", file_name);
    fflush(stdout);

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    printf("length: %llu\n", file_length);

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    nwos_log_md5sum(file_name, md5_digest);
    nwos_log_sha1sum(file_name, sha1_digest);
    nwos_log_sha256sum(file_name, sha256_digest);
    nwos_log_sha512sum(file_name, sha512_digest);


    /* see if there is a file that matches completely */

    if (!nwos_find_matching_file_from_size_md5_sha1(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref))
    {
	nwos_generate_new_id(&file_ref);

	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
	nwos_find_or_create_private_class_definition("FILE", &file_class_ref);
#endif

	nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

#ifndef PUBLIC_MODE
	if (nwos_find_public_matching_file_from_size_md5_sha1(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &public_file_ref))
	{
	    copy_reference(&file_obj.header.object.clone_of, &public_file_ref);
	    snprintf(msg, sizeof(msg), "file - ref: %08x  clone_of: %08x",
		     nwos_ref_to_word(ref), nwos_ref_to_word(&public_file_ref));
	    nwos_log(msg);
	}
#endif
	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 32;
	file_obj.size[1] = file_length >> 24;
	file_obj.size[2] = file_length >> 16;
	file_obj.size[3] = file_length >> 8;
	file_obj.size[4] = file_length;

	nwos_create_md5(md5_digest, &file_obj.md5sum);

	nwos_create_sha1(sha1_digest, &file_obj.sha1sum);

	nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
	nwos_create_sha512(sha512_digest, &file_obj.sha512sum);

	nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);
	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to this sha1sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha256sum);  /* add to this sha256sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha512sum);  /* add to this sha512sum object's reference list */

	nwos_add_to_references(&file_ref, &file_class_ref);
    }

    /* at this point we have a file object, and we need to either find or create a path object */

    assert(nwos_create_file_path(file_name, &path_ref) != ERROR_OCCURRED);

    return nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}


static uint64 read_file(const char* path,
			uint8 md5_digest[MD5_DIGEST_SIZE],
			uint8 sha1_digest[SHA1_DIGEST_SIZE],
			uint8 sha256_digest[SHA256_DIGEST_SIZE],
			uint8 sha512_digest[SHA512_DIGEST_SIZE],
			ObjRef* block_list_ref)
{
    FILE* fp;
    size_t bytes_read;
    uint64 file_length;
    ObjRef data_block_ref;
    DataObject data_obj;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    uint8 ivec[IVEC_SIZE];
    int seq = 0;
    char msg[128];


    memset(ivec, 0, sizeof(ivec));

    /* create a reference list that is a list of the data blocks for the file */
    nwos_create_reference_list(NULL, block_list_ref);

    snprintf(msg, sizeof(msg), "read_file %s  block_list: %02x%02x%02x%02x", path,
	   block_list_ref->id[0], block_list_ref->id[1], block_list_ref->id[2], block_list_ref->id[3]);
    nwos_log(msg);


    /* now open the file and start creating the data block objects */

    fp = fopen(path, "rb");

    file_length = 0;

    md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
    sha1_init_ctx(&sha1_context);
    sha256_init_ctx(&sha256_context);
    sha512_init_ctx(&sha512_context);

    memset(data_obj.flags, 0, sizeof(data_obj.flags));

    bytes_read = DATA_STORAGE_SIZE;   /* go through loop at least once */

    while (bytes_read == DATA_STORAGE_SIZE)
    {
	bytes_read = fread(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp);

	if (bytes_read < DATA_STORAGE_SIZE)    /* eof or error */
	{
	    if (ferror(fp))
	    {
		perror(path);
		exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
	    }
	}

	file_length = file_length + bytes_read;

	if (bytes_read > 0)   /* we have data to store */
	{
	    md5_process_bytes(data_obj.storage, bytes_read, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(data_obj.storage, bytes_read, &sha1_context);    /* include this data in the sha1 checksum */
	    sha256_process_bytes(data_obj.storage, bytes_read, &sha256_context);    /* include this data in the sha256 checksum */
	    sha512_process_bytes(data_obj.storage, bytes_read, &sha512_context);    /* include this data in the sha512 checksum */

	    nwos_generate_new_id(&data_block_ref);

	    memcpy(&data_obj.id, &data_block_ref, sizeof(ObjRef));

#if 0
	    printf("data block seq: %d ref: %02x%02x%02x%02x\n", seq, data_block_ref.id[0], data_block_ref.id[1], data_block_ref.id[2], data_block_ref.id[3]);
#endif

	    nwos_crc32_calculate((uint8*) &data_obj.storage, bytes_read, data_obj.checksum);

	    /* fill in the remainder with random data */
	    while (bytes_read < DATA_STORAGE_SIZE) data_obj.storage[bytes_read++] = random();

	    nwos_write_object_to_disk_and_encrypt(&data_block_ref, &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[seq%NUM_STORED_SEQ]);

	    nwos_add_to_reference_list(&data_block_ref, block_list_ref);

	    seq++;
	}
    }

    fclose(fp);

    snprintf(msg, sizeof(msg), "length: %llu", file_length);
    nwos_log(msg);
    printf("%s\n", msg);

    md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
    sha1_finish_ctx(&sha1_context, sha1_digest);
    sha256_finish_ctx(&sha256_context, sha256_digest);
    sha512_finish_ctx(&sha512_context, sha512_digest);

    return file_length;
}



/* Create new file - returns FOUND_EXISTING if an exact file match already existed      */
/*          (exact match meaning size, md5, sha1, date, and path are all identical).    */
/*                                                                                      */
/* There are several possible cases of duplication that have to be dealt with:          */
/*   Path is the same but the file is different (file with same name as another file).  */
/*   File is the same but the name is different (copy of a file with a different name). */
/*   Path and File are the same but the date is different.                              */
/*   Everything is identical, but the file isn't stored in the system yet.              */
/*   Everything is identical.                                                           */
/*                                                                                      */
/*  Matching  Matching  Matching  Matching   Data    Allow      Action      Return                            */
/*    Path     Assoc      Time      File    Stored  Multiple                                                  */
/*     T         T         X         T        T        X      Import Data IMPORTED_DATA                       */
/*     T         T         X         T        F        X      Do Nothing  FOUND_EXISTING                      */

/*     T         F         X         T        X        T      Create Path CREATED_NEW_PATH_FOR_EXISTING_FILE  */
/*     T         F         X         T        X        F      Do Nothing  DUPLICATE_FILE                      */

/*     F         X         X         T        X        T      Create Path CREATED_NEW_PATH_FOR_EXISTING_FILE  */
/*     F         X         X         T        X        F      Do Nothing  DUPLICATE_FILE                      */

/*     T         X         X         F        X        X      Do Nothing  DUPLICATE_PATH                      */
/*     F         X         X         F        X        X      Create all  CREATED_NEW                         */
/*                                                                                                            */

ObjCreateResult nwos_create_file(const char* path, ObjRef* ref, bool allow_multiple_paths)
{
    size_t length;
    struct stat stat_struct;
    bool file_found;
    bool path_found;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    uint64 file_length;
    C_struct_File file_obj;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    uint8 md5_digest2[MD5_DIGEST_SIZE];
    uint8 sha1_digest2[SHA1_DIGEST_SIZE];
    uint8 sha256_digest2[SHA256_DIGEST_SIZE];
    uint8 sha512_digest2[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;

    void_reference(ref);   /* in case it fails */


    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1073741824)
    {
	fprintf(stderr, "Cannot store files larger than 1 gigabyte at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);


    /* first find out if we already have this file stored */

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    file_found = nwos_find_matching_file_from_size_md5_sha1(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref);


    /* second find out if we already have this path */

    path_found = nwos_find_file_path(path, &path_ref);


    if (file_found && path_found)
    {
	if (nwos_find_path_and_file_association(&path_ref, &file_ref, NULL, ref))
	{
	    nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj)); /* make sure we already have the data */

	    if (is_void_reference(&file_obj.block_list))   /* file data is not stored in the system */
	    {
		file_length = read_file(path, md5_digest2, sha1_digest2, sha256_digest2, sha512_digest2, &file_obj.block_list);

		/* verify that the checksums match what they were before */
		assert(memcmp(md5_digest, md5_digest2, sizeof(md5_digest)) == 0);
		assert(memcmp(sha1_digest, sha1_digest2, sizeof(sha1_digest)) == 0);
		assert(memcmp(sha256_digest, sha256_digest2, sizeof(sha256_digest)) == 0);
		assert(memcmp(sha512_digest, sha512_digest2, sizeof(sha512_digest)) == 0);

		if (is_void_reference(&file_obj.md5sum))
		{
		    nwos_create_md5(md5_digest, &file_obj.md5sum);
		    nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to the md5sum object's reference list */
		    nwos_log_md5sum(path, md5_digest);
		}

		if (is_void_reference(&file_obj.sha1sum))
		{
		    nwos_create_sha1(sha1_digest, &file_obj.sha1sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to the sha1sum object's reference list */
		    nwos_log_sha1sum(path, sha1_digest);
		}

		if (is_void_reference(&file_obj.sha256sum))
		{
		    nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha256sum); /* add to the sha256sum object's reference list */
		    nwos_log_sha256sum(path, sha256_digest);
		}

		if (is_void_reference(&file_obj.sha512sum))
		{
		    nwos_create_sha512(sha512_digest, &file_obj.sha512sum);
		    nwos_add_to_references(&file_ref, &file_obj.sha512sum); /* add to the sha512sum object's reference list */
		    nwos_log_sha512sum(path, sha512_digest);
		}

		nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

		nwos_overwrite_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

		return IMPORTED_DATA;
	    }

	    return FOUND_EXISTING;
	}

	if (allow_multiple_paths)
	{
	    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	    assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);

	    return CREATED_NEW_PATH_FOR_EXISTING_FILE;
	}

	return DUPLICATE_FILE;
    }
    else if (file_found)   /* matching file was found, but path was not */
    {
	if (allow_multiple_paths)
	{
	    assert(nwos_create_file_path(path, &path_ref) == CREATED_NEW);

	    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	    assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);

	    return CREATED_NEW_PATH_FOR_EXISTING_FILE;
	}

	return DUPLICATE_FILE;
    }
    else if (path_found)   /* matching path was found, but file was not */
    {
	return DUPLICATE_PATH;
    }
    else
    {
	printf("creating file: %s ", path);
	fflush(stdout);

	assert(nwos_create_file_path(path, &path_ref) == CREATED_NEW);

	memset(&file_obj, 0, sizeof(file_obj));  /* zero it out */

	nwos_find_or_create_private_class_definition("FILE", &file_class_ref);

	nwos_generate_new_id(&file_ref);

	nwos_fill_in_common_header(&file_obj.header.common, &file_ref, &file_class_ref);

	file_length = read_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_obj.block_list);

	nwos_create_md5(md5_digest, &file_obj.md5sum);
	nwos_create_sha1(sha1_digest, &file_obj.sha1sum);
	nwos_create_sha256(sha256_digest, &file_obj.sha256sum);
	nwos_create_sha512(sha512_digest, &file_obj.sha512sum);

	nwos_add_to_references(&file_ref, &file_obj.md5sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha1sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha256sum);   /* add to this md5sum object's reference list */
	nwos_add_to_references(&file_ref, &file_obj.sha512sum);   /* add to this md5sum object's reference list */

	nwos_log_md5sum(path, md5_digest);
	nwos_log_sha1sum(path, sha1_digest);
	nwos_log_sha256sum(path, sha256_digest);
	nwos_log_sha512sum(path, sha512_digest);


	/* store file size as 4 bytes in big endian order */
	file_obj.size[0] = file_length >> 32;
	file_obj.size[1] = file_length >> 24;
	file_obj.size[2] = file_length >> 16;
	file_obj.size[3] = file_length >> 8;
	file_obj.size[4] = file_length;

	nwos_create_reference_list(&file_ref, &file_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &file_obj.header.object, sizeof(ObjectHeader), file_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&file_ref, &file_obj, sizeof(file_obj));

	nwos_add_to_references(&file_ref, &file_class_ref);

#if 0
	ptr_file_obj = malloc(sizeof(C_struct_File) + num_blocks * sizeof(ObjRef));
	nwos_read_object_from_disk(ref, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef));
	assert(memcmp(kludge, ptr_file_obj, sizeof(C_struct_File) + num_blocks * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_file_object_size);  /* read the other way */
	assert(memcmp(ptr_file_obj, kludge, sizeof(C_struct_File_Path) + length) == 0);
#endif

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);
    }

    return CREATED_NEW;
}


/* Add new revision of existing file - returns DUPLICATE_FILE if an exact file match already existed */
/*          (exact match meaning size, md5, sha1, date, and path are all identical).    */

ObjCreateResult nwos_add_new_revision_of_file(const char* path, ObjRef* ref)
{
    size_t length;
    struct stat stat_struct;
    ObjRef assoc_class_ref;
    ObjRef file_class_ref;
    ObjRef path_ref;
    ObjRef file_ref;
    EveryObject header;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    ObjRef object_class;
    uint64 file_length;
    C_struct_File new_file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association old_assoc_obj;
    C_struct_Path_And_File_Association new_assoc_obj;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;

    void_reference(ref);   /* in case it fails */


    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1073741824)
    {
	fprintf(stderr, "Cannot handle files larger than 1 gigabyte at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);


    /* first find out if we already have this file stored */

    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    if (nwos_find_matching_file_from_size_md5_sha1(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref))
    {
	return DUPLICATE_FILE;
    }


    /* second find out if we already have this path */

    if (!nwos_find_file_path(path, &path_ref))   /* this file must already exist */
    {
	return PATH_NOT_FOUND;
    }

    nwos_read_object_headers_from_disk(&path_ref, &header);

    ref_list = nwos_malloc_reference_list(&header.object.references);
    num_refs = ref_list->common_header.num_refs;

    /* Must make sure the File class is up to date before searching for the old file, */
    /* in case it is of the old flavor (it has to be updated before rewritting it.    */

    nwos_find_or_create_private_class_definition("FILE", &file_class_ref);

    assert(nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));

    memset(&old_assoc_obj, 0, sizeof(old_assoc_obj));  /* zero it out */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    assert(is_same_object(&assoc_obj.path, &path_ref));

	    if (is_void_reference(&assoc_obj.header.object.next_version))
	    {
		if (!is_void_reference(&old_assoc_obj.header.common.id))  /* already found a file */
		{
		    fprintf(stderr, "Cannot handle multiple files with the same name at this time\n");
		    return MULTIPLE_FILES;
		}

		memcpy(&old_assoc_obj, &assoc_obj, sizeof(old_assoc_obj));
	    }
	}
    }

    printf("adding file revision: %s\n", path);

    printf("previous assoc: %08x  time: %s\n", nwos_ref_to_word(&old_assoc_obj.header.common.id),
	   nwos_time_stamp_to_string(old_assoc_obj.modification_time));

    nwos_generate_new_id(&file_ref);

    printf("new file: %08x  ", nwos_ref_to_word(&file_ref));
    fflush(stdout);

    memset(&new_file_obj, 0, sizeof(new_file_obj));  /* zero it out */

    nwos_fill_in_common_header(&new_file_obj.header.common, &file_ref, &file_class_ref);

    file_length = read_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest, &new_file_obj.block_list);

    nwos_create_md5(md5_digest, &new_file_obj.md5sum);
    nwos_create_sha1(sha1_digest, &new_file_obj.sha1sum);
    nwos_create_sha256(sha256_digest, &new_file_obj.sha256sum);
    nwos_create_sha512(sha512_digest, &new_file_obj.sha512sum);

    nwos_add_to_references(&file_ref, &new_file_obj.md5sum);   /* add to this md5sum object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha1sum);   /* add to this md5sum object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha256sum);   /* add to this sha256 object's reference list */
    nwos_add_to_references(&file_ref, &new_file_obj.sha512sum);   /* add to this sha512 object's reference list */

    nwos_log_md5sum(path, md5_digest);
    nwos_log_sha1sum(path, sha1_digest);
    nwos_log_sha256sum(path, sha256_digest);
    nwos_log_sha512sum(path, sha512_digest);


    /* store file size as 4 bytes in big endian order */
    new_file_obj.size[0] = file_length >> 32;
    new_file_obj.size[1] = file_length >> 24;
    new_file_obj.size[2] = file_length >> 16;
    new_file_obj.size[3] = file_length >> 8;
    new_file_obj.size[4] = file_length;

    nwos_create_reference_list(&file_ref, &new_file_obj.header.object.references);

    nwos_crc32_calculate((uint8*) &new_file_obj.header.object, sizeof(ObjectHeader), new_file_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &new_file_obj.size[0], sizeof(C_struct_File) - sizeof(EveryObject), new_file_obj.header.common.data_chksum);

    nwos_write_object_to_disk(&file_ref, &new_file_obj, sizeof(new_file_obj));

    nwos_add_to_references(&file_ref, &new_file_obj.header.common.class_definition);

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    assert(nwos_find_or_create_path_and_file_association(&path_ref, &file_ref, mod_time, ref) == CREATED_NEW);

    /* update the old association object's next_version */

    copy_reference(&old_assoc_obj.header.object.next_version, ref);

    nwos_crc32_calculate((uint8*) &old_assoc_obj.header.object, sizeof(ObjectHeader), old_assoc_obj.header.common.header_chksum);

    nwos_overwrite_object_to_disk(&old_assoc_obj.header.common.id, &old_assoc_obj, sizeof(old_assoc_obj));

    /* update the new association object's prev_version */

    nwos_read_object_from_disk(ref, &new_assoc_obj, sizeof(new_assoc_obj));

    copy_reference(&new_assoc_obj.header.object.prev_version, &old_assoc_obj.header.common.id);

    nwos_crc32_calculate((uint8*) &new_assoc_obj.header.object, sizeof(ObjectHeader), new_assoc_obj.header.common.header_chksum);

    nwos_overwrite_object_to_disk(ref, &new_assoc_obj, sizeof(new_assoc_obj));

    return CREATED_NEW_REVISION;
}


/* Check a file's md5sum */

CheckFileResult nwos_check_file_checksums(const char* path)
{
    size_t length;
    struct stat stat_struct;
    FILE* fp;
    size_t bytes_read;
    ObjRef path_ref;
    ObjRef file_ref;
    uint8 buffer[4096];
    uint64 file_length;
    int num_blocks;
    C_struct_File file_obj;
    struct md5_ctx md5_context;    /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    struct sha256_ctx sha256_context;
    struct sha512_ctx sha512_context;
    C_struct_MD5sum md5_object;
    C_struct_SHA1sum sha1_object;
    C_struct_SHA256sum sha256_object;
    C_struct_SHA512sum sha512_object;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    TimeStamp mod_time;
    CheckFileResult result = File_Not_Found;

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return ERROR_OCCURRED;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return ERROR_OCCURRED;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return ERROR_OCCURRED;
    }

    length = strlen(path);

    assert(length <= 255);
    
    /* first find out if we already have this path */

    assert(false);   /* need to fix file_path to assoc below */

    if (nwos_find_file_path(path, &path_ref))   /* it's ok, found the file */
    {
#if 0
	nwos_file_path_to_file(&path_ref, &file_ref);
#endif
	nwos_read_object_from_disk(&file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

	file_length = file_size_to_uint64(file_obj.size);

	printf("file size: %llu\n", file_length);
	/* now open the file and start creating the data block objects */

	fp = fopen(path, "rb");

	file_length = 0;
	num_blocks = 0;

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
	sha1_init_ctx(&sha1_context);   /* and the SHA1 context */
	sha256_init_ctx(&sha256_context);   /* and the SHA256 context */
	sha512_init_ctx(&sha512_context);   /* and the SHA512 context */

	bytes_read = sizeof(buffer);   /* go through loop at least once */

	while (bytes_read == sizeof(buffer))
	{
	    bytes_read = fread(buffer, sizeof(uint8), sizeof(buffer), fp);

	    if (bytes_read < sizeof(buffer))    /* eof or error */
	    {
		if (ferror(fp))
		{
		    perror(path);
		    exit(1);            /* FIX THIS - needs to clean up and exit gracefully!! */
		}
	    }

	    file_length = file_length + bytes_read;

	    if (bytes_read > 0)   /* we have data to store */
	    {
		md5_process_bytes(buffer, bytes_read, &md5_context);    /* include this data in the md5 checksum */
		sha1_process_bytes(buffer, bytes_read, &sha1_context);    /* and in the sha1 checksum */
		sha256_process_bytes(buffer, bytes_read, &sha256_context);    /* and in the sha256 checksum */
		sha512_process_bytes(buffer, bytes_read, &sha512_context);    /* and in the sha512 checksum */
	    }
	}

	fclose(fp);

	printf("length: %llu\n", file_length);

	md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
	sha1_finish_ctx(&sha1_context, sha1_digest);  /* and the sha1 sum */
	sha256_finish_ctx(&sha256_context, sha256_digest);  /* and the sha256 sum */
	sha512_finish_ctx(&sha512_context, sha512_digest);  /* and the sha512 sum */


	fflush(stdout);

	if (file_length != file_size_to_uint64(file_obj.size))
	{
	    printf("file size mismatch\n");
	}

	nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

	assert(false);
#if 0
	if (memcmp(file_obj.modification_time, mod_time, sizeof(TimeStamp)) != 0)
	{
	    printf("time mismatch\n");
	}
#endif
	nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

	if (memcmp(md5_digest, md5_object.md5sum, sizeof(md5_digest)) == 0)
	{
	    int j;
	    printf("MD5 sum OK: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
	    printf("\n");

	    result = Checksums_Match;    /* found file and MD5 was ok */
	}
	else
	{
	    int j;
	    printf("MD5 checksum error, expected: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
	    printf("\n");

	    result = MD5_Sum_Mismatch;    /* found file but MD5 was bad */
	}


	nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object));

	if (memcmp(sha1_digest, sha1_object.sha1sum, sizeof(sha1_digest)) == 0)
	{
	    int j;
	    printf("SHA1 sum OK: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA1 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA1 checksum error, expected: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA1_Sum_Mismatch;    /* found file but SHA1 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}


	nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_object, sizeof(sha256_object));

	if (memcmp(sha256_digest, sha256_object.sha256sum, sizeof(sha256_digest)) == 0)
	{
	    int j;
	    printf("SHA256 sum OK: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA256 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA256 checksum error, expected: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_object.sha256sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA256_Sum_Mismatch;    /* found file but SHA256 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}


	nwos_read_object_from_disk(&file_obj.sha512sum, &sha512_object, sizeof(sha512_object));

	if (memcmp(sha512_digest, sha512_object.sha512sum, sizeof(sha512_digest)) == 0)
	{
	    int j;
	    printf("SHA512 sum OK: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_digest[j]);
	    printf("\n");

	    /* MD5 compare set result, since SHA512 was ok leave it */
	}
	else
	{
	    int j;
	    printf("SHA512 checksum error, expected: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_object.sha512sum[j]);
	    printf("\n                    received: ");
	    for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_digest[j]);
	    printf("\n");

	    if (result == Checksums_Match)
	    {
		result = SHA512_Sum_Mismatch;    /* found file but SHA512 was bad */
	    }
	    else
	    {
		result = Checksums_Mismatch;
	    }
	}
    }

    return result;   /* didn't find file */
}



/* Read file */

bool nwos_restore_file(ObjRef* assoc_ref, const char* path)
{
    bool result = true;
    FILE* fp;
    int i;
    uint64 file_length;
    uint32 num_blocks;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    DataObject data_obj;
    struct timeval tv[2];      /* [0] is access time and [1] is modification time, see man utimes */
    struct md5_ctx md5_context;           /* MD5 checksum context */
    struct sha1_ctx sha1_context;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    C_struct_MD5sum md5_object;
    C_struct_SHA1sum sha1_object;
    ReferenceList* ref_list_ptr;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];

    memset(ivec, 0, sizeof(ivec));

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));  /* read the path and file assoc. object */

    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));  /* read the file object */

    file_length = file_size_to_uint64(file_obj.size);

    printf("file size: %llu\n", file_length);

    ref_list_ptr = nwos_malloc_reference_list(&file_obj.block_list);

    num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

    assert(num_blocks == ref_list_ptr->common_header.num_refs);

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);


    /* now open the file and start writing the data block objects */

    fp = fopen(path, "wb");

    if (fp == NULL)
    {
	perror(path);
	exit(1);
    }
    else  /* file opened ok */
    {
	i = 0;

	md5_init_ctx(&md5_context);   /* initialize the MD5 checksum context */
	sha1_init_ctx(&sha1_context);   /* and the SHA1 context */

	while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
	{
	    assert(nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]));

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		printf("checksum error in file block - sequence: %d\n", i);
		exit(1);
	    }

	    md5_process_bytes(data_obj.storage, DATA_STORAGE_SIZE, &md5_context);    /* include this data in the md5 checksum */
	    sha1_process_bytes(data_obj.storage, DATA_STORAGE_SIZE, &sha1_context);    /* and in the sha1 checksum */

	    if (fwrite(data_obj.storage, sizeof(uint8), DATA_STORAGE_SIZE, fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}

	assert(nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]));

	nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	{
	    printf("checksum error in file block - sequence: %d\n", i);
	    exit(1);
	}

	md5_process_bytes(data_obj.storage, file_length, &md5_context);    /* include this data in the md5 checksum */
	sha1_process_bytes(data_obj.storage, file_length, &sha1_context);    /* and in the sha1 checksum */

	if (fwrite(data_obj.storage, sizeof(uint8), file_length, fp) != file_length)
	{
	    perror(path);
	    exit(1);
	}

	if (fclose(fp) != 0)
	{
	    perror(path);
	    exit(1);
	}

	if (result)   /* the file was ok, restore it's modification time */
	{
	    nwos_convert_time_stamp_to_timeval(assoc_obj.modification_time, &tv[1]);
	    tv[0] = tv[1];  /* copy modification time into access time since we don't save the access time */
	    utimes(path, tv);
	}

	md5_finish_ctx(&md5_context, md5_digest);   /* finish computing the md5 sum */
	sha1_finish_ctx(&sha1_context, sha1_digest);  /* and the sha1 sum */

	if (result)
	{
	    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

	    if (memcmp(md5_digest, md5_object.md5sum, sizeof(md5_digest)) == 0)
	    {
		int j;
		printf("MD5 sum OK: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("MD5 checksum error, expected: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_digest[j]);
		printf("\n");

		result = false;
	    }

	    nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object));

	    if (memcmp(sha1_digest, sha1_object.sha1sum, sizeof(sha1_digest)) == 0)
	    {
		int j;
		printf("SHA1 sum OK: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
		printf("\n");
	    }
	    else
	    {
		int j;
		printf("SHA1 checksum error, expected: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
		printf("\n                    received: ");
		for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_digest[j]);
		printf("\n");

		result = false;
	    }
	}

	nwos_free_reference_list(ref_list_ptr);
	ref_list_ptr = NULL;
    }

    return result;
}


/* Compare file - does not verify checksums (just compares the data) */

bool nwos_file_is_identical(ObjRef* assoc_ref, const char* path, MatchCode match)
{
    bool result = true;
    FILE* fp;
    int i;
    struct stat stat_struct;
    uint64 file_length;
    uint32 num_blocks;
    C_struct_File file_obj;
    C_struct_Path_And_File_Association assoc_obj;
    DataObject data_obj;
    struct timeval tv;
    ReferenceList* ref_list_ptr;
    uint8 ivec[IVEC_SIZE];
    uint8 chksum[4];
    uint8 buffer[DATA_STORAGE_SIZE];

    memset(ivec, 0, sizeof(ivec));

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return false;
    }

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));  /* read the path and file assoc. object */

    nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));  /* read the file object */

    file_length = file_size_to_uint64(file_obj.size);

    printf("file size: %llu\n", file_length);

    if (stat_struct.st_size != file_length)
    {
	fprintf(stderr, "File size mismatch\n");
	return false;
    }

    ref_list_ptr = nwos_malloc_reference_list(&file_obj.block_list);

    num_blocks = (file_length + DATA_STORAGE_SIZE - 1) / DATA_STORAGE_SIZE;

    assert(num_blocks == ref_list_ptr->common_header.num_refs);

//    printf("num_blocks: %u  block_list: %02x%02x%02x%02x\n", num_blocks,
//	   file_obj.block_list.id[0],
//	   file_obj.block_list.id[1],
//	   file_obj.block_list.id[2],
//	   file_obj.block_list.id[3]);


    /* now open the file and start writing the data block objects */

    fp = fopen(path, "rb");

    if (fp == NULL)
    {
	perror(path);
	exit(1);
    }
    else  /* file opened ok */
    {
	i = 0;

	while (file_length > DATA_STORAGE_SIZE)   /* do all except the last block */
	{
	    assert(nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]));

	    nwos_crc32_calculate((uint8*) &data_obj.storage, sizeof(data_obj.storage), chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		printf("checksum error in file block - sequence: %d\n", i);
		exit(1);
	    }

	    if (fread(buffer, sizeof(uint8), sizeof(buffer), fp) != DATA_STORAGE_SIZE)
	    {
		perror(path);
		exit(1);
	    }

	    if (memcmp(&data_obj.storage, buffer, DATA_STORAGE_SIZE) != 0)
	    {
		result = false;
		break;
	    }

	    file_length -= DATA_STORAGE_SIZE;

	    i++;
	}


	if (result)   /* file compared ok thus far */
	{
	    assert(nwos_read_object_from_disk_and_decrypt(&ref_list_ptr->references[i], &data_obj, sizeof(data_obj), ivec, nwos_random_sequence[i%NUM_STORED_SEQ]));

	    nwos_crc32_calculate((uint8*) &data_obj.storage, file_length, chksum);

	    if (memcmp(chksum, data_obj.checksum, sizeof(chksum)) != 0)
	    {
		printf("checksum error in file block - sequence: %d\n", i);
		exit(1);
	    }

	    if (fread(buffer, sizeof(uint8), file_length, fp) != file_length)
	    {
		perror(path);
		exit(1);
	    }

	    if (memcmp(&data_obj.storage, buffer, file_length) != 0)
	    {
		result = false;
	    }

	    if (fclose(fp) != 0)
	    {
		perror(path);
		exit(1);
	    }
	}

	if (result && match != IgnoreTime)   /* the file was ok, compare it's modification time */
	{
	    nwos_convert_time_stamp_to_timeval(assoc_obj.modification_time, &tv);

	    if (stat_struct.st_mtimespec.tv_sec != tv.tv_sec || stat_struct.st_mtimespec.tv_nsec / 1000 != tv.tv_usec)
	    {
		printf("Time values mismatch - file %ld.%ld  stored: %ld.%ld\n",
		       stat_struct.st_mtimespec.tv_sec, stat_struct.st_mtimespec.tv_nsec / 1000, tv.tv_sec, tv.tv_usec);
		result = false;
	    }
	}
    }

    nwos_free_reference_list(ref_list_ptr);
    ref_list_ptr = NULL;

    return result;
}


/* Returns true if the file is stored on some backup media */

bool nwos_file_has_backup(ObjRef* assoc_ref)
{
    ObjRef disc_list_class_ref;
    ObjRef disc_copy_class_ref;
    ObjRef class_ref;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Disc_Copy copy_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ReferenceList* assoc_ref_list_ptr;
    ReferenceList* disc_ref_list_ptr;
    int num_assoc_refs;
    int num_disc_refs;
    int i;
    int j;
    int count;

    if (!nwos_find_private_class_definition("DISC LIST", &disc_list_class_ref)) return false;
    if (!nwos_find_private_class_definition("DISC COPY", &disc_copy_class_ref)) return false;

    nwos_read_object_from_disk(assoc_ref, &assoc_obj, sizeof(assoc_obj));  /* read the path and file assoc. object */

    assoc_ref_list_ptr = nwos_malloc_reference_list(&assoc_obj.header.object.references);

    num_assoc_refs = assoc_ref_list_ptr->common_header.num_refs;

    for (i = 0; i < num_assoc_refs; i++)
    {
	nwos_get_object_class(&assoc_ref_list_ptr->references[i], &class_ref);

	if (is_same_object(&class_ref, &disc_list_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&assoc_ref_list_ptr->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    disc_ref_list_ptr = nwos_malloc_reference_list(&ptr_disc_obj->header.object.references);

	    /* as long as we had to read the object, verify that this file is in the list */

	    count = nwos_decode_variable_sized_count(ptr_disc_obj->count);
	    for (j = 0; j < count; j++)
	    {
		if (is_same_object(assoc_ref, &ptr_disc_obj->files[j])) break;
	    }
	    assert(j < count);   /* something very wrong if it wasn't in the list */

	    /* now find any disc copies that exist */

	    num_disc_refs = disc_ref_list_ptr->common_header.num_refs;

	    for (j = 0; j < num_disc_refs; j++)
	    {
		nwos_get_object_class(&disc_ref_list_ptr->references[j], &disc_copy_class_ref);

		if (is_same_object(&class_ref, &disc_list_class_ref))
		{
		    nwos_read_object_from_disk(&disc_ref_list_ptr->references[j], &copy_obj, sizeof(copy_obj));
		    assert(is_same_object(&assoc_ref_list_ptr->references[i], &copy_obj.disc_list));
		    break;
		}
	    }
	    
	    nwos_free_reference_list(disc_ref_list_ptr);
	    disc_ref_list_ptr = NULL;

	    if (j < num_disc_refs) break;  /* found a disc_copy in references */
	}
    }

    nwos_free_reference_list(assoc_ref_list_ptr);
    assoc_ref_list_ptr = NULL;

    return i < num_assoc_refs;;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* File association (Path <-> File) object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

bool nwos_find_path_and_file_association_by_index(ObjRef* path_ref, ObjRef* file_ref, int index, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int count;

    void_reference(ref);  /* in case we don't find it */


    /* for now just return false if there is no private class definition yet */

    if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;


    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    count = 0;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    if (is_same_object(&assoc_obj.file, file_ref) && is_same_object(&assoc_obj.path, path_ref))   /* found a match */
	    {
		if (count == index)
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
		else
		{
		    count++;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


static bool find_path_and_file_association(bool public, ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    ObjRef assoc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now just return false if there is no private class definition yet */

#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));
#else
    if (public)
    {
	assert(nwos_reference_type(path_ref) == Public_Reference);
	assert(nwos_reference_type(file_ref) == Public_Reference);
	if (!nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;
    }
    else
    {
	assert(nwos_reference_type(path_ref) == Private_Reference);
	assert(nwos_reference_type(file_ref) == Private_Reference);
	if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref)) return false;
    }
#endif

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    if (is_same_object(&assoc_obj.file, file_ref) && is_same_object(&assoc_obj.path, path_ref))   /* found a match */
	    {
		if (mod_time == NULL || memcmp(assoc_obj.modification_time, mod_time, sizeof(mod_time)) == 0)
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


bool nwos_find_public_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    assert(nwos_reference_type(path_ref) == Public_Reference);
    assert(nwos_reference_type(file_ref) == Public_Reference);

    return find_path_and_file_association(true, path_ref, file_ref, mod_time, ref);
}


bool nwos_find_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    return find_path_and_file_association(nwos_in_public_mode(), path_ref, file_ref, mod_time, ref);
}



bool nwos_find_matching_path_and_file_association(const char* dir, const char* file_name, ObjRef* ref, MatchCode match)
{
    size_t length;
    struct stat stat_struct;
    uint64 file_length;
    uint8 md5_digest[MD5_DIGEST_SIZE];
    uint8 sha1_digest[SHA1_DIGEST_SIZE];
    uint8 sha256_digest[SHA256_DIGEST_SIZE];
    uint8 sha512_digest[SHA512_DIGEST_SIZE];
    ObjRef path_ref;
    ObjRef file_ref;
    TimeStamp mod_time;
    bool path_found;
    bool file_found;
    char path[PATH_MAX];

    void_reference(ref);  /* in case we don't find it */

    if (dir == NULL)      /* no directory specified */
    {
	strncpy(path, file_name, PATH_MAX);
    }
    else
    {
	strncpy(path, dir, PATH_MAX);
	if (path[strlen(path) - 1] != '/')   /* the directory didn't end with a '/' so add one */
	{
	    strncat(path, "/", PATH_MAX);
	}
	strncat(path, file_name, PATH_MAX);
    }

    if (stat(path, &stat_struct) != 0)
    {
	perror(path);
	return false;       /* FIX THIS to handle this more gracefully */
    }

    if (stat_struct.st_size == 0)
    {
	fprintf(stderr, "Cannot handle empty files at this time\n");
	return false;
    }

    if (stat_struct.st_size > 1099511627775LL)
    {
	fprintf(stderr, "Cannot handle files larger than 1 terabyte at this time\n");
	return false;
    }


    length = strlen(file_name);

    assert(length <= 255);
    
    file_length = checksum_file(path, md5_digest, sha1_digest, sha256_digest, sha512_digest);

    path_found = nwos_find_file_path(file_name, &path_ref);

    file_found = nwos_find_matching_file_from_size_md5_sha1(file_length, md5_digest, sha1_digest, sha256_digest, sha512_digest, &file_ref);

    printf("(found path: %s  file: %s) ", path_found ? "yes" : "no", file_found ? "yes" : "no");

    if (!path_found || !file_found) return false;

    if (match == IgnoreTime)
    {
	return nwos_find_path_and_file_association(&path_ref, &file_ref, NULL, ref);
    }

    nwos_convert_timespec_to_time_stamp(&stat_struct.st_mtimespec, mod_time);

    return nwos_find_path_and_file_association(&path_ref, &file_ref, mod_time, ref);
}



/* Find existing file association or create new */

ObjCreateResult nwos_find_or_create_path_and_file_association(ObjRef* path_ref, ObjRef* file_ref, TimeStamp mod_time, ObjRef* ref)
{
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association temp;
    ObjRef assoc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
#ifndef PUBLIC_MODE
    EveryObject path_header;
    EveryObject file_header;
    ObjRef public_assoc_ref;
    char msg[128];
#endif


    /* first find out if we already have this association */

    if (!nwos_find_path_and_file_association(path_ref, file_ref, mod_time, ref))   /* didn't find existing association */
    {
	memset(&assoc_obj, 0, sizeof(assoc_obj));  /* zero it out */

#ifdef PUBLIC_MODE
	assert(nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref));
#else
	nwos_find_or_create_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref);
#endif

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&assoc_obj.header.common, ref, &assoc_class_ref);

#ifndef PUBLIC_MODE
	nwos_read_object_headers_from_disk(path_ref, &path_header);
	nwos_read_object_headers_from_disk(file_ref, &file_header);
	if (!is_void_reference(&path_header.object.clone_of) && !is_void_reference(&file_header.object.clone_of) &&
	    nwos_find_public_path_and_file_association(&path_header.object.clone_of, &file_header.object.clone_of, NULL, &public_assoc_ref))
	{
	    copy_reference(&assoc_obj.header.object.clone_of, &public_assoc_ref);
	    snprintf(msg, sizeof(msg), "Create path and file association - path: %08x  file: %08x  assoc: %08x",
		     nwos_ref_to_word(path_ref), nwos_ref_to_word(file_ref), nwos_ref_to_word(ref));
	    nwos_log(msg);
	    snprintf(msg, sizeof(msg), "                          public - path: %08x  file: %08x  assoc: %08x",
		     nwos_ref_to_word(&path_header.object.clone_of), nwos_ref_to_word(&file_header.object.clone_of),
		     nwos_ref_to_word(&public_assoc_ref));
	    nwos_log(msg);
	}
#endif
	copy_reference(&assoc_obj.path, path_ref);
	copy_reference(&assoc_obj.file, file_ref);
	memcpy(assoc_obj.modification_time, mod_time, sizeof(assoc_obj.modification_time));

	nwos_create_reference_list(ref, &assoc_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &assoc_obj.header.object, sizeof(ObjectHeader), assoc_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &assoc_obj.path, sizeof(C_struct_Path_And_File_Association) - sizeof(EveryObject), assoc_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &assoc_obj, sizeof(assoc_obj));

	nwos_add_to_references(ref, &assoc_class_ref);
	nwos_add_to_references(ref, file_ref);
	nwos_add_to_references(ref, path_ref);

	/* read back to verify */
	nwos_read_object_from_disk(ref, &temp, sizeof(temp));
	assert(memcmp(&temp, &assoc_obj, sizeof(C_struct_Path_And_File_Association)) == 0);

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc List object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/

size_t nwos_get_disc_list_object_size(void* disc_list_obj)
{
    uint32 count = nwos_decode_variable_sized_count(((C_struct_Disc_List*)disc_list_obj)->count);

    assert(0 < count && count <= MAX_FILES_PER_DISC_LIST);

    return sizeof(C_struct_Disc_List) + count * sizeof(ObjRef);
}


bool nwos_find_disc_list(char id[12], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if the private class doesn't exist */

    if (!nwos_find_private_class_definition("DISC LIST", &disc_class_ref)) return false;


    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (memcmp(ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id)) == 0)   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}


/* returns number of files added (zero if none) or:                                                 */
/*   -1 if empty directories were found,                                                            */
/*   -2 if empty files were found,                                                                  */
/*   -3 if empty files and directories were found,                                                  */
/*   -4 if invalid files (symbolic links, devices, etc.) were found,                                */
/*   -5 if both empty directories and invalid files were found,                                     */
/*   -6 if both empty and invalid files were found,                                                 */
/*   -7 if empty files, invalid files, and empty directories were found,                            */
/*   -8 if maximum files were reached,                                                              */
/*   -9 if maximum files were reached and empty directories were found,                             */
/*  -10 if maximum files were reached and empty files were found,                                   */
/*  -11 if maximum files were reached and empty files and directories were found,                   */
/*  -12 if maximum files were reached and invalid files (symbolic links, devices, etc.) were found, */
/*  -13 if maximum files were reached and empty directories and invalid files were found.           */
/*  -14 if maximum files were reached and empty and invalid files were found,                       */
/*  -15 if maximum files were reached, empty directories and empty and invalid files were found.    */

int nwos_read_files_disc_list(const char* root_dir, const char* subdirectory, char* file_names[MAX_FILES_PER_DISC_LIST], int num_files)
{
    DIR* dp;
    struct dirent *dir_entry;
    char path[PATH_MAX];
    char* subpath;
    size_t path_len;
    struct stat stat_struct;
    const char* invalid_type;
    int call_result;
    int result = 0;


    assert(root_dir != NULL && root_dir[0] != '\0');

    path_len = strlcpy(path, root_dir, sizeof(path));

    if (path_len + 1 >= sizeof(path))
    {
	fprintf(stderr, "root directory: %s is too long for buffer size: %d\n", root_dir, PATH_MAX);
	exit(1);
    }

    if (path[path_len - 1] != '/')   /* if there wasn't a slash at the end of the root directory, add one */
    {
	path[path_len] = '/';
	path_len++;
	path[path_len] = '\0';
    }

    subpath = &path[path_len];   /* save a pointer to just the file or subdirectory */

    if (subdirectory != NULL)
    {
	path_len = strlcat(path, subdirectory, sizeof(path));

	if (path_len >= sizeof(path))
	{
	    fprintf(stderr, "combined root directory: %s and subdirectory: %s are too long for buffer size: %d\n",
		    root_dir, subdirectory, PATH_MAX);
	}
    }

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    assert(path_len == strlen(path));

	    if (path[path_len - 1] != '/')   /* if there isn't a slash at the end of the path, add one */
	    {
		path[path_len] = '/';
		path[path_len + 1] = '\0';
	    }

	    if (strlcat(path, dir_entry->d_name, sizeof(path)) >= sizeof(path))
	    {
		fprintf(stderr, "combined path: %s and %s are too long for buffer size: %d\n",
			subdirectory, dir_entry->d_name, PATH_MAX);
	    }

	    if (lstat(path, &stat_struct) != 0)
	    {
		perror(path);
		exit(1);
	    }

	    invalid_type = nwos_check_invalid_type(stat_struct.st_mode);

	    if (invalid_type != NULL)
	    {
		fprintf(stderr, "ERROR: '%s' is a %s and cannot be in a disc list!\n", path, invalid_type);

		if (result < 0)   /* already had an error */
		{
		    result = -(-result | 4);  /* merge in a -4 */
		}
		else              /* this is the first error */
		{
		    result = -4;
		}
	    }
	    else if (S_ISDIR(stat_struct.st_mode))
	    {
		if (result < 0)       /* an error has already occurred in this invocation */
		{
		    call_result = nwos_read_files_disc_list(root_dir, subpath, file_names, result);  /* keep checking */

		    if (call_result > 0)          /* normal return */
		    {
			                                       /* no changes to result, since it is already an error */
		    }
		    else if (call_result < 0)     /* error return */
		    {
			result = -(-result | -call_result);    /* combine the errors (-3 and -5 results in -7) */
		    }
		    else                          /* empty directory */
		    {
			result = -(-result | 1);               /* merge in empty directory error (-1) */
		    }
		}
		else
		{
		    if (num_files < 0)       /* some error has occurred prior to this invocation */
		    {
			call_result = nwos_read_files_disc_list(root_dir, subpath, file_names, num_files);  /* keep checking */
		    }
		    else                     /* no errors have occurred yet */
		    {
			call_result = nwos_read_files_disc_list(root_dir, subpath, file_names, num_files + result);
		    }

		    assert(result >= 0);          /* make sure we didn't somehow get here with an error result already */

		    if (call_result > 0)          /* normal return */
		    {
			result += call_result;        /* add number of files found to result */
		    }
		    else if (call_result < 0)     /* error return */
		    {
			result = call_result;        /* pass the error on up */
		    }
		    else                          /* empty directory */
		    {
			result = -1;                 /* pass empty directory error up */
		    }
		}

		if (call_result == 0)
		{
		    fprintf(stderr, "ERROR: subdirectory '%s' is empty!\n", path);
		}
	    }
	    else   /* must be a regular file */
	    {
		assert(S_ISREG(stat_struct.st_mode));

		/* first check for an empty file */

		if (stat_struct.st_size == 0)
		{
		    fprintf(stderr, "ERROR: file '%s' is empty, this version cannot handle empty files.\n", path);

		    if (result < 0)   /* already had an error */
		    {
			result = -(-result | 2);  /* merge in a -2 */
		    }
		    else              /* this is the first error */
		    {
			result = -2;
		    }
		}

		if (num_files >= 0 && result >= 0)
		{
		    if (num_files + result >= MAX_FILES_PER_DISC_LIST)
		    {
			fprintf(stderr, "ERROR: Too many files in directory '%s', this version can only handle a maximum of %d files.\n",
				root_dir, MAX_FILES_PER_DISC_LIST);
			result = -8;
		    }
		    else
		    {
			assert(num_files + result < MAX_FILES_PER_DISC_LIST);

			file_names[num_files + result] = malloc(strlen(subpath) + 1);

			if (file_names[num_files + result] == NULL)
			{
			    perror("allocating memory for file name");
			    exit(1);
			}

			strcpy(file_names[num_files + result], subpath);

			/* printf("%s\n", file_names[num_files + result]); */

			result++;
		    }
		}
	    }

	    path[path_len] = '\0';    /* truncate back to just the subdirectory */
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }

    return result;
}



bool nwos_find_matching_disc_list(ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    int j;
    int k;
    int count = 0;

    void_reference(ref);  /* in case we don't find it */


    /* first count the files */

    for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
    {
	if (!is_void_reference(&files[i])) count++;
    }

    assert(count > 0);

    /* for now return false if the private disc list class doesn't exist */

    if (!nwos_find_private_class_definition("DISC LIST", &disc_class_ref)) return false;


    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

	    /* remember ptr_disc_obj points to the kludge buffer */

	    if (nwos_decode_variable_sized_count(ptr_disc_obj->count) == count)   /* found a possible match */
	    {
		for (j = 0; j < MAX_FILES_PER_DISC_LIST; j++)
		{
		    if (!is_void_reference(&files[j]))
		    {
			for (k = 0; k < count; k++)
			{
			    if (is_same_object(&files[j], &ptr_disc_obj->files[k])) break;
			}

			if (k == count) break;   /* didn't find a match this can't be the list */
		    }
		}

		if (j == MAX_FILES_PER_DISC_LIST)   /* found a match for each one */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    break;
		}
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc list or create new */

ObjCreateResult nwos_create_disc_list(char id[12], ObjRef files[MAX_FILES_PER_DISC_LIST], ObjRef* ref)
{
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_disc_obj = (C_struct_Disc_List*)kludge;
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;
    int i;
    int count = 0;


    /* first find out if we already have this disc */

    if (!nwos_find_disc_list(id, ref))   /* didn't find it */
    {
	memset(kludge, 0, sizeof(kludge));  /* zero it out */

	/* remember ptr_path_obj points to the kludge buffer */

	nwos_find_or_create_private_class_definition("DISC LIST", &disc_class_ref);

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_disc_obj->header.common, ref, &disc_class_ref);

	memcpy(&ptr_disc_obj->id, id, sizeof(ptr_disc_obj->id));

	for (i = 0; i < MAX_FILES_PER_DISC_LIST; i++)
	{
	    if (!is_void_reference(&files[i]))
	    {
		copy_reference(&ptr_disc_obj->files[count], &files[i]);
		count++;

		nwos_add_to_references(ref, &files[i]);
	    }
	}

	nwos_encode_variable_sized_count(count, ptr_disc_obj->count);

	assert(nwos_get_disc_list_object_size(ptr_disc_obj) == sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));

	nwos_create_reference_list(ref, &ptr_disc_obj->header.object.references);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->header.object, sizeof(ObjectHeader), ptr_disc_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_disc_obj->id, nwos_get_disc_list_object_size(ptr_disc_obj) - sizeof(EveryObject), ptr_disc_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, nwos_get_disc_list_object_size(ptr_disc_obj));

	nwos_add_to_references(ref, &disc_class_ref);

	ptr_disc_obj = malloc(sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));
	nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef));
	assert(memcmp(kludge, ptr_disc_obj, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &nwos_get_disc_list_object_size);  /* read the other way */
	assert(memcmp(ptr_disc_obj, kludge, sizeof(C_struct_Disc_List) + count * sizeof(ObjRef)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Disc Copy object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing disc copy */

bool nwos_find_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Disc_Copy disc_obj;
    ObjRef disc_class_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    /* for now return false if there isn't a disc copy private class definition */

    if (!nwos_find_private_class_definition("DISC COPY", &disc_class_ref)) return false;

    nwos_read_class_definition(&disc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &disc_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &disc_obj, sizeof(disc_obj));

	    if (is_same_object(&disc_obj.disc_list, disc_list) && disc_obj.copy_number == copy_num)   /* found a match */
	    {
		/* if there is a newer version get it */
		while (!is_void_reference(&disc_obj.header.object.next_version))
		{
		    nwos_read_object_from_disk(&disc_obj.header.object.next_version, &disc_obj, sizeof(disc_obj));
		    assert(is_same_object(&disc_obj.disc_list, disc_list));
		    assert(disc_obj.copy_number == copy_num);
		}

		copy_reference(ref, &disc_obj.header.common.id);
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_disc_copy(ObjRef* disc_list, int copy_num, ObjRef* location, ObjRef* ref)
{
    C_struct_Disc_Copy disc_copy_obj;
    C_struct_Disc_Copy* ptr_disc_obj;
    ObjRef disc_class_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this disc copy */

    if (nwos_find_disc_copy(disc_list, copy_num, ref))   /* already exists */
    {
	nwos_read_object_from_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	if (is_same_object(&disc_copy_obj.location, location))
	{
	    /* add verification here */
	}
	else  /* location changed, create new object */
	{
	    nwos_find_or_create_private_class_definition("DISC COPY", &disc_class_ref);

	    /* verify ref is == to object ID in header */
	    assert(is_same_object(ref, &disc_copy_obj.header.common.id));
	    assert(is_void_reference(&disc_copy_obj.header.object.next_version));

	    nwos_generate_new_id(ref);

	    copy_reference(&disc_copy_obj.header.object.next_version, ref);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object,
				 sizeof(ObjectHeader),
				 disc_copy_obj.header.common.header_chksum);

	    nwos_overwrite_object_to_disk(&disc_copy_obj.header.common.id, &disc_copy_obj, sizeof(disc_copy_obj));

	    /* clear out next pointer that we just wrote */
	    void_reference(&disc_copy_obj.header.object.next_version);

	    /* copy the old object into the previous pointer */
	    copy_reference(&disc_copy_obj.header.object.prev_version, &disc_copy_obj.header.common.id);

	    /* put the new id in */
	    copy_reference(&disc_copy_obj.header.common.id, ref);

	    /* change the location */
	    copy_reference(&disc_copy_obj.location, location);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object,
				 sizeof(ObjectHeader),
				 disc_copy_obj.header.common.header_chksum);

	    nwos_crc32_calculate((uint8*) &disc_copy_obj.disc_list,
				 sizeof(disc_copy_obj) - sizeof(EveryObject),
				 disc_copy_obj.header.common.data_chksum);

	    nwos_write_object_to_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	    nwos_add_to_references(ref, &disc_class_ref);
	    nwos_add_to_references(ref, &disc_copy_obj.disc_list);


	    nwos_add_to_references(ref, location);
	}
    }
    else   /* didn't find it */
    {
	nwos_find_or_create_private_class_definition("DISC COPY", &disc_class_ref);

	memset(&disc_copy_obj, 0, sizeof(disc_copy_obj));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&disc_copy_obj.header.common, ref, &disc_class_ref);

	copy_reference(&disc_copy_obj.disc_list, disc_list);
	copy_reference(&disc_copy_obj.location, location);
	disc_copy_obj.copy_number = copy_num;

	nwos_create_reference_list(ref, &disc_copy_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.header.object, sizeof(ObjectHeader), disc_copy_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &disc_copy_obj.disc_list, sizeof(disc_copy_obj) - sizeof(EveryObject), disc_copy_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &disc_copy_obj, sizeof(disc_copy_obj));

	nwos_add_to_references(ref, &disc_class_ref);

	nwos_add_to_references(ref, disc_list);
	nwos_add_to_references(ref, location);

	ptr_disc_obj = malloc(sizeof(C_struct_Disc_Copy));
	nwos_read_object_from_disk(ref, ptr_disc_obj, sizeof(C_struct_Disc_Copy));
	assert(memcmp(&disc_copy_obj, ptr_disc_obj, sizeof(C_struct_Disc_Copy)) == 0);

	free(ptr_disc_obj);
	ptr_disc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}



/*-------------------------------------------------------------------------------------------------------------------*/
/* Storage Location object stuff */
/*-------------------------------------------------------------------------------------------------------------------*/


/* find existing storage location */

bool nwos_find_storage_location(const char* location, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Storage_Location location_obj;
    ObjRef location_class_ref;
    ObjRef object_class;
    ObjRef name_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    void_reference(ref);  /* in case we don't find it */


    if (!nwos_find_private_name(location, &name_ref))
    {
	return false;
    }


    /* for now return false if storage location private class definition doesn't exist */

    if (!nwos_find_private_class_definition("STORAGE LOCATION", &location_class_ref)) return false;


    nwos_read_class_definition(&location_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("num_refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &location_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &location_obj, sizeof(location_obj));

	    if (is_same_object(&location_obj.name, &name_ref))   /* found a match */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		break;
	    }
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i != num_refs);   /* return true if we found it */
}



/* Find existing disc copy or create new */

ObjCreateResult nwos_create_storage_location(const char* location, ObjRef* ref)
{
    C_struct_Storage_Location location_obj;
    C_struct_Storage_Location* ptr_loc_obj;
    ObjRef location_class_ref;
    ObjRef name_ref;
    ObjCreateResult result = FOUND_EXISTING;


    /* first find out if we already have this location */

    if (!nwos_find_storage_location(location, ref))   /* didn't find it */
    {
	nwos_create_name(location, &name_ref);

	nwos_find_or_create_private_class_definition("STORAGE LOCATION", &location_class_ref);

	memset(&location_obj, 0, sizeof(location_obj));

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&location_obj.header.common, ref, &location_class_ref);

	copy_reference(&location_obj.name, &name_ref);

	nwos_create_reference_list(ref, &location_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &location_obj.header.object, sizeof(ObjectHeader), location_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &location_obj.name, sizeof(location_obj) - sizeof(EveryObject), location_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &location_obj, sizeof(location_obj));

	nwos_add_to_references(ref, &location_class_ref);

	nwos_add_to_references(ref, &name_ref);

	ptr_loc_obj = malloc(sizeof(C_struct_Storage_Location));
	nwos_read_object_from_disk(ref, ptr_loc_obj, sizeof(C_struct_Storage_Location));
	assert(memcmp(&location_obj, ptr_loc_obj, sizeof(C_struct_Storage_Location)) == 0);

	free(ptr_loc_obj);
	ptr_loc_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* Directory Processing */
/*-------------------------------------------------------------------------------------------------------------------*/

const char* nwos_check_invalid_type(mode_t mode)
{
    if (S_ISCHR(mode))
    {
	return "character device";
    }

    if (S_ISBLK(mode))
    {
	return "block device";
    }

    if (S_ISFIFO(mode))
    {
	return "FIFO (named pipe)";
    }

    if (S_ISLNK(mode))
    {
	return "symbolic link";
    }

    if (S_ISSOCK(mode))
    {
	return "socket";
    }

    return NULL;
}


void nwos_check_directory_is_all_ok(const char* path)
{
    struct stat stat_struct;
    DIR* dp;
    struct dirent *dir_entry;
    const char* invalid_type;
    const char* p;
    char full_path[PATH_MAX];
    int i;

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    // Construct the full path name

	    i = 0;

	    for (p = path; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX) full_path[i++] = '/';

	    for (p = dir_entry->d_name; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX)
	    {
		full_path[i] = '\0';
	    }
	    else
	    {
		fprintf(stderr, "ERROR: path %s/%s exceeds maximum size of a path (PATH_MAX = %d)\n",
			path, dir_entry->d_name, PATH_MAX);
		exit(1);
	    }

	    if (lstat(full_path, &stat_struct) != 0)
	    {
		perror(full_path);
		exit(1);
	    }

	    invalid_type = nwos_check_invalid_type(stat_struct.st_mode);

	    if (invalid_type != NULL)
	    {
		fprintf(stderr, "ERROR: %s is a %s.\n", full_path, invalid_type);
		exit(1);
	    }

	    if (S_ISDIR(stat_struct.st_mode))    // it's not a normal file, investigate further
	    {
		nwos_check_directory_is_all_ok(full_path);
	    }    
	    else if (!S_ISREG(stat_struct.st_mode))
	    {
		fprintf(stderr, "ERROR: %s is unrecognized type of file.\n", full_path);
		exit(1);
	    }
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }
}


/*-------------------------------------------------------------------------------------------------------------------*/
/* File Object Updates */
/*-------------------------------------------------------------------------------------------------------------------*/

void nwos_update_file_001_object_to_current(void* object, size_t size)
{
    C_struct_File_001 file_001_obj;
    C_struct_File* file_obj_ptr = (C_struct_File*) object;
    ObjRef file_class_ref;

    memcpy(&file_001_obj, object, sizeof(file_001_obj));  /* save the 001 data */

    /* change the class definition to the latest */
#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    /* if a the newer class doesn't exist yet, change the class definition to the latest public */
    if (is_same_object(&file_class_ref, &file_obj_ptr->header.common.class_definition))
    {
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
    }
#endif
    copy_reference(&file_obj_ptr->header.common.class_definition, &file_class_ref);

    file_obj_ptr->size[0] = 0;  /* most significant byte didn't exist before */
    file_obj_ptr->size[1] = file_001_obj.size[0];
    file_obj_ptr->size[2] = file_001_obj.size[1];
    file_obj_ptr->size[3] = file_001_obj.size[2];
    file_obj_ptr->size[4] = file_001_obj.size[3];

    copy_reference(&file_obj_ptr->md5sum,     &file_001_obj.md5sum);
    copy_reference(&file_obj_ptr->sha1sum,    &file_001_obj.sha1sum);
    void_reference(&file_obj_ptr->sha256sum);
    void_reference(&file_obj_ptr->sha512sum);
    copy_reference(&file_obj_ptr->media,      &file_001_obj.media);
    copy_reference(&file_obj_ptr->block_list, &file_001_obj.block_list);

    nwos_crc32_calculate((uint8*) &file_obj_ptr->size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj_ptr->header.common.data_chksum);
}


void nwos_update_file_002_object_to_current(void* object, size_t size)
{
    C_struct_File_002 file_002_obj;
    C_struct_File* file_obj_ptr = (C_struct_File*) object;
    ObjRef file_class_ref;

    memcpy(&file_002_obj, object, sizeof(file_002_obj));  /* save the 002 data */

    /* change the class definition to the latest */
#ifdef PUBLIC_MODE
    assert(nwos_find_public_class_definition("FILE", &file_class_ref));
#else
    assert(nwos_find_private_class_definition("FILE", &file_class_ref));

    /* if a the newer class doesn't exist yet, change the class definition to the latest public */
    if (is_same_object(&file_class_ref, &file_obj_ptr->header.common.class_definition))
    {
	assert(nwos_find_public_class_definition("FILE", &file_class_ref));
    }
#endif
    copy_reference(&file_obj_ptr->header.common.class_definition, &file_class_ref);

    file_obj_ptr->size[0] = file_002_obj.size[0];
    file_obj_ptr->size[1] = file_002_obj.size[1];
    file_obj_ptr->size[2] = file_002_obj.size[2];
    file_obj_ptr->size[3] = file_002_obj.size[3];
    file_obj_ptr->size[4] = file_002_obj.size[4];

    copy_reference(&file_obj_ptr->md5sum,     &file_002_obj.md5sum);
    copy_reference(&file_obj_ptr->sha1sum,    &file_002_obj.sha1sum);
    copy_reference(&file_obj_ptr->sha256sum,  &file_002_obj.sha256sum);
    void_reference(&file_obj_ptr->sha512sum);
    copy_reference(&file_obj_ptr->media,      &file_002_obj.media);
    copy_reference(&file_obj_ptr->block_list, &file_002_obj.block_list);

    nwos_crc32_calculate((uint8*) &file_obj_ptr->size[0], sizeof(C_struct_File) - sizeof(EveryObject), file_obj_ptr->header.common.data_chksum);
}


