/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: verify_file.c,v $
-- Revision 1.18  2008/06/23 04:46:13  jsedwards
-- Added calls to nwos_normalize_path to clean up the paths for processing
-- directories.
--
-- Revision 1.17  2008/04/27 15:09:26  jsedwards
-- Added a print indicating the file was shredded when the shredding option
-- was used.  Also added a warning that is printed when the program is
-- finished that the shredding is minimal and the files could still be read.
--
-- Revision 1.16  2008/04/27 14:46:46  jsedwards
-- Added a very minimal --shred option.  NOTE: the shred only makes one pass
-- of random data.  It is very possible that files could still be read by
-- someone with the proper equipment.
--
-- Revision 1.15  2008/04/27 02:00:49  jsedwards
-- Added usage printing.
--
-- Revision 1.14  2008/04/26 15:01:50  jsedwards
-- Added code to allow recursive verification of directories.
--
-- Revision 1.13  2008/03/20 14:03:50  jsedwards
-- Added --public option.
--
-- Revision 1.12  2008/02/03 01:04:28  jsedwards
-- Changed DEFAULT_TYPE_RO to READ_ONLY and changed path variable to const char
-- instead of char.
--
-- Revision 1.11  2007/08/12 21:29:18  jsedwards
-- Remove call to set security level because this program doesn't create
-- objects there is no need for a security level.
--
-- Revision 1.10  2007/08/12 20:45:00  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.9  2007/07/21 14:57:01  jsedwards
-- Added 'match' parameter in call to nwos_file_is_identical so it will ignore
-- the time stamp on the file if desired.
--
-- Revision 1.8  2007/07/21 11:41:30  jsedwards
-- Added option to read compressed file instead of default storage.
--
-- Revision 1.7  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.6  2007/06/13 21:20:58  jsedwards
-- Change to pass null to find_matching_path_and_file function for the
-- new directory parameter (which makes it work the same as before).
--
-- Revision 1.5  2007/06/05 23:50:16  jsedwards
-- Added option to ignore time stamp on files.
--
-- Revision 1.4  2007/03/18 19:55:53  jsedwards
-- Enabled verification for removal based on file having a backup copy on a disc.
--
-- Revision 1.3  2007/02/28 14:02:49  jsedwards
-- Added test to make sure file is stored in system before trying to compare it.
--
-- Revision 1.2  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.1  2007/01/27 17:37:00  jsedwards
-- Program to verify file if a file is stored in the system and if desired
-- delete it.
--
*/

#include <assert.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>

#include "objectify.h"


static void display_usage(char* prog_name)
{
    fprintf(stderr, "usage: %s [file or directory list]\n", prog_name);
    fprintf(stderr, "   or: %s --help\n", prog_name);
    fprintf(stderr, "\n");
    fprintf(stderr, "  --compressed file\tuse file for archive.\n");
    fprintf(stderr, "  --help\t\tdisplay this message.\n");
    fprintf(stderr, "  --ignore-time\t\tdon't require time stamp to match.\n");
    fprintf(stderr, "  --public\t\tverify files against public data.\n");
    fprintf(stderr, "  -r, --recursive\trecurse through directories.\n");
    fprintf(stderr, "  --remove\t\terase files if they verify correctly.\n");
    fprintf(stderr, "  --shred\t\twrite random data over file and then remove.  See note below!\n");

    fprintf(stderr, "PLEASE NOTE: the --shred option is very minimal in this version.  With the\n");
    fprintf(stderr, "   proper equipment it is very possible that your data could still be read.\n");

    fprintf(stderr, "\n");
}


const char* check_invalid_type(mode_t mode)
{
    if (S_ISCHR(mode))
    {
	return "character device";
    }

    if (S_ISBLK(mode))
    {
	return "block device";
    }

    if (S_ISFIFO(mode))
    {
	return "FIFO (named pipe)";
    }

    if (S_ISLNK(mode))
    {
	return "symbolic link";
    }

    if (S_ISSOCK(mode))
    {
	return "socket";
    }

    if (!S_ISREG(mode) && !S_ISDIR(mode))
    {
	return "unknown file type";
    }

    return NULL;
}


#include <fcntl.h>
void shred(const char* path)
{
    struct stat stat_struct;
    int fd;
    ssize_t bytes_written;
    ssize_t total_written = 0;
    uint8 buffer[512];
    int i;

    if (lstat(path, &stat_struct) != 0)
    {
	perror(path);
	exit(1);
    }

    fd = open(path, O_WRONLY | O_SYNC);

    if (fd < 0)
    {
	perror(path);
	exit(1);
    }

    while (total_written < stat_struct.st_size)
    {
	for (i = 0; i < sizeof(buffer); i++) buffer[i] = random();

	bytes_written = write(fd, buffer, sizeof(buffer));

	if (bytes_written != sizeof(buffer))
	{
	    perror(path);
	    exit(1);
	}

	total_written += bytes_written;
    }

    if (fdatasync(fd) != 0)
    {
	perror(path);
	exit(1);
    }

    if (close(fd) != 0)
    {
	perror(path);
	exit(1);
    }
}


typedef enum { Do_Not_Remove, Just_Remove, Shred_Before_Remove } Remove_Option;

void process_file(const char* path, MatchCode match, Remove_Option remove_option)
{
    ObjRef ref;
    struct stat stat_struct;
    const char* invalid_type;

    printf("%s: ", path);
    fflush(stdout);

    if (lstat(path, &stat_struct) != 0)
    {
	perror(path);
	exit(1);
    }

    invalid_type = check_invalid_type(stat_struct.st_mode);

    if (invalid_type != NULL)
    {
	printf("is a %s, ignored.\n", invalid_type);
    }
    else if (S_ISDIR(stat_struct.st_mode))    // it's not a normal file, investigate further
    {
	printf("is a directory and -r or --recursive was not specified, ignored.\n");
    }
    else if (nwos_find_matching_path_and_file_association(NULL, path, &ref, match))
    {
	if (remove_option != Do_Not_Remove)
	{
	    if (nwos_file_is_stored(&ref))
	    {
		assert(nwos_file_is_identical(&ref, path, match));

		if (remove_option == Shred_Before_Remove)
		{
		    shred(path);
		    printf("shredded and ");
		}

		if (unlink(path) != 0)
		{
		    perror(path);
		    exit(1);
		}

		printf("removed\n");
	    }
	    else if (nwos_file_has_backup(&ref))
	    {
		if (remove_option == Shred_Before_Remove)
		{
		    shred(path);
		    printf("shredded and ");
		}

		if (unlink(path) != 0)
		{
		    perror(path);
		    exit(1);
		}

		printf("removed\n");
	    }
	    else
	    {
		printf("no known backup, not removed\n");
	    }
	}
	else
	{
	    printf("found: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
	}
    }
    else
    {
	printf("NOT found\n");
    }
}


void process_directory(const char* path, MatchCode match, Remove_Option remove_option)
{
    DIR* dp;
    struct dirent *dir_entry;
    const char* p;
    char full_path[PATH_MAX];
    int i;
    struct stat stat_struct;

    dp = opendir(path);
    if (dp == NULL)
    {
	perror(path);
	exit(1);
    }

    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    // Construct the full path name

	    i = 0;

	    if (strcmp(path, ".") != 0)
	    {
		for (p = path; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

		if (i < PATH_MAX && full_path[i-1] != '/') full_path[i++] = '/';
	    }

	    for (p = dir_entry->d_name; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

	    if (i < PATH_MAX)
	    {
		full_path[i] = '\0';
	    }
	    else
	    {
		fprintf(stderr, "ERROR: path %s/%s exceeds maximum size of a path (PATH_MAX = %d)\n",
			path, dir_entry->d_name, PATH_MAX);
		exit(1);
	    }

	    if (lstat(full_path, &stat_struct) != 0)
	    {
		perror(full_path);
		exit(1);
	    }

	    if (S_ISDIR(stat_struct.st_mode))
	    {
		process_directory(full_path, match, remove_option);
	    }
	    else
	    {
		process_file(full_path, match, remove_option);
	    }
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(path);
	exit(1);
    }

    if (remove_option != Do_Not_Remove)
    {
	if (strcmp(path, ".") == 0)
	{
	    printf("Warning: will not remove current working directory: '.'!\n");
	}
	else if (rmdir(path) != 0)
	{
	    perror(path);
	}
    }
}


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    int i;
    Remove_Option remove_option = Do_Not_Remove;
    bool do_public = false;
    bool do_recursive = false;
    MatchCode match = MatchAll;
    char* path = DEFAULT_FILE;
    char normalized_path[PATH_MAX];

    printf("\n");

    if (argc < 2)
    {
	display_usage(argv[0]);
	exit(1);
    }

    for (i = 1; i < argc && argv[i][0] == '-'; i++)
    {
	if (strcmp(argv[i], "--remove") == 0)
	{
	    if (remove_option == Shred_Before_Remove)
	    {
		fprintf(stderr, "NOTE: --shred option implies --remove\n");
	    }
	    else
	    {
		remove_option = Just_Remove;	// Don't shred it
	    }
	}
	else if (strcmp(argv[i], "--shred") == 0)
	{
	    if (remove_option == Just_Remove)
	    {
		fprintf(stderr, "NOTE: --shred option implies --remove\n");
	    }

	    remove_option = Shred_Before_Remove;	// Do shred it
	}
	else if (strcmp(argv[i], "--public") == 0)
	{
	    do_public = true;
	}
	else if (strcmp(argv[i], "--recursive") == 0 || strcmp(argv[i], "-r") == 0)
	{
	    do_recursive = true;
	}
	else if (strcmp(argv[i], "--ignore-time") == 0)
	{
	    match = IgnoreTime;
	}
	else if (strcmp(argv[i], "--compressed") == 0)
	{
	    i++;
	    path = argv[i];
	}
	else if (strcmp(argv[i], "--help") == 0)
	{
	    display_usage(argv[0]);
	    exit(1);
	}
	else
	{
	    fprintf(stderr, "Unknown option: %s\n\n", argv[i]);
	    display_usage(argv[0]);
	    exit(1);
	}
    }

    nwos_log_arguments(argc, argv);

    if (do_public && path != NULL)   /* both --public and --compressed were specified */
    {
	fprintf(stderr, "WARNING: because --public was specified --compressed file is ignored!\n");
	path = NULL;
    }

    if (do_public)
    {
	nwos_initialize_objectify(NULL, 0, 0, PUBLIC, NULL);

	root_object_ref.id[0] = 0;
	root_object_ref.id[1] = 0;
	root_object_ref.id[2] = 0;
	root_object_ref.id[3] = 1;

	nwos_set_root_object(&root_object_ref);
    }
    else
    {
	nwos_get_key_from_password(big_key, sizeof(big_key));

	memcpy(bf_key, big_key, 16);
	linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
	memcpy(root_object_ref.id, big_key+20, 4);
	serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

	nwos_initialize_objectify(bf_key, linear, serial, READ_ONLY, path);

	nwos_set_root_object(&root_object_ref);
    }

    for (/* use i as left by loop above */; i < argc; i++)
    {
	if (strcmp(argv[i], ".") == 0 && strcmp(argv[i], "./") == 0)
	{
	    if (do_recursive)
	    {
		process_directory(".", match, remove_option);
	    }
	    else
	    {
		printf("'.' is a directory and -r or --recursive was not specified, ignored.\n");
	    }
	}
	else if (strlen(argv[i]) + 1 > sizeof(normalized_path))
	{
	    fprintf(stderr, "ERROR: the path '%s' is too long for the buffer (exceeds %d characters), skipped\n", argv[i], sizeof(normalized_path));
	}
	else
	{
	    nwos_normalize_path(normalized_path, argv[i], sizeof(normalized_path));

	    if (normalized_path[0] == '.' && normalized_path[1] == '.' && normalized_path[2] == '/')
	    {
		fprintf(stderr, "Cannot verify from parent directory (%s), will be skipped!\n", argv[i]);
	    }
	    else if (do_recursive)
	    {
		process_directory(normalized_path, match, remove_option);
	    }
	    else
	    {
		process_file(normalized_path, match, remove_option);
	    }
	}
    }

    if (remove_option == Shred_Before_Remove)
    {
	printf("\n");
	printf(" **************************************************************************************\n");
	printf("  WARNING: the shredding done by this version of this program is very minimal!  It is\n");
	printf("   very possible, with the right equipment, that someone could still read the files!\n");
	printf(" *************************************************************************************\n");
	printf("\n");
    }

    nwos_terminate_objectify();

    return 0;
}


