/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: export_c_structs.c,v $
-- Revision 1.12  2008/03/12 11:41:53  jsedwards
-- Added call to nwos_log_arguments(0, NULL) to disable logging.
--
-- Revision 1.11  2008/02/16 12:15:30  jsedwards
-- Change AccessType to PUBLIC.
--
-- Revision 1.10  2007/08/17 15:54:32  jsedwards
-- Change check for "Sha1" to "Sha" and any number, so it will match "Sha1",
-- "Sha256", or "Sha512".
--
-- Revision 1.9  2007/07/27 17:09:55  jsedwards
-- Changed to deal with class definitions that have revisions.
--
-- Revision 1.8  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.7  2007/06/19 14:28:15  jsedwards
-- Eliminate option to read from public.obj file because now public file is
-- always separate from the private storage.
--
-- Revision 1.6  2007/03/08 13:43:24  jsedwards
-- Change to allow reading public objects from a public objects file instead
-- of the drive or partition.
--
-- Revision 1.5  2007/03/08 13:09:08  jsedwards
-- Removed printing of number of classes (not good inside .h file).
--
-- Revision 1.4  2007/03/08 12:25:56  jsedwards
-- Deleted unused size routines, changed to not generate reference list and
-- write warning about not editing at beginning of file.
--
-- Revision 1.3  2007/03/08 02:33:52  jsedwards
-- Mostly working now.
--
-- Revision 1.2  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.1  2006/12/05 12:31:19  jsedwards
-- Initial version, not functional!!
--
*/


#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "objectify_private.h"


char* current_filename;   /* kludgy, but needed for printing */

FILE* ref_list_fp;

#define MAX_LINE_LENGTH 256
#define MAX_REF_LIST_LINE_LENGTH 8192


static size_t get_class_object_size(void* class_obj)
{
    return sizeof(C_struct_Class_Definition) + (((C_struct_Class_Definition*)class_obj)->count * sizeof(ObjRef));
}


char *class_to_type(ObjRef* class_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];

    nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size);
    nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

    if (strcmp(name, "Object Reference") == 0) return "ObjRef";

    if (strcmp(name, "Byte") == 0) return "uint8";

    if (strcmp(name, "Character") == 0) return "char";

    if (strcmp(name, "Time Stamp") == 0) return "TimeStamp";

    /* sprintf(result, "%02x%02x%02x%02x",
               class_ref->id[0], class_ref->id[1], class_ref->id[2], class_ref->id[3]); */

    return "** ERROR **";
}


void output_feature(ObjRef* feature_ref)
{
    C_struct_Feature_Definition feature_obj;
    char label[32];
    int i;

    nwos_read_object_from_disk(feature_ref, &feature_obj, sizeof(feature_obj));

    nwos_name_to_string(&feature_obj.label, label, sizeof(label));

    for (i = 0; label[i] != '\0'; i++)
    {
	if (label[i] == ' ')
	{
	    label[i] = '_';
	}
	else if (isupper(label[i]))
	{
	    label[i] = tolower(label[i]);
	}
    }

    if (feature_obj.count == 1)
    {
	printf("    %s %s;\n", class_to_type(&feature_obj.class), label);
    }
    else
    {
	printf("    %s %s[%d];\n", class_to_type(&feature_obj.class), label, feature_obj.count);
    }
}

void output_class(ObjRef* class_ref)
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    char name[32];
    int i;
    int revision;

    nwos_read_variable_sized_object_from_disk(class_ref, kludge, sizeof(kludge), &get_class_object_size);
    nwos_name_to_string(&class_def_obj_ptr->name, name, sizeof(name));

    /* don't create structures for the built in types */

    if (strcmp(name, "Byte") != 0 &&
	strcmp(name, "Character") != 0 &&
	strcmp(name, "Object Reference") != 0 &&
	strcmp(name, "Reference List") != 0 &&
	strcmp(name, "Time Stamp") != 0)
    {
	for (i = 0; name[i] != '\0'; i++) if (name[i] == ' ') name[i] = '_';

	if (name[0] == 'U' && name[1] == 's' && name[2] == '_')
	{
	    name[1] = 'S';
	}
	else if (name[0] == 'C' && name[1] == 'd' && name[2] == '_')
	{
	    name[1] = 'D';
	}
	else if (name[0] == 'M' && name[1] == 'd' && name[2] == '5')
	{
	    name[1] = 'D';
	}
	else if (name[0] == 'S' && name[1] == 'h' && name[2] == 'a' && isdigit(name[3]))
	{
	    name[1] = 'H';
	    name[2] = 'A';
	}

	printf("typedef struct {\n");

	printf("    EveryObject header;\n");

	for (i = 0; i < class_def_obj_ptr->count; i++)
	{
	    output_feature(&class_def_obj_ptr->feature[i]);
	}

	if (is_void_reference(&class_def_obj_ptr->header.object.next_version))  /* no newer version */
	{
	    printf("} C_struct_%s;\n\n", name);
	}
	else   /* newer version of this class exists */
	{
	    revision = 1;
	    while (!is_void_reference(&class_def_obj_ptr->header.object.prev_version))
	    {
		revision++;

		nwos_read_variable_sized_object_from_disk(&class_def_obj_ptr->header.object.prev_version, kludge, sizeof(kludge), &get_class_object_size);
	    }

	    printf("} C_struct_%s_%03d;\n\n", name, revision);
	}
    }
}

int main(int argc, char* argv[])
{
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_Class_Definition* class_def_obj_ptr = (C_struct_Class_Definition*)kludge;
    ObjRef root_ref;
    ObjRef obj_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;

    nwos_log_arguments(0, NULL);   /* disable logging */

    if (argc == 1)
    {
	nwos_initialize_objectify(NULL, 0, 0, PUBLIC, NULL);    /* public only */ 
    }
    else
    {
	fprintf(stderr, "usage: %s [public.objects.file]\n", argv[0]);
    }

    root_ref.id[0] = 0;
    root_ref.id[1] = 0;
    root_ref.id[2] = 0;
    root_ref.id[3] = 1;

    nwos_set_root_object(&root_ref);


    printf("/* This file was created by the 'export_c_structs' program from the objectify */\n");
    printf("/* public objects.  It should not be edited to make additions or changes, the */\n");
    printf("/* objectify classes should be changed and then regenerate this file.         */\n");
    printf("\n\n");


    nwos_read_variable_sized_object_from_disk(&nwos_public_class_definition_class_ref, kludge, sizeof(kludge), &get_class_object_size);

    ref_list = nwos_malloc_reference_list(&class_def_obj_ptr->header.object.references);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &obj_class);

	if (is_same_object(&obj_class, &nwos_public_class_definition_class_ref))
	{
	    output_class(&ref_list->references[i]);
	}
    }

    nwos_free_reference_list(ref_list);
	 
    nwos_terminate_objectify();

    return 0;
}

