/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: objectify_private.h,v $
-- Revision 1.56  2007/09/15 18:41:53  jsedwards
-- Added ivec variable to Ref_List_First_Block and Ref_List_Extra_Block
-- structures.
--
-- Revision 1.55  2007/07/31 18:25:28  jsedwards
-- Changed 'reserved' element of Chunk_Info structure to 'index'.
--
-- Revision 1.54  2007/07/28 01:07:02  jsedwards
-- Left the "nwos_" prefix off update_file_001_object_to_current, added it.
--
-- Revision 1.53  2007/07/28 01:03:49  jsedwards
-- Added external definition for update_file_001_object_to_current.
--
-- Revision 1.52  2007/07/07 20:28:40  jsedwards
-- Move declaration of nwos_log to objectify.h.
--
-- Revision 1.51  2007/07/03 00:41:43  jsedwards
-- Added Chunk_Info structure for chunk info table (previously chunk_index).
--
-- Revision 1.50  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.49  2007/06/26 14:32:47  jsedwards
-- Removed private_generate_new_completely_random_id, now only used in disk_io.c.
--
-- Revision 1.48  2007/06/26 13:03:07  jsedwards
-- Changed RESERVED_PRIVATE_BLOCKS into three definitions:
--  RESERVED_PRIVATE_CHUNKS = number of chunks between 0x10000000 and 0xffffffff
--  RESERVED_PRIVATE_BLOCKS = RESERVED_PRIVATE_CHUNKS * USABLE_BLOCKS_PER_CHUNK
--  MAXIMUM_VALID_PRIVATE_REFERENCE = RESERVED_PUBLIC_BLOCKS +
--                                                  RESERVED_PRIVATE_BLOCKS - 1
--
-- Revision 1.47  2007/06/26 12:31:11  jsedwards
-- Added RESERVED_PRIVATE_BLOCKS define.
--
-- Revision 1.46  2007/06/26 12:00:22  jsedwards
-- Added "used_chunks" element to Disk_Header and external declarations
-- for "nwos_total_private_chunks", "nwos_used_private_chunks", and
-- "nwos_allocate_new_chunk".
--
-- Revision 1.45  2007/06/25 05:37:36  jsedwards
-- Added external references needed when split disk I/O functions out of
-- objectify.c and created new disk_io.c file.
--
-- Revision 1.44  2007/06/20 03:45:44  jsedwards
-- Put old 0021 Disk Header structure back in for expand_sparse.
--
-- Revision 1.43  2007/06/19 18:49:46  jsedwards
-- Removed public info from Disk Header because the public objects are now
-- separate from the private objects.
--
-- Revision 1.42  2007/05/26 14:51:08  jsedwards
-- Change create_manufacturer to have a list of URLs instead of just one.
--
-- Revision 1.41  2007/05/25 13:39:36  jsedwards
-- Add external declarations for create_url and create_public_manufacturer.
--
-- Revision 1.40  2007/05/24 11:51:51  jsedwards
-- Add external declaration for create public acronym.
--
-- Revision 1.39  2007/04/05 01:46:36  jsedwards
-- Added external declaration for 'nwos_hash_ref'.
--
-- Revision 1.38  2007/03/27 11:38:41  jsedwards
-- Add external declarations for total and used, public and private block
-- counts so that 'expand_sparse' can access them.
--
-- Revision 1.37  2007/03/26 12:35:08  jsedwards
-- Change disk header again to save time disk was prepped.
--
-- Revision 1.36  2007/03/25 12:57:26  jsedwards
-- Changed disk header yet again, separate time stamp for public and private.
--
-- Revision 1.35  2007/02/27 12:22:00  jsedwards
-- Added RESERVED_PUBLIC_BLOCKS back in and made hash_uint32_ref(), write_block(),
-- set_bit_in_map(), clear_bit_in_map(), and test_bit_in_map() external.
--
-- Revision 1.34  2007/01/27 01:51:37  jsedwards
-- Added external declaration for 'verbose' boolean.
--
-- Revision 1.33  2007/01/04 17:38:20  jsedwards
-- Add external declaration for generate_new_completely_random_id.
--
-- Revision 1.32  2006/12/30 03:49:04  jsedwards
-- Moved NUM_PUBLIC_IDS and NUM_PRIVATE_IDS to objectify.c because they are no
-- longer used by the main code (only needed now when reading an old archive).
--
-- Revision 1.31  2006/12/20 12:33:06  jsedwards
-- Changed to have two class definition refs, public and private.
--
-- Revision 1.30  2006/12/15 15:00:26  jsedwards
-- Removed total_encrypted_blocks and used_encrypted_blocks from disk header,
-- because now there are only two spaces: public and private-encrypted.
--
-- Revision 1.29  2006/12/14 13:21:33  jsedwards
-- Added external declaration of clone_class_definition function.
--
-- Revision 1.28  2006/12/08 11:30:35  jsedwards
-- Add disk header structure.
--
-- Revision 1.27  2006/12/08 10:39:57  jsedwards
-- Remove old convert 0004 to 0005 function declarations and add declaration
-- for private_create_city which is needed by the big_bang.
--
-- Revision 1.26  2006/12/07 14:07:34  jsedwards
-- Removed external references to setup routines that are no longer used.
--
-- Revision 1.25  2006/11/11 15:24:02  jsedwards
-- Added external declaration for nwos_log function.
--
-- Revision 1.24  2006/11/11 14:18:06  jsedwards
-- Added external declarations for nwos_malloc and nwos_free functions.
--
-- Revision 1.23  2006/11/11 13:57:25  jsedwards
-- Add external declaration for generate_new_public_id function.
--
-- Revision 1.22  2006/11/11 13:34:57  jsedwards
-- Added external declaration of next_public_ref.
--
-- Revision 1.21  2006/11/11 12:01:05  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.20  2006/11/10 13:32:05  jsedwards
-- Moved Ref_List_First_Block and Ref_List_Extra_Block definitions from
-- objectify.c to objectify_private.h so that import_public_csv can use
-- them also.
--
-- Revision 1.19  2006/11/10 04:11:54  jsedwards
-- Add "write_public_object_to_disk" external declaration.
--
-- Revision 1.18  2006/10/26 01:35:58  jsedwards
-- Replaced main trunk version (1.17) with latest version from alpha_05_branch.
--
-- Revision 1.14.2.15  2006/10/22 13:09:30  jsedwards
-- Removed RESERVED_PUBLIC_BLOCKS and PRIVATE_BLOCKS_ON_DISK defines, they
-- are both now variables based upon values read from disk.
--
-- Revision 1.14.2.14  2006/10/11 13:26:28  jsedwards
-- Added bit_map_block_mask definition.
--
-- Revision 1.14.2.13  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.14.2.12  2006/10/06 04:35:48  jsedwards
-- Renamed and rearranged the definitions of public and private, blocks and
-- ids.
--
-- Revision 1.14.2.11  2006/10/05 12:47:21  jsedwards
-- Added definitions for number of private blocks (id's) and number of
-- private blocks on the disk.
--
-- Revision 1.14.2.10  2006/09/27 13:25:13  jsedwards
-- Added reference for nwos_block_used function.
--
-- Revision 1.14.2.9  2006/09/26 13:15:35  jsedwards
-- Fixed the number of public reserved blocks (256 MB instead of 128 MB).
--
-- Revision 1.14.2.8  2006/09/26 13:00:53  jsedwards
-- Moved DEFAULT_FILE definition to config.h and added RESERVED_PUBLIC_BLOCKS
-- definition.
--
-- Revision 1.14.2.7  2006/09/18 01:36:26  jsedwards
-- Add declarations and definitions so that file.c can access block read and
-- write encryption routines directly.
--
-- Revision 1.14.2.6  2006/09/16 14:23:53  jsedwards
-- Add declaration of create_reference_list_with_existing_id.
--
-- Revision 1.14.2.5  2006/09/06 13:15:25  jsedwards
-- Removed nwos_seed_sequence function and instead pass pointers to the values
-- in the call to next_sequence function.
--
-- Revision 1.14.2.4  2006/09/03 13:54:55  jsedwards
-- Moved definition of DEFAULT_FILE here from objectify.c so the big_bang can
-- use it to create the file.
--
-- Revision 1.14.2.3  2006/09/03 11:03:24  jsedwards
-- Added declaration of overwrite function.
--
-- Revision 1.14.2.2  2006/09/02 01:09:51  jsedwards
-- Add "ref" parameter to fill_in_common_header so it can put it in header.
--
-- Revision 1.14.2.1  2006/08/18 12:55:57  jsedwards
-- Added new setup_file() routine for creating file and file path classes.
--
-- Revision 1.14  2006/01/12 02:58:28  jsedwards
-- Added external declaration of "update spelling from 0004 to 0005" routine.
--
-- Revision 1.13  2006/01/09 03:26:10  jsedwards
-- Added external reference for "upgrade persons from 0004 to 0005" routine.
--
-- Revision 1.12  2006/01/06 14:14:42  jsedwards
-- Removed external declaration of "create all fifty states".  Added external
-- declarations for "setup address", "upgrade states from 0004 to 0005", and
-- "upgrade months from 0004 to 0005".
--
-- Revision 1.11  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.10  2005/12/31 17:56:23  jsedwards
-- Added declrations for "create_class_definition" and "setup_person" routines.
--
-- Revision 1.9  2005/12/28 13:04:08  jsedwards
-- Added declarations for sequence generator routines in security.c.
--
-- Revision 1.8  2005/12/27 19:47:23  jsedwards
-- Added declaration for create root object routine.
--
-- Revision 1.7  2005/12/27 18:09:27  jsedwards
-- Changed parameters to "fill_in_common_header" and "create_english_language"
-- routines.  Added external references for two global variables: "class_
-- definition_class_ref" and "reference_list_class_ref".  Also added external
-- declarations for "set_spelling_class_ref", "set_name_class_ref", and
-- "remove_object" routines.
--
-- Revision 1.6  2005/12/21 17:10:31  jsedwards
-- Add 'nwos' to 'create_all_area_codes' routine name.
--
-- Revision 1.5  2005/12/21 17:07:36  jsedwards
-- Add declaration of create_all_area_codes.
--
-- Revision 1.4  2005/12/21 04:01:12  jsedwards
-- Added declarations for creating all states and all months.
--
-- Revision 1.3  2005/12/11 16:48:13  jsedwards
-- Added "create_english_language" declaration.
--
-- Revision 1.2  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.1  2005/11/26 15:19:24  jsedwards
-- External references that are private to the system (shouldn't be used by
-- 'user' programs).
--
*/

#ifndef OBJECTIFY_PRIVATE_H
#define OBJECTIFY_PRIVATE_H

#include "objectify.h"

#define RESERVED_PUBLIC_BLOCKS 0x10000000    /* IDs 0x00000000 to 0x0fffffff reserved */

#define BIT_MAP_BLOCKS 32
#define BIT_MAP_BYTES (BIT_MAP_BLOCKS * FILE_BLOCK_SIZE)

#define BLOCKS_IN_CHUNK (BIT_MAP_BYTES * 8)
#define BIT_MAP_BLOCK_MASK (~(BLOCKS_IN_CHUNK-1))
#define USABLE_BLOCKS_PER_CHUNK (BLOCKS_IN_CHUNK - BIT_MAP_BLOCKS)   /* 65536 - 32 = 65504 */


/* some blocks don't come out even when dividing usable chunks per block, so don't allocate them */

#define RESERVED_PRIVATE_CHUNKS ((0xffffffff - RESERVED_PUBLIC_BLOCKS) / USABLE_BLOCKS_PER_CHUNK)
/*                              ((0xffffffff - 0x10000000)             / 65504)    = 61470 */

#define RESERVED_PRIVATE_BLOCKS (RESERVED_PRIVATE_CHUNKS * USABLE_BLOCKS_PER_CHUNK)
/*                                                 61470 * 65504                   = 4026530880 */
/*                                                                                 = 0xeffffc40 */

#define MAXIMUM_VALID_PRIVATE_REFERENCE (RESERVED_PUBLIC_BLOCKS + RESERVED_PRIVATE_BLOCKS - 1)
/*                                                   0x10000000 + 0xeffffc40 - 1   = 0xfffffc3f */

#define CHUNK_SIZE (BLOCKS_IN_CHUNK * FILE_BLOCK_SIZE)

#define IVEC_SIZE 8  /* don't know for sure, the man page doesn't say, i made the buffer larger and looked at if afterwards. */

#define NUM_STORED_SEQ 4096     /* number of sequence tables that are stored, 4096 = 1 MB worth and uses 1 MB worth of RAM */

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp last_prep_disk;

    uint8 total_blocks[4];
    uint8 used_blocks[4];
    TimeStamp last_change;

    uint8 block_offset_to_chunks[4];
    uint8 used_chunks[4];
    uint8 unused7[4];
    uint8 unused8[4];
} Disk_Header;

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp last_prep_disk;

    uint8 total_public_blocks[4];
    uint8 used_public_blocks[4];
    TimeStamp public_last_change;

    uint8 total_private_blocks[4];
    uint8 used_private_blocks[4];
    TimeStamp private_last_change;

    uint8 unused5[4];
    uint8 next_private_chunk[4];
    uint8 unused6[4];
    uint8 unused7[4];
} Disk_Header_0021;

typedef struct {
    char magic_number[4];
    char version_string[4];
    TimeStamp time_last_change;

    uint8 total_public_blocks[4];
    uint8 total_private_blocks[4];
    uint8 unused1[4];
    uint8 unused2[4];

    uint8 used_public_blocks[4];
    uint8 used_private_blocks[4];
    uint8 unused3[4];
    uint8 unused4[4];

    uint8 unused5[4];
    uint8 next_private_chunk[4];
    uint8 unused6[4];
    uint8 unused7[4];
} Disk_Header_0016_to_0020;

extern uint32 nwos_total_public_blocks;
extern uint32 nwos_total_private_blocks;
extern uint32 nwos_used_public_blocks;
extern uint32 nwos_used_private_blocks;


typedef struct {
  uint32 ref;
  uint16 used;
  uint16 index;
} Chunk_Info;

extern uint32 nwos_total_private_chunks;
extern uint32 nwos_used_private_chunks;


#define MAX_REFS_IN_REF_LIST (((FILE_BLOCK_SIZE - sizeof(CommonHeader)) / sizeof(ObjRef)) - 1)
#define MAX_REFS_IN_SIDECAR  (((FILE_BLOCK_SIZE - 12) / sizeof(ObjRef)) - 1)

typedef struct ref_list_extra_block {
    struct ref_list_extra_block* next_block_ptr;
    int dirty;
    ObjRef id;
    uint8 checksum[4];
    ObjRef refs[MAX_REFS_IN_SIDECAR];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_Extra_Block;

typedef struct {
    struct ref_list_extra_block* next_block_ptr;
    ReferenceList list;
    ObjRef refs[MAX_REFS_IN_REF_LIST];
    ObjRef next_block_ref;
    uint8 ivec[IVEC_SIZE];
} Ref_List_First_Block;


extern bool verbose;

extern uint8 nwos_random_sequence[NUM_STORED_SEQ][FILE_BLOCK_SIZE];     /* resequencing tables */

extern ObjRef nwos_public_class_definition_class_ref;
extern ObjRef nwos_private_class_definition_class_ref;
extern ObjRef nwos_reference_list_class_ref;
extern ObjRef nwos_next_public_ref;


extern void nwos_create_class_definition(char* name_of_class);
extern void nwos_clone_class_definition(ObjRef* class_to_clone_ref, ObjRef* clone_ref);


extern void* nwos_malloc(size_t size);
extern void  nwos_free(void* ptr);

extern void nwos_fill_in_common_header(CommonHeader* common, ObjRef* ref, ObjRef* class_definition_ref);

extern uint32 nwos_hash_uint32_ref(uint32 ref);
extern uint32 nwos_hash_ref(ObjRef* ref);

extern bool nwos_object_exists(ObjRef* ref);
extern void nwos_generate_new_completely_random_id(ObjRef* ref);
extern void nwos_generate_new_public_id(ObjRef* ref);
extern void nwos_set_bit_in_map(uint32 block);
extern void nwos_clear_bit_in_map(uint32 block);
extern bool nwos_test_bit_in_map(uint32 block);
extern bool nwos_block_used(ObjRef* ref);
extern bool nwos_read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE]);
extern void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE]);
extern void nwos_write_object_to_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_write_public_object_to_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_overwrite_object_to_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_read_object_from_disk(ObjRef* ref, void* object, size_t size);
extern void nwos_create_english_language(ObjRef* ref);
extern void nwos_create_all_twelve_months();
extern void nwos_create_all_area_codes();
extern void nwos_set_spelling_class_ref(ObjRef* spelling_class_ref);
extern void nwos_set_name_class_ref(ObjRef* name_class_ref);
extern bool nwos_create_root(ObjRef* ref);
extern void nwos_remove_object(ObjRef* ref);
extern void nwos_create_reference_list_with_existing_id(ObjRef* for_obj, ObjRef* ref_list);
extern void nwos_flush_dirty_ref_lists();
extern void nwos_create_public_acronym(char* acronym, ObjRef* ref);
extern uint32 nwos_next_sequence(uint32* linear_seed, uint32* serial_seed);

extern void nwos_read_object_from_disk_and_decrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE]);
extern void nwos_write_object_to_disk_and_encrypt(ObjRef* ref, void* object, size_t size, uint8 ivec[IVEC_SIZE], uint8 seq_table[FILE_BLOCK_SIZE]);

extern char* nwos_public_path;
extern char* nwos_private_path;
extern void  nwos_initialize_disk_io(StorageType type, char* path);
extern void  nwos_terminate_disk_io();
extern void  nwos_allocate_new_chunk(uint32 ref);
extern void nwos_generate_new_closely_spaced_id(ObjRef* last_ref, int density, ObjRef* new_ref);
extern void nwos_generate_new_1_in_N_id(ObjRef* last_ref, uint32 n, ObjRef* new_ref);

extern void nwos_private_create_city(ObjRef* city_class_ref, ObjRef* name_ref, ObjRef* state_ref, ObjRef* city_ref);

extern ObjCreateResult nwos_create_url(char* url, ObjRef* ref);

extern void nwos_create_public_manufacturer(char* brand_name, char* acronym, char* url_list[], ObjRef* ref);

extern void nwos_update_file_001_object_to_current(void* object, size_t size);

#endif

