/*
--             This file is part of the New World OS project
--                Copyright (C) 2006, 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: import_file.c,v $
-- Revision 1.15  2007/09/02 19:45:31  jsedwards
-- Added calls to set the block estimate.
--
-- Revision 1.14  2007/09/01 21:52:57  jsedwards
-- Changed 'encryption_options' table to 'security_options' to match up with
-- changes made earlier (Aug 12th).
--
-- Revision 1.13  2007/08/12 20:45:00  jsedwards
-- Change all of the "Encryption Level" stuff to "Security Level" because it
-- doesn't really change the encryption at all, all it does is change the
-- randomization of where objects are stored.
--
-- Revision 1.12  2007/08/06 02:22:45  jsedwards
-- Added "high" security option.
--
-- Revision 1.11  2007/07/24 04:47:03  jsedwards
-- Added "--add-revision" option and code to add new revision to a file that
-- is already in the system.
--
-- Revision 1.10  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.9  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.8  2007/01/27 14:59:03  jsedwards
-- Changed to handle new possible return values (duplicate path or file).
--
-- Revision 1.7  2007/01/17 13:17:43  jsedwards
-- Added security level options.
--
-- Revision 1.6  2007/01/07 20:31:55  jsedwards
-- Added call to log arguments.
--
-- Revision 1.5  2007/01/07 03:27:06  jsedwards
-- Added call to set encryption level (Low by default).
--
-- Revision 1.4  2006/12/07 14:10:25  jsedwards
-- Removed call to file_setup, no longer needed.
--
-- Revision 1.3  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.4  2006/10/13 12:58:52  jsedwards
-- Added printing of time to import each file.
--
-- Revision 1.1.2.3  2006/10/13 11:48:27  jsedwards
-- Add call to flush block maps after each file is read.
--
-- Revision 1.1.2.2  2006/10/03 12:53:07  jsedwards
-- Changed so that instead of calling a separate routine after initialize to
-- change the already opened storage, you call it now with a type of storage
-- parameter and a path to the storage.  The problem with the other way was
-- if you tried reading a compressed file on another machine it tried to open
-- the default file which didn't exist.
--
-- Revision 1.1.2.1  2006/09/18 01:45:37  jsedwards
-- Program to store a file into objectify.
--
*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"


struct options {
  Security_Level level;
  char* option;
  char* description;
} security_options[] = 
  {
    { Security_Minimal,  "--minimal-security",  "minimal"  },
    { Security_Very_Low, "--very-low-security", "very low" },
    { Security_Low,      "--low-security",      "low"      },
    { Security_Medium,   "--medium-security",   "medium"   },
    { Security_High,     "--high-security",     "high"   },
    { Security_Extreme,  "--extreme-security",  "extreme"  },
  };

#define NUM_OPTIONS (sizeof(security_options)/sizeof(struct options))


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef ref;
    ObjCreateResult result;
    int i;
    int j;
    time_t start_time;
    int result_counts[NUM_OBJ_RESULTS];
    bool Unknown_result_in_switch_statement = false;

    for (i = 0; i < NUM_OBJ_RESULTS; i++) result_counts[i] = 0;

    printf("\n");

    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    for (j = 0; j < NUM_OPTIONS; j++)
	    {
		if (strcmp(argv[i], security_options[j].option) == 0) break;
	    }

	    if (j == NUM_OPTIONS)
	    {
		if (strcmp(argv[i], "--add-revision") == 0)
		{
		    i++;

		    if (i == argc || argv[i][0] == '-')
		    {
			fprintf(stderr, "Error: a file must be specified after --add-revision\n");
			exit(1);
		    }
		}
		else
		{
		    fprintf(stderr, "Unknown option: %s\n", argv[i]);
		    exit(1);
		}
	    }
	}
    }

    nwos_log_arguments(argc, argv);

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_set_security_level(Security_Low);


    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    if (strcmp(argv[i], "--add-revision") == 0)
	      {
		i++;    /* point to the file argument */

		assert(i < argc && argv[i][0] != '-');

		nwos_set_block_estimate(nwos_estimate_blocks_for_file(argv[i]));

		start_time = time(NULL);

		result = nwos_add_new_revision_of_file(argv[i], &ref);

		result_counts[result]++;

		switch(result)
		{
		  case CREATED_NEW_REVISION:
		    printf("file created: %02x%02x%02x%02x   time: %d seconds\n", 
			   ref.id[0], ref.id[1], ref.id[2], ref.id[3], 
			   (int) (time(NULL) - start_time));
		    break;

		  case PATH_NOT_FOUND:
		    fprintf(stderr, "Error: no previous version of '%s' exists, will be skipped\n", argv[i]);
		    break;

		  case MULTIPLE_FILES:
		    fprintf(stderr, "Error: multiple files with the name %s exist, ", argv[i]);
		    fprintf(stderr, "cannot cope with them.\n");
		    break;

		  case DUPLICATE_FILE:
		    printf("WARNING: a file with the same size, md5, and sha1 sums already exists.\n");
		    printf("         In this version it will be skipped!!\n");
		    break;

		  case ERROR_OCCURRED:
		    break;

		  default:      /* keep the compiler form complaining */
		    assert(Unknown_result_in_switch_statement);
		    break;
		}

	    }
	    else
	    {
		for (j = 0; j < NUM_OPTIONS; j++)
		{
		    if (strcmp(argv[i], security_options[j].option) == 0) break;
		}

		assert(j < NUM_OPTIONS);

		printf("setting security level to: %s\n", security_options[j].description);

		nwos_set_security_level(security_options[j].level);
	    }
	}
	else
	{
	    nwos_set_block_estimate(nwos_estimate_blocks_for_file(argv[i]));

	    start_time = time(NULL);

	    result = nwos_create_file(argv[i], &ref);

	    result_counts[result]++;

	    switch(result)
	    {
	      case CREATED_NEW:
		printf("file created: %02x%02x%02x%02x   time: %d seconds\n", 
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3], 
		       (int) (time(NULL) - start_time));
		break;

	      case FOUND_EXISTING:
		printf("existing file found: %02x%02x%02x%02x\n",
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
		break;

	      case DUPLICATE_PATH:
		printf("WARNING: a file by the name of: %s already exists and --add-revision.\n", argv[i]);
		printf("         was not specified, it will be skipped!!\n");
		break;

	      case DUPLICATE_FILE:
		printf("WARNING: a file with the same size, md5, and sha1 sums already exists.\n");
		printf("         In this version it will be skipped!!\n");
		break;

	      case ERROR_OCCURRED:
		break;

	      default:   /* keep the compiler form complaining */
		assert(Unknown_result_in_switch_statement);
		break;
	    }

	    nwos_flush_bit_maps();
	}
    }

    printf("files created:   %d\n", result_counts[CREATED_NEW]);
    printf("new revisions:   %d\n", result_counts[CREATED_NEW_REVISION]);
    printf("existing files:  %d\n", result_counts[FOUND_EXISTING]);
    printf("path not found:  %d\n", result_counts[PATH_NOT_FOUND]);
    printf("duplicate paths: %d\n", result_counts[DUPLICATE_PATH]);
    printf("duplicate files: %d\n", result_counts[DUPLICATE_FILE]);
    printf("multiple files:  %d\n", result_counts[MULTIPLE_FILES]);
    printf("files w/errors:  %d\n", result_counts[ERROR_OCCURRED]);

    nwos_terminate_objectify();

    return 0;
}


