/*
--             This file is part of the New World OS project
--                    Copyright (C) 2007 QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: patch_compressed.c,v $
-- Revision 1.6  2007/11/20 05:45:31  jsedwards
-- Fix comparison of reference id's.
--
-- Revision 1.5  2007/11/17 21:41:41  jsedwards
-- Added codee to determine if it is an older wrapped id file or the new
-- sequential file and decide where to insert the new blocks using the
-- appropriate method.
--
-- Revision 1.4  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.3  2007/06/18 13:07:38  jsedwards
-- Fixed to close output file.
--
-- Revision 1.2  2007/03/24 14:09:25  jsedwards
-- Added loop to finish writing the patch file blocks to the output file
-- after file 1 has reached end of file (blocks were added at the end).
--
-- Revision 1.1  2007/03/15 13:44:59  jsedwards
-- Initial version.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>    /* define sleep() */

#include "objectify_private.h"

uint32 convert_4_uint8_to_uint32(uint8 byte[4])
{
  return ((uint32)byte[0] << 24) | ((uint32)byte[1] << 16) | ((uint32)byte[2] << 8) | (uint32)byte[3];
}


static int version_string_to_int(char string[4])
{
    int i;
    int result = 0;

    for (i = 0; i < 4; i++) if (string[i] != '0') break;

    while (i < 4)
    {
	if (!isdigit(string[i])) return -1;

	result = result * 10 + string[i] - '0';

	i++;
    }

    return result;
}


int main(int argc, char* argv[])
{
    FILE* fp1;
    FILE* fp2;
    FILE* fp3 = NULL;
    uint8 buf1[FILE_BLOCK_SIZE];
    uint8 buf2[FILE_BLOCK_SIZE];
    int block1;
    int block2;
    size_t read1;
    size_t read2;
    size_t write3;
    Disk_Header header1;
    Disk_Header header2;
    bool wrapped_file = false;     /* true if older file that wasn't sequencial */
    int header1_version;
    int header2_version;


    if (argc != 4)
    {
	fprintf(stderr, "usage: %s old_file patch_file new_file\n", argv[0]);
	exit(1);
    }


    /* Open the old file and check it */

    fp1 = fopen(argv[1], "r");
    if (fp1 == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    read1 = fread(buf1, 1, sizeof(buf1), fp1);

    if (read1 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp1))
	{
	    perror(argv[1]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[1]);
	}
	fclose(fp1);
	exit(1);
    }

    memcpy(&header1, buf1, sizeof(header1));

    if (memcmp(header1.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify file: %s\n", argv[1]);
	fclose(fp1);
	exit(1);
    }


    /* Open the patch file and check it */

    fp2 = fopen(argv[2], "r");
    if (fp2 == NULL)
    {
	perror(argv[2]);
	exit(1);
    }

    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    /* Verify the old file header and the patch file header are exactly the same */

    if (memcmp(buf1, buf2, FILE_BLOCK_SIZE) != 0)
    {
	fprintf(stderr, "Old file doesn't match patch file!\n");
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    /* Now read in the new file header */

    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    if (read2 != FILE_BLOCK_SIZE)
    {
	if (ferror(fp2))
	{
	    perror(argv[2]);
	}
	else
	{
	    fprintf(stderr, "Unexpected end of file: %s\n", argv[2]);
	}
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    memcpy(&header2, buf2, sizeof(header2));

    if (memcmp(header2.magic_number, "NWOS", 4) != 0)
    {
	fprintf(stderr, "Not an Objectify patch file: %s\n", argv[2]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }


    header1_version = version_string_to_int(header1.version_string);

    if (header1_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[1]);
	exit(1);
    }

    header2_version = version_string_to_int(header2.version_string);

    if (header2_version < 1)
    {
	fprintf(stderr, "Invalid version string in file: %s\n", argv[2]);
	exit(1);
    }

    if (header1_version < 23 && header2_version < 23)
    {
	wrapped_file = true;   /* it's an old file */
    }
    else if (header1_version < 23 || header2_version < 23)
    {
	fprintf(stderr, "ERROR: '%s' is an old wrapped file, but file '%s' is not, cannot diff!\n",
		header1_version < 23 ? argv[1] : argv[2],
		header1_version < 23 ? argv[2] : argv[1]);
	exit(1);
    }


    fp3 = fopen(argv[3], "w");

    if (fp3 == NULL)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	exit(1);
    }

    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

    if (write3 != FILE_BLOCK_SIZE)
    {
	perror(argv[3]);
	fclose(fp1);
	fclose(fp2);
	fclose(fp3);
	exit(1);
    }

    block1 = 1;
    block2 = 1;

    read1 = fread(buf1, 1, sizeof(buf1), fp1);
    read2 = fread(buf2, 1, sizeof(buf2), fp2);

    while (!feof(fp1) && !feof(fp2) && read1 == FILE_BLOCK_SIZE && read2 == FILE_BLOCK_SIZE)
    {
	if (buf1[0] != 0 || buf1[1] != 0 || buf1[2] != 0 || buf1[3] != 0)
	{
	    printf("\n%s block %d - first four bytes not zero: %02x%02x%02x%02x\n",
		   argv[1], block1, buf1[0], buf1[1], buf1[2], buf1[3]);
	    break;
	}

	if (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] == buf2[7])
	{
	    /* blocks have the same id number, must have changed */

	    if (memcmp(buf1, buf2, FILE_BLOCK_SIZE) != 0)
	    {
		printf("%02x%02x%02x%02x: changed\n",
		       buf1[4], buf1[5], buf1[6], buf1[7]);

		write3 = fwrite(buf2, 1, sizeof(buf2), fp3);
	    }
	    else   /* block stayed the same write the old block */
	    {
		write3 = fwrite(buf1, 1, sizeof(buf1), fp3);
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[2]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }

	    read1 = fread(buf1, 1, sizeof(buf1), fp1);
	    block1++;
	    read2 = fread(buf2, 1, sizeof(buf2), fp2);
	    block2++;
	}
	else   /* id is different, assume this an add, since right now can't delete */
	{
	    if (wrapped_file)
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] == buf2[0] && buf1[5] == buf2[1] && buf1[6] == buf2[2] && buf1[7] == buf2[3])
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    buf2[0] = 0;
		    buf2[1] = 0;
		    buf2[2] = 0;
		    buf2[3] = 0;

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }
	    else   /* new non-wrapped file that is sequential */
	    {
		/* if we've found the proper place to insert it */
		if (buf1[4] > buf2[4] || 
		    (buf1[4] == buf2[4] && buf1[5] > buf2[5]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] > buf2[6]) ||
		    (buf1[4] == buf2[4] && buf1[5] == buf2[5] && buf1[6] == buf2[6] && buf1[7] > buf2[7])
		    )
		{
		    printf("%02x%02x%02x%02x: added\n",
			   buf2[4], buf2[5], buf2[6], buf2[7]);

		    write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

		    read2 = fread(buf2, 1, sizeof(buf2), fp2);
		    block2++;
		}
		else  /* not time to insert yet, just write the old block */
		{
		    write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

		    read1 = fread(buf1, 1, sizeof(buf1), fp1);
		    block1++;
		}
	    }

	    if (write3 != FILE_BLOCK_SIZE)
	    {
		perror(argv[3]);
		fclose(fp1);
		fclose(fp2);
		fclose(fp3);
		exit(1);
	    }
	}

    }

    /* finish writing output */

    while (!feof(fp1) && read1 == FILE_BLOCK_SIZE)
    {
	write3 = fwrite(buf1, 1, sizeof(buf1), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read1 = fread(buf1, 1, sizeof(buf1), fp1);
	block1++;
    }

    /* more in the patch file */

    while (!feof(fp2) && read2 == FILE_BLOCK_SIZE)
    {
	printf("%02x%02x%02x%02x: added\n", buf2[4], buf2[5], buf2[6], buf2[7]);

	write3 = fwrite(buf2, 1, sizeof(buf2), fp3);

	if (write3 != FILE_BLOCK_SIZE)
	{
	    perror(argv[3]);
	    fclose(fp1);
	    fclose(fp2);
	    fclose(fp3);
	    exit(1);
	}

	read2 = fread(buf2, 1, sizeof(buf2), fp2);
	block1++;
    }

    printf("\n");


    if (memcmp(header1.version_string, header2.version_string, 4) != 0)
    {
	fprintf(stderr, "WARNING, versions don't match - %s: %c%c%c%c  %s: %c%c%c%c\n",
		argv[1], 
		header1.version_string[0], header1.version_string[1],
		header1.version_string[2], header1.version_string[3],
		argv[2], 
		header2.version_string[0], header2.version_string[1],
		header2.version_string[2], header2.version_string[3]);
    }


    if (!feof(fp1) && read1 != FILE_BLOCK_SIZE)
    {
	perror(argv[1]);
    }

    fclose(fp1);

    if (!feof(fp2) && read2 != FILE_BLOCK_SIZE)
    {
	perror(argv[2]);
    }

    fclose(fp2);

    if (fclose(fp3) != 0)
    {
	perror(argv[3]);
    }

    return 0;
}
