/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: time_stamp.h,v $
-- Revision 1.2  2007/07/01 19:44:13  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.1  2006/12/22 14:01:37  jsedwards
-- Copy of 0014 version from root directory.
--
-- Revision 1.6  2006/12/01 14:39:22  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.5  2006/11/11 12:01:07  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.3.2.5  2006/09/14 11:43:45  jsedwards
-- Changed values in inlined functions from 'int' to 'uint'.
--
-- Revision 1.3.2.4  2006/09/13 13:33:22  jsedwards
-- Added static, inline functions to insert values into time stamp.  Also
-- added asserts to export functions.
--
-- Revision 1.3.2.3  2006/09/10 20:41:13  jsedwards
-- Added static inline routines to extract the individual values from a time
-- stamp.
--
-- Revision 1.3.2.2  2006/08/24 11:56:38  jsedwards
-- Added convert_time_struct_to_timeval routine.
--
-- Revision 1.3.2.1  2006/08/23 13:22:26  jsedwards
-- Added nwos_convert_timespec_to_time_stamp function.
--
-- Revision 1.3  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.2  2005/11/26 15:15:24  jsedwards
-- Added 'nwos' prefix to function name and changed parameter to TimeStamp
-- instead of uint8[8].
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
-- Revision 1.2  2005/11/23 20:22:02  jsedwards
-- Changed to store as 8 bytes instead of a 64 bit integer.  This has the
-- advantage of working the same for both big and little endian machines.
--
-- Revision 1.1  2004/10/31 18:23:39  jsedwards
-- Moved time stamp definitions from 'types.h' into here.
--
*/

#ifndef TIME_STAMP_H
#define TIME_STAMP_H

#include <assert.h>

#include "types.h"


/* changed to a more broken out format for the following reasons:
--  1. no endian-ness problems
--  2. it is only written once, so we suffer the conversion penalty once
--
-- It is broken down as follows:
--     12 bits - year - good until the year 4095, if this junk lasts
--                       that long I appologize.
--      4 bits - month - 1 to 12
--      5 bits - day of month - 1 to 31
--      3 bits - day of week - 1 to 7
--  ------
--     24 bits - date
--
--      3 bits - unused
--      5 bits - hour - 0 to 23
--      6 bits - minute - 0 to 59
--      6 bits - second - 0 to 59
--     20 bits - microseconds - 0 to 999,999
--   -----
--     40 bits - time
*/

typedef uint8 TimeStamp[8];
extern void nwos_get_time_stamp(TimeStamp time_stamp);
extern void nwos_convert_timespec_to_time_stamp(struct timespec* ts, TimeStamp time_stamp);
extern void nwos_convert_time_stamp_to_timeval(TimeStamp time_stamp, struct timeval* tv);


static inline uint nwos_extract_year_from_time_stamp(TimeStamp stamp)
{
    uint year = (stamp[0] << 4) | (stamp[1] >> 4);

    //    assert(2004 <= year);   /* this program didn't exist until 2004 */

    return year;
}

static inline uint nwos_extract_month_from_time_stamp(TimeStamp stamp)
{
    uint month = stamp[1] & 0xF;

    assert(1 <= month && month <= 12);

    return month;
}

static inline uint nwos_extract_day_of_month_from_time_stamp(TimeStamp stamp)
{
    uint day_of_month = stamp[2] >> 3;

    assert(1 <= day_of_month && day_of_month <= 31);

    return day_of_month;
}

static inline uint nwos_extract_day_of_week_from_time_stamp(TimeStamp stamp)
{
    uint day_of_week = stamp[2] & 0x7;

    assert(1 <= day_of_week && day_of_week <= 7);

    return day_of_week;
}

static inline uint nwos_extract_hour_from_time_stamp(TimeStamp stamp)
{
    uint hour = stamp[3] & 0x1F;

    assert(0 <= hour && hour <= 23);

    return hour;
}

static inline uint nwos_extract_minute_from_time_stamp(TimeStamp stamp)
{
    uint minute = stamp[4] >> 2;

    assert(0 <= minute && minute <= 59);

    return minute;
}

static inline uint nwos_extract_second_from_time_stamp(TimeStamp stamp)
{
    uint second = ((stamp[4] << 4) | (stamp[5] >> 4)) & 0x3F;

    assert(0 <= second && second <= 59);

    return second;
}

static inline uint nwos_extract_microsecond_from_time_stamp(TimeStamp stamp)
{
    uint usec = ((stamp[5] << 16) | (stamp[6] << 8) | stamp[7]) & 0xFFFFF;

    assert(0 <= usec && usec <= 999999);

    return usec;
}


static inline void nwos_insert_year_into_time_stamp(uint year, TimeStamp stamp)
{
  //    assert(2004 <= year);   /* this program didn't exist until 2004 */

    stamp[0] = year >> 4;
    stamp[1] = (year << 4) | (stamp[1] & 0xF);
}

static inline void nwos_insert_month_into_time_stamp(uint month, TimeStamp stamp)
{
    assert(1 <= month && month <= 12);

    stamp[1] = (stamp[1] & 0xF0) | (month & 0xF);
}

static inline void nwos_insert_day_of_month_into_time_stamp(uint day_of_month, TimeStamp stamp)
{
    assert(1 <= day_of_month && day_of_month <= 31);

    stamp[2] = (day_of_month << 3) | (stamp[2] & 0x7);
}

static inline void nwos_insert_day_of_week_into_time_stamp(uint day_of_week, TimeStamp stamp)
{
    assert(1 <= day_of_week && day_of_week <= 7);

    stamp[2] = (stamp[2] & 0xF8) | (day_of_week & 0x7);
}

static inline void nwos_insert_hour_into_time_stamp(uint hour, TimeStamp stamp)
{
    assert(0 <= hour && hour <= 23);

    stamp[3] = (hour & 0x1F);
}

static inline void nwos_insert_minute_into_time_stamp(uint minute, TimeStamp stamp)
{
    assert(0 <= minute && minute <= 59);

    stamp[4] = (minute << 2) | (stamp[4] & 0x3);
}

static inline void nwos_insert_second_into_time_stamp(uint second, TimeStamp stamp)
{
    assert(0 <= second && second <= 59);

    stamp[4] = (stamp[4] & 0xFC) | (second >> 4);
    stamp[5] = (second << 4) | (stamp[5] & 0xF);
}

static inline void nwos_insert_microsecond_into_time_stamp(uint usec, TimeStamp stamp)
{
    assert(0 <= usec && usec <= 999999);

    stamp[5] = (stamp[5] & 0xF0) | (usec >> 16);
    stamp[6] = usec >> 8;
    stamp[7] = usec;
}



#endif

