/*             This file is part of the New World OS project
--                 Copyright (C) 2004-2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: disk_io.c,v $
-- Revision 1.12  2007/06/28 19:06:09  jsedwards
-- Add log message when generating a random ID or 1 in N ID.
--
-- Revision 1.11  2007/06/28 13:26:06  jsedwards
-- Removed line in allocate_new_chunk that added the blocks used for the bit
-- map to the used_blocks count.  In 0023 the blocks used count doesn't
-- include the bit map blocks.
--
-- Revision 1.10  2007/06/28 04:37:32  jsedwards
-- Fix the stupid binary search thing yet again.
--
-- Revision 1.9  2007/06/28 02:43:41  jsedwards
-- Tweak the binary search for id a bit more.
--
-- Revision 1.8  2007/06/28 02:36:37  jsedwards
-- Make binary search for reference more robust.
--
-- Revision 1.7  2007/06/27 01:13:42  jsedwards
-- Fix nwos_generate_new_1_in_N_id to generate new id if current one doesn't
-- fit in a chunk.
--
-- Revision 1.6  2007/06/26 20:02:37  jsedwards
-- Changed algorithm to generate random id to generate id in existing chunk
-- instead of completely random.
--
-- Revision 1.5  2007/06/26 16:46:06  jsedwards
-- Moved code to compute number of blocks used in an entry to the read
-- cache entry function.
--
-- Revision 1.4  2007/06/26 14:31:44  jsedwards
-- Rename private_generate_new_completely_random_id to
-- generate_new_completely_random_id and make it static.
--
-- Revision 1.3  2007/06/26 14:21:34  jsedwards
-- Moved random id generation functions from objectify.c.
--
-- Revision 1.2  2007/06/26 13:20:53  jsedwards
-- Changed to use new chunk index to convert reference ids to block numbers on
-- disk, instead of wrapping them around the disk size.
--
-- Revision 1.1  2007/06/25 05:28:54  jsedwards
-- New file created from variables, functions and code related to disk I/O,
--  taken from the objectfy.c file.
--
*/

#define _LARGEFILE64_SOURCE

#include <assert.h>
#include <byteswap.h>
#include <endian.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>


#include "objectify_private.h"


static Disk_Header disk_header;

char*  nwos_public_path = DEFAULT_PUBLIC_PATH;
char*  nwos_private_path;

uint32 nwos_total_private_blocks;
uint32 nwos_used_public_blocks;
uint32 nwos_used_private_blocks;

uint32 nwos_block_offset_to_chunks;
uint32 nwos_total_private_chunks = 0;
uint32 nwos_used_private_chunks;

static int    public_file_desc;
static int    private_file_desc;
static bool    read_only;
static bool    modified;
static ObjRef* storage_index;
static int     storage_count;

static uint32* chunk_index;
static bool chunk_index_modified;


#define BIT_MAP_CACHE_SIZE 256

typedef struct {
  uint32 chunk;
  int age;
  bool dirty;
  uint16 used;
  uint8* map;
} Bit_Map_Cache_Entry;

static Bit_Map_Cache_Entry bit_map_cache[BIT_MAP_CACHE_SIZE];

static int bit_map_tick = 0;

static void write_bit_map(Bit_Map_Cache_Entry* entry);


void special_update_total_blocks_for_expand_0021(uint32 new_reserved_blocks)
{
    nwos_total_private_blocks = new_reserved_blocks;
    nwos_uint32_to_4_uint8(&nwos_total_private_blocks, disk_header.total_blocks);
}



/*****************************************/
/* Function that indexes compressed file */
/*****************************************/

#define INDEX_ALLOC_SIZE 262144 

void index_compressed_file()
{
    int i;
    int size_of_index = INDEX_ALLOC_SIZE;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    uint8 header[FILE_BLOCK_SIZE];
    char* index_path = NULL;
    FILE* index_fp;

#ifdef NO_INDEX_FILE
    index_fp = NULL;
#else
    assert(private_file_desc > 0);

    /* first see if there is an existing index file */

    i = strlen(nwos_private_path);
    index_path = nwos_malloc(i+5);  /* 4 for .ndx and 1 for null */
    strcpy(index_path, nwos_private_path);

    if (index_path[i-4] == '.' && index_path[i-3] == 'o' && index_path[i-2] == 'b' && index_path[i-1] == 'j')
    {
	i = i - 4;   /* write over the old .obj */
    }

    index_path[i++] = '.';
    index_path[i++] = 'n';
    index_path[i++] = 'd';
    index_path[i++] = 'x';
    index_path[i++] = '\0';

    fprintf(stderr, "index_path: %s\n", index_path);

    index_fp = fopen(index_path, "r");
#endif

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    /* read the first block, it contains the header */
    bytes_read = read(private_file_desc, header, sizeof(header));

    assert(bytes_read == sizeof(header));

    /* if there is an index file see if the header matches */
    if (index_fp != NULL)
    {
	bytes_read = fread(buffer, 1, FILE_BLOCK_SIZE, index_fp);
	assert(bytes_read == sizeof(buffer));

	fprintf(stderr, "Index file found: ");
	fflush(stderr);

	if (memcmp(header, buffer, FILE_BLOCK_SIZE) == 0)   /* we have a winner */
	{
	    fprintf(stderr, "match\n");
	}
	else
	{
	    fprintf(stderr, "no match\n");
	    fclose(index_fp);
	    index_fp = NULL;
	}
    }
    else /* no index file */
    {
	errno = 0;   /* clear errno for later */
    }

    fprintf(stderr, "Indexing");
    fflush(stderr);

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += INDEX_ALLOC_SIZE;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	if (index_fp != NULL)   /* read index from file */
	{
	    bytes_read = fread(&storage_index[i], sizeof(ObjRef), 1, index_fp);

	    if (bytes_read != 1) break;
	}
	else
	{
	    bytes_read = read(private_file_desc, buffer, sizeof(buffer));

	    if (bytes_read != sizeof(buffer)) break;

	    memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));
	}

	i++;
    }

    storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);

    if (index_fp != NULL)
    {
	if (ferror(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
	fclose(index_fp);
    }
    else
    {
	if (errno != 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}
#ifndef NO_INDEX_FILE
	fprintf(stderr, "Writing index file: %s\n", index_path);

	index_fp = fopen(index_path, "w");

	if (fwrite(header, 1, sizeof(header), index_fp) != sizeof(header))
	{
	    perror(index_path);
	    exit(1);
	}

	if (fwrite(&storage_index[1], sizeof(ObjRef), i, index_fp) != i)
	{
	    perror(index_path);
	    exit(1);
	}

	if (fclose(index_fp))
	{
	    perror(index_path);
	    exit(1);
	}
#endif
    }

    assert(bytes_read == 0);

    if (index_path != NULL)
    {
	nwos_free(index_path);
    }
}



/**********************************************/
/* Function to initialize the disk I/O module */
/**********************************************/

void initialize_random_number_generator()
{
    struct timespec ts;

    clock_gettime(CLOCK_REALTIME, &ts);
    srandom((uint32)ts.tv_sec ^ (uint32)ts.tv_nsec);

#ifdef DISABLE_SECURITY_FEATURES
    randomxxx = 0x00000100;     /* start objects here */
#endif
}


void nwos_initialize_disk_io(StorageType type, char* path)
{
    size_t chunk_index_size_in_bytes;
    char log_msg[128];

    /* make sure the storage is something we can deal with */
    assert(type == Drive_Or_Partition_RO || type == Drive_Or_Partition_RW || 
	   type == Sparse_File_RO || type == Sparse_File_RW || type == Compressed_File_RO);

    assert(nwos_private_path == NULL);   /* make sure this is only called once */
    assert(private_file_desc == 0);
    assert(public_file_desc == 0);

    initialize_random_number_generator();

    nwos_private_path = path;

    /********************************/
    /* open the public objects file */
    /********************************/

    public_file_desc = open(nwos_public_path, O_RDONLY | O_LARGEFILE);

    if (public_file_desc < 0)
    {
	perror(nwos_public_path);
	exit(1);
    }

    if (read(public_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
    {
	snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_public_path);
	perror(log_msg);
	exit(1);
    }

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    nwos_4_uint8_to_uint32(disk_header.used_blocks, &nwos_used_private_blocks);

    snprintf(log_msg, sizeof(log_msg), " used public blocks: %9u", nwos_used_public_blocks);
    nwos_log(log_msg);


    /***************************************************/
    /* if a private objects file was specified open it */
    /***************************************************/

    if (nwos_private_path != NULL)
    {
	read_only = ((type & 1) == 0);

	if (read_only)
	{
	    private_file_desc = open(nwos_private_path, O_RDONLY | O_LARGEFILE);
	}
	else
	{
	    private_file_desc = open(nwos_private_path, O_RDWR | O_LARGEFILE);
	}

	if (private_file_desc < 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	if (type == Compressed_File_RO)
	{
	    index_compressed_file();   /* build an index into the compressed file */

	    if (lseek(private_file_desc, 0LL, SEEK_SET) < 0)
	    {
		perror("rewinding after indexing");
		exit(1);
	    }
	}

	if (read(private_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
	{
	    snprintf(log_msg, sizeof(log_msg), "reading disk header from: %s", nwos_private_path);
	    perror(log_msg);
	    exit(1);
	}

	if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
	{
	    fprintf(stderr, "Missing magic number in disk header\n");
	    exit(1);
	}

	if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
	{
	    fprintf(stderr, "Incorrect version string in disk header\n");
	    exit(1);
	}

	nwos_4_uint8_to_uint32(disk_header.block_offset_to_chunks, &nwos_block_offset_to_chunks);
	nwos_4_uint8_to_uint32(disk_header.total_blocks, &nwos_total_private_blocks);
	nwos_4_uint8_to_uint32(disk_header.used_blocks, &nwos_used_private_blocks);
	nwos_4_uint8_to_uint32(disk_header.used_chunks, &nwos_used_private_chunks);

	nwos_total_private_chunks = nwos_total_private_blocks / BLOCKS_IN_CHUNK;

	if (lseek(private_file_desc, FILE_BLOCK_SIZE, SEEK_SET) < 0)
	{
	    perror("chunk index");
	    exit(1);
	}

	chunk_index_size_in_bytes = nwos_total_private_chunks * sizeof(*chunk_index);
	chunk_index = nwos_malloc(chunk_index_size_in_bytes);

	if (read(private_file_desc, chunk_index, chunk_index_size_in_bytes) != chunk_index_size_in_bytes)
	{
	    snprintf(log_msg, sizeof(log_msg), "reading chunk index from: %s", nwos_private_path);
	    perror(log_msg);
	    exit(1);
	}

#if __BYTE_ORDER == __LITTLE_ENDIAN
	{
	  int i;
	  for (i = 0; i < (int)nwos_total_private_chunks; i++)
	  {
	      chunk_index[i] = bswap_32(chunk_index[i]);
	  }
	}
#endif

	chunk_index_modified = false;

	if (type == Drive_Or_Partition_RO || type == Drive_Or_Partition_RW)
	{
	    assert(nwos_total_private_blocks > 0);
	}

	snprintf(log_msg, sizeof(log_msg), "   private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
    }
}



/*********************************************/
/* Function to terminate the disk I/O module */
/*********************************************/

void nwos_terminate_disk_io()
{
    int i;
    size_t chunk_index_size_in_bytes;
    char log_msg[128];

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);

	    bit_map_cache[i].chunk = 0;

	    free(bit_map_cache[i].map);
	    bit_map_cache[i].map = NULL;
	}
    }

    if (modified)
    {
#ifdef PUBLIC_MODE
	snprintf(log_msg, sizeof(log_msg), "  used public blocks: %9u",
		 nwos_ref_to_word(&nwos_next_public_ref));
	nwos_log(log_msg);

	copy_reference((ObjRef*)&disk_header.used_blocks, &nwos_next_public_ref);

	nwos_get_time_stamp(disk_header.last_change);
#else
	nwos_uint32_to_4_uint8(&nwos_used_private_blocks, disk_header.used_blocks);

	snprintf(log_msg, sizeof(log_msg), "  private blocks: %9u  used: %9u", nwos_total_private_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);

	nwos_get_time_stamp(disk_header.last_change);
#endif

	if (chunk_index_modified)
	{
	    snprintf(log_msg, sizeof(log_msg), "  chunk index modified - new size: %u",
		     nwos_used_private_chunks);
	    nwos_log(log_msg);

	    chunk_index_size_in_bytes = nwos_total_private_chunks * sizeof(*chunk_index);

#if __BYTE_ORDER == __LITTLE_ENDIAN
	    for (i = 0; i < (int)nwos_total_private_chunks; i++)
	    {
		chunk_index[i] = bswap_32(chunk_index[i]);
	    }
#endif
	    if (lseek(private_file_desc, (off_t)FILE_BLOCK_SIZE, SEEK_SET) < 0)
	    {
		perror(nwos_private_path);
		exit(1);
	    }

	    if (write(private_file_desc, chunk_index, chunk_index_size_in_bytes) != chunk_index_size_in_bytes)
	    {
		perror(nwos_private_path);
		exit(1);
	    }

	    nwos_uint32_to_4_uint8(&nwos_used_private_chunks, disk_header.used_chunks);

	    chunk_index_modified = false;
	}

	if (lseek(private_file_desc, (off_t)0, SEEK_SET) < 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	if (write(private_file_desc, &disk_header, sizeof(disk_header)) != sizeof(disk_header))
	{
	    perror(nwos_private_path);
	    exit(1);
	}
    }
    else
    {
	nwos_log("  No changes made");
    }

    if (close(private_file_desc) < 0)
    {
	perror(nwos_private_path);
	exit(1);
    }

    private_file_desc = 0;
    nwos_private_path = NULL;

    if (close(public_file_desc) < 0)
    {
	perror(nwos_public_path);
	exit(1);
    }

    public_file_desc = 0;
    nwos_public_path = NULL;
}


/*************************************/
/* This function creates a new chunk */
/*************************************/

void nwos_allocate_new_chunk(uint32 ref)
{
    uint32 offset;
    static uint8* chunk;

    assert(RESERVED_PUBLIC_BLOCKS <= ref && ref <= MAXIMUM_VALID_PRIVATE_REFERENCE);

    assert(nwos_hash_uint32_ref(ref) == 0);

    assert(nwos_used_private_chunks == 0 || ref >= chunk_index[nwos_used_private_chunks-1] + USABLE_BLOCKS_PER_CHUNK);

    offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;

    chunk_index[nwos_used_private_chunks] = ref - offset;

    if (chunk == NULL)   /* first time this has been called, allocate memory */
    {
	chunk = malloc(CHUNK_SIZE);

	if (chunk == NULL)
	{
	    perror("allocate memory for creating new chunk");
	    exit(1);
	}

	memset(chunk, 0, CHUNK_SIZE);

	chunk[0] = 0xff;
	chunk[1] = 0xff;
	chunk[2] = 0xff;
	chunk[3] = 0xff;
    }

    offset = nwos_block_offset_to_chunks + (nwos_used_private_chunks * BLOCKS_IN_CHUNK);

    if (lseek(private_file_desc, (off_t)offset << 8, SEEK_SET) < 0)
    {
	perror(nwos_private_path);
	exit(1);
    }

    if (write(private_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	perror(nwos_private_path);
	exit(1);
    }

    nwos_used_private_chunks += 1;

    chunk_index_modified = true;
}


/**********************************************************************************************/
/* This function returns amount of private space used on disk 0.1 = 10%, 0.5 = 50%, 0.9 = 90% */
/**********************************************************************************************/

float nwos_private_space_used()
{
    float result;

    result = (float)nwos_used_private_blocks / (float)nwos_total_private_blocks;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}


/*********************************************************************************************/
/* this function maps a uint32 reference to a block number                                   */
/*                                                                                           */
/* The new formula is to skip over the bit maps instead of leaving holes for them, so every  */
/* 32 bit number is available for use.                                                       */
/*********************************************************************************************/

uint32 nwos_hash_uint32_ref(uint32 ref)
{
#ifdef PUBLIC_MODE
    assert(ref < nwos_used_public_blocks);
    return ref;
#else
    int lwr;
    int upr;
    int mid = 0;
    uint32 result = 0;
    uint32 chunk;
    uint32 offset;

    if (ref < RESERVED_PUBLIC_BLOCKS)
    {
	result = ref;
    }
    else if (ref <= MAXIMUM_VALID_PRIVATE_REFERENCE && nwos_used_private_chunks > 0)
    {
	lwr = 1;

	upr = nwos_used_private_chunks;

	while (lwr <= upr)
	{
	    mid = (upr + lwr) / 2;

	    if (ref < chunk_index[mid - 1])
	    {
		upr = mid - 1;
	    }
	    else if (ref >= chunk_index[mid - 1] + USABLE_BLOCKS_PER_CHUNK)
	    {
		lwr = mid + 1;
	    }
	    else
	    {
		mid = mid - 1;  /* adjust to 0 based index */
		break;
	    }
	}

	if (lwr <= upr)
	{
	    assert(0 <= mid && mid < nwos_used_private_chunks);
	    assert(chunk_index[mid] <= ref && ref < chunk_index[mid] + USABLE_BLOCKS_PER_CHUNK);

	    chunk = (uint32) mid;

	    offset = (ref - RESERVED_PUBLIC_BLOCKS) % USABLE_BLOCKS_PER_CHUNK;

	    result = nwos_block_offset_to_chunks + (chunk * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS + offset;
	}

//	printf("total blocks: %u chunks: %u  chunk: %u  offset: %u  result: %u (%08x)\n",
//	       nwos_total_private_blocks, nwos_total_private_chunks, chunk, offset, result, result);
    }

    return result;
#endif
}


uint32 nwos_hash_ref(ObjRef* ref)
{
    uint32 uint32_ref;

    uint32_ref = ((uint32)(ref->id[0]) << 24) | ((uint32)(ref->id[1]) << 16) | 
                                  ((uint32)(ref->id[2]) << 8) | ((uint32)(ref->id[3]));

    return nwos_hash_uint32_ref(uint32_ref);
}


static inline off_t ref_to_offset(ObjRef* ref) 
{
    uint32 hash;

    assert(FILE_BLOCK_SIZE == 256);

    hash = nwos_hash_ref(ref);

    assert(hash != 0);

    return (off_t) hash << 8;
}


/*********************************************/
/* Functions to manipulate the bit map cache */
/*********************************************/

static int8 bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

void write_bit_map(Bit_Map_Cache_Entry* entry)
{
    char log_msg[80];

    snprintf(log_msg, sizeof(log_msg), "write bit_map: %08x", entry->chunk);

    nwos_log(log_msg);

    assert(nwos_block_offset_to_chunks <= entry->chunk && entry->chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(private_file_desc, (off_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map lseek chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "write_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    entry->dirty = false;

    modified = true;
}


void read_bit_map_into_cache(Bit_Map_Cache_Entry* entry)
{
    int i;
    int blocks_used = 0;
    char log_msg[80];
#if 0
    snprintf(log_msg, sizeof(log_msg), "write bit_map: %08x\n", entry->chunk);

    nwos_log(log_msg);
#endif
    assert(nwos_block_offset_to_chunks <= entry->chunk && entry->chunk < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    if (lseek(private_file_desc, (off_t)entry->chunk << 8, SEEK_SET) < 0)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map lseek chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    if (entry->map == NULL)
    {
	entry->map = malloc(BIT_MAP_BYTES);
	assert(entry->map != NULL);
    }

    if (read(private_file_desc, entry->map, BIT_MAP_BYTES) != BIT_MAP_BYTES)
    {
	snprintf(log_msg, sizeof(log_msg), "read_bit_map write chunk:%08x", entry->chunk);
	perror(log_msg);
	exit(1);
    }

    /* don't count the blocks used by the bit map itself */
    for (i = BIT_MAP_BLOCKS / 8; i < BIT_MAP_BYTES; i++)
    {
	blocks_used += bits_in_byte[entry->map[i]];
    }

    assert(blocks_used <= BLOCKS_IN_CHUNK);

    entry->used = blocks_used;
    entry->dirty = false;
}


static Bit_Map_Cache_Entry* find_bit_map_in_cache(uint32 block)
{
    int i;
    Bit_Map_Cache_Entry* result = NULL;

    assert(nwos_block_offset_to_chunks <= block && block < nwos_block_offset_to_chunks + nwos_total_private_blocks);

    /* assert((block % CHUNK_SIZE) != 0);  what was this for? */

    /* see if it's already in the cache */
    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk == (block & BIT_MAP_BLOCK_MASK)) break;
    }

    if (i < BIT_MAP_CACHE_SIZE)   /* found it */
    {
	result = &bit_map_cache[i];
    }
    else                           /* didn't find it */
    {
	/* find an empty one or find the oldest */

	result = bit_map_cache;   /* so we have an age to compare to */

	for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
	{
	    if (bit_map_cache[i].chunk == 0)
	    {
		result = &bit_map_cache[i];
		break;
	    }

	    if (bit_map_cache[i].age < result->age)
	    {
		result = &bit_map_cache[i];
	    }
	}

	if (i == BIT_MAP_CACHE_SIZE && result->dirty)    /* didn't find an empty one, write the oldest one out */
	{
	    write_bit_map(result);
	}

	result->chunk = block & BIT_MAP_BLOCK_MASK;

	read_bit_map_into_cache(result);
    }

    bit_map_tick++;
    result->age = bit_map_tick;

    assert(result != NULL);

    return result;
}


void nwos_set_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) == 0)    /* don't count a block that was already used */
    {
	entry->map[byte_num] |= (0x80 >> bit_num);

/* printf("set_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;
	entry->used++;

	nwos_used_private_blocks++;
    }
}


void nwos_clear_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    if ((entry->map[byte_num] & (0x80 >> bit_num)) != 0)    /* don't count a block that was already clear */
    {
	entry->map[byte_num] &= ~(0x80 >> bit_num);

/* printf("clear_bit_in_map block: %08x  chunk: %08x  byte: %d  bit: %d\n", block, entry->chunk, byte_num, bit_num); */

	entry->dirty = true;
	entry->used--;

	nwos_used_private_blocks--;
    }
}


bool nwos_test_bit_in_map(uint32 block)
{
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return (entry->map[byte_num] & (0x80 >> bit_num)) != 0;
}


void nwos_flush_bit_maps()
{
    int i;

    for (i = 0; i < BIT_MAP_CACHE_SIZE; i++)
    {
	if (bit_map_cache[i].chunk != 0 && bit_map_cache[i].dirty)
	{
	    write_bit_map(&bit_map_cache[i]);
	}
    }
}


#ifndef PUBLIC_MODE

static float chunk_usage(Bit_Map_Cache_Entry* entry)
{
    float result;

    result = (float)entry->used / (float)USABLE_BLOCKS_PER_CHUNK;

    assert(0.0f <= result && result <= 1.0f);

    return result;
}
#endif


bool nwos_block_used(ObjRef* ref)
{
#ifdef PUBLIC_MODE
    uint8 block[FILE_BLOCK_SIZE];

    return  nwos_read_block(ref, block) && !is_void_reference((ObjRef*)&block[4]);
#else
    uint32 block;
    int byte_num;
    int bit_num;
    Bit_Map_Cache_Entry* entry;

    block = nwos_hash_ref(ref);

    entry = find_bit_map_in_cache(block);

    byte_num = (block % BLOCKS_IN_CHUNK) / 8;
    assert(byte_num < BIT_MAP_BYTES);
    bit_num = block % 8;

    return ((entry->map[byte_num] & (0x80 >> bit_num)) != 0);
#endif
}



/**********************************/
/* Functions to generate new id's */
/**********************************/

static void generate_new_completely_random_id(ObjRef* ref)
{
    int i;
    int bit;
    int byte;
    int chunk = 0;
    Bit_Map_Cache_Entry* entry;
    uint16 quota;
    uint32 word;
#ifdef LOG_ID_GENERATION
    char msg[128];
#endif

    assert(nwos_used_private_chunks > 0);

    if (nwos_used_private_chunks == 1)
    {
	entry = find_bit_map_in_cache(nwos_block_offset_to_chunks + BIT_MAP_BLOCKS);
    }
    else
    {
#define UBPC USABLE_BLOCKS_PER_CHUNK

	for (quota = UBPC / 4; quota <= UBPC; quota += UBPC / 16)
	{
	    chunk = random() % nwos_used_private_chunks;

	    entry = find_bit_map_in_cache(nwos_block_offset_to_chunks +
					  (chunk * BLOCKS_IN_CHUNK) + BIT_MAP_BLOCKS);

	    if (entry->used < quota) break;
	}
    }

    assert(entry->used < USABLE_BLOCKS_PER_CHUNK);

    byte = random() % BIT_MAP_BYTES;

    if (entry->map[byte] == 0xff)   /* didn't get lucky */
    {
	for (i = 1; i < BIT_MAP_BYTES; i++)   /* find the nearest empty */
	{
	    if (byte - i >= 0 && entry->map[byte - i] != 0xff)
	    {
		byte -= i;
		break;
	    }

	    if (byte + i < BIT_MAP_BYTES && entry->map[byte + i] != 0xff)
	    {
		byte += i;
		break;
	    }

	    assert(0 <= byte - i || byte + i < BIT_MAP_BYTES);
	}
    }

    assert(0 <= byte && byte < BIT_MAP_BYTES);
    assert(entry->map[byte] != 0xff);

    bit = random() % 8;

    if ((entry->map[byte] & (0x80 >> bit)) != 0)
    {
	for (i = 1; i < 8; i++)   /* find the nearest empty */
	{
	    if (bit - i >= 0 && (entry->map[byte] & (0x80 >> (bit - i))) == 0)
	    {
		bit -= i;
		break;
	    }

	    if (bit + i < 8 && (entry->map[byte] & (0x80 >> (bit + i))) == 0)
	    {
		bit += i;
		break;
	    }

	    assert(0 <= bit - i || bit + i < 8);
	}
    }

    assert(0 <= bit && bit < 8);

    word = chunk_index[chunk] + byte * 8 + bit;
    nwos_word_to_ref(word, ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "generate_new_completely_random_id(%08x)", word);
    nwos_log(msg);
#endif

    assert(!nwos_block_used(ref));
}


void nwos_generate_new_completely_random_id(ObjRef* ref)
{
    generate_new_completely_random_id(ref);
    nwos_set_bit_in_map(nwos_hash_ref(ref));
}


#ifndef PUBLIC_MODE
/* density is how many blocks to use 1-7 out of 8 blocks. */

void nwos_generate_new_closely_spaced_id(ObjRef* last_ref, int density, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 block;
    uint32 hash;
    float  private_space_used;
    float  chunk_used;
    int    count;
    int    byte_num;
    int    bit_num;
    uint8  byte;

    bool closely_spaced_broken = false;
    assert(closely_spaced_broken);

    assert(1 <= density && density <= 7);

    private_space_used = nwos_private_space_used();

    /* for now don't do anything if disc is more than 90% full */
    assert(private_space_used < 0.9f);

    if (is_void_reference(last_ref))
    {
	for (count = 0; count < 10000; count++)
	{
	    generate_new_completely_random_id(new_ref);

	    block = nwos_ref_to_word(new_ref);

	    chunk_used = chunk_usage(find_bit_map_in_cache(block));

	    if (chunk_used < private_space_used + 0.01f)   /* if less than 1% over average */
	    {
		break;
	    }
	}

	assert(count < 10000);

	block--;    /* adjust for increment below */
    }
    else
    {
	block = nwos_ref_to_word(last_ref);
    }

    while (true)
    {
	block++;

	if ((random() % 8) < density)
	{
	    if (entry == NULL || entry->chunk != (block & BIT_MAP_BLOCK_MASK))
	    {
		entry = find_bit_map_in_cache(nwos_hash_uint32_ref(block));
	    }

	    hash = nwos_hash_uint32_ref(block);

	    assert(entry->chunk == (hash & BIT_MAP_BLOCK_MASK));

	    byte_num = (hash % BLOCKS_IN_CHUNK) / 8;
	    assert(byte_num < BIT_MAP_BYTES);
	    bit_num = hash % 8;

	    byte = entry->map[byte_num];

	    if (bits_in_byte[byte] < density && (byte & (0x80 >> bit_num)) == 0) break;
	}
    }

    nwos_set_bit_in_map(nwos_hash_uint32_ref(block));

    nwos_word_to_ref(block, new_ref);
}


/* generate approximately 1 block out of N */

void nwos_generate_new_1_in_N_id(ObjRef* last_ref, uint32 n, ObjRef* new_ref)
{
    Bit_Map_Cache_Entry* entry = NULL;
    uint32 mask;
    uint32 block;
    uint32 hash;
    int    byte_num;
    int    bit_num;
    uint8  byte;
    int    i;
#ifdef LOG_ID_GENERATION
    char   msg[128];
#endif

    /* make sure it's a binary multiple between 8 and 32768 */

    for (i = 3; i < 16; i++) if ((1 << i) == n) break;

    assert(i < 16);


    mask = n - 1;

    if (is_void_reference(last_ref))
    {
	generate_new_completely_random_id(new_ref);

	block = nwos_ref_to_word(new_ref);
    }
    else
    {
	/* pick random bit in next byte */
	block = ((nwos_ref_to_word(last_ref) & ~mask) | (random() & mask)) + n;
    }

    while (true)
    {
	hash = nwos_hash_uint32_ref(block);

	if (hash == 0 || (~block & ~mask) == 0)  /* continue this somewhere else */
	{
	    generate_new_completely_random_id(new_ref);

	    block = nwos_ref_to_word(new_ref);

	    hash = nwos_hash_uint32_ref(block);
	}	    

	if (entry == NULL || entry->chunk != (block & BIT_MAP_BLOCK_MASK))
	{
	    entry = find_bit_map_in_cache(hash);
	}

	hash = nwos_hash_uint32_ref(block);

	assert(entry->chunk == (hash & BIT_MAP_BLOCK_MASK));

	byte_num = (hash % BLOCKS_IN_CHUNK) / 8;
	assert(byte_num < BIT_MAP_BYTES);
	bit_num = hash % 8;

	byte = entry->map[byte_num];
	
	if ((byte & (0x80 >> bit_num)) == 0) break;

	block++;
    }

    nwos_set_bit_in_map(nwos_hash_uint32_ref(block));

    nwos_word_to_ref(block, new_ref);

#ifdef LOG_ID_GENERATION
    snprintf(msg, sizeof(msg), "nwos_generate_new_1_in_N_id(%08x)", block);
    nwos_log(msg);
#endif
}
#endif


/*****************************************/
/* Function to read a block from storage */
/*****************************************/

bool nwos_read_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char ref_str[64];
    off_t offset;
    int i;
    int lower;
    int upper;
    int file_desc;
    uint32 ref_hash;
    uint32 mid_hash;

    assert(!is_void_reference(ref));

    /*    printf("Read block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]); */


#if NO_DISK_ACCESS

    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] != NULL)
    {
	memcpy(block, cache[i], FILE_BLOCK_SIZE);
    }

    return cache[i] != NULL;

#else
    if (nwos_ref_to_word(ref) < RESERVED_PUBLIC_BLOCKS)
    {
	file_desc = public_file_desc;
	offset = (off_t) nwos_ref_to_word(ref) * FILE_BLOCK_SIZE;
    }
    else if (storage_index == NULL)     /* normal storage */
    {
	file_desc = private_file_desc;
	offset = ref_to_offset(ref);
    }
    else
    {
	file_desc = private_file_desc;

	lower = 1;
	upper = storage_count;

	ref_hash = nwos_hash_ref(ref);

	while (lower <= upper)
	{
	    i = (lower + upper) / 2;

	    mid_hash = nwos_hash_ref(&storage_index[i-1]);

	    if (mid_hash > ref_hash)
	    {
		upper = i - 1;
	    }
	    else if (mid_hash < ref_hash)
	    {
		lower = i + 1;
	    }
	    else
	    {
		i = i - 1;
		break;
	    }
	}

	assert(lower <= upper);

	assert(is_same_object(&storage_index[i], ref));

	offset = (off_t)i * FILE_BLOCK_SIZE;
    }

    if (lseek(file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(ref_str, sizeof(ref_str), "nwos_read_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(ref_str);
	exit(1);
    }

    return read(file_desc, block, FILE_BLOCK_SIZE) == FILE_BLOCK_SIZE;

#endif
}


/****************************************/
/* Function to write a block to storage */
/****************************************/

void nwos_write_block(ObjRef* ref, uint8 block[FILE_BLOCK_SIZE])
{
    char log_msg[128];
    off_t offset;

#ifdef PUBLIC_MODE
    assert(nwos_reference_type(ref) == Public_Reference);
#else
    assert(nwos_reference_type(ref) != Public_Reference);
#endif

    assert(!is_void_reference(ref));
    assert((block[0] & 0x7f) == 0 && block[1] == 0 && block[2] == 0 && block[3] == 0);
    assert(is_same_object((ObjRef*)block + 1, ref));

    snprintf(log_msg, sizeof(log_msg), "write_block - ref:%02x%02x%02x%02x block:%08x",
	    ref->id[0], ref->id[1], ref->id[2], ref->id[3], nwos_hash_ref(ref));
    nwos_log(log_msg);

#if NO_DISK_ACCESS
{
    int i;
    for (i = 0; i < 4096 && cache[i] != NULL; i++)
    {
	if (is_same_object(ref, (ObjRef*)&cache[i][4]))
	{
	    printf("WARNING - rewritting the same block: %02x%02x%02x%02x\n", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	    break;
	}
    }

    assert(i < 4096);

    if (cache[i] == NULL)
    {
	cache[i] = malloc(FILE_BLOCK_SIZE);
	assert(cache[i] != NULL);
    }

    memcpy(cache[i], block, FILE_BLOCK_SIZE);
}
#else
    assert(storage_index == NULL);   /* at this time if there is a storage index it is read only */

    offset = ref_to_offset(ref);
    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "write_block lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "write_block write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    modified = true;
#endif
}


/************************************************/
/* Remove an object (by writing zero's over it) */
/************************************************/

void nwos_remove_object(ObjRef* ref)
{
    char log_msg[128];
    uint8 block[FILE_BLOCK_SIZE];
    off_t offset;

    assert(nwos_object_exists(ref));

    snprintf(log_msg, sizeof(log_msg), "nwos_remove_object - %02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
    nwos_log(log_msg);

    memset(block, 0, sizeof(block));

    assert(sizeof(block) == FILE_BLOCK_SIZE);

    offset = ref_to_offset(ref);
    if (lseek(private_file_desc, offset, SEEK_SET) < 0)
    {
	uint32 offset_upr = (uint32)(offset >> 32);
	uint32 offset_lwr = (uint32)offset;
	snprintf(log_msg, sizeof(log_msg), "remove_object lseek ref:%02x%02x%02x%02x offset:%x%08x",
		ref->id[0], ref->id[1], ref->id[2], ref->id[3], offset_upr, offset_lwr);

	perror(log_msg);
	exit(1);
    }

    if (write(private_file_desc, block, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	snprintf(log_msg, sizeof(log_msg), "nwos_remove_object write ref:%02x%02x%02x%02x", ref->id[0], ref->id[1], ref->id[2], ref->id[3]);

	perror(log_msg);
	exit(1);
    }

    nwos_clear_bit_in_map(nwos_hash_ref(ref));

    modified = true;

    assert(!nwos_object_exists(ref));
}



