/*
--                This file is part of the New World OS project
--                       Copyright (C) 2006 QRW Software
--              J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                         http://www.qrwsoftware.com
--                         http://nwos.sourceforge.com
--
--  This program is free software; you can redistribute it and/or modify it 
--  under the terms of the GNU General Public License as published by the 
--  Free Software Foundation; either version 2 of the License, or (at your 
--  option) any later version.
--
--  This program is distributed in the hope that it will be useful, but
--  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
--  for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to:
--
--      Free Software Foundation, Inc.
--      51 Franklin Street, Fifth Floor
--      Boston, MA   02110-1301
--      USA
--
--  On the internet: http://www.fsf.org/licenses/gpl.html
--
-- $Log: ref_to_path.c,v $
-- Revision 1.10  2006/04/13 13:06:39  jsedwards
-- Moved two bytes of the references, which have name embedded, from the left
-- to the right end so that the name is an even 20 digits (100 bits), and the
-- three least significant nibbles are the version number 0 to 4095.
--
-- Revision 1.9  2006/04/12 12:01:07  jsedwards
-- Added decoding of description references.
--
-- Revision 1.8  2006/04/09 14:52:07  jsedwards
-- Moved "definitions" subdomain from "things" to "nwos" domain and added
-- the "values" domain.
--
-- Revision 1.7  2006/04/09 03:25:01  jsedwards
-- Changed defintions and attributes so that reference bytes 4 to 15 are
-- converted into a name for the object (5 bits at a time).
--
-- Revision 1.6  2006/04/09 02:37:23  jsedwards
-- Added "const" qualifier to each "reference" parameter.
--
-- Revision 1.5  2006/04/08 13:56:56  jsedwards
-- Added defintions and logs.
--
-- Revision 1.4  2006/04/08 13:50:01  jsedwards
-- Added decoding of cardinal numbers.
--
-- Revision 1.3  2006/04/08 13:42:43  jsedwards
-- Finish the binary code conversion.
--
-- Revision 1.2  2006/04/08 13:37:48  jsedwards
-- Added code to convert binary code references to a path.
--
-- Revision 1.1  2006/03/26 15:48:58  jsedwards
-- Initial version can only convert one byte binary codes.
--
*/

/* The routines in this file convert a reference to a path */

#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "domains.h"
#include "reference.h"


#define GLOBAL_TOP_LEVEL_DIRECTORY "/obj/"
#define NWOS_DIRECTORY                  "nwos/"
#define ATTRIBUTES_DIRECTORY                 "attributes/"
#define DEFINITIONS_DIRECTORY                "definitions/"
#define DESCRIPTIONS_DIRECTORY               "descriptions/"
#define VALUES_DIRECTORY                     "values/"
#define LOGS_DIRECTORY                       "logs/"

#define BINARY_CODES_DIRECTORY          "binary_codes/"
#define ONE_BYTE_CODE_DIRECTORY                      "one_byte/"

#define NUMBERS_DIRECTORY               "numbers/"
#define CARDINAL_DIRECTORY                      "cardinal/"
#define ONE_DIGIT_DIRECTORY                              "one_digit/"

#define THINGS_DIRECTORY                "things/"


/*----------------------------------------------------*/
/* This is where we come if the reference is invalid. */
/* Doesn't return!                                    */
/*----------------------------------------------------*/

static void bad_reference(const reference ref)
{
    int i;

    fputs("Invalid reference:", stderr); 

    for (i = 0; i < sizeof(reference); i++)
    {
	if (i % 4 == 0) fputc(' ', stderr);

	fprintf(stderr, "%02x", ref[i]);
    }

    fputc('\n', stderr);
}


/*----------------------------------------------------*/
/* Add a directory to the path.                       */
/* And first make sure it won't over flow the buffer. */
/*----------------------------------------------------*/

static void add_to_path(char path[NWOS_PATH_MAX], char* dir)
{
    assert((strlen(path) + strlen(dir) + 1) < NWOS_PATH_MAX);

    strcat(path, dir);
}

/*--------------------------------------------------------------------------------------------*/
/* Convert the reference bytes 2 to 15 into a name (last 3 nibbles are the version 0-4095).   */
/*--------------------------------------------------------------------------------------------*/

static void convert_ref_to_name(const reference ref, char path[NWOS_PATH_MAX])
{
    int i;
    char name[26];  /* 20 possible alpha digits + underscore + 4 version digits + null */
    int vers;

    name[0] = ref[2] >> 3;
    name[1] = (ref[2] << 2 | ref[3] >> 6) & 0x1f;
    name[2] = (ref[3] >> 1) & 0x1f;
    name[3] = (ref[3] << 4 | ref[4] >> 4) & 0x1f;
    name[4] = (ref[4] << 1 | ref[5] >> 7) & 0x1f;
    name[5] = (ref[5] >> 2) & 0x1f;
    name[6] = (ref[5] << 3 | ref[6] >> 5) & 0x1f;
    name[7] = ref[6] & 0x1f;

    name[8]  = ref[7] >> 3;
    name[9]  = (ref[7] << 2 | ref[8] >> 6) & 0x1f;
    name[10] = (ref[8] >> 1) & 0x1f;
    name[11] = (ref[8] << 4 | ref[9] >> 4) & 0x1f;
    name[12] = (ref[9] << 1 | ref[10] >> 7) & 0x1f;
    name[13] = (ref[10] >> 2) & 0x1f;
    name[14] = (ref[10] << 3 | ref[11] >> 5) & 0x1f;
    name[15] = ref[11] & 0x1f;

    name[16]  = ref[12] >> 3;
    name[17]  = (ref[12] << 2 | ref[13] >> 6) & 0x1f;
    name[18] = (ref[13] >> 1) & 0x1f;
    name[19] = (ref[13] << 4 | ref[14] >> 4) & 0x1f;

    /* find the last character (non-zero) to remove trailing space/underscore. */
    for (i = 19; i > 0; i--)
    {
	if (name[i] != 0) break;
    }

    /* now add the version (12 bits) on the end */

    vers = (ref[14] << 8 | ref[15]) & 0xfff;

    name[i+1] = '_';
    name[i+2] = '0' + (vers / 1000);
    vers = vers % 1000;
    name[i+3] = '0' + (vers / 100);
    vers = vers % 100;
    name[i+4] = '0' + (vers / 10);
    vers = vers % 10;
    name[i+5] = '0' + vers;
    name[i+6] = '\0';

    while (i >= 0)
    {
	if (name[i] == 0)
	{
	    name[i] = '_';
	}
	else if (name[i] > 26)
	{
	    bad_reference(ref);
	    return;
	}
	else
	{
	    name[i] = name[i] + 'a' - 1;
	}
	i--;
    }

    /* check it will fit in the output buffer */

    assert((strlen(path) + strlen(name) + 1) < NWOS_PATH_MAX);

    strcat(path, name);

#if DEBUG
    printf("path_to_name: %s\n", path);
#endif
}


/**************/
/* NWOS Stuff */
/**************/

/*------------------------------------------------------*/
/* Add the log number to path and the file name.        */
/*                                                      */
/* NOTE: for the time being this code assumes that an   */
/*       log only uses the lower 16 bits of the         */
/*       reference, allowing a maximum of 65536.        */
/*       This will need to be expanded later, it is     */
/*       enough for the current implementation.         */
/*------------------------------------------------------*/

static void log_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    char* hexdigits = "0123456789abcdef";
    char* p;
    int i;

    add_to_path(path, LOGS_DIRECTORY);

    /* check to verify all unused bytes of reference are zero */

    for (i = 1; i < sizeof(reference) - 2; i++)
    {
	if (ref[i] != 0) bad_reference(ref);   /* this doesn't return */
    }

    p = path + strlen(path);   /* point to last character + 1 */

    assert(i == sizeof(reference) - 2);

    *p++ = hexdigits[ref[i] >> 4];
    *p++ = hexdigits[ref[i] & 0xF];
    i++;
    *p++ = hexdigits[ref[i] >> 4];
    *p++ = hexdigits[ref[i] & 0xF];

    *p = '\0';
}


/*--------------------------------------*/
/* Add the nwos path and the file name. */
/*--------------------------------------*/

static void nwos_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    add_to_path(path, NWOS_DIRECTORY);

    switch (ref[0])
    {
      case DOMAIN_ATTRIBUTES:
	add_to_path(path, ATTRIBUTES_DIRECTORY);
	convert_ref_to_name(ref, path);   /* convert bytes 2 to 15 into a name */
        break;

      case DOMAIN_DEFINITIONS:
	add_to_path(path, DEFINITIONS_DIRECTORY);
	convert_ref_to_name(ref, path);   /* convert bytes 2 to 15 into a name */
        break;

      case DOMAIN_DESCRIPTIONS:
	add_to_path(path, DESCRIPTIONS_DIRECTORY);
	convert_ref_to_name(ref, path);   /* convert bytes 2 to 15 into a name */
        break;

      case DOMAIN_VALUES:
	add_to_path(path, VALUES_DIRECTORY);
	convert_ref_to_name(ref, path);   /* convert bytes 2 to 15 into a name */
        break;

      case DOMAIN_LOGS:
	log_to_path(ref, path);
        break;

      /* if we get here it's unknown */
      default:
	bad_reference(ref);
    }
}


/****************/
/* Binary Codes */
/****************/

/*----------------------------------------------------------*/
/* Add the file name of a one byte binary code to the path. */
/*----------------------------------------------------------*/

static void one_byte_code_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    int i;
    char *p;
    uint8 mask;

    add_to_path(path, ONE_BYTE_CODE_DIRECTORY);

    /* check to verify all unused bytes of reference are zero */

    for (i = 2; i < sizeof(reference); i++)
    {
	if (ref[i] != 0) bad_reference(ref);   /* this doesn't return */
    }

    p = path + strlen(path);   /* point to last character + 1 */

    for (i = 7; i >= 0; i--)
    {
	mask = 1 << i;   /* create a mask for the bit we are testing */
	if ((ref[1] & mask) == 0)
	{
	    *p = '0';
	}
	else
	{
	    *p = '1';
	}
	p++;
    }

    *p = '\0';
}


/*---------------------------------------------*/
/* Add the binary code path and the file name. */
/*---------------------------------------------*/

static void binary_code_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    add_to_path(path, BINARY_CODES_DIRECTORY);

    switch (ref[0] & 0x0F)
    {
      case 1:
	one_byte_code_to_path(ref, path);
        break;

      /* if we get here we can't handle that size yet */
      default:
	bad_reference(ref);
    }
}


/***********/
/* Numbers */
/***********/

static void cardinal_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    char* dec_digits = "0123456789";
    char* p;
    int i;

    add_to_path(path, CARDINAL_DIRECTORY);

    /* check to verify all unused bytes of reference are zero */

    for (i = 1; i < sizeof(reference) - 1; i++)
    {
	if (ref[i] != 0) bad_reference(ref);   /* this doesn't return */
    }

    assert(i == sizeof(reference) - 1);

    /* for now we can only handle single digits */
    if (ref[i] > 9) bad_reference(ref);

    p = path + strlen(path);   /* point to last character + 1 */

    *p++ = dec_digits[ref[i]];

    *p = '\0';
}


static void number_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    add_to_path(path, NUMBERS_DIRECTORY);

    switch (ref[0])
    {
      case DOMAIN_CARDINAL_NUMBERS:
	cardinal_to_path(ref, path);
        break;

      /* if we get here it's a type of number we can't handle yet */
      default:
	bad_reference(ref);
    }
}


/**********/
/* Things */
/**********/

static void thing_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    add_to_path(path, THINGS_DIRECTORY);

    switch (ref[0])
    {
      /* if we get here it's a type of number we can't handle yet */
      default:
	bad_reference(ref);
    }
}


/**********************************/
/* Top Level Reference Conversion */
/**********************************/

/*-----------------------------------------------*/
/* Convert a reference code to a full path name. */
/*-----------------------------------------------*/

void nwos_ref_to_path(const reference ref, char path[NWOS_PATH_MAX])
{
    strcpy(path, GLOBAL_TOP_LEVEL_DIRECTORY);

    switch (ref[0] & 0xF0)
    {
      case DOMAIN_NWOS:
	nwos_to_path(ref, path);
	break;

      case DOMAIN_BINARY_CODES:
	binary_code_to_path(ref, path);
	break;

      case DOMAIN_NUMBERS:
	number_to_path(ref, path);
	break;

      case DOMAIN_THINGS:
        thing_to_path(ref, path);
	break;

      /* if we get here we couldn't decode the reference */
      default:
        bad_reference(ref);   /* this doesn't return */
    }

    add_to_path(path, ".obj");   /* for now append a .obj to indicate no signature */
}

