/*             This file is part of the New World OS project
--                   Copyright (C) 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- This is an experimental program that takes a compressed file and tries to
-- find the smallest sparse file that doesn't have any id block overlaps.
--
-- $Log: find_smallest_sparse.c,v $
-- Revision 1.1  2007/06/23 11:40:29  jsedwards
-- Move to attic.
--
-- Revision 1.2  2007/04/21 09:29:15  jsedwards
-- Changed to print time every 10 minutes insted of a period every so many
-- passes.
--
-- Revision 1.1  2007/04/21 09:23:45  jsedwards
-- Initial version.
--
*/


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "objectify.h"


static ObjRef* storage_index;
static int     storage_count;


void index_compressed_file(char* storage_path)
{
    int i;
    int size_of_index = 262144;
    size_t bytes_read;
    uint8 buffer[FILE_BLOCK_SIZE];
    int obj_file_desc;

    obj_file_desc = open(storage_path, O_RDONLY);

    if (obj_file_desc < 0)
    {
	perror(storage_path);
	exit(1);
    }

    storage_index = malloc(size_of_index * sizeof(ObjRef));   /* allocate space for the first batch */

    assert(storage_index != NULL);

    fprintf(stderr, "Indexing");
    fflush(stderr);

    /* read the first block, it contains the header */
    bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

    assert(bytes_read == sizeof(buffer));

    i = 1;
    while (1)
    {
	if (i % 262144 == 1)
	{
	    fprintf(stderr, ".");
	    fflush(stderr);
	}

	if (i == size_of_index)   /* need to allocate more space */
	{
	    size_of_index += 262144;
	    storage_index = realloc(storage_index, size_of_index * sizeof(ObjRef));
	    assert(storage_index != NULL);
	}

	assert(i < size_of_index);

	bytes_read = read(obj_file_desc, buffer, sizeof(buffer));

	if (bytes_read != sizeof(buffer)) break;

	memcpy(&storage_index[i], &buffer[4], sizeof(ObjRef));

	i++;
    }

    if (errno != 0)
    {
	perror(storage_path);
	exit(1);
    }

    assert(bytes_read == 0);

    storage_count = i;

    fprintf(stderr, "\nblocks: %d\n", i);
}


int main(int argc, char* argv[])
{
    int i;
    int test_size;
    uint8* bit_map;
    int max_overlaps;
    int overlaps;
    int block;
    int byte;
    int bit;
    uint32 id;
    time_t now;
    time_t then;

    index_compressed_file(argv[1]);

    bit_map = malloc(storage_count);

    if (bit_map == NULL)
    {
	perror("allocating memory for bit map");
	exit(1);
    }

    max_overlaps = 2147483647;
    then = time(NULL);

    for (test_size = storage_count; test_size < storage_count * 8; test_size++)
    {
	memset(bit_map, 0, storage_count);

	overlaps = 0;
	for (i = 0; i < storage_count; i++)
	{
	    id = nwos_ref_to_word(&storage_index[i]);
	    block = (int) (id % (uint32)test_size);
	    byte = block / 8;
	    bit = block % 8;
	    if ((bit_map[byte] & (0x80 >> bit)) == 0)
	    {
		bit_map[byte] |= 0x80 >> bit;
	    }
	    else
	    {
		overlaps++;
		if (overlaps == max_overlaps) break;
		assert(overlaps < max_overlaps);
	    }
	}

	assert(i <= storage_count);

	now = time(NULL);

	if (i == storage_count)
	{
	    printf("size: %d  overlaps: %d - %s", test_size, overlaps, ctime(&now));
	    fflush(stdout);
	    max_overlaps = overlaps;
	    then = now;
	}

	if (now - then > 600)
	{
	    printf("size: %d - %s", test_size, ctime(&now));
	    fflush(stdout);
	    then = now;
	}
    }

    return 0;
}


