/*             This file is part of the New World OS project
--                Copyright (C) 2006, 2007  QRW Software
--             J. Scott Edwards - j.scott.edwards@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: prep_disk.c,v $
-- Revision 1.29  2007/04/14 15:42:25  jsedwards
-- Changed to handle errors in size specification more gracefully.
--
-- Revision 1.28  2007/04/14 12:53:38  jsedwards
-- Change to create or turncate the storage when it is opened for writing,
-- if it is a file.
--
-- Revision 1.27  2007/04/14 12:50:10  jsedwards
-- Removed commented out variable.
--
-- Revision 1.26  2007/04/14 12:49:21  jsedwards
-- Add code to break out of write_chunk loop if an error occurred, otherwise
-- the warning about a chunk not being a multiple of 512 will be triggered by
-- an error.
--
-- Revision 1.25  2007/04/14 12:44:52  jsedwards
-- Changed name of 'extra_argument' boolean variable to 'is_file' which is
-- hopefully more descriptive of what it means.
--
-- Revision 1.24  2007/04/14 12:39:05  jsedwards
-- Only write the remaining public blocks if there are more than 1 chunk.
--
-- Revision 1.23  2007/04/14 12:36:49  jsedwards
-- Move storage open after asking yes or no to write it.
--
-- Revision 1.22  2007/03/27 11:36:41  jsedwards
-- Change to have separate public and private last change times.
--
-- Revision 1.21  2007/03/14 13:15:12  jsedwards
-- Added LARGEFILE define for Linux.
--
-- Revision 1.20  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.19  2007/01/02 11:20:23  jsedwards
-- Changed to not create a private root object class because I can't think of
-- any reason to create one right now.  There isn't a list of root objects
-- in the private area.
--
-- Revision 1.18  2006/12/30 03:50:13  jsedwards
-- Removed assert that verified the NUM_PUBLIC_IDS was a multiple of
-- BLOCKS_IN_CHUNK.  NUM_PUBLIC_IDS no longer used.
--
-- Revision 1.17  2006/12/28 23:15:54  jsedwards
-- Added logging of block counts.
--
-- Revision 1.16  2006/12/27 14:16:42  jsedwards
-- Added some log messages.
--
-- Revision 1.15  2006/12/25 12:20:54  jsedwards
-- Ifdef out set_public_root_object function when compiled "don't copy public"
-- mode.
--
-- Revision 1.14  2006/12/21 13:08:36  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.13  2006/12/20 12:33:07  jsedwards
-- Changed to have two class definition refs, public and private.
--
-- Revision 1.12  2006/12/16 13:08:23  jsedwards
-- Added option to not create the private root and class definition objects.
--
-- Revision 1.11  2006/12/16 12:39:42  jsedwards
-- Added disclaimers back in.
--
-- Revision 1.10  2006/12/15 15:02:47  jsedwards
-- Changed back to only having two areas on the disk, public and private
-- encrypted.
--
-- Revision 1.9  2006/12/14 22:30:43  jsedwards
-- Add ifdef to build version that doesn't copy the public blocks or setup
-- the private root and class definition objects.
--
-- Revision 1.8  2006/12/14 14:21:31  jsedwards
-- Added code to copy public objects from file and setup private root and
-- class defintion.
--
-- Revision 1.7  2006/12/05 14:57:14  jsedwards
-- Moved root object creation code from big_bang.c (also #if'd out).
--
-- Revision 1.6  2006/12/05 14:23:49  jsedwards
-- Moved user warnings and security stuff from big_bang.c to here (#if'd out).
--
-- Revision 1.5  2006/12/04 14:55:44  jsedwards
-- Moved next available private chunk to bytes 52-55 in header.
--
-- Revision 1.4  2006/12/04 04:44:23  jsedwards
-- Changed for new header layout.
--
-- Revision 1.3  2006/12/03 17:56:56  jsedwards
-- Changed for the new disk layout: first section contains public objects,
-- second section private unencrypted objects, and third section private
-- encrypted objects.
--
-- Revision 1.2  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.5  2006/10/22 12:32:30  jsedwards
-- Added code to handle variable number of reserved public blocks.
--
-- Revision 1.1.2.4  2006/10/09 13:10:35  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.3  2006/10/07 22:28:40  jsedwards
-- Changed to write block bit map into the first block of every 2048 block
-- chunk.
--
-- Revision 1.1.2.2  2006/09/26 20:15:51  jsedwards
-- Added warning and verification before nuking the disk or partition.
--
-- Revision 1.1.2.1  2006/09/26 14:03:50  jsedwards
-- Kludged program to prepare a disk or partition for being used with
-- objectify.
--
*/

#define _LARGEFILE64_SOURCE
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "objectify_private.h"



/**********************************************************/
/* Determine if path is a regular file or a block device. */
/**********************************************************/

bool is_device(char* path)
{
    assert(path[0] == '/');   /* must be a full path specified */

    return path[1] == 'd' && path[2] == 'e' && path[3] == 'v' && path[4] == '/';
}


/****************************************************/
/* Print the usage information, is can be different */
/* depending upon if writing to a file or device.   */
/****************************************************/

void print_usage(char* name, bool specify_private_size)
{
    assert((CHUNK_SIZE % 1048576) == 0);  /* chunks must be a multiple of 1M */

    if (specify_private_size == false)
    {
	fprintf(stderr, "usage: %s [--no-private] public_file public_blocks\n\n", name);
    }
    else
    {
	fprintf(stderr, "usage: %s [--no-private] public_file public_blocks private_blocks\n\n", name);
    }

    fprintf(stderr, "  --no-private: don't create the initial private blocks (if importing old data)\n");
    fprintf(stderr, "  size can be specified in %dM chunks, megabytes 'M', or gigabytes 'G'.\n", CHUNK_SIZE / 1048576);
    fprintf(stderr, "  it must be a multiple of %dM.\n\n", CHUNK_SIZE / 1048576);
}


void goodbye()
{
    fprintf(stderr, "\n");
    fprintf(stderr, "I'm sorry you don't wish to use this software.  If you have any comments,\n");
    fprintf(stderr, "questions, or concerns please e-mail me at `j.scott.edwards.nwos@gmail.com'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "-------->   Program terminated without preparing disk!   <--------\n");
    fprintf(stderr, "\n");
    exit(1);
}


void agree_no_warranty()
{
    char buffer[16];

                  /*          1         2         3         4         5         6         7         8 */
                  /* 12345678901234567890123456789012345678901234567890123456789012345678901234567890 */
    fprintf(stderr, "\n");
    fprintf(stderr, "Objectify version %s\n", RELEASE_VERSION);
    fprintf(stderr, "\n");
    fprintf(stderr, "Copyright (C) 2005-2007  J. Scott Edwards - QRW Software\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY; THE ENTIRE RISK AS TO THE\n");
    fprintf(stderr, "QUALITY AND PERFORMANCE OF THIS PROGRAM IS WITH YOU.  For details read the\n");
    fprintf(stderr, "`GNU General Public License' described in the file `LICENSE', which should\n");
    fprintf(stderr, "have been distributed with this software.  If it is not included, it can be\n");
    fprintf(stderr, "read online at `http://www.gnu.org/copyleft/gpl.html'.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This is free software, and you are welcome to redistribute it under certain\n");
    fprintf(stderr, "conditions; also see the `LICENSE' file for details.  If you have any\n");
    fprintf(stderr, "comments or questions you can e-mail me at `j.scott.edwards.nwos@gmail.com'\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you wish to use this program, enter \"I agree\" at the prompt.  By doing\n");
    fprintf(stderr, "so you agree to the terms specified in the GNU General Public License.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you agree with the license terms of this software? ");
    fflush(stderr);

    fgets(buffer, sizeof(buffer), stdin);

    if (strcmp(buffer, "I agree\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


void understand_security()
{
    char buffer[16];

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "In addition to the above there are two things you should be aware of before\n");
    fprintf(stderr, "using this software:\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "1) This software attempts to make your data secure by encrypting the data\n");
    fprintf(stderr, "when it is written to the disk.  This should make your data secure, but I\n");
    fprintf(stderr, "am NOT a security expert and I CANNOT guarantee your data cannot be read\n");
    fprintf(stderr, "by someone if they can access your disk drive.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "2) This version of Objectify uses a long pass phrase (a minimum of 34\n");
    fprintf(stderr, "characters) to encrypt the objects.  Be aware that if you forget or lose\n");
    fprintf(stderr, "the pass phrase, it is UNLIKELY you will be able to recover your data.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "If you still wish to use this program, enter \"I understand\" at the prompt.\n");
    fprintf(stderr, "By doing so you acknowledge that you understand that the security methods in\n");
    fprintf(stderr, "this program are not guaranteed.  You also acknowledge that if you forget or\n");
    fprintf(stderr, "otherwise lose the pass phrase, YOUR DATA WILL BE LOST FOREVER.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Do you understand this? ");
    fflush(stderr);

    fgets(buffer, sizeof(buffer), stdin);

    if (strcmp(buffer, "I understand\n") != 0) goodbye();

    fprintf(stderr, "\n");
    fprintf(stderr, "*******************************************************************************\n");
}


/********************************************/
/* Convert size string to number of blocks. */
/* Must be a multiple of chunk_size.        */
/********************************************/

#define SIZE_ERROR 0xffffffff      /* return value if error occurred */

uint32 convert_size(char* size)
{
    char* p;
    uint32 result = 0;

    assert(strlen(size) > 0);   /* bug if called with empty string */

    for (p = size; isdigit(*p); p++)
    {
	result = result * 10 + (*p - '0');
    }

    if (toupper(*p) == 'M')
    {
	p++;   /* after this point we want the value in megabytes so just ignore it */
    }
    else if (toupper(*p) == 'G')
    {
	result = result * 1024;    /* change gigabytes to megabytes */
	p++;
    }
    else
    {
	result = result * (CHUNK_SIZE / 1048576);  /* change chunks to megabytes */
	p++;
    }

    /* at this point the result in in megabytes */

    if (!isdigit(*size) || *p != 0)
    {
	fprintf(stderr, "Invalid size: %s\n", size);
	result = SIZE_ERROR;
    }

    if (result == 0 || (result % (CHUNK_SIZE / 1048576)) != 0)
    {
	fprintf(stderr, "Size must be a multiple of: %dM\n", CHUNK_SIZE / 1048576);
	result = SIZE_ERROR;
    }

    if (result > 1048576)
    {
	fprintf(stderr, "Size must be less than a 1024 gigabytes\n");
	result = SIZE_ERROR;
    }

    return result * (1048576 / FILE_BLOCK_SIZE);   /* convert megabytes to blocks */
}


/********************************************************************/
/* Write the given number of chunks to the device or file.          */
/* If num_chunks == 0 it means write until EOF.  Returns the number */
/* blocks written (NOT chunks) because the last may not be even.    */
/********************************************************************/

uint32 write_chunks(int fd, uint8* chunk, uint32 num_chunks)
{
    size_t bytes_written;
    uint32 blocks = 0;

    while (num_chunks == 0 || blocks < (num_chunks * BLOCKS_IN_CHUNK))
    {
	bytes_written = write(fd, chunk, CHUNK_SIZE);

	if (bytes_written < 0) break;    /* exit if error occurred */

	if (bytes_written % 512 != 0)
	{
	    fprintf(stderr, "bytes written not a multiple of 512: %zd\n", bytes_written);
	    close(fd);
	    exit(1);
	}

	blocks += bytes_written / 256;

	if (bytes_written < CHUNK_SIZE || blocks % 4194304 == 0)
	{
	    printf("%d - %d\n", (blocks / 4194304), blocks);
	    fflush(stdout);
	}

	if (bytes_written < CHUNK_SIZE) break;
    }

    if (errno != 0)
    {
	perror("after last write");
	exit(1);
    }

    return blocks;
}


/* this is the same as the normal routine in objectify.c except it doesn't translate the reference for the disk */

#ifndef DONT_COPY_PUBLIC_BLOCKS
static void set_public_root_object(ObjRef* ref)
{
    C_struct_Root root_obj;
    ObjRef verify_ref;

    nwos_read_object_from_disk(ref, &root_obj, sizeof(root_obj));
    copy_reference(&nwos_public_class_definition_class_ref, &root_obj.class_definition_class);
    copy_reference(&nwos_reference_list_class_ref, &root_obj.reference_list_class);

    assert(nwos_find_public_class_definition("REFERENCE LIST", &verify_ref));
    assert(is_same_object(&verify_ref, &nwos_reference_list_class_ref));
}
#endif


/***************************************************************************/
/* Program to prepare a disk or file for NWOS storage.                     */
/* Now there is a bit map of used blocks as the first block in every chunk */
/* 8192 * 8 = 65536 blocks * 256 bytes per block = 16777216 bytes in chunk */
/***************************************************************************/

int main(int argc, char* argv[])
{
    int obj_fd;
    uint8 *chunk;
    char answer[8];
    int arg_index = 1;
    bool is_file = false;
    bool do_private_blocks = true;
    char* public_file_name;

    uint32 reserved_public_blocks;
    uint32 reserved_private_blocks;
    uint32 public_blocks;
    uint32 blocks;
    uint32 blocks_written;
    uint32 num_chunks;
    float chunks;
    Disk_Header disk_header;
    FILE* pub_fp;
    size_t bytes_read;
    ObjRef root_obj_ref;
    ObjRef private_root_obj_ref;
    ObjRef public_root_class_ref;
    /* ObjRef private_root_class_ref; */
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    char msg[128];
    TimeStamp save_public_time;
#if 0
    ObjRef root_object_reference;
    bool upgrade = false;
//    struct stat stat_struct;
#endif

    agree_no_warranty();   /* make sure he or she agrees with the licensing */

    printf("\n");

    nwos_log_arguments(argc, argv);


    /**************************/
    /* sort out the arguments */
    /**************************/

    if (!is_device(DEFAULT_FILE))
    {
	is_file = true;
    }

    if (*argv[arg_index] == '-')   /* option specified */
    {
	if (strcmp(argv[arg_index], "--no-private") != 0)
	{
	    fprintf(stderr, "\nunknown option: %s\n\n", argv[arg_index]);
	    print_usage(argv[0], is_file);
	    exit(1);
	}

	do_private_blocks = false;

	arg_index++;
    }


    /* must have 2 or 3 arguments */

    if (is_file)
    {
	if (argc - arg_index != 3)
	{
	    print_usage(argv[0], is_file);
	    exit(1);
	}
    }
    else
    {
	if (argc - arg_index != 2)
	{
	    print_usage(argv[0], false);
	    exit(1);
	}
    }

    public_file_name = argv[arg_index];
    arg_index++;


    /* convert arguments */

    reserved_public_blocks = convert_size(argv[arg_index]);
    arg_index++;

    if (reserved_public_blocks == 0 || reserved_public_blocks == SIZE_ERROR)
    {
	print_usage(argv[0], is_file);
	exit(1);
    }

    printf("reserved public chunks: %u\n", reserved_public_blocks / BLOCKS_IN_CHUNK);


    if (!is_file)
    {
	reserved_private_blocks = 0;   /* zero means unlimited */
    }
    else
    {
	reserved_private_blocks = convert_size(argv[arg_index]);

	if (reserved_private_blocks == 0 || reserved_private_blocks == SIZE_ERROR)
	{
	    print_usage(argv[0], is_file);
	    exit(1);
	}

	printf("\nreserved private chunks: %u\n", reserved_private_blocks / BLOCKS_IN_CHUNK);
    }


    chunk = malloc(CHUNK_SIZE);

    if (chunk == NULL)
    {
	perror("allocating chunk");
	exit(1);
    }


    /************************************/
    /* write public portion of the disk */
    /************************************/

    /* first copy the public blocks in */

    memset(chunk, 0, CHUNK_SIZE);

#ifdef DONT_COPY_PUBLIC_BLOCKS
    public_blocks = 1;
    nwos_log("DON'T COPY PUBLIC BLOCKS");
#else
    pub_fp = fopen(public_file_name, "r");

    if (pub_fp == NULL)
    {
	perror(public_file_name);
	exit(1);
    }

    bytes_read = fread(chunk, 1, CHUNK_SIZE, pub_fp);

    if (bytes_read < FILE_BLOCK_SIZE)
    {
	fprintf(stderr, "Public file: %s too small\n", public_file_name);
	exit(1);
    }

    if (bytes_read == CHUNK_SIZE)
    {
	fprintf(stderr, "Public file: %s too big, this version cannot handle > 1 chunk\n", public_file_name);
	exit(1);
    }

    fclose(pub_fp);

    memcpy(&disk_header, chunk, sizeof(disk_header));   /* copy the header portion */

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, sizeof(disk_header.magic_number)) != 0)
    {
	fprintf(stderr, "file: %s is not a NWOS file\n", public_file_name);
	exit(1);
    }
    
    if (memcmp(disk_header.version_string, VERSION_STRING, sizeof(disk_header.version_string)) != 0)
    {
	fprintf(stderr, "file: %s incorrect version: %c%c%c%c should be: %s\n", public_file_name, 
		disk_header.version_string[0], disk_header.version_string[1],
		disk_header.version_string[2], disk_header.version_string[3],
		VERSION_STRING);
	exit(1);
    }

    nwos_4_uint8_to_uint32(disk_header.used_public_blocks, &public_blocks);

    snprintf(msg, sizeof(msg), "Public file: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c  blocks: %u",
	     nwos_extract_year_from_time_stamp(disk_header.public_last_change),
	     nwos_extract_month_from_time_stamp(disk_header.public_last_change),
	     nwos_extract_day_of_month_from_time_stamp(disk_header.public_last_change),
	     nwos_extract_hour_from_time_stamp(disk_header.public_last_change),
	     nwos_extract_minute_from_time_stamp(disk_header.public_last_change),
	     nwos_extract_second_from_time_stamp(disk_header.public_last_change),
	     disk_header.version_string[0], disk_header.version_string[1],
	     disk_header.version_string[2], disk_header.version_string[3],
	     public_blocks);
    nwos_log(msg);
    printf("%s\n", msg);
#endif
    
    printf("\n");
    printf("WARNING: all of the contents of %s will be ERASED!!!\n", DEFAULT_FILE);
    printf("\n");
    printf("Do you want to continue? (enter `yes' to erase %s) ", DEFAULT_FILE);

    fflush(stdout);

    fgets(answer, sizeof(answer), stdin);

    printf("\n");

    if (strcmp(answer, "yes\n") != 0)
    {
	exit(1);
    }


    /***************************/
    /* Open the storage device */
    /***************************/

    if (is_file)
    {
	obj_fd = open(DEFAULT_FILE, O_WRONLY | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    }
    else
    {
	obj_fd = open(DEFAULT_FILE, O_WRONLY);    /* don't need large file if not seeking? */
    }

    if (obj_fd < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }


    /**************************/
    /* Copy the public blocks */
    /**************************/

    printf("Copying %d public blocks...\n", public_blocks);
    fflush(stdout);

    blocks_written = write_chunks(obj_fd, chunk, 1);

    if (blocks_written < BLOCKS_IN_CHUNK)
    {
	fprintf(stderr, "Error: writing public blocks\n");
	close(obj_fd);
	exit(1);
    }

    blocks = blocks_written;

    /* if more than 1 chuck was allocated for public blocks, write the rest */

    if ((reserved_public_blocks / BLOCKS_IN_CHUNK) > 1)
    {
	printf("Writing remaining public chunks...\n");
	fflush(stdout);

	/* the public blocks don't have bit maps (at least for now) */

	memset(chunk, 0, CHUNK_SIZE);

	blocks_written = write_chunks(obj_fd, chunk, (reserved_public_blocks / BLOCKS_IN_CHUNK) - 1);

	if (blocks_written < reserved_public_blocks - BLOCKS_IN_CHUNK)
	{
	    fprintf(stderr, "Error: not enough space for public blocks\n");
	    close(obj_fd);
	    exit(1);
	}

	blocks += blocks_written;
    }


    /*************************************/
    /* write private portion of the disk */
    /*************************************/

    printf("Writing private chunks...\n");
    fflush(stdout);

    /* private blocks have bit maps so mark off 32 bits for the bit map itself */
    chunk[0] = 0xff;
    chunk[1] = 0xff;
    chunk[2] = 0xff;
    chunk[3] = 0xff;

    blocks_written = write_chunks(obj_fd, chunk, reserved_private_blocks / BLOCKS_IN_CHUNK);

    if (blocks_written < reserved_private_blocks)
    {
	fprintf(stderr, "Error: not enough space for private chunks\n");
	close(obj_fd);
	exit(1);
    }

    blocks += blocks_written;


    chunks = (float)blocks / (float)BLOCKS_IN_CHUNK;

    printf("Total chunks: %.1f\n", chunks);


    /*********************************/
    /* finally write the disk header */
    /*********************************/

    if (lseek(obj_fd, 0, SEEK_SET) < 0)
    {
	perror("lseek 0");
	exit(1);
    }

    memcpy(save_public_time, disk_header.public_last_change, sizeof(save_public_time));

    memset(&disk_header, 0, sizeof(disk_header));
    
    memcpy(disk_header.magic_number, MAGIC_NUMBER, 4);
    memcpy(disk_header.version_string, VERSION_STRING, 4);

    nwos_get_time_stamp(disk_header.last_prep_disk);

    memcpy(disk_header.private_last_change, disk_header.last_prep_disk, sizeof(disk_header.private_last_change));
    memcpy(disk_header.public_last_change, save_public_time, sizeof(disk_header.public_last_change));

    if (is_device(DEFAULT_FILE))
    {
	reserved_private_blocks = blocks - reserved_public_blocks;
    }
    else
    {
	assert(reserved_public_blocks + reserved_private_blocks == blocks);
    }


    nwos_uint32_to_4_uint8(&reserved_public_blocks, disk_header.total_public_blocks);
    nwos_uint32_to_4_uint8(&reserved_private_blocks, disk_header.total_private_blocks);

    nwos_uint32_to_4_uint8(&public_blocks, disk_header.used_public_blocks);


    /* compute the number of private blocks used */

    num_chunks = (reserved_private_blocks + BLOCKS_IN_CHUNK - 1) / BLOCKS_IN_CHUNK;  /* round up */

    blocks = num_chunks * BIT_MAP_BLOCKS;

    nwos_uint32_to_4_uint8(&blocks, disk_header.used_private_blocks);


    /* start the unencrypted private blocks at the beginning */

    memcpy(disk_header.next_private_chunk, disk_header.total_public_blocks, 4);

    memset(chunk, 0, FILE_BLOCK_SIZE);

    memcpy(chunk, &disk_header, sizeof(disk_header));

    if (write(obj_fd, chunk, FILE_BLOCK_SIZE) != FILE_BLOCK_SIZE)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (close(obj_fd) != 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    snprintf(msg, sizeof(msg), "    public blocks: %9u  used: %9u", reserved_public_blocks, public_blocks);
    nwos_log(msg);
    snprintf(msg, sizeof(msg), "   private blocks: %9u  used: %9u", reserved_private_blocks, blocks);
    nwos_log(msg);

#ifndef DONT_COPY_PUBLIC_BLOCKS
    if (do_private_blocks)
    {
	understand_security();   /* make sure he or she understands the risks */

	nwos_get_key_from_password(big_key, sizeof(big_key));

	memcpy(bf_key, big_key, 16);
	linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
	memcpy(private_root_obj_ref.id, big_key+20, 4);
	serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

	nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);

	root_obj_ref.id[0] = 0;
	root_obj_ref.id[1] = 0;
	root_obj_ref.id[2] = 0;
	root_obj_ref.id[3] = 1;

	set_public_root_object(&root_obj_ref);

	assert(nwos_find_public_class_definition("ROOT", &public_root_class_ref));


	/* create the private root object */

	nwos_set_encryption_level(Encryption_Extreme);

	printf("private root object id: %02x%02x%02x%02x\n", 
	       private_root_obj_ref.id[0],
	       private_root_obj_ref.id[1],
	       private_root_obj_ref.id[2],
	       private_root_obj_ref.id[3]);

	nwos_clone_class_definition(&nwos_public_class_definition_class_ref, &nwos_private_class_definition_class_ref);

#if 0
	/* currently I cannot think of a reason to have a private root class because there isn't a list of root objects */

	printf("private class definition id: %02x%02x%02x%02x\n", 
	       nwos_private_class_definition_class_ref.id[0],
	       nwos_private_class_definition_class_ref.id[1],
	       nwos_private_class_definition_class_ref.id[2],
	       nwos_private_class_definition_class_ref.id[3]);

	nwos_clone_class_definition(&public_root_class_ref, &private_root_class_ref);

	printf("private root class definition id: %02x%02x%02x%02x\n", 
	       private_root_class_ref.id[0],
	       private_root_class_ref.id[1],
	       private_root_class_ref.id[2],
	       private_root_class_ref.id[3]);
#endif

	assert(nwos_create_root(&private_root_obj_ref));

	nwos_terminate_objectify();
    }
#endif

    return 0;
}

