/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: disk_usage.c,v $
-- Revision 1.7  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.6  2007/02/11 14:41:26  jsedwards
-- Change all 'off64_t' and 'lseek64' references to 'off_t' and 'lseek',
-- because BSD doesn't dig the whole brain damaged 64 bit thing.
--
-- Revision 1.5  2006/12/03 21:43:03  jsedwards
-- Changed for the new disk layout: first section contains public objects,
-- second section private unencrypted objects, and third section private
-- encrypted objects.
--
-- Revision 1.4  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.10  2006/10/22 16:01:42  jsedwards
-- Change so number of blocks and reserved public blocks are read from disk
-- instead of being define constants.
--
-- Revision 1.1.2.9  2006/10/19 01:45:48  jsedwards
-- Fixed format specifiers for off64_t type.
--
-- Revision 1.1.2.8  2006/10/17 01:38:02  jsedwards
-- Added code to keep track of then number of blocks used in each chunk.
--
-- Revision 1.1.2.7  2006/10/15 12:30:03  jsedwards
-- Change to print blocks used in each print instead of a running total.
--
-- Revision 1.1.2.6  2006/10/15 12:17:54  jsedwards
-- Change to print output every 16 gigabytes instead of 10 gigabytes.
--
-- Revision 1.1.2.5  2006/10/14 12:33:58  jsedwards
-- Added in blocks skipped over at the beginning and stop when reached
-- the capacity of the disk.
--
-- Revision 1.1.2.4  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.3  2006/10/10 07:02:37  jsedwards
-- Removed code that verified the bits_in_byte table (since it has been
-- verified there is no longer a need for it) and added printing blocks
-- used for every gigabyte printed.
--
-- Revision 1.1.2.2  2006/10/10 06:56:08  jsedwards
-- Changed to 8192 byte bit maps to speed up this program (with 256 byte bit
-- maps this program took around an hour to run on a 250 GB disk).
--
-- Revision 1.1.2.1  2006/10/08 13:50:02  jsedwards
-- Program to count the number of blocks used on the disk using the
-- block bit maps.
--
*/


#define _LARGEFILE64_SOURCE

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s\n", program);
}


static int bits_in_byte[256] = 
{
/*        0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f */
/* 0 */   0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
/* 1 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 2 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 3 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 4 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 5 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 6 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* 7 */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* 8 */   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
/* 9 */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* a */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* b */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* c */   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
/* d */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* e */   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
/* f */   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

int chunk_counts[8192];


uint32 calculate_blocks_used(int obj_fd, uint32 start, uint32 num_blocks)
{
    char ref_str[128];
    off_t ref;
    size_t bytes_read;
    uint8 map[BIT_MAP_BYTES];
    /* uint32 chunk_used; */
    uint32 last_total = 0;
    uint32 blocks_used = 0;
    int i;
    int count;

    for (ref = start; ref < start + num_blocks; ref += BLOCKS_IN_CHUNK)
    {
	if (lseek(obj_fd, ref << 8, SEEK_SET) < 0)
	{
	    snprintf(ref_str, sizeof(ref_str), "lseek ref:%08x", (uint32)ref);
	    perror(ref_str);
	    exit(1);
	}

	bytes_read = read(obj_fd, map, sizeof(map));

	if (bytes_read != sizeof(map))
	{
	    printf("error: incomplete bit map: %08x bytes_read: %zd\n", (uint32)(ref), bytes_read);
	    break;
	}

	/* chunk_used = 0; */

	for (i = 0; i < bytes_read; i++)
	{
#if 0
	    blocks_total = blocks_used + blocks_free + chunk_used;

	    if (blocks_total >= blocks_on_disk)
	    {
		/* account for blocks not an even 8 */
		blocks_free -= (blocks_total - blocks_on_disk);
		break;
	    }
#endif
	    count = bits_in_byte[map[i]];
	    /* chunk_used += count; */
	    blocks_used += count;

	    /* blocks_free += (8 - count); */
	}

	if (ref % (4194304 /* * 16 */) == 4194304 - BLOCKS_IN_CHUNK)
	{
	    printf("%u - %u  used: %d\n", (uint32)(ref / 4194304), (uint32)ref, blocks_used - last_total);
	    fflush(stdout);
	    last_total = blocks_used;
	}

	/* if (ref > 0x0fffffff) printf("%d\n", chunk_used); */

	/* blocks_used += chunk_used; */

	/* assert(32 <= chunk_used && chunk_used <= 8192); */

	/* chunk_counts[chunk_used - 32]++; */
    }

    return blocks_used;
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    char ref_str[128];
    uint32 blocks_on_disk;
    uint32 reserved_public_blocks;
    uint32 reserved_private_blocks;
    uint32 reserved_encrypted_blocks;
    uint32 used_public_blocks;
    uint32 used_private_blocks;
    uint32 used_encrypted_blocks;
    uint32 used_blocks;
    uint8 header[32];

    if (argc != 1)
    {
	print_usage(argv[0]);
	exit(1);
    }

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    if (read(obj_file_desc, header, sizeof(header)) != sizeof(header))
    {
	snprintf(ref_str, sizeof(ref_str), "reading disk header from: %s", DEFAULT_FILE);
	perror(ref_str);
	exit(1);
    }

    if (memcmp(&header[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&header[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    reserved_public_blocks = ((uint32)header[8] << 24) | ((uint32)header[9]<< 16) | ((uint32)header[10] << 8) | ((uint32)header[11]);

    reserved_private_blocks = ((uint32)header[12] << 24) | ((uint32)header[13]<< 16) | ((uint32)header[14] << 8) | ((uint32)header[15]);

    blocks_on_disk = ((uint32)header[16] << 24) | ((uint32)header[17]<< 16) | ((uint32)header[18] << 8) | ((uint32)header[19]);

    reserved_encrypted_blocks = blocks_on_disk - reserved_public_blocks - reserved_private_blocks;

    used_public_blocks = ((uint32)header[20] << 24) | ((uint32)header[21]<< 16) | ((uint32)header[22] << 8) | ((uint32)header[23]);

    assert(reserved_public_blocks % BLOCKS_IN_CHUNK == 0);
    assert(reserved_private_blocks % BLOCKS_IN_CHUNK == 0);


    printf("Calculating private blocks used...\n");

    used_private_blocks = calculate_blocks_used(obj_file_desc, reserved_public_blocks, reserved_private_blocks);


    printf("Calculating encrypted blocks used...\n");

    used_encrypted_blocks = calculate_blocks_used(obj_file_desc, reserved_public_blocks + reserved_private_blocks, reserved_encrypted_blocks);


    /*    printf("%u+ - %u  used: %d\n", (uint32)(ref / 4194304), (uint32)ref, blocks_used - last_total); */

    close(obj_file_desc);

#if 0
    for (i = 0; i < 8192; i++)
    {
	if (chunk_counts[i] > 0)
	{
	    printf("used: %d  chunks: %d\n", i, chunk_counts[i]);
	}
    }
#endif

    printf("Reserved for public:    %9u  used: %9u %5.2f\n", 
	   reserved_public_blocks, used_public_blocks, (float)used_public_blocks/(float)reserved_public_blocks);
    printf("Reserved for private:   %9u  used: %9u %5.2f\n",
	   reserved_private_blocks, used_private_blocks, (float)used_private_blocks/(float)reserved_private_blocks);
    printf("Reserved for encrypted: %9u  used: %9u %5.2f\n", 
	   reserved_encrypted_blocks, used_encrypted_blocks, (float)used_encrypted_blocks/(float)reserved_encrypted_blocks);
    used_blocks = used_public_blocks + used_private_blocks + used_encrypted_blocks;
    printf("Total blocks on disk:   %9u  used: %9u %5.2f\n", 
	   blocks_on_disk, used_blocks, (float)used_blocks/(float)blocks_on_disk);

    fflush(stdout);

#if 0
    blocks_total = blocks_used + blocks_free;

    printf("blocks total: %d  used: %d  free: %d  usage: %.1f%%\n",
	   blocks_total, blocks_used, blocks_free, ((float)blocks_used / (float)blocks_free) );
#endif

    return 0;
}

