/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: check_block_maps.c,v $
-- Revision 1.8  2007/03/27 11:32:09  jsedwards
-- Change time_last_change in disk_header to private_last_change.
--
-- Revision 1.7  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.6  2006/12/16 13:10:24  jsedwards
-- Make sure last_total is always initialized.
--
-- Revision 1.5  2006/12/16 11:35:59  jsedwards
-- Change for new disk layout and check public blocks too.
--
-- Revision 1.4  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:22  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.9  2006/10/22 16:02:58  jsedwards
-- Add print statement to print used in final blocks.
--
-- Revision 1.1.2.8  2006/10/22 14:19:23  jsedwards
-- Changed so that blocks_on_disk and reserved_public_blocks are read off
-- disk instead of defined constants.
--
-- Revision 1.1.2.7  2006/10/18 13:18:51  jsedwards
-- Changed printf formats for off64_t to 'z' instead of 'll'.
--
-- Revision 1.1.2.6  2006/10/15 12:30:02  jsedwards
-- Change to print blocks used in each print instead of a running total.
--
-- Revision 1.1.2.5  2006/10/15 12:13:14  jsedwards
-- Changed to correctly account for the first chunk on the disk as all used.
--
-- Revision 1.1.2.4  2006/10/12 12:09:57  jsedwards
-- Added kludge to add in stats for the section of the disk (reserved public)
-- that it currently skips over.  Also included the block maps themselves.
--
-- Revision 1.1.2.3  2006/10/10 07:32:36  jsedwards
-- Move bit map defines to objectify_private.h instead of spread all over.
--
-- Revision 1.1.2.2  2006/10/09 13:13:14  jsedwards
-- Change bit map size from 256 to 8192 (to improve speed, disk_usage program
-- took over an hour to run with 256 byte bit maps).
--
-- Revision 1.1.2.1  2006/10/08 13:44:09  jsedwards
-- Program to verify the 256 byte block maps are correct on the disk.
--
*/


#define _LARGEFILE64_SOURCE

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s\n", program);
}


bool block_all_zeros(uint8* block)
{
    int i;

    for (i = 0; i < FILE_BLOCK_SIZE; i++)
    {
	if (block[i] != 0) break;
    }

    return i == FILE_BLOCK_SIZE;
}


bool reference_matches(uint32 ref, uint8 *block)
{
    return (uint8)(ref >> 24) == block[4] && (uint8)(ref >> 16) == block[5] && (uint8)(ref >> 8) == block[6] && (uint8)ref == block[7];
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    char ref_str[128];
    uint8 *chunk;
    int i;
    int j;
    size_t bytes_read;
    int num_blocks;
    int bit_num;
    int byte_num;
    uint32 public_blocks_used = 0;
    uint32 private_blocks_used = 0;
    uint32 private_blocks_free = 0;
    uint32 private_blocks_total;
    uint32 block;
    uint32 last_total = 0;
    uint32 total_public_blocks;
    uint32 total_private_blocks;
    uint32 used_public_blocks;
    uint32 used_private_blocks;
    Disk_Header disk_header;

    if (argc != 1)
    {
	print_usage(argv[0]);
	exit(1);
    }

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    /* allocate buffer for reading chunks */

    chunk = malloc(CHUNK_SIZE);

    assert(chunk != NULL);

    if (read(obj_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
    {
	snprintf(ref_str, sizeof(ref_str), "reading first chunk from: %s", DEFAULT_FILE);
	perror(ref_str);
	close(obj_file_desc);
	exit(1);
    }

    memcpy(&disk_header, chunk, sizeof(disk_header));

    if (memcmp(disk_header.magic_number, MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(disk_header.version_string, VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    printf("Last change: %u-%02u-%02u %02u:%02u:%02u  version: %c%c%c%c\n",
	   nwos_extract_year_from_time_stamp(disk_header.private_last_change),
	   nwos_extract_month_from_time_stamp(disk_header.private_last_change),
	   nwos_extract_day_of_month_from_time_stamp(disk_header.private_last_change),
	   nwos_extract_hour_from_time_stamp(disk_header.private_last_change),
	   nwos_extract_minute_from_time_stamp(disk_header.private_last_change),
	   nwos_extract_second_from_time_stamp(disk_header.private_last_change),
	   disk_header.version_string[0], disk_header.version_string[1],
	   disk_header.version_string[2], disk_header.version_string[3]);

    public_blocks_used++;   /* count 1 for the header */

    /* get what the disk thinks */
    nwos_4_uint8_to_uint32(disk_header.total_public_blocks, &total_public_blocks);
    nwos_4_uint8_to_uint32(disk_header.total_private_blocks, &total_private_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_public_blocks, &used_public_blocks);
    nwos_4_uint8_to_uint32(disk_header.used_private_blocks, &used_private_blocks);

    num_blocks = 0;

    /* check the first chunk which has the public blocks (for now public blocks < 1 chunk) */

    for (block = 1; block < BLOCKS_IN_CHUNK; block++)
    {
	if (block_all_zeros(chunk + block * FILE_BLOCK_SIZE))
	{
	    break;
	}

	if (!reference_matches(block, chunk + block * FILE_BLOCK_SIZE))
	{
	    printf("Reference mismatch block: %u  stored: %02x%02x%02x%02x\n", block,
		   *(chunk + block * 256 + 0),
		   *(chunk + block * 256 + 1),
		   *(chunk + block * 256 + 2),
		   *(chunk + block * 256 + 3));
	}

	public_blocks_used++;
    }

    for ( ; block < total_public_blocks; block++)
    {
	if (block % BLOCKS_IN_CHUNK == 0)
	{
	    if (read(obj_file_desc, chunk, CHUNK_SIZE) != CHUNK_SIZE)
	    {
		snprintf(ref_str, sizeof(ref_str), "reading public chunk from: %s  block: %u", DEFAULT_FILE, block);
		perror(ref_str);
		close(obj_file_desc);
		exit(1);
	    }
	}

	if (!block_all_zeros(chunk + (block % BLOCKS_IN_CHUNK) * FILE_BLOCK_SIZE))
	{
	    printf("Error: non-zero block: %u past the end of public blocks\n", block);

	    if (!reference_matches(block, chunk + block * FILE_BLOCK_SIZE))
	    {
		printf("Reference mismatch block: %u  stored: %02x%02x%02x%02x\n", block,
		       *(chunk + block * 256 + 0),
		       *(chunk + block * 256 + 1),
		       *(chunk + block * 256 + 2),
		       *(chunk + block * 256 + 3));
	    }

	    public_blocks_used++;
	}

	if (block % (total_public_blocks / 100) == 0)
	{
	    if ((block / (total_public_blocks / 100)) % 10 == 0)
	    {
		printf("%u", block / (total_public_blocks / 100));
	    }
	    else
	    {
		printf(".");
	    }
	    fflush(stdout);
	}
    }
    printf("\n");

    printf("public blocks used: %u  stored: %u - %s\n", public_blocks_used, used_public_blocks,
	   public_blocks_used == used_public_blocks ? "OK" : "MISMATCH");

    printf("public blocks free: %u  total: %u\n", total_public_blocks - used_public_blocks, total_public_blocks);

    fflush(stdout);


    /*****************************/
    /* now do the private blocks */
    /*****************************/

    for (block = 0; block < total_private_blocks; block += BLOCKS_IN_CHUNK)
    {
	bytes_read = read(obj_file_desc, chunk, CHUNK_SIZE);

	private_blocks_used += BIT_MAP_BLOCKS;

	for (i = 8192; i < bytes_read; i += FILE_BLOCK_SIZE)
	{
	    for (j = 0; j < 256; j++)
	    {
		if (chunk[i+j] != 0) break;
	    }

	    byte_num = (i / 256) / 8;
	    bit_num = (i / 256) % 8;

	    if (j < 256)   /* block not empty */
	    {
		private_blocks_used++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) == 0)
		{
		    printf("bit clear that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", block, i, byte_num, bit_num);
//		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[(byte_num&0xffffff00)+j], j%32 == 31 ? '\n' : ' ');
		    for (j = 0; j < 8192; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	    else
	    {
		private_blocks_free++;

		if ((chunk[byte_num] & (0x80 >> bit_num)) != 0)
		{
		    printf("bit set that shouldn't be, chunk: %x  offset: %05x  byte: %d bit: %d\n", block, i, byte_num, bit_num);
//		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[(byte_num&0xffffff00)+j], j%32 == 31 ? '\n' : ' ');
		    for (j = 0; j < 8192; j++) printf("%02x%c", chunk[j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    for (j = 0; j < 256; j++) printf("%02x%c", chunk[i+j], j%32 == 31 ? '\n' : ' ');
		    printf("\n");
		    exit(1);
		}
	    }
	}

	if (bytes_read != CHUNK_SIZE)
	{
	    printf("last block read: %08zx\n", block + (bytes_read / FILE_BLOCK_SIZE));
	    break;
	}

	if (block % 4194304 == 0)
	{
	    printf("%u - %u  used: %d\n", block / 4194304, block, private_blocks_used - last_total);
	    fflush(stdout);
	    last_total = private_blocks_used;
	}
    }

    printf("%u+ - %u  used: %d\n", block / 4194304, block, private_blocks_used - last_total);
    fflush(stdout);

    close(obj_file_desc);

    private_blocks_total = private_blocks_used + private_blocks_free;

    printf("private blocks used: %u  stored: %u - %s\n", private_blocks_used, used_private_blocks,
	   private_blocks_used == used_private_blocks ? "OK" : "MISMATCH");

    printf("private blocks free: %u  total: %u  stored: %u - %s\n", private_blocks_free,
	   private_blocks_total, total_private_blocks,
	   private_blocks_total == total_private_blocks ? "OK" : "MISMATCH");

    printf("usage: %.1f%%\n", ((float)private_blocks_used / (float)total_private_blocks) );

    return 0;
}

