/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package; see the file LICENSE.   If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: word.c,v $
-- Revision 1.16  2006/11/11 12:01:07  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.15  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.14.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.14  2006/01/01 21:49:02  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.13  2005/12/31 17:44:19  jsedwards
-- Changed find_word to look for a thing with a certain class instead of a
-- certain thing.  I.E. to look for a "month" instead of "January".
--
-- Revision 1.12  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.11  2005/12/29 17:01:41  jsedwards
-- Changed to just read spelling obj header to get reference list instead of
-- the whole variable sized object.
--
-- Revision 1.10  2005/12/27 19:46:34  jsedwards
-- Removed two unused variables.
--
-- Revision 1.9  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.8  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.7  2005/12/11 18:27:28  jsedwards
-- Removed debugging print messages.
--
-- Revision 1.6  2005/12/11 16:47:05  jsedwards
-- Fixed error message to say "word" instead of "year".
-- Split "find_word" function out of "create_word" function.
--
-- Revision 1.5  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.4  2005/12/05 05:22:46  jsedwards
-- Change to call read_reference_list_from_disk for reference lists instead
-- of read_object_from_disk.  Also moved calculating checksums down to just
-- before writing object, so that no changes occur after computing them.
--
-- Revision 1.3  2005/12/04 04:13:03  jsedwards
-- Added 'nwos' prefix to create_xxxx function names and eliminated the
-- 'referring' object parameter from all of them.
--
-- Revision 1.2  2005/12/03 21:52:20  jsedwards
-- Changed to use "spelling" class to store letters.
--
-- Revision 1.1  2005/12/03 21:49:14  jsedwards
-- Initial version copied from name.c.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"


void nwos_setup_word()
{
    ObjRef english_lang_ref;

   /*-----------------------------------------------------------------*/
   /* This too has to be done in the correct order, since words need  */
   /* a language and language objects need words to name them.        */
   /*-----------------------------------------------------------------*/

    printf ("Creating Word class definition.\n");
    nwos_create_class_definition("WORD");

    printf ("Creating Language class definition.\n");
    nwos_create_class_definition("LANGUAGE");

    printf ("Creating English Language object\n");
    nwos_create_english_language(&english_lang_ref);

    printf ("Creating Abbreviation class definition.\n");
    nwos_create_class_definition("ABBREVIATION");
}


bool nwos_find_word(char* word, ObjRef* thing_class, ObjRef* lang, ObjRef* ref)
{
    C_struct_Word word_obj;
    EveryObject spelling_header;
    EveryObject thing_header;
    ObjRef word_class_ref;
    ObjRef spelling_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_spellings;
    int i;
    bool result = false;

    assert(nwos_object_exists(lang));

    assert(nwos_find_class_definition("WORD", &word_class_ref));

    if (nwos_find_spelling(word, &spelling_ref))
    {
	nwos_read_object_headers_from_disk(&spelling_ref, &spelling_header);

	ref_list_size = nwos_reference_list_size(&spelling_header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading spelling reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&spelling_header.object.references, ref_list, ref_list_size);

	num_spellings = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_spellings (in find_word): %d\n", num_spellings); */

	for (i = 0; i < num_spellings; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &word_class_ref))   /* it is a word object */
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &word_obj, sizeof(word_obj));

		if (is_same_object(&word_obj.language, lang))
		{
		    /* get the class for the "thing" this word points to */
		    nwos_read_object_headers_from_disk(&word_obj.thing, &thing_header);
		    if (is_same_object(&thing_header.common.class_definition, thing_class))
		    {
			memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
			result = true;
			break;
		    }
		}
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }

    return result;
}


ObjCreateResult nwos_create_word(char* word, ObjRef* thing, ObjRef* lang, ObjRef* ref)
{
    C_struct_Word word_obj;
    EveryObject thing_header;
    ObjRef word_class_ref;
    ObjRef spelling_ref;
    ObjCreateResult result = CREATED_NEW;

    assert(nwos_object_exists(lang));

    assert(nwos_find_class_definition("WORD", &word_class_ref));

    nwos_read_object_headers_from_disk(thing, &thing_header);

    if (!nwos_find_word(word, &thing_header.common.class_definition, lang, ref))   /* didn't find it */
    {
	nwos_create_spelling(word, &spelling_ref);

	memset(&word_obj, 0, sizeof(word_obj));  /* zero it out */

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&word_obj.header.common, ref, &word_class_ref);

	memcpy(&word_obj.language, lang, sizeof(ObjRef));
	nwos_add_to_references(ref, lang);

	memcpy(&word_obj.thing, thing, sizeof(ObjRef));
	nwos_add_to_references(ref, thing);

	memcpy(&word_obj.spelling, &spelling_ref, sizeof(ObjRef));
	nwos_add_to_references(ref, &spelling_ref);

	nwos_create_reference_list(ref, &word_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &word_obj.header.object, sizeof(ObjectHeader), word_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &word_obj.thing, sizeof(word_obj) - sizeof(EveryObject), word_obj.header.common.data_chksum);

	nwos_write_object_to_disk(ref, &word_obj, sizeof(word_obj));

        nwos_add_to_references(ref, &word_class_ref);
    }

    return result;
}


