/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: us_state.c,v $
-- Revision 1.17  2006/11/11 12:01:07  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.16  2006/10/26 01:51:29  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.15.2.2  2006/10/19 01:47:08  jsedwards
-- Fix format specifier for uint32 which is unsigned instead of long unsigned
-- now.
--
-- Revision 1.15.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.15  2006/01/09 13:56:53  jsedwards
-- Restored call to create reference list for state object that was
-- accidentally deleted in revision 1.12.
--
-- Revision 1.14  2006/01/09 03:45:29  jsedwards
-- Added assert to verify that the capital in 0004 state objects was void.
--
-- Revision 1.13  2006/01/09 03:09:37  jsedwards
-- Added test function to return true if there is a state with the given name.
-- Added a routine to describe a state.  Added code to the upgrade routine
-- to create the "CITY" class and to print out more information about the
-- upgrade.
--
-- Revision 1.12  2006/01/06 14:12:49  jsedwards
-- Added code to upgrade a 0004 state object to a 0005 state object (which
-- entails adding the capitol city).
--
-- Revision 1.11  2006/01/04 18:56:43  jsedwards
-- Added routine to add a bunch of additional major cities (only Alaska to
-- California are entered in this version).  Moved print of "City created" to
-- interactive creation so it doesn't print during "additional city" creation.
--
-- Revision 1.10  2006/01/03 02:58:45  jsedwards
-- Fixed bug in "create us city" routine, where it wasn't reading the class
-- reference list.
--
-- Revision 1.9  2006/01/02 19:43:37  jsedwards
-- Added creation of the US City class.  Added "find state by name", "create
-- us city", "add city", and "list cities in state" routines.  NOTE: there is
-- something wrong with this version, it can't find the name Salt Lake City or
-- Oklahoma City (yet it can find Jefferson City and Carson City).
--
-- Revision 1.8  2006/01/02 19:24:16  jsedwards
-- Moved remaining state information (capital city and slogan) from states.txt
-- file and into state_info.  Changed debugging printf to print reference id.
--
-- Revision 1.7  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.6  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.5  2005/12/29 16:52:19  jsedwards
-- Reordered the state info entries so that they are in order of states
-- joining instead of alphabetical by postal code.
--
-- Revision 1.4  2005/12/29 16:38:07  jsedwards
-- Changed to just read spelling obj header to get reference list instead of
-- the whole variable sized object.
--
-- Revision 1.3  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.2  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.1  2005/12/21 04:03:42  jsedwards
-- Initial version that creates all 50 states and finds state from postal code.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"

/* forward references */
static void create_all_fifty_states();
static void create_additional_cities();


/*----------------------------------------*/
/* Create classes and stuff for US States */
/*----------------------------------------*/

void nwos_setup_us_states()
{
    printf ("Creating US State class definition.\n");
    nwos_create_class_definition("US STATE");

    printf ("Creating US State class definition.\n");
    nwos_create_class_definition("US CITY");

    /* this has to be done after US STATE and US CITY classes are created */
    printf ("Creating 50 US State objects\n");
    create_all_fifty_states();

    /* this has to be done after all of the states are created (obviously) */
    printf ("Creating additional US city objects\n");
    create_additional_cities();
}


bool nwos_find_state_from_postal_code(char* postal_code, ObjRef* ref)
{
    EveryObject spelling_header;
    ObjRef state_class_ref;
    ObjRef spelling_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_spellings;
    int i;
    bool result = false;

    assert(strlen(postal_code) == 2);

    assert(nwos_find_class_definition("US STATE", &state_class_ref));

    if (nwos_find_spelling(postal_code, &spelling_ref))
    {
	nwos_read_object_headers_from_disk(&spelling_ref, &spelling_header);

	ref_list_size = nwos_reference_list_size(&spelling_header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading spelling reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&spelling_header.object.references, ref_list, ref_list_size);

	num_spellings = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_spellings (in find_postal_code): %d\n", num_spellings); */

	for (i = 0; i < num_spellings; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &state_class_ref))   /* it is a state object */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }

    return result;
}


bool nwos_find_state_from_name(char* name, ObjRef* ref)
{
    EveryObject name_header;
    ObjRef state_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_names;
    int i;
    bool result = false;

    assert(nwos_find_class_definition("US STATE", &state_class_ref));

    if (nwos_find_name(name, &name_ref))
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list_size = nwos_reference_list_size(&name_header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading name reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&name_header.object.references, ref_list, ref_list_size);

	num_names = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_names (in find_postal_code): %d\n", num_names); */

	for (i = 0; i < num_names; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &state_class_ref))   /* it is a state object */
	    {
		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }

    return result;
}


bool nwos_any_states_named(char* name)
{
    ObjRef state_ref;

    return nwos_find_state_from_name(name, &state_ref);
}


/* local routine that does the actual city object creation, needed by create all 50 states to create capital cities */
/* WARNING: DOES NOT add city to state references because if it is "create all 50 states" calling, the state object */
/* has not been written to disk yet and so we can't call add_to_references yet.  So the caller must add it to state refs. */

static void create_city(ObjRef* city_class_ref, ObjRef* name_ref, ObjRef* state_ref, ObjRef* city_ref)
{
    C_struct_US_City city_obj;

    memset(&city_obj, 0, sizeof(city_obj));  /* zero it out */

    nwos_generate_new_id(city_ref);

    nwos_fill_in_common_header(&city_obj.header.common, city_ref, city_class_ref);

    nwos_create_reference_list(city_ref, &city_obj.header.object.references);

    memcpy(&city_obj.name, name_ref, sizeof(ObjRef));
    memcpy(&city_obj.state, state_ref, sizeof(ObjRef));

    nwos_crc32_calculate((uint8*) &city_obj.header.object, sizeof(ObjectHeader), city_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &city_obj.name, sizeof(city_obj) - sizeof(EveryObject), city_obj.header.common.data_chksum);

    nwos_write_object_to_disk(city_ref, &city_obj, sizeof(city_obj));

    nwos_add_to_references(city_ref, city_class_ref);
    nwos_add_to_references(city_ref, name_ref);
}



ObjCreateResult nwos_create_us_city(char* city, ObjRef* state_ref, ObjRef* city_ref)
{
    EveryObject name_header;
    C_struct_US_City city_obj;
    ObjRef city_class_ref;
    ObjRef name_ref;
    ObjRef object_class;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int i;
    ObjCreateResult result = CREATED_NEW;

    assert(!is_void_reference(state_ref));

    /* first find out if we already have this city in this state */

    assert(nwos_find_class_definition("US CITY", &city_class_ref));

    if (nwos_create_name(city, &name_ref) == FOUND_EXISTING)    /* see if this city already exists */
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list_size = nwos_reference_list_size(&name_header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&name_header.object.references, ref_list, ref_list_size);

	num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&city_class_ref, &object_class))
	    {
		nwos_read_object_from_disk(&ref_list->references[i], &city_obj, sizeof(city_obj));

		if (is_same_object(&city_obj.state, state_ref))   /* found a match */
		{
		    memcpy(city_ref, &ref_list->references[i], sizeof(ObjRef));
		    result = FOUND_EXISTING;
		    break;
		}
	    }
	}

	free(ref_list);
	ref_list = NULL;
    }

    if (result != FOUND_EXISTING)   /* didn't find it */
    {
	create_city(&city_class_ref, &name_ref, state_ref, city_ref);
	nwos_add_to_references(city_ref, state_ref);
    }

    return result;
}


static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}


#define MAX_NAMES 20
#define MAX_NAME_SIZE 64

void nwos_add_city(ObjRef* city_ref)
{
    C_struct_US_City city_obj;
    C_struct_US_State state_obj;
    ObjRef name_ref;
    ObjRef state_ref;
    ObjRef object_class;
    EveryObject name_header;
    ObjRef city_class_ref;
    ObjRef city_refs[MAX_NAMES];
    int city_count;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    int selection;
    char temp[80];
    char buffer[4];
    bool ok;
    int i;
    int j;
    char city[MAX_NAME_SIZE];
    char state[MAX_NAME_SIZE];

    nwos_find_class_definition("US CITY", &city_class_ref);

    get_input("city", city, sizeof(city));

  /* first find the name */

    assert(nwos_find_class_definition("US CITY", &city_class_ref));

    selection = -1;    /* default to create new city */
    city_count = 0;

    if (nwos_find_name(city, &name_ref))
    {
	nwos_read_object_headers_from_disk(&name_ref, &name_header);

	ref_list_size = nwos_reference_list_size(&name_header.object.references);

	ref_list = malloc(ref_list_size);

	if (ref_list == NULL) 
	{
	    perror("reading city name reference list");
	    exit(1);
	}

	nwos_read_reference_list_from_disk(&name_header.object.references, ref_list, ref_list_size);

	num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

	/* printf("name num refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &city_class_ref))    /* add it to the list */
	    {
		memcpy(&city_refs[city_count], &ref_list->references[i], sizeof(ObjRef));
		city_count++;
	    }

	    if (city_count == MAX_NAMES)
	    {
		printf("/nSorry, right now this program can only deal with a maximum of %d possible names.\n", MAX_NAMES);
		break;
	    }
	}

	free(ref_list);

	if (city_count > 0)  /* ask which one we want */
	{
	    printf("\nI have found cities in the following states with that name\n\n");

	    for (j = 0; j < city_count; j++)    /* see if we already have this one */
	    {
		nwos_read_object_from_disk(&city_refs[j], &city_obj, sizeof(city_obj));

		nwos_read_object_from_disk(&city_obj.state, &state_obj, sizeof(state_obj));

		nwos_name_to_string(&state_obj.name, temp, sizeof(temp));

		printf("  %d: %s\n", j, temp);
	    }

	    printf("\nIf it is in one of these states, enter the number, otherwise just hit enter to create a new city.\n\n");

	    ok = false;
	    while (!ok)
	    {
		get_input("selection number", buffer, sizeof(buffer));

		printf("\n");

		if (buffer[0] == '\0')
		{
		    ok = true;   /* create a new one */
		}
		else if (isdigit(buffer[0]) && buffer[1] == '\0')   /* single digit */
		{
		    selection = buffer[0] - '0';
		    ok = (selection <= city_count);
		}
		else if (isdigit(buffer[0]) && buffer[2] == '\0')   /* two digits */
		{
		    selection = (buffer[0] - '0') * 10 + (buffer[1] - '0');
		    ok = (selection < city_count);
		}

		if (!ok) printf("Please enter the number of one of the cities listed.\n");
	    }
	}
    }

    if (selection != -1)     /* selected an existing city */
    {
	memcpy(city_ref, &city_refs[selection], sizeof(ObjRef));
    }
    else                     /* need to add a new one */
    {
	snprintf(temp, sizeof(temp), "What state is %s in (name or 2 letter code)", city);

	ok = false;
	while (!ok)
	{
	    get_input(temp, state, sizeof(state));

	    printf("\n");

	    if (strlen(state) == 2)
	    {
		ok = nwos_find_state_from_postal_code(state, &state_ref);
	    }
	    else
	    {
		ok = nwos_find_state_from_name(state, &state_ref);
	    }

	    if (!ok) printf("Can't find state: %s, please re-enter.\n\n", state);
	}

	nwos_create_us_city(city, &state_ref, city_ref);
	printf("Created city: %08x\n", nwos_ref_to_word(city_ref));
    }
}


void nwos_describe_state(char* state)
{
    C_struct_US_State state_obj;
    C_struct_US_City city_obj;
    ObjRef state_ref;
    char temp[80];
    bool ok;

    if (strlen(state) == 2)
    {
	ok = nwos_find_state_from_postal_code(state, &state_ref);
    }
    else
    {
        ok = nwos_find_state_from_name(state, &state_ref);
    }

    if (!ok)
    {
	printf("I can't find the state of %s\n", state);
	return;
    }
    else
    {
	nwos_read_object_from_disk(&state_ref, &state_obj, sizeof(state_obj));

	nwos_name_to_string(&state_obj.name, temp, sizeof(temp));
	printf("Name: %s\n", temp);

	printf("Number %u\n", state_obj.number);

	nwos_date_to_string(&state_obj.date, temp, sizeof(temp));
	printf("Joined: %s\n", temp);

	nwos_read_object_from_disk(&state_obj.capital, &city_obj, sizeof(city_obj));

	nwos_name_to_string(&city_obj.name, temp, sizeof(temp));
	printf("Capital: %s\n", temp);
    }

    printf("\n");
}


void nwos_list_cities_in_state(char* state)
{
    C_struct_US_City city_obj;
    ObjRef state_ref;
    ObjRef object_class;
    EveryObject state_header;
    ObjRef city_class_ref;
    ReferenceList* ref_list;
    size_t ref_list_size;
    int num_refs;
    char temp[80];
    bool ok;
    int i;

    nwos_find_class_definition("US CITY", &city_class_ref);

    if (strlen(state) == 2)
    {
	ok = nwos_find_state_from_postal_code(state, &state_ref);
    }
    else
    {
        ok = nwos_find_state_from_name(state, &state_ref);
    }

    if (!ok)
    {
	printf("I can't find the state of %s\n", state);
	return;
    }

    nwos_read_object_headers_from_disk(&state_ref, &state_header);

    ref_list_size = nwos_reference_list_size(&state_header.object.references);

    ref_list = malloc(ref_list_size);

    if (ref_list == NULL) 
    {
	perror("reading state for list cities reference list");
	exit(1);
    }

    nwos_read_reference_list_from_disk(&state_header.object.references, ref_list, ref_list_size);

    num_refs = (ref_list_size - sizeof(CommonHeader)) / sizeof(ObjRef);

    /* printf("state num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &city_class_ref))    /* add it to the list */
	{
	    /* printf("found a city: %08lx\n", nwos_ref_to_word(&ref_list->references[i])); */
	    nwos_read_object_from_disk(&ref_list->references[i], &city_obj, sizeof(city_obj));
	    nwos_name_to_string(&city_obj.name, temp, sizeof(temp));
	    /* printf("name: %08lx - %s\n", nwos_ref_to_word(&city_obj.name), temp); */
	    printf("%s\n", temp);
	}
    }

    free(ref_list);
}



typedef struct 
{
  char  code[3];
  uint8 number;
  uint8 month;
  uint8 day;
  uint16 year;
  char* abbr;
  char* name;
  char* capital;
  char* slogan;
} state_info_struct;

#define  Jan  1
#define  Feb  2
#define  Mar  3
#define  Apr  4
#define  May  5
#define  Jun  6
#define  Jul  7
#define  Aug  8
#define  Sep  9
#define  Oct  10
#define  Nov  11
#define  Dec  12


static state_info_struct state_info[] =
  {
    { "DE",  1, Dec,  7, 1787, "Del.",   "Delaware",       "Dover",          "The First State" },
    { "PA",  2, Dec, 12, 1787, "Pa.",    "Pennsylvania",   "Harrisburg",     "Memories Last A Lifetime" },
    { "NJ",  3, Dec, 18, 1787, "N.J.",   "New Jersey",     "Trenton",        "New Jersey and You Are Perfect Together" },
    { "GA",  4, Jan,  2, 1788, "Ga.",    "Georgia",        "Atlanta",        "Georgia On My Mind" },
    { "CT",  5, Jan,  9, 1788, "Conn.",  "Connecticut",    "Hartford",       "We're Full Of Surprises" },
    { "MA",  6, Feb,  6, 1788, "Mass.",  "Massachusetts",  "Boston",         "We'd Love To Show You Around" },
    { "MD",  7, Apr, 28, 1788, "Md.",    "Maryland",       "Annapolis",      "So Many Things To Do, So Close Together" },
    { "SC",  8, May, 23, 1788, "S.C.",   "South Carolina", "Columbia",       "Smiling Faces Beautiful Places" },
    { "NH",  9, Jun, 21, 1788, "N.H.",   "New Hampshire",  "Concord",        "The Road Less Traveled" },
    { "VA", 10, Jun, 25, 1788, "Va.",    "Virginia",       "Richmond",       "Virginia Is For Lovers" },
    { "NY", 11, Jul, 26, 1788, "N.Y.",   "New York",       "Albany",         "I Love New York" },
    { "NC", 12, Nov, 21, 1789, "N.C.",   "North Carolina", "Raleigh",        "A Better Place To Be" },
    { "RI", 13, May, 29, 1790, "R.I.",   "Rhode Island",   "Providence",     "And Providence Plantations" },
    { "VT", 14, Mar,  4, 1791, "Vt.",    "Vermont",        "Montpelier",     "Green Mountain State" },
    { "KY", 15, Jun,  1, 1792, "Ky.",    "Kentucky",       "Frankfort",      "Always In Season" },
    { "TN", 16, Jun,  1, 1796, "Tenn.",  "Tennessee",      "Nashville",      "Sounds Good To Me" },
    { "OH", 17, Mar,  1, 1803, NULL,     "Ohio",           "Columbus",       "Heart Of It All" },
    { "LA", 18, Apr, 30, 1812, "La.",    "Louisiana",      "Baton Rouge",    "Come As You Are, Leave Different" },
    { "IN", 19, Dec, 11, 1816, "Ind.",   "Indiana",        "Indianapolis",   "The Welcome Mats Always Out", },
    { "MS", 20, Dec, 10, 1817, "Miss.",  "Mississippi",    "Jackson",        "The South's Warmest Welcome" },
    { "IL", 21, Dec,  3, 1818, "Ill.",   "Illinois",       "Springfield",    "A Million Miles From Monday" },
    { "AL", 22, Dec, 14, 1819, "Ala.",   "Alabama",        "Montgomery",     "Unforgettable" },
    { "ME", 23, Mar, 15, 1820, NULL,     "Maine",          "Augusta",        "Maine Is On The Move" },
    { "MO", 24, Aug, 10, 1821, "Mo.",    "Missouri",       "Jefferson City", "Show Me Missouri" },
    { "AR", 25, Jun, 15, 1836, "Ark.",   "Arkansas",       "Little Rock",    "The Natural State" },
    { "MI", 26, Jan, 26, 1837, "Mich.",  "Michigan",       "Lansing",        "Great Things To See And Do" },
    { "FL", 27, Mar,  3, 1845, "Fla.",   "Florida",        "Tallahassee",    "Sunshine State" },
    { "TX", 28, Dec, 29, 1845, "Tex.",   "Texas",          "Austin",         "It's Like A Whole Other Country" },
    { "IA", 29, Dec, 28, 1846, NULL,     "Iowa",           "Des Moines",     "Come Be Our Guest" },
    { "WI", 30, May, 29, 1848, "Wis.",   "Wisconsin",      "Madison",        "Stay Just A Little Bit Longer" },
    { "CA", 31, Sep,  9, 1850, "Calif.", "California",     "Sacramento",     "Find Yourself Here" },
    { "MN", 32, May, 11, 1858, "Minn.",  "Minnesota",      "Saint Paul",     "Explore Minnesota" },
    { "OR", 33, Feb, 14, 1859, "Ore.",   "Oregon",         "Salem",          "Things Look Different Here" },
    { "KS", 34, Jan, 29, 1861, "Kans.",  "Kansas",         "Topeka",         "Heart Of America's West" },
    { "WV", 35, Jun, 20, 1863, "W.Va.",  "West Virginia",  "Charleston",     "Wild And Wonderful" },
    { "NV", 36, Oct, 31, 1864, "Nev.",   "Nevada",         "Carson City",    "Discover Both Sides Of Nevada" },
    { "NE", 37, Mar,  1, 1867, "Nebr.",  "Nebraska",       "Lincoln",        "Genuine" },
    { "CO", 38, Aug,  1, 1875, "Colo.",  "Colorado",       "Denver",         "Colorful Colorado" },
    { "ND", 39, Nov,  2, 1889, "N.D.",   "North Dakota",   "Bismarck",       "Discover The Spirit" },
    { "SD", 40, Nov,  2, 1889, "S.D.",   "South Dakota",   "Pierre",         "Great Faces Great Places" },
    { "MT", 41, Nov,  8, 1889, "Mont.",  "Montana",        "Helena",         "Big Sky Country" },
    { "WA", 42, Nov, 11, 1889, "Wash.",  "Washington",     "Olympia",        "A Little Trip To The Extraordinary" },
    { "ID", 43, Jul,  3, 1890, NULL,     "Idaho",          "Boise",          "Discover Idaho" },
    { "WY", 44, Jul, 10, 1890, "Wyo.",   "Wyoming",        "Cheyenne",	     "Like No Place On Earth", },
    { "UT", 45, Jan,  4, 1896, NULL,     "Utah",           "Salt Lake City", "The Greatest Snow On Earth" },
    { "OK", 46, Nov, 16, 1907, "Okla.",  "Oklahoma",       "Oklahoma City",  "Native America" },
    { "NM", 47, Jan,  6, 1912, "N.M.",   "New Mexico",     "Santa Fe",       "Land of Enchantment" },
    { "AZ", 48, Feb, 14, 1912, "Ariz.",  "Arizona",        "Phoenix",        "Grand Canyon State" },
    { "AK", 49, Jan,  3, 1959, NULL,     "Alaska",         "Juneau",         "North! To Alaska" },
    { "HI", 50, Aug, 21, 1959, NULL,     "Hawaii",         "Honolulu",       "The Islands Of Aloha" },
  };

#define NUM_STATES (sizeof(state_info)/sizeof(state_info_struct))

static void create_all_fifty_states()
{
    C_struct_US_State state_obj;
    ObjRef state_ref;
    ObjRef state_class_ref;
    ObjRef city_class_ref;
    ObjRef city_name_ref;
    size_t length;
    int i;

    assert(nwos_find_class_definition("US STATE", &state_class_ref));
    assert(nwos_find_class_definition("US CITY", &city_class_ref));

    for (i = 0; i < NUM_STATES; i++)
    {
	memset(&state_obj, 0, sizeof(state_obj));  /* zero it out */

	nwos_generate_new_id(&state_ref);

	nwos_fill_in_common_header(&state_obj.header.common, &state_ref, &state_class_ref);

	length = strlen(state_info[i].name);

	printf("creating postal code: %s\n", state_info[i].code);

	/* create postal code, first make sure spelling exists */
	if (nwos_find_spelling(state_info[i].code, &state_obj.postal_code))
	{
	    /* make sure we haven't screwed up and the state doesn't already exist */
	    assert(!nwos_find_state_from_postal_code(state_info[i].code, &state_ref));
	}
	else
	{
	    nwos_create_spelling(state_info[i].code, &state_obj.postal_code);
	}

	printf("creating date: %d-%d-%d\n", state_info[i].year, state_info[i].month, state_info[i].day);

	nwos_create_date(state_info[i].year, state_info[i].month, state_info[i].day, &state_obj.date);

	state_obj.number = state_info[i].number;

	nwos_create_name(state_info[i].name, &state_obj.name);

	nwos_create_reference_list(&state_ref, &state_obj.header.object.references);

	nwos_create_name(state_info[i].capital, &city_name_ref);

	create_city(&city_class_ref, &city_name_ref, &state_ref, &state_obj.capital);

	nwos_add_to_reference_list(&state_obj.capital, &state_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &state_obj.header.object, sizeof(ObjectHeader), state_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &state_obj.name, sizeof(state_obj) - sizeof(EveryObject), state_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&state_ref, &state_obj, sizeof(state_obj));

	printf("created state: %s  reference: %02x%02x%02x%02x\n", state_info[i].name, 
	       state_ref.id[0], state_ref.id[1], state_ref.id[2],  state_ref.id[3]);

	assert(nwos_object_exists(&state_ref));

	nwos_add_to_references(&state_ref, &state_obj.postal_code);
	nwos_add_to_references(&state_ref, &state_obj.date);
	nwos_add_to_references(&state_ref, &state_obj.name);
	nwos_add_to_references(&state_ref, &state_class_ref);
    }
}

#if 0
void nwos_upgrade_states_from_0004_to_0005()
{
    C_struct_US_State state_obj;
    C_struct_US_City city_obj;
    ObjRef state_ref;
    ObjRef city_class_ref;
    ObjRef city_name_ref;
    char state_name[64];
    char city_name[64];
    int i;

    /* add the City class */
    printf ("    Creating US State class definition.\n");
    nwos_create_class_definition("US CITY");

    /* add the capitals to the states */

    assert(nwos_find_class_definition("US CITY", &city_class_ref));

    for (i = 0; i < NUM_STATES; i++)
    {
	assert(nwos_find_state_from_postal_code(state_info[i].code, &state_ref));

	nwos_read_object_from_disk(&state_ref, &state_obj, sizeof(state_obj));

	assert(state_obj.number == state_info[i].number);

	if (memcmp(state_obj.header.common.version, "0004", 4) != 0)
	{
	    printf("    State %s is not version 0004, skipping it!\n", state_info[i].code);
	    continue;
	}

	assert(is_void_reference(&state_obj.capital));

	/* update to the new version */
	memcpy(state_obj.header.common.version, HEADER_VERSION, sizeof(state_obj.header.common.version));

	nwos_create_name(state_info[i].capital, &city_name_ref);

	create_city(&city_class_ref, &city_name_ref, &state_ref, &state_obj.capital);

	/* we have to do this because the "create_city" routine doesn't */
	nwos_add_to_reference_list(&state_obj.capital, &state_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &state_obj.header.object, sizeof(ObjectHeader), state_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &state_obj.name, sizeof(state_obj) - sizeof(EveryObject), state_obj.header.common.data_chksum);

	nwos_remove_object(&state_ref);   /* remove the old state object */

	nwos_write_object_to_disk(&state_ref, &state_obj, sizeof(state_obj));   /* write the new one */

	/* read it back and print it out to verify */

	nwos_read_object_from_disk(&state_obj.capital, &city_obj, sizeof(city_obj));

	nwos_name_to_string(&city_obj.name, city_name, sizeof(city_name));

	nwos_name_to_string(&state_obj.name, state_name, sizeof(state_name));

	printf("    Added capital: %s to %s.\n", city_name, state_name);
    }

    /* finally add the additional cities, although this doesn't serve much purpose */
    printf ("    Creating additional US city objects\n");
    create_additional_cities();
}
#endif

typedef struct 
{
  char  code[3];
  char* name;
} city_info_struct;


static city_info_struct additional_city_info[] =
  {
    { "AK", "Anchorage" },
    { "AK", "Bethel" },
    { "AK", "Fairbanks" },

    { "AL", "Anniston" },
    { "AL", "Auburn" },
    { "AL", "Birmingham" },
    { "AL", "Decatur" },
    { "AL", "Dothan" },
    { "AL", "Florence" },
    { "AL", "Gadsden" },
    { "AL", "Huntsville" },
    { "AL", "Madison" },
    { "AL", "Mobile" },
    { "AL", "Tuscaloosa" },

    { "AZ", "Bullhead City" },
    { "AZ", "Casa Grande" },
    { "AZ", "Chandler" },
    { "AZ", "Flagstaff" },
    { "AZ", "Florence" },
    { "AZ", "Gilbert" },
    { "AZ", "Glendale" },
    { "AZ", "Kingman" },
    { "AZ", "Lake Havasu City" },
    { "AZ", "Mesa" },
    { "AZ", "Page" },
    { "AZ", "Phoenix" },
    { "AZ", "Scottsdale" },
    { "AZ", "Tempe" },
    { "AZ", "Prescott" },
    { "AZ", "Marana" },
    { "AZ", "Tuscon" },
    { "AZ", "Yuma" },

    { "AR", "Arkadelphia" },
    { "AR", "Conway" },
    { "AR", "Fayetteville" },
    { "AR", "Fort Smith" },
    { "AR", "Hot Springs" },
    { "AR", "Jonesboro" },
    { "AR", "Pine Bluff" },
    { "AR", "Springdale" },
    { "AR", "Texarkana" },
    { "AR", "Van Buren" },

    { "CA", "Alhambra" },
    { "CA", "Anaheim" },
    { "CA", "Apple Valley" },
    { "CA", "Atascadero" },
    { "CA", "Bakersfield" },
    { "CA", "Barstow" },
    { "CA", "Berkeley" },
    { "CA", "Beverly Hills" },
    { "CA", "Burbank" },
    { "CA", "Carlsbad" },
    { "CA", "Chula Vist" },
    { "CA", "Concord" },
    { "CA", "Corona" },
    { "CA", "Coronado" },
    { "CA", "Cupertino" },
    { "CA", "Daly City" },
    { "CA", "East Los Angeles" },
    { "CA", "El Cajon" },
    { "CA", "El Centro" },
    { "CA", "El Monte" },
    { "CA", "Encinitas" },
    { "CA", "Eureka" },
    { "CA", "Folsom" },
    { "CA", "Foster City" },
    { "CA", "Freemont" },
    { "CA", "Fullerton" },
    { "CA", "Glendale" },
    { "CA", "Grass Valley" },
    { "CA", "Hesperia" },
    { "CA", "Huntington Beach" },
    { "CA", "Indio" },
    { "CA", "Inglewood" },
    { "CA", "Irvine" },
    { "CA", "Hayward" },
    { "CA", "Escondido" },
    { "CA", "La Jolla" },
    { "CA", "Lancaster" },
    { "CA", "Lompoc" },
    { "CA", "Long Beach" },
    { "CA", "Los Angeles" },
    { "CA", "Los Banos" },
    { "CA", "Los Gatos" },
    { "CA", "Milpitas" },
    { "CA", "Marina" },
    { "CA", "Menlo Park" },
    { "CA", "Merced" },
    { "CA", "Monterey" },
    { "CA", "Modesto" },
    { "CA", "Moreno Valley" },
    { "CA", "Morgan Hill" },
    { "CA", "Moutain View" },
    { "CA", "National City" },
    { "CA", "Oakland" },
    { "CA", "Oceanside" },
    { "CA", "Oxnard" },
    { "CA", "Palm Springs" },
    { "CA", "Palmdale" },
    { "CA", "Pasadena" },
    { "CA", "Redding" },
    { "CA", "Redwood City" },
    { "CA", "Riverside" },
    { "CA", "San Bernardino" },
    { "CA", "San Bruno" },
    { "CA", "San Diego" },
    { "CA", "San Francisco" },
    { "CA", "San Jose" },
    { "CA", "San Luis Obispo" },
    { "CA", "San Mateo" },
    { "CA", "San Rafael" },
    { "CA", "Santa Ana" },
    { "CA", "Santa Barbara" },
    { "CA", "Santa Clara" },
    { "CA", "Santa Clarita" },
    { "CA", "Santa Cruz" },
    { "CA", "Santa Maria" },
    { "CA", "Santa Rosa" },
    { "CA", "Seaside" },
    { "CA", "Simi Valley" },
    { "CA", "Sonoma" },
    { "CA", "Sunnyvale" },
    { "CA", "South Lake Tahoe" },
    { "CA", "Spring Valley" },
    { "CA", "Stockton" },
    { "CA", "Temecula" },
    { "CA", "Thousand Oaks" },
    { "CA", "Vallejo" },
    { "CA", "Ventura" },
    { "CA", "Victorville" },
    { "CA", "Walnut Creek" },
    { "CA", "Watsonville" },
    { "CA", "West Covina" },
    { "CA", "West Hollywood" },
    { "CA", "West Sacramento" },
    { "CA", "Yuba City" },

    { "KS", "Kansas City" },
    { "MA", "Boston" },
    { "MO", "Kansas City" },
    { "NV", "Las Vegas" },
    { "NV", "Wendover" },
    { "NY", "New York" },
    { "PA", "Philidelphia" },
    { "UT", "Bountiful" },
    { "UT", "Cedar City" },
    { "UT", "Coalville" },
    { "UT", "Clearfield" },
    { "UT", "Draper" },
    { "UT", "Heber" },
    { "UT", "Kaysville" },
    { "UT", "Layton" },
    { "UT", "Ogden" },
    { "UT", "Orem" },
    { "UT", "North Salt Lake" },
    { "UT", "Park City" },
    { "UT", "Provo" },
    { "UT", "Roy" },
    { "UT", "South Ogden" },
    { "UT", "South Salt Lake" },
    { "UT", "St. George" },
    { "UT", "Tooele" },
    { "UT", "Wendover" },
    { "VA", "Chantilly" },
    { "VA", "Fairfax" },
    { "VA", "Haymarket" },
    { "VA", "Manassas" },
    { "WY", "Evanston" },
    { "WY", "Rock Springs" },
  };
static const int num_additional_cities = sizeof(additional_city_info) / sizeof(city_info_struct);


void create_additional_cities()
{
    int i;
    ObjRef state_ref;
    ObjRef city_ref;

    for (i = 0; i < num_additional_cities; i++)
    {
	/* if this is the first one or if the state is different from the previous, find the state reference */
	if (i == 0 || strcmp(additional_city_info[i].code, additional_city_info[i-1].code) != 0)
	{
	    assert(nwos_find_state_from_postal_code(additional_city_info[i].code, &state_ref));
	}
	nwos_create_us_city(additional_city_info[i].name, &state_ref, &city_ref);
    }
}


