--             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file COPYING.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: nwos_character.e,v $
-- Revision 1.17  2006/11/11 12:29:18  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.16  2006/02/19 20:03:05  jsedwards
-- Changed "to_binary_code" feature to "to_binary_codes" which returns the
-- array of characters instead of the binary_codes object.
--
-- Revision 1.15  2006/02/19 19:56:24  jsedwards
-- Added "to_binary_code" routine to locate the correct binary code object for
-- the given character set.
--
-- Revision 1.14  2006/02/19 19:07:06  jsedwards
-- Wrapped debug print statement when object read in, in "debug" so they only
-- print when debug is passed to compiler.
--
-- Revision 1.13  2006/02/19 15:55:18  jsedwards
-- Changed the name of NWOS_HEADER to NWOS_OBJECT.
--
-- Revision 1.12  2006/02/19 14:55:13  jsedwards
-- Removed loading of character set object, it is done automatically now.
--
-- Revision 1.11  2006/02/19 14:08:10  jsedwards
-- Changed attributes to routines that make sure the object is loaded before
-- returning the object.
--
-- Revision 1.10  2006/02/17 14:06:36  jsedwards
-- Added make_unresolved to creation list.
--
-- Revision 1.9  2006/02/13 04:27:02  jsedwards
-- Changed "character" attribute to "symbol" so it can point to any symbol
-- (letters, numbers, punctuation marks, etc.).  Also changed to non-deferred
-- class so it can exist for punctuation marks, numbers, etc.
--
-- Revision 1.8  2006/02/11 18:37:41  jsedwards
-- Changed the arrays from fixed_array to just array, so classes are more
-- consistent.  Filled in the read_body_from_stream routine.  Commented out
-- the class invariant because it fails when created with make_unresolved.
--
-- Revision 1.7  2006/02/09 12:54:56  jsedwards
-- Added reference parameter to initialize_header call.
--
-- Revision 1.6  2006/02/08 14:14:40  jsedwards
-- Add dummy read_body_from_stream routine.
--
-- Revision 1.5  2006/02/05 05:03:26  jsedwards
-- Changed "character" to be a letter instead of a reference.
--
-- Revision 1.4  2006/02/04 15:39:45  jsedwards
-- Moved reference back into predefined references.
--
-- Revision 1.3  2006/02/04 03:45:11  jsedwards
-- Updated to new format.
--
-- Revision 1.2  2006/02/02 02:09:57  jsedwards
-- Removed maximum character set calculation and next character set feature.
--
-- Revision 1.1  2006/01/29 18:17:06  jsedwards
-- Initial version.
--


class NWOS_CHARACTER

inherit NWOS_OBJECT

creation make, make_unresolved


feature

   symbol: NWOS_SYMBOL is  -- letters, numbers, punctuation marks, dollar signs, other symbols
      do
         load_if_not_already
         Result := symbol_obj
      ensure
         Result /= Void
      end

   character_set_list: ARRAY[NWOS_ASCII_CHARACTER] is  -- this should be a more general class
      do
         load_if_not_already
         Result := char_set_list_obj
      ensure
         Result /= Void
      end


feature

   to_binary_codes(char_set: NWOS_REFERENCE): ARRAY[CHARACTER] is
      local
         i: INTEGER
      do
         -- if we haven't done found it or if the character set changed
         if last_character_set = Void or else not last_character_set.is_equal(char_set) then
            !!last_character_set.copy(char_set)

            from
               i := character_set_list.lower
            until
               i > character_set_list.upper
                  or else
               character_set_list.item(i).class_definition.identifier.is_equal(char_set)
            loop
               i := i + 1
            end

            if i > character_set_list.upper then
               std_error.put_string("NWOS_CHARACTER.to_binary_code - character set not found: ")
               std_error.put_string(char_set.to_string)
               std_error.put_new_line
               die_with_code(exit_failure_code)
            end

            check
               character_set_list.item(i).character.identifier.is_equal(identifier) -- make sure this is the correct one
            end

            last_binary_codes := character_set_list.item(i).binary_code.codes
         end

         Result := last_binary_codes
      ensure
         Result /= Void
      end

feature

   add_character_set(char_set: NWOS_ASCII_CHARACTER) is
      do
         character_set_list.add_last(char_set)
      end


feature {NONE}

   write_body_to_stream(stream: BINARY_OUTPUT_STREAM) is
      local
         i: INTEGER
      do
         symbol_obj.class_definition.identifier.write_to_stream(stream)
         symbol_obj.identifier.write_to_stream(stream)

         from
            i := char_set_list_obj.lower
         variant
            char_set_list_obj.upper - i
         until
            i > char_set_list_obj.upper
         loop
            char_set_list_obj.item(i).class_definition.identifier.write_to_stream(stream)
            char_set_list_obj.item(i).identifier.write_to_stream(stream)
            i := i + 1
         end

         Void_reference.write_to_stream(stream)
      end

   read_body_from_stream(stream: BINARY_INPUT_STREAM) is
      local
         class_ref: NWOS_REFERENCE
         object_ref: NWOS_REFERENCE
         char_set: NWOS_ASCII_CHARACTER
      do
         !!class_ref.make_from_stream(stream)
         !!object_ref.make_from_stream(stream)

         symbol_obj ?= object_ref.get_object(class_ref)

         if symbol_obj = Void then
            std_error.put_string("NWOS_CHARACTER.read_body_from_stream: symbol failed to load%N")
            die_with_code(exit_failure_code)
         end

         !!char_set_list_obj.with_capacity(5, 1)

         from
            !!class_ref.make_from_stream(stream)
         until
            class_ref.is_void
         loop
            !!object_ref.make_from_stream(stream)
debug
std_output.put_string("NWOS_CHARACTER.read_body_from_stream")
std_output.put_string("%N  class: ")
std_output.put_string(class_ref.to_string)
std_output.put_string("%N  object: ")
std_output.put_string(object_ref.to_string)
std_output.put_new_line
end
            char_set ?= object_ref.get_object(class_ref)

            if char_set = Void then
               std_error.put_string("NWOS_CHARACTER.read_body_from_stream: char_set failed to load%N")
               die_with_code(exit_failure_code)
            end

            char_set_list_obj.add_last(char_set)

            !!class_ref.make_from_stream(stream)
         end
      end

feature {NONE}

   make(sym: like symbol) is
      do
         initialize_header(next_identifier)
         symbol_obj := sym
         !!char_set_list_obj.with_capacity(2, 1)  -- for now
      end

   next_identifier: NWOS_REFERENCE is
      do
         !!Result.copy(Characters_ref)
         Characters_ref.increment
      end


feature {NONE}  -- actual storage

   symbol_obj: like symbol

   char_set_list_obj: like character_set_list

   last_character_set: NWOS_REFERENCE

   last_binary_codes: like to_binary_codes


feature {NONE}  -- once again the kludgy class definition stuff

   class_revision: STRING is
      once
         !!Result.copy("$Revision: 1.17 $")
         Result.remove_prefix("$Revision: ")
         Result.remove_suffix(" $")
      end

   class_reference: NWOS_REFERENCE is
      do
         Result := Class_character_ref
      end


--invariant

--   character_set_list /= Void

end

