--             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file COPYING.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: nwos_ascii_character.e,v $
-- Revision 1.13  2006/02/19 15:55:18  jsedwards
-- Changed the name of NWOS_HEADER to NWOS_OBJECT.
--
-- Revision 1.12  2006/02/19 03:14:30  jsedwards
-- Changed attributes to routines so they can make sure the object is loaded
-- before returning values.
--
-- Revision 1.11  2006/02/11 18:35:28  jsedwards
-- Filled in read_body_from_string routine and commented out the class
-- invariant because it fails if created with make unresolved.
--
-- Revision 1.10  2006/02/10 13:55:34  jsedwards
-- Changed "load" in creation clause to "make_unresolved".
--
-- Revision 1.9  2006/02/09 13:02:23  jsedwards
-- Changed to the new method of doing the class defintion stuff, with the
-- "class reference" and "class revision".  Also added reference parameter
-- to initialize_header call.
--
-- Revision 1.8  2006/02/08 14:24:04  jsedwards
-- Added "load" to creation clause, a dummy read_body_from_stream routine,
-- and class reference to class definition "make" call.
--
-- Revision 1.7  2006/02/05 16:53:15  jsedwards
-- Removed {NONE} from "character" and "binary_code" features so other
-- objects could read them.
--
-- Revision 1.6  2006/02/04 15:39:45  jsedwards
-- Moved reference back into predefined references.
--
-- Revision 1.5  2006/02/04 14:56:09  jsedwards
-- Changed from the class_ref to the deferred class_definition.
--
-- Revision 1.4  2006/02/04 03:31:12  jsedwards
-- Added "class ref" definition back.
--
-- Revision 1.3  2006/02/04 02:40:43  jsedwards
-- Changed to deal with a real character object instead of just a reference.
--
-- Revision 1.2  2006/02/03 20:26:59  jsedwards
-- Changed so that the least significant byte of the reference is the ascii
-- code + 0x80.  It's easier for us humans to follow it that way, doesn't
-- matter to the computer.
--
-- Revision 1.1  2006/02/03 14:22:59  jsedwards
-- Initial version, saves a dummy reference as the character instead of the
-- real character object reference.
--


class NWOS_ASCII_CHARACTER

    -- this object is a mapping of a character (UPPER_CASE_CHARACTER for 
    -- example) to a binary code object.

inherit NWOS_OBJECT -- this class should inherit from character_set_character or something

creation make, make_unresolved

feature

   character: NWOS_CHARACTER is
      do
         load_if_not_already
         Result := character_obj
      ensure
         Result /= Void
      end

   binary_code: NWOS_BINARY_CODE is
      do
         load_if_not_already
         Result := binary_code_obj
      ensure
         Result /= Void
      end


feature {NONE}

   write_body_to_stream(stream: BINARY_OUTPUT_STREAM) is
      do
         character_obj.class_definition.identifier.write_to_stream(stream)
         character_obj.identifier.write_to_stream(stream)

         binary_code_obj.class_definition.identifier.write_to_stream(stream)
         binary_code_obj.identifier.write_to_stream(stream)
      end

   read_body_from_stream(stream: BINARY_INPUT_STREAM) is
      local
         class_ref: NWOS_REFERENCE
         object_ref: NWOS_REFERENCE
      do
         !!class_ref.make_from_stream(stream)
         !!object_ref.make_from_stream(stream)

         character_obj ?= object_ref.get_object(class_ref)

         if character_obj = Void then
            std_error.put_string("NWOS_ASCII_CHARACTER.read_body_from_stream: character failed to load%N")
            die_with_code(exit_failure_code)
         end

         !!class_ref.make_from_stream(stream)
         !!object_ref.make_from_stream(stream)

         binary_code_obj ?= object_ref.get_object(class_ref)

         if binary_code_obj = Void then
            std_error.put_string("NWOS_ASCII_CHARACTER.read_body_from_stream: binary_code failed to load%N")
            die_with_code(exit_failure_code)
         end
      end

   make(char: like character; code: like binary_code) is
      do
         initialize_header(next_identifier)

         character_obj := char

         binary_code_obj := code
         binary_code_obj.add_reference(Current)
      end


feature

  -- put the ascii character codes + 0x80 in the last byte of the reference

   upper_case_a_last_byte: CHARACTER is
      once
         Result := (('A').code + 128).to_character
      end

   upper_case_z_last_byte: CHARACTER is
      once
         Result := (('Z').code + 128).to_character
      end

   lower_case_a_last_byte: CHARACTER is
      once
         Result := (('a').code + 128).to_character
      end

   next_identifier: NWOS_REFERENCE is
      do
         -- if this is the first time we need to increment it to get to the upper case A reference
         from
         until
            Ascii_chars_ref.last >= upper_case_a_last_byte
         loop
            Ascii_chars_ref.increment
         end

         !!Result.copy(Ascii_chars_ref)

         -- increment it once unless we just passed Z, then increment it until we hit lower case a
         from
            Ascii_chars_ref.increment
         until
           Ascii_chars_ref.last <= upper_case_z_last_byte
              or else
           Ascii_chars_ref.last >= lower_case_a_last_byte
         loop
            Ascii_chars_ref.increment
         end
      ensure
         Ascii_chars_ref.last = Result.last.next
            or else
         Ascii_chars_ref.last = lower_case_a_last_byte
      end


feature {NONE}  -- actual storage

   character_obj: like character

   binary_code_obj: like binary_code


feature {NONE}  -- once again the kludgy class definition stuff

   class_revision: STRING is
      once
         !!Result.copy("$Revision: 1.13 $")
         Result.remove_prefix("$Revision: ")
         Result.remove_suffix(" $")
      end

   class_reference: NWOS_REFERENCE is
      do
         Result := Class_ascii_character_ref
      end


--invariant

--   character /= Void
--   binary_code /= Void

end

