/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: log_disc.c,v $
-- Revision 1.7  2006/11/19 16:40:22  jsedwards
-- Moved creation of disc list to separate program (disc_list) and made this
-- program just verify the disc list create the disc_copy object.
--
-- Revision 1.6  2006/11/18 14:34:58  jsedwards
-- Changed so that discs can be stored in one of three different binders.
--
-- Revision 1.5  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/11/07 14:03:35  jsedwards
-- Add assert to verify file_path didn't come back void.
--
-- Revision 1.3  2006/11/06 13:54:07  jsedwards
-- Changed wording in print statement because disc_list now stores the
-- references to the file_path object instead of the file object.
--
-- Revision 1.2  2006/11/05 21:32:18  jsedwards
-- Add code to create storage_location and disc_copy objects.
--
-- Revision 1.1  2006/11/04 18:56:22  jsedwards
-- Program to scan all the files on a CD or DVD and make a disk_list object
-- of it.
--
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"


extern void nwos_setup_file();


int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef ref;
    int i;
    time_t start_time;
    char id[13];
    char copy_num[3] = "";
    char binder_num[3] = "";
    char *p;
    bool ok = false;
    int binder;
    ObjRef files[MAX_FILES_PER_DISC_LIST];
    ObjRef location_ref[3];
    ObjRef disc_list_ref;
    bool missing_files = false;
    bool bad_checksum = false;
    CheckFileResult check_file_result;
    uint8 kludge[MAX_SIZE_DISC_LIST];
    C_struct_Disc_List* ptr_list_obj = (C_struct_Disc_List*)kludge;

    if (argc - 1 > MAX_FILES_PER_DISC_LIST)
    {
	fprintf(stderr, "Too many files specified, currently can only handle %u files max\n", (unsigned)MAX_FILES_PER_DISC_LIST);
	exit(1);
    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_setup_file();

    nwos_create_storage_location("Backups", &location_ref[0]);
    nwos_create_storage_location("ISOs", &location_ref[1]);
    nwos_create_storage_location("Music", &location_ref[2]);


    /* first verify that all the listed files are already in the system */

    memset(files, 0, sizeof(files));

    for (i = 1; i < argc; i++)
    {
	if (!nwos_find_file_path(argv[i], &files[i-1]))
	{
	    missing_files = true;
	}
    }

    if (missing_files)
    {
	printf("The following files are not in the system:\n\n");

	for (i = 1; i < argc; i++)
	{
	    if (is_void_reference(&files[i-1]))
	    {
		printf("   %s\n", argv[i]);
	    }
	}

	printf("\n");
    }


    /* find and verify the disc */

    if (!nwos_find_matching_disc_list(files, &disc_list_ref))
    {
	printf("No matching disc list found!\n");
	exit(1);
    }

    nwos_read_variable_sized_object_from_disk(&disc_list_ref, kludge, sizeof(kludge), &nwos_get_disc_list_object_size);

    memcpy(id, ptr_list_obj->id, sizeof(id) - 1);
    id[sizeof(id) - 1] = '\0';

    printf("disc list %02x%02x%02x%02x: %s\n",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3], id);

    for (i = 1; i < argc; i++)
    {
	start_time = time(NULL);

	check_file_result = nwos_check_file_md5sum(argv[i]);

	assert(check_file_result != File_Not_Found);   /* since we verified this above it should always find it */

	if (check_file_result != MD5_Sum_Match) 
	{
	    bad_checksum = true;
	}
    }

    if (bad_checksum)
    {
	printf("This version of code cannot create a log with a bad checksum.\n");
	exit(1);
    }


    /* create record of this disc */

    ok = false;
    while (!ok)
    {
	printf("Copy number: ");
	fflush(stdout);

	fgets(copy_num, sizeof(copy_num), stdin);

	p = strchr(copy_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(copy_num, sizeof(copy_num), stdin);
		p = strchr(copy_num, '\n');   /* find the newline char */
	    }
	    printf("copy number can only be 1 digit in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(copy_num) != 1)
	    {
		printf("copy number can only be 1 digit in this version\n");
	    }
	    else if (copy_num[0] < '1' || copy_num[0] > '9')
	    {
		printf("copy number must be a 1 digit number in this version\n");
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }

    ok = false;
    while (!ok)
    {
	printf("Binder (0=Backups, 1=ISOs, 2=Music): ");
	fflush(stdout);

	fgets(binder_num, sizeof(binder_num), stdin);

	p = strchr(binder_num, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(binder_num, sizeof(binder_num), stdin);
		p = strchr(binder_num, '\n');   /* find the newline char */
	    }
	    printf("copy number can only be 1 digit in this version\n");
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(binder_num) != 1)
	    {
		printf("storage binder can only be 1 digit\n");
	    }
	    else if (binder_num[0] < '0' || binder_num[0] > '2')
	    {
		printf("storage binder must be 0, 1, or 2\n");
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }


    binder = binder_num[0] - '0';

    assert(binder == 0 || binder == 1 || binder == 2);

    assert(nwos_create_disc_copy(&disc_list_ref, copy_num[0] - '0', &location_ref[binder], &ref) == CREATED_NEW);

    printf("disc copy: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);

    nwos_terminate_objectify();

    return 0;
}


