/*             This file is part of the New World OS project
--                   Copyright (C) 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: compress_sparse.c,v $
-- Revision 1.5  2006/11/11 12:01:01  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/11/06 13:52:54  jsedwards
-- Changed to skip over public blocks for now.
--
-- Revision 1.3  2006/11/02 11:49:28  jsedwards
-- Fixed all cases where 'z' was used as a format for 'off64_t' values because
-- the older compiler complains.
--
-- Revision 1.2  2006/10/26 01:51:23  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.8  2006/10/22 12:45:21  jsedwards
-- Change to use the number of blocks stored on disk instead of
-- BLOCKS_ON_DISK #define.
--
-- Revision 1.1.2.7  2006/10/22 12:40:47  jsedwards
-- Corrected version string error message.
--
-- Revision 1.1.2.6  2006/10/19 01:42:47  jsedwards
-- Fixed format specifiers for uint32, which is now an int instead of a long,
-- and off64_t.
--
-- Revision 1.1.2.5  2006/10/15 16:31:42  jsedwards
-- Changed to use block maps to find blocks to write instead of scanning
-- the entire drive.  Time reduced from 87 to 63 minutes.
--
-- Revision 1.1.2.4  2006/10/15 12:10:08  jsedwards
-- Change to read an entire chunk at a time and to skip over the block maps.
--
-- Revision 1.1.2.3  2006/09/19 14:11:57  jsedwards
-- Added printing of number of blocks.
--
-- Revision 1.1.2.2  2006/09/17 13:52:11  jsedwards
-- Fix argument count bug and open file read only instead of read-write.
--
-- Revision 1.1.2.1  2006/09/17 13:21:39  jsedwards
-- Program to compress the sparse objectify file into a non-sparse file.
--
*/


#define _LARGEFILE64_SOURCE

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>

#include "objectify_private.h"

static void print_usage(char *program)
{
    fprintf(stderr, "usage: %s output-file\n", program);
}


int main(int argc, char* argv[])
{
    int obj_file_desc;
    off64_t chunk;
    uint8 block_map[BIT_MAP_BYTES];
    uint8 block[FILE_BLOCK_SIZE];
    int i;
    size_t bytes_read;
    FILE* ofp;
    int num_blocks;
    uint32 public_blocks;
    uint32 blocks_on_disk;
    char msg[128];

    if (argc != 2)
    {
	print_usage(argv[0]);
	exit(1);
    }

    if (*argv[1] == '-')
    {
	fprintf(stderr, "error: this program doesn't have any options\n");
	print_usage(argv[0]);
	exit(1);
    }


    /* Open the storage drive and verify the header info */

    obj_file_desc = open(DEFAULT_FILE, O_RDONLY | O_LARGEFILE);

    if (obj_file_desc < 0)
    {
	perror(DEFAULT_FILE);
	exit(1);
    }

    bytes_read = read(obj_file_desc, block, sizeof(block));

    if (bytes_read != sizeof(block))
    {
	perror("reading first block");
	exit(1);
    }

    if (memcmp(&block[0], MAGIC_NUMBER, 4) != 0)
    {
	fprintf(stderr, "Missing magic number in disk header\n");
	exit(1);
    }

    if (memcmp(&block[4], VERSION_STRING, 4) != 0)
    {
	fprintf(stderr, "Incorrect version string in disk header\n");
	exit(1);
    }

    public_blocks = ((uint32)block[8] << 24) | ((uint32)block[9]<< 16) | ((uint32)block[10] << 8) | ((uint32)block[11]);

    blocks_on_disk = ((uint32)block[12] << 24) | ((uint32)block[13]<< 16) | ((uint32)block[14] << 8) | ((uint32)block[15]);
    ofp = fopen(argv[1], "w");

    if (ofp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    printf("header: %c%c%c%c %c%c%c%c\n",
	   block[0], block[1], block[2], block[3], 
	   block[4], block[5], block[6], block[7]);
    printf("total blocks on disk: %08u  public blocks: %08u\n",
	   blocks_on_disk, public_blocks);
    fflush(stdout);

    fwrite(block, 1, FILE_BLOCK_SIZE, ofp);   /* write the first 256 bytes always */

    num_blocks = 0;

    // for now we can skip over public blocks because they should always stay the same
    for (chunk = public_blocks; chunk < blocks_on_disk; chunk += BLOCKS_IN_CHUNK)
    {
	if (lseek64(obj_file_desc, (off64_t)chunk << 8, SEEK_SET) < 0)
	{
	    sprintf(msg, "lseek64 chunk:%08x", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	bytes_read = read(obj_file_desc, block_map, sizeof(block_map));

	if (bytes_read != sizeof(block_map))
	{
	    sprintf(msg, "reading block map: %u", (uint32)chunk);
	    perror(msg);
	    exit(1);
	}

	/* scan block map (skip over the blocks for the block map itself) */
	for (i = BIT_MAP_BLOCKS; i < BLOCKS_IN_CHUNK; i++)
	{
	    if ((block_map[i/8] & (0x80 >> (i%8))) != 0)
	    {
		if (lseek64(obj_file_desc, (chunk + i) << 8, SEEK_SET) < 0)
		{
		    sprintf(msg, "lseek64 block:%08x", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}

		bytes_read = read(obj_file_desc, block, sizeof(block));

		if (bytes_read != sizeof(block))
		{
		    sprintf(msg, "reading block: %u", (uint32)(chunk + i));
		    perror(msg);
		    exit(1);
		}
		
		printf("id: %02x%02x%02x%02x\n", block[4], block[5], block[6], block[7]);
		fflush(stdout);

		if (fwrite(block, 1, sizeof(block), ofp) != sizeof(block))
		{
		    perror(argv[1]);
		    close(obj_file_desc);
		    exit(1);
		}

		num_blocks++;
	    }
	}
    }

    printf("Number of blocks: %d\n", num_blocks);

    if (fclose(ofp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    close(obj_file_desc);

    return 0;
}

